<?php
/**
 * Verifizierungsstatus-Seite für User
 * Zeigt je nach Status unterschiedliche Inhalte:
 * - pending: Wartend - bitte Dokument hochladen
 * - approved: Erfolgreich verifiziert - Weiterleitung
 * - rejected: Abgelehnt - Grund anzeigen
 */

session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$user = getCurrentUser();

if (!$user) {
    header('Location: login.php');
    exit;
}

// Unternehmensdaten laden
$db = new Database();
$conn = $db->getConnection();

$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/BE-Logo-FINAL_Schwarz-1.png';
$appName = $companySettings['app_name'] ?? 'B&E Gruppe';

// Verifizierungsstatus
$verification_status = $user['verification_status'] ?? 'pending';

// Wenn approved, zur Startseite weiterleiten
if ($verification_status === 'approved') {
    header('Location: index.php');
    exit;
}

// Prüfe ob Dokument bereits hochgeladen wurde
$has_document = !empty($user['verification_document']);

// Status-spezifische Informationen
$status_config = [
    'pending' => [
        'icon' => 'fas fa-hourglass-half',
        'title' => 'Verifizierung läuft',
        'message' => 'Ihr Konto wird von einem Administrator geprüft.',
        'submessage' => 'Sie erhalten eine E-Mail, sobald Ihre Verifizierung abgeschlossen ist.',
        'bg_color' => '#fff3cd',
        'icon_color' => '#ffc107',
        'show_upload' => !$has_document, // Nur zeigen wenn kein Dokument vorhanden
        'upload_message' => 'Falls noch nicht geschehen, laden Sie bitte ein Identitätsdokument hoch.'
    ],
    'rejected' => [
        'icon' => 'fas fa-times-circle',
        'title' => 'Verifizierung abgelehnt',
        'message' => 'Ihr Antrag wurde leider nicht genehmigt.',
        'submessage' => 'Grund: ' . htmlspecialchars($user['verification_reason'] ?? 'Nicht angegeben'),
        'bg_color' => '#f8d7da',
        'icon_color' => '#dc3545',
        'show_upload' => true, // Immer erlauben wenn abgelehnt
        'upload_message' => 'Sie können ein neues Dokument hochladen und erneut einen Antrag stellen.'
    ]
];

$current_status = $status_config[$verification_status] ?? $status_config['pending'];
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($current_status['title']); ?> - <?php echo htmlspecialchars($appName); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            /* Volvo Shadow Guidelines - 5 Elevation Levels */
            --shadow-1: 0 1px 2px rgba(0, 0, 0, 0.1); /* Content switcher, Tabs */
            --shadow-2: 0 2px 4px rgba(0, 0, 0, 0.12); /* Buttons, Slider, Tooltip */
            --shadow-4: 0 4px 8px rgba(0, 0, 0, 0.15); /* Cards, Dropdown, Search fields, Snackbar */
            --shadow-8: 0 8px 16px rgba(0, 0, 0, 0.2); /* Dialog, Date picker, Time picker */
            --shadow-16: 0 16px 32px rgba(0, 0, 0, 0.25); /* Modal dialog */
            
            --primary-gradient: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            --card-shadow: var(--shadow-4);
        }
        
        /* User Profile Dropdown - Volvo Design System */
        .user-profile-dropdown {
            position: relative;
        }
        
        .user-profile-trigger {
            background: transparent;
            color: #1a1a1a;
            border: none;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.5rem 1rem;
            cursor: pointer;
            border-radius: 4px;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            font-family: inherit;
            font-size: 0.9rem;
            font-weight: 500;
        }
        
        .user-profile-trigger:hover,
        .user-profile-trigger:focus {
            background: #f8f9fa;
            color: #1a1a1a;
        }
        
        .user-profile-trigger:active {
            background: #e9ecef;
            color: #1a1a1a;
        }
        
        .user-profile-trigger:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .user-profile-trigger.active {
            background: #f8f9fa;
        }
        
        .user-dropdown-icon {
            font-size: 0.75rem;
            color: #6c757d;
            transition: transform 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .user-profile-trigger.active .user-dropdown-icon {
            transform: rotate(180deg);
        }
        
        .user-dropdown-menu {
            position: absolute;
            top: calc(100% + 0.5rem);
            right: 0;
            background: #ffffff;
            border-radius: 4px;
            box-shadow: var(--shadow-8);
            min-width: 200px;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 1000;
            border: 1px solid #e9ecef;
            overflow: hidden;
        }
        
        .user-dropdown-menu.show {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }
        
        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: #1a1a1a;
            text-decoration: none;
            font-size: 0.9rem;
            font-weight: 500;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            border: none;
            background: none;
            width: 100%;
            text-align: left;
        }
        
        .dropdown-item:hover {
            background: #f8f9fa;
            color: #1a1a1a;
        }
        
        .dropdown-item:active {
            background: #e9ecef;
        }
        
        .dropdown-item i {
            width: 18px;
            text-align: center;
            color: #6c757d;
        }
        
        .dropdown-item:hover i {
            color: #0056b3;
        }
        
        /* User Avatar in Navbar - Volvo Design System */
        .user-avatar-nav {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            overflow: hidden;
            background: linear-gradient(135deg, #0056b3 0%, #003057 100%);
            position: relative;
        }
        
        .user-avatar-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 50%;
        }
        
        .user-avatar-initials {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #ffffff;
            font-weight: 600;
            font-size: 0.875rem;
            letter-spacing: 0.01em;
            background: linear-gradient(135deg, #0056b3 0%, #003057 100%);
        }
        
        .user-name-nav {
            font-weight: 500;
            font-size: 0.9rem;
            white-space: nowrap;
        }
        
        @media (max-width: 768px) {
            .user-name-nav {
                display: none;
            }
            
            .user-profile-trigger {
                padding: 0.5rem;
                min-width: 44px;
                justify-content: center;
            }
            
            .user-dropdown-menu {
                right: 0;
                left: auto;
            }
        }
        
        .verification-container {
            max-width: 700px;
            margin: 4rem auto;
            padding: 2rem;
        }
        
        .verification-card {
            background: white;
            border-radius: 4px;
            padding: 2.5rem;
            box-shadow: var(--card-shadow);
            text-align: center;
            border: 1px solid #e9ecef;
        }
        
        .status-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: <?php echo $current_status['bg_color']; ?>;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1.5rem;
            font-size: 2.5rem;
            color: <?php echo $current_status['icon_color']; ?>;
            box-shadow: var(--shadow-2);
            border: 2px solid <?php echo $current_status['icon_color']; ?>;
        }
        
        .verification-title {
            font-size: 1.5rem;
            font-weight: 600;
            color: #1a1a1a;
            margin-bottom: 1rem;
        }
        
        .verification-message {
            font-size: 1rem;
            color: #495057;
            margin-bottom: 0.5rem;
            line-height: 1.6;
        }
        
        .verification-submessage {
            font-size: 0.9rem;
            color: #6c757d;
            margin-bottom: 2rem;
        }
        
        /* Upload Section - Volvo Design System */
        .upload-section {
            background: #ffffff;
            border-radius: 4px;
            padding: 0;
            margin-bottom: 2rem;
            border: none;
        }
        
        .upload-header {
            margin-bottom: 1.5rem;
        }
        
        .upload-title {
            font-size: 1rem;
            font-weight: 600;
            color: #1a1a1a;
            margin: 0 0 0.5rem 0;
        }
        
        .upload-instructions {
            color: #6c757d;
            font-size: 0.875rem;
            margin: 0;
        }
        
        .upload-area {
            background: #ffffff;
            border: 1px dashed #dee2e6;
            border-radius: 4px;
            padding: 3rem 2rem;
            margin-bottom: 1.5rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
        }
        
        .upload-area:hover {
            border-color: #adb5bd;
            background: #f8f9fa;
        }
        
        .upload-area.dragover {
            border-color: #0056b3;
            background: #f8f9fa;
            border-style: solid;
        }
        
        .upload-area-content {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 1rem;
        }
        
        .upload-area-icon {
            font-size: 2.5rem;
            color: #6c757d;
        }
        
        .upload-area-text {
            color: #495057;
            font-size: 0.9rem;
            margin: 0;
        }
        
        .file-input-wrapper {
            position: relative;
            display: flex;
            justify-content: center;
            margin-bottom: 1rem;
        }
        
        .file-input {
            display: none;
        }
        
        .file-input-label {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            background: #1a1a1a;
            color: white;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 500;
            font-size: 0.9rem;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            border: none;
            box-shadow: none;
        }
        
        .file-input-label:hover {
            background: #2d2d2d;
            color: white;
        }
        
        .file-input-label:active {
            background: #0d0d0d;
            color: white;
        }
        
        .file-input-label:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .file-input-label:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .upload-message {
            margin-top: 1rem;
            font-size: 0.875rem;
            color: #6c757d;
            text-align: center;
        }
        
        .uploaded-files-list {
            margin-top: 1.5rem;
        }
        
        .uploaded-file-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 4px;
            margin-bottom: 0.75rem;
        }
        
        .uploaded-file-item.success {
            background: #d4edda;
            border-color: #c3e6cb;
        }
        
        .uploaded-file-item.error {
            background: #f8d7da;
            border-color: #f5c6cb;
            border-left: 4px solid #dc3545;
        }
        
        .uploaded-file-item.loading {
            background: #f8f9fa;
            border-color: #e9ecef;
        }
        
        .file-status-icon {
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            font-size: 0.75rem;
        }
        
        .file-status-icon.success {
            background: #28a745;
            color: white;
        }
        
        .file-status-icon.error {
            background: #dc3545;
            color: white;
        }
        
        .file-status-icon.loading {
            background: #6c757d;
            color: white;
        }
        
        .file-info {
            flex: 1;
            min-width: 0;
        }
        
        .file-name {
            font-weight: 500;
            color: #1a1a1a;
            font-size: 0.9rem;
            margin-bottom: 0.25rem;
            word-break: break-word;
        }
        
        .file-size {
            color: #6c757d;
            font-size: 0.875rem;
        }
        
        .file-actions {
            display: flex;
            gap: 0.5rem;
            align-items: center;
        }
        
        .file-delete-btn {
            background: transparent;
            border: none;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 4px;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .file-delete-btn:hover {
            background: #e9ecef;
            color: #dc3545;
        }
        
        .file-error-message {
            margin-top: 0.5rem;
            color: #dc3545;
            font-size: 0.875rem;
        }
        
        /* Alerts - Volvo Design System */
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 4px;
            margin-bottom: 1.5rem;
            display: none;
            border: 1px solid;
        }
        
        .alert.show {
            display: block;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border-color: #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border-color: #f5c6cb;
        }
        
        .alert-info {
            background: #d1ecf1;
            color: #0c5460;
            border-color: #bee5eb;
        }
        
        .btn-group {
            display: flex;
            gap: 1rem;
            justify-content: center;
            margin-top: 2rem;
        }
        
        /* Buttons - Volvo Design System */
        .btn {
            padding: 0.625rem 1.25rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 500;
            font-size: 0.9rem;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: none;
        }
        
        .btn-primary {
            background: #0056b3;
            color: white;
        }
        
        .btn-primary:hover,
        .btn-primary:focus {
            background: #0066cc;
            color: white;
        }
        
        .btn-primary:active {
            background: #004c99;
            color: white;
        }
        
        .btn-primary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .btn-secondary {
            background: #ffffff;
            color: #1a1a1a;
            border: 1px solid #dee2e6;
        }
        
        .btn-secondary:hover,
        .btn-secondary:focus {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .btn-secondary:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .btn-secondary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
        }
        
        .info-box {
            background: #e7f3ff;
            border-left: 4px solid #0056b3;
            border-radius: 4px;
            padding: 1rem 1.25rem;
            margin-bottom: 2rem;
            text-align: left;
        }
        
        .info-box i {
            color: #0056b3;
            margin-right: 0.5rem;
        }
        
        .document-status-card {
            background: #ffffff;
            border-radius: 4px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            border: 1px solid #e9ecef;
            box-shadow: var(--shadow-2);
        }
        
        .document-status-header {
            display: flex;
            align-items: center;
            gap: 1.25rem;
            margin-bottom: 1.5rem;
            padding-bottom: 1.25rem;
            border-bottom: 1px solid #e8e8e8;
        }
        
        .document-status-icon {
            width: 48px;
            height: 48px;
            background: #f8f9fa;
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #28a745;
            font-size: 1.25rem;
            border: 1px solid #e9ecef;
        }
        
        .document-status-info h3 {
            margin: 0 0 0.25rem 0;
            color: #1a1a1a;
            font-size: 1.15rem;
            font-weight: 600;
        }
        
        .document-status-date {
            color: #6c757d;
            font-size: 0.875rem;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .status-timeline {
            margin: 1.5rem 0;
        }
        
        .timeline-step {
            display: flex;
            align-items: flex-start;
            gap: 1.5rem;
            margin-bottom: 1.5rem;
            position: relative;
        }
        
        .timeline-step:not(:last-child)::after {
            content: '';
            position: absolute;
            left: 27px;
            top: 48px;
            width: 1px;
            height: calc(100% + 0.5rem);
            background: #e0e0e0;
        }
        
        .timeline-icon {
            width: 36px;
            height: 36px;
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            z-index: 1;
            position: relative;
            font-size: 0.875rem;
        }
        
        .timeline-step.completed .timeline-icon {
            background: #d4edda;
            color: #28a745;
            border: 1px solid #c3e6cb;
        }
        
        .timeline-step.active .timeline-icon {
            background: #fff3cd;
            color: #ffc107;
            border: 1px solid #ffeaa7;
            animation: pulse-subtle 2s infinite;
        }
        
        .timeline-step.pending .timeline-icon {
            background: #f8f9fa;
            color: #6c757d;
            border: 1px solid #e9ecef;
        }
        
        @keyframes pulse-subtle {
            0%, 100% {
                opacity: 1;
            }
            50% {
                opacity: 0.8;
            }
        }
        
        .timeline-content {
            flex: 1;
            padding-top: 0.5rem;
        }
        
        .timeline-content strong {
            display: block;
            color: #1a1a1a;
            font-size: 0.95rem;
            margin-bottom: 0.25rem;
            font-weight: 600;
        }
        
        .timeline-step.pending .timeline-content strong {
            color: #9e9e9e;
        }
        
        .timeline-content span {
            display: block;
            color: #6c757d;
            font-size: 0.875rem;
        }
        
        .timeline-step.pending .timeline-content span {
            color: #9e9e9e;
        }
        
        .document-status-message {
            background: #f8f9fa;
            border-radius: 4px;
            padding: 1rem 1.25rem;
            margin-top: 1.5rem;
            border-left: 4px solid #0056b3;
        }
        
        .document-status-message i {
            color: #0056b3;
            margin-right: 0.75rem;
            font-size: 0.95rem;
        }
        
        .document-status-message p {
            margin: 0;
            color: #495057;
            line-height: 1.6;
            font-size: 0.95rem;
        }
        
        .document-status-message strong {
            color: #1a1a1a;
            font-weight: 600;
        }
        
        /* Benachrichtigungs-Badge im Modal */
        .notification-badge {
            position: absolute;
            top: -8px;
            right: -8px;
            background: linear-gradient(135deg, #ff6b6b 0%, #ee5a6f 100%);
            color: white;
            border-radius: 50%;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.75rem;
            font-weight: 700;
            box-shadow: 0 2px 8px rgba(255, 107, 107, 0.4);
            animation: pulse-badge 2s infinite;
        }
        
        @keyframes pulse-badge {
            0%, 100% {
                transform: scale(1);
                box-shadow: 0 2px 8px rgba(255, 107, 107, 0.4);
            }
            50% {
                transform: scale(1.1);
                box-shadow: 0 4px 12px rgba(255, 107, 107, 0.6);
            }
        }
        
        .document-status-card {
            position: relative;
        }
        
        .notification-reminder {
            background: #fff3cd;
            border: 1px solid #ffc107;
            border-radius: 4px;
            padding: 1rem 1.25rem;
            margin-top: 1.25rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            box-shadow: var(--shadow-1);
        }
        
        .notification-reminder-icon {
            color: #f57c00;
            font-size: 1.25rem;
            flex-shrink: 0;
        }
        
        .notification-reminder-text {
            flex: 1;
            color: #856404;
            font-size: 0.9rem;
            line-height: 1.5;
        }
        
        .notification-reminder-text strong {
            color: #f57c00;
            font-weight: 600;
        }
        
        .push-notification-box {
            background: #e3f2fd;
            border-radius: 4px;
            padding: 1.5rem;
            margin: 2rem 0;
            border: 1px solid #bbdefb;
            box-shadow: var(--shadow-2);
        }
        
        .push-notification-header {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 1rem;
        }
        
        .push-notification-header i {
            font-size: 1.5rem;
            color: #0056b3;
        }
        
        .push-notification-header h3 {
            margin: 0;
            color: #003057;
            font-size: 1.1rem;
            font-weight: 600;
        }
        
        .push-notification-text {
            color: #424242;
            margin-bottom: 1.5rem;
            line-height: 1.6;
        }
        
        .push-status-message {
            margin-top: 1rem;
            padding: 0.75rem 1rem;
            border-radius: 4px;
            display: none;
            font-size: 0.9rem;
            border: 1px solid;
        }
        
        .push-status-message.show {
            display: block;
        }
        
        .push-status-message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .push-status-message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .push-status-message.info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        #enable-push-btn {
            width: 100%;
            justify-content: center;
        }
        
        #enable-push-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .upload-disabled-notice {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 4px;
            padding: 1.25rem;
            margin: 2rem 0;
            display: flex;
            align-items: flex-start;
            gap: 1rem;
        }
        
        .upload-disabled-icon {
            width: 40px;
            height: 40px;
            background: #e9ecef;
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #6c757d;
            font-size: 1.1rem;
            flex-shrink: 0;
        }
        
        .upload-disabled-content {
            flex: 1;
        }
        
        .upload-disabled-content strong {
            display: block;
            color: #1a1a1a;
            font-size: 1rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .upload-disabled-content p {
            margin: 0;
            color: #6c757d;
            font-size: 0.9rem;
            line-height: 1.5;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar" style="position: fixed; top: 0; left: 0; right: 0; z-index: 1000; background: white; backdrop-filter: blur(20px); box-shadow: 0 2px 12px rgba(0,0,0,0.08);">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo" style="height: 50px;">
                    <div class="brand-text">
                        <h1 style="font-size: 1.5rem; color: #003057;"><?php echo htmlspecialchars($appName); ?></h1>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <?php 
                    $user_profile_image = getUserProfileImage($user);
                    $user_initials = getUserInitials($user);
                    $user_display_name = getUserDisplayName($user);
                    
                    // Prüfe ob der eingeloggte User ein Admin ist
                    $is_admin = false;
                    if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] && isset($_SESSION['user_id'])) {
                        // Lade User-Daten
                        $stmt = $conn->prepare("SELECT id, email, admin_user_id FROM users WHERE id = ? AND is_active = 1");
                        $stmt->execute([$_SESSION['user_id']]);
                        $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($user_data) {
                            // Prüfe ob User direkt mit Admin verknüpft ist
                            if (!empty($user_data['admin_user_id'])) {
                                // Prüfe ob verknüpfter Admin aktiv ist
                                $stmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ? AND is_active = 1");
                                $stmt->execute([$user_data['admin_user_id']]);
                                $admin_user = $stmt->fetch();
                                $is_admin = (bool)$admin_user;
                            } else {
                                // Fallback: Suche nach E-Mail in admin_users
                                $user_email = $user_data['email'] ?? '';
                                if (!empty($user_email)) {
                                    $stmt = $conn->prepare("SELECT id FROM admin_users WHERE email = ? AND is_active = 1");
                                    $stmt->execute([$user_email]);
                                    $admin_user = $stmt->fetch();
                                    $is_admin = (bool)$admin_user;
                                }
                            }
                        }
                    }
                    // Oder prüfe ob der User bereits als Admin eingeloggt ist
                    if (!$is_admin && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
                        $is_admin = true;
                    }
                ?>
                <div class="user-profile-dropdown">
                    <button type="button" class="user-profile-trigger" onclick="toggleUserDropdown(event)">
                        <div class="user-avatar-nav">
                            <?php if ($user_profile_image): ?>
                                <img src="<?php echo htmlspecialchars($user_profile_image); ?>" 
                                     alt="<?php echo htmlspecialchars($user_display_name); ?>" 
                                     class="user-avatar-img"
                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="user-avatar-initials" style="display: none;">
                                    <?php echo htmlspecialchars($user_initials); ?>
                                </div>
                            <?php else: ?>
                                <div class="user-avatar-initials">
                                    <?php echo htmlspecialchars($user_initials); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <span class="user-name-nav"><?php echo htmlspecialchars($user_display_name); ?></span>
                        <i class="fas fa-chevron-down user-dropdown-icon"></i>
                    </button>
                    <div class="user-dropdown-menu" id="userDropdownMenu">
                        <a href="profile.php" class="dropdown-item">
                            <i class="fas fa-user"></i>
                            <span>Profil</span>
                        </a>
                        <?php if ($is_admin): ?>
                        <a href="admin/" class="dropdown-item">
                            <i class="fas fa-cog"></i>
                            <span>Admin</span>
                        </a>
                        <?php endif; ?>
                        <a href="api/logout.php" class="dropdown-item">
                            <i class="fas fa-sign-out-alt"></i>
                            <span>Abmelden</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div style="margin-top: 80px;"></div>

    <!-- Verification Container -->
    <div class="verification-container">
        <div class="verification-card">
            <div class="status-icon">
                <i class="<?php echo htmlspecialchars($current_status['icon']); ?>"></i>
            </div>
            
            <h1 class="verification-title"><?php echo htmlspecialchars($current_status['title']); ?></h1>
            
            <p class="verification-message">
                <?php echo htmlspecialchars($current_status['message']); ?>
            </p>
            
            <p class="verification-submessage">
                <?php echo htmlspecialchars($current_status['submessage']); ?>
            </p>
            
            <?php if ($user['verification_status'] === 'pending' && !empty($user['verification_document'])): ?>
            <div class="document-status-card">
                <div class="document-status-header">
                    <div class="document-status-icon">
                        <i class="fas fa-file-check"></i>
                    </div>
                    <div class="document-status-info">
                        <h3>Dokument erfolgreich hochgeladen</h3>
                        <p class="document-status-date">
                            <i class="fas fa-calendar-alt"></i>
                            Hochgeladen am: <?php 
                                $uploadDate = $user['updated_at'] ?? date('Y-m-d H:i:s');
                                echo date('d.m.Y \u\m H:i', strtotime($uploadDate));
                            ?>
                        </p>
                    </div>
                </div>
                <div class="document-status-content">
                    <div class="status-timeline">
                        <div class="timeline-step completed">
                            <div class="timeline-icon">
                                <i class="fas fa-check"></i>
                            </div>
                            <div class="timeline-content">
                                <strong>Dokument hochgeladen</strong>
                                <span>Ihr Verifikationsdokument wurde erfolgreich übermittelt</span>
                            </div>
                        </div>
                        <div class="timeline-step active">
                            <div class="timeline-icon">
                                <i class="fas fa-hourglass-half"></i>
                            </div>
                            <div class="timeline-content">
                                <strong>In Bearbeitung</strong>
                                <span>Wir prüfen Ihr Dokument so schnell wie möglich</span>
                            </div>
                        </div>
                        <div class="timeline-step pending">
                            <div class="timeline-icon">
                                <i class="fas fa-clock"></i>
                            </div>
                            <div class="timeline-content">
                                <strong>Verifizierung abgeschlossen</strong>
                                <span>Sie erhalten eine Benachrichtigung per E-Mail und Push</span>
                            </div>
                        </div>
                    </div>
                    <div class="document-status-message">
                        <i class="fas fa-info-circle"></i>
                        <p><strong>Was passiert jetzt?</strong><br>
                        Ihr Dokument wurde an unser Team übermittelt und wird nun geprüft. Sie erhalten eine Benachrichtigung, sobald die Verifizierung abgeschlossen ist. Dies kann in der Regel 1-2 Werktage dauern.</p>
                    </div>
                    <div class="notification-reminder">
                        <i class="fas fa-bell notification-reminder-icon"></i>
                        <div class="notification-reminder-text">
                            <strong>Wichtig:</strong> Sie erhalten eine Push-Benachrichtigung, sobald Ihre Verifizierung abgeschlossen ist. Bitte aktivieren Sie Push-Benachrichtigungen, um sofort informiert zu werden.
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Alerts -->
            <div id="alert-message" class="alert"></div>
            
            <!-- Upload Section - nur zeigen wenn erlaubt -->
            <?php if ($current_status['show_upload']): ?>
            <div class="upload-section" id="upload-section">
                <div class="upload-header">
                    <h3 class="upload-title">Dateien hochladen</h3>
                    <p class="upload-instructions">Png, .jpg oder .svg Dateien mit max. 1 GB pro Datei.</p>
                </div>
                <div class="upload-area" id="upload-area">
                    <div class="upload-area-content">
                        <i class="fas fa-cloud-upload-alt upload-area-icon"></i>
                        <p class="upload-area-text">Dateien hier ablegen oder klicken zum Durchsuchen</p>
                    </div>
                </div>
                <div class="file-input-wrapper">
                    <input type="file" id="verification_document" name="verification_document" 
                           class="file-input" 
                           accept="image/jpeg,image/png,image/jpg,application/pdf">
                    <label for="verification_document" class="file-input-label">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <span>Dateien hinzufügen</span>
                    </label>
                </div>
                <div class="upload-message">
                    <?php echo htmlspecialchars($current_status['upload_message']); ?>
                </div>
                <div id="uploaded-files-list" class="uploaded-files-list"></div>
            </div>
            <?php elseif ($verification_status === 'pending' && $has_document): ?>
            <div class="upload-disabled-notice">
                <div class="upload-disabled-icon">
                    <i class="fas fa-lock"></i>
                </div>
                <div class="upload-disabled-content">
                    <strong>Dokument bereits eingereicht</strong>
                    <p>Ihr Verifikationsdokument wurde bereits hochgeladen und wird derzeit geprüft. Sie können erst ein neues Dokument hochladen, wenn die Prüfung abgeschlossen ist.</p>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Push-Benachrichtigungen Aktivierung -->
            <div class="push-notification-box" id="push-notification-box">
                <div class="push-notification-header">
                    <i class="fas fa-bell"></i>
                    <h3>Push-Benachrichtigungen aktivieren</h3>
                </div>
                <p class="push-notification-text">
                    Erhalten Sie sofortige Benachrichtigungen über den Status Ihrer Verifizierung direkt auf Ihrem Gerät.
                </p>
                <button id="enable-push-btn" class="btn btn-primary" onclick="enablePushNotifications()">
                    <i class="fas fa-bell-slash"></i>
                    <span id="push-btn-text">Benachrichtigungen aktivieren</span>
                </button>
                <div id="push-status-message" class="push-status-message"></div>
            </div>
            
            <div class="btn-group">
                <a href="index.php" class="btn btn-primary">
                    <i class="fas fa-home"></i>
                    <span>Zur Startseite</span>
                </a>
                <a href="profile.php" class="btn btn-secondary">
                    <i class="fas fa-user"></i>
                    <span>Zum Profil</span>
                </a>
            </div>
        </div>
    </div>
    
    <script>
        // User Dropdown Toggle Funktion
        function toggleUserDropdown(event) {
            event.stopPropagation();
            const menu = document.getElementById('userDropdownMenu');
            const trigger = event.currentTarget;
            
            if (menu && trigger) {
                const isOpen = menu.classList.contains('show');
                
                // Schließe alle anderen Dropdowns
                document.querySelectorAll('.user-dropdown-menu').forEach(m => {
                    m.classList.remove('show');
                });
                document.querySelectorAll('.user-profile-trigger').forEach(t => {
                    t.classList.remove('active');
                });
                
                // Toggle aktuelles Dropdown
                if (!isOpen) {
                    menu.classList.add('show');
                    trigger.classList.add('active');
                } else {
                    menu.classList.remove('show');
                    trigger.classList.remove('active');
                }
            }
        }
        
        // Schließe Dropdown beim Klick außerhalb
        document.addEventListener('click', function(event) {
            if (!event.target.closest('.user-profile-dropdown')) {
                document.querySelectorAll('.user-dropdown-menu').forEach(menu => {
                    menu.classList.remove('show');
                });
                document.querySelectorAll('.user-profile-trigger').forEach(trigger => {
                    trigger.classList.remove('active');
                });
            }
        });
        
        const fileInput = document.getElementById('verification_document');
        const uploadSection = document.getElementById('upload-section');
        const alertMessage = document.getElementById('alert-message');
        
        <?php if ($current_status['show_upload']): ?>
        const uploadArea = document.getElementById('upload-area');
        const uploadedFilesList = document.getElementById('uploaded-files-list');
        
        // Drag & Drop für Upload-Area
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, preventDefaults, false);
            document.body.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadArea.addEventListener(eventName, () => {
                uploadArea.classList.add('dragover');
            }, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, () => {
                uploadArea.classList.remove('dragover');
            }, false);
        });
        
        uploadArea.addEventListener('drop', (e) => {
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                fileInput.files = files;
                handleFileUpload(files[0]);
            }
        }, false);
        
        // Klick auf Upload-Area öffnet File-Dialog
        uploadArea.addEventListener('click', () => {
            fileInput.click();
        });
        
        // File Input Handler
        fileInput.addEventListener('change', function(e) {
            if (this.files.length > 0) {
                handleFileUpload(this.files[0]);
            }
        });
        
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
        }
        
        function addFileToList(file, status, errorMessage = null) {
            const fileItem = document.createElement('div');
            fileItem.className = `uploaded-file-item ${status}`;
            
            let iconHtml = '';
            let statusText = '';
            
            if (status === 'success') {
                iconHtml = '<i class="fas fa-check"></i>';
                statusText = '';
            } else if (status === 'error') {
                iconHtml = '<i class="fas fa-exclamation-triangle"></i>';
                statusText = errorMessage ? `<div class="file-error-message">${errorMessage}</div>` : '';
            } else if (status === 'loading') {
                iconHtml = '<i class="fas fa-spinner fa-spin"></i>';
                statusText = '<span style="color: #6c757d; font-size: 0.875rem;">Wird hochgeladen...</span>';
            }
            
            fileItem.innerHTML = `
                <div class="file-status-icon ${status}">
                    ${iconHtml}
                </div>
                <div class="file-info">
                    <div class="file-name">${file.name}</div>
                    <div class="file-size">(${formatFileSize(file.size)})</div>
                    ${statusText}
                </div>
                <div class="file-actions">
                    ${status === 'loading' ? '<button type="button" class="file-delete-btn" onclick="this.closest(\'.uploaded-file-item\').remove()"><i class="fas fa-times"></i></button>' : ''}
                    ${status !== 'loading' ? '<button type="button" class="file-delete-btn" onclick="this.closest(\'.uploaded-file-item\').remove()"><i class="fas fa-trash"></i></button>' : ''}
                </div>
            `;
            
            uploadedFilesList.appendChild(fileItem);
        }
        
        async function handleFileUpload(file) {
            // Validiere Dateigröße (max 1 GB = 1024 MB)
            const maxSize = 1024 * 1024 * 1024; // 1 GB
            if (file.size > maxSize) {
                addFileToList(file, 'error', 'Datei zu groß. Maximum 1 GB erlaubt.');
                showAlert('Datei zu groß. Maximum 1 GB erlaubt.', 'error');
                return;
            }
            
            // Validiere Dateityp
            const allowedTypes = ['image/jpeg', 'image/png', 'image/jpg', 'application/pdf'];
            if (!allowedTypes.includes(file.type)) {
                addFileToList(file, 'error', 'Ungültiger Dateityp. Nur JPG, PNG und PDF erlaubt.');
                showAlert('Ungültiger Dateityp. Nur JPG, PNG und PDF erlaubt.', 'error');
                return;
            }
            
            // Zeige Datei in Liste mit Loading-Status
            addFileToList(file, 'loading');
            showAlert('Datei wird hochgeladen...', 'info');
            
            const formData = new FormData();
            formData.append('verification_document', file);
            
            try {
                const response = await fetch('api/upload_verification_document.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                // Entferne Loading-Item
                const loadingItem = uploadedFilesList.querySelector('.uploaded-file-item.loading');
                if (loadingItem) {
                    loadingItem.remove();
                }
                
                if (data.success) {
                    addFileToList(file, 'success');
                    showAlert('Dokument erfolgreich hochgeladen! Ihr Konto wird nun geprüft.', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    addFileToList(file, 'error', data.message || 'Fehler beim Hochladen');
                    showAlert(data.message || 'Fehler beim Hochladen', 'error');
                }
            } catch (error) {
                // Entferne Loading-Item
                const loadingItem = uploadedFilesList.querySelector('.uploaded-file-item.loading');
                if (loadingItem) {
                    loadingItem.remove();
                }
                
                addFileToList(file, 'error', 'Verbindungsfehler');
                showAlert('Verbindungsfehler', 'error');
            }
        }
        <?php endif; ?>
        
        function showAlert(message, type) {
            if (alertMessage) {
                alertMessage.textContent = message;
                alertMessage.className = `alert alert-${type} show`;
                
                // Auto-hide nach 5 Sekunden (außer bei Errors)
                if (type !== 'error') {
                    setTimeout(() => {
                        alertMessage.classList.remove('show');
                    }, 5000);
                }
            }
        }
        
        // Push-Benachrichtigungen
        let pushNotificationScriptLoaded = false;
        
        async function loadPushNotificationScript() {
            if (pushNotificationScriptLoaded) return;
            
            return new Promise((resolve, reject) => {
                const script = document.createElement('script');
                script.src = 'assets/js/push-notifications.js';
                script.onload = () => {
                    pushNotificationScriptLoaded = true;
                    resolve();
                };
                script.onerror = reject;
                document.head.appendChild(script);
            });
        }
        
        async function enablePushNotifications() {
            const btn = document.getElementById('enable-push-btn');
            const btnText = document.getElementById('push-btn-text');
            const statusMessage = document.getElementById('push-status-message');
            
            btn.disabled = true;
            btnText.textContent = 'Wird aktiviert...';
            statusMessage.className = 'push-status-message info show';
            statusMessage.textContent = 'Push-Benachrichtigungen werden aktiviert...';
            
            try {
                // Lade Push-Notification-Script falls noch nicht geladen
                await loadPushNotificationScript();
                
                // Prüfe ob Service Worker unterstützt wird
                if (!('serviceWorker' in navigator)) {
                    throw new Error('Service Worker wird von diesem Browser nicht unterstützt');
                }
                
                // Registriere Service Worker falls noch nicht registriert
                if (!navigator.serviceWorker.controller) {
                    const registration = await navigator.serviceWorker.register('/service-worker.js');
                    console.log('[Service Worker] Registriert:', registration.scope);
                }
                
                // Warte bis Service Worker bereit ist
                const registration = await navigator.serviceWorker.ready;
                
                // Verwende die Funktionen aus push-notifications.js
                if (typeof subscribeToPushNotifications === 'function') {
                    await subscribeToPushNotifications();
                    
                    statusMessage.className = 'push-status-message success show';
                    statusMessage.textContent = '✅ Push-Benachrichtigungen erfolgreich aktiviert! Sie erhalten nun Benachrichtigungen über den Status Ihrer Verifizierung.';
                    btnText.innerHTML = '<i class="fas fa-bell"></i> Benachrichtigungen aktiviert';
                    btn.style.background = '#4caf50';
                } else {
                    throw new Error('Push-Notification-Funktionen nicht verfügbar');
                }
            } catch (error) {
                console.error('[Push] Fehler:', error);
                statusMessage.className = 'push-status-message error show';
                
                if (error.message.includes('not allowed') || error.message.includes('permission')) {
                    statusMessage.textContent = '❌ Benachrichtigungen wurden nicht erlaubt. Bitte erlauben Sie Benachrichtigungen in Ihren Browser-Einstellungen.';
                } else if (error.message.includes('not supported')) {
                    statusMessage.textContent = '❌ Push-Benachrichtigungen werden von diesem Browser nicht unterstützt.';
                } else {
                    statusMessage.textContent = '❌ Fehler beim Aktivieren: ' + error.message;
                }
                
                btnText.textContent = 'Erneut versuchen';
                btn.disabled = false;
            }
        }
        
        // Prüfe Status beim Laden der Seite
        window.addEventListener('load', async () => {
            try {
                await loadPushNotificationScript();
                
                if (typeof checkSubscriptionStatus === 'function') {
                    const isSubscribed = await checkSubscriptionStatus();
                    const btn = document.getElementById('enable-push-btn');
                    const btnText = document.getElementById('push-btn-text');
                    const statusMessage = document.getElementById('push-status-message');
                    
                    if (isSubscribed) {
                        btnText.innerHTML = '<i class="fas fa-bell"></i> Benachrichtigungen aktiviert';
                        btn.style.background = '#4caf50';
                        btn.disabled = true;
                        statusMessage.className = 'push-status-message success show';
                        statusMessage.textContent = '✅ Push-Benachrichtigungen sind bereits aktiviert.';
                    }
                }
            } catch (error) {
                console.error('[Push] Fehler beim Prüfen des Status:', error);
            }
        });
    </script>
    
    <!-- Service Worker Registration -->
    <script>
        if ('serviceWorker' in navigator) {
            window.addEventListener('load', () => {
                navigator.serviceWorker.register('/service-worker.js')
                    .then((registration) => {
                        console.log('[Service Worker] Registriert:', registration.scope);
                    })
                    .catch((error) => {
                        console.error('[Service Worker] Registrierung fehlgeschlagen:', error);
                    });
            });
        }
    </script>
</body>
</html>
