<?php
/**
 * Test-Script: Prüft wie viele Radsätze über die RESY API abrufbar sind
 */

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/api/resy_rest_api_complete.php';

header('Content-Type: text/html; charset=utf-8');

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>RESY API - Radsätze Test</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #333; }
        .section { margin: 20px 0; padding: 15px; background: #f9f9f9; border-radius: 4px; }
        .success { color: #28a745; font-weight: bold; }
        .error { color: #dc3545; font-weight: bold; }
        .info { color: #17a2b8; }
        table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        th, td { padding: 8px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #4CAF50; color: white; }
        tr:hover { background-color: #f5f5f5; }
        .count { font-size: 24px; font-weight: bold; color: #4CAF50; }
    </style>
</head>
<body>
<div class='container'>
    <h1>🔍 RESY API - Radsätze Anzahl Test</h1>";

try {
    // Lade alle Autohäuser aus der Datenbank
    $db = new Database();
    $pdo = $db->getConnection();
    
    $stmt = $pdo->query("SELECT id, name, resy_username, resy_dealer_id, resy_customer_no FROM autohaus ORDER BY id");
    $autohaeuser = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($autohaeuser)) {
        echo "<div class='error'>Keine Autohäuser in der Datenbank gefunden!</div>";
        exit;
    }
    
    echo "<div class='section'>";
    echo "<h2>📊 Ergebnisse für alle Autohäuser</h2>";
    
    $totalWheelsets = 0;
    $totalPickupOrders = 0;
    $totalInventory = 0;
    $totalReturnOrders = 0;
    
    foreach ($autohaeuser as $autohaus) {
        $autohausId = $autohaus['id'];
        $autohausName = $autohaus['name'];
        
        echo "<div class='section'>";
        echo "<h3>🏢 Autohaus: {$autohausName} (ID: {$autohausId})</h3>";
        
        try {
            $api = new ResyRestAPIComplete($autohausId);
            
            // Test 1: getAllWheelsetsSmart()
            echo "<h4>1. getAllWheelsetsSmart() - Alle Radsätze</h4>";
            $startTime = microtime(true);
            $wheelsets = $api->getAllWheelsetsSmart();
            $duration = round((microtime(true) - $startTime) * 1000, 2);
            
            $wheelsetCount = is_array($wheelsets) ? count($wheelsets) : 0;
            $totalWheelsets += $wheelsetCount;
            
            echo "<div class='count'>Anzahl Radsätze: {$wheelsetCount}</div>";
            echo "<div class='info'>Ladezeit: {$duration} ms</div>";
            
            // Test 2: Pickup Orders
            echo "<h4>2. Pickup Orders</h4>";
            try {
                $pickupOrders = $api->getPickupOrders();
                $pickupCount = is_array($pickupOrders) ? count($pickupOrders) : 0;
                $totalPickupOrders += $pickupCount;
                echo "<div class='info'>Anzahl Pickup Orders: {$pickupCount}</div>";
                
                // Zähle eindeutige Radsätze in Pickup Orders
                $uniqueWheelsets = [];
                foreach ($pickupOrders as $order) {
                    if (isset($order['wheelSetNumber'])) {
                        $uniqueWheelsets[$order['wheelSetNumber']] = true;
                    }
                }
                echo "<div class='info'>Eindeutige Radsätze in Pickup Orders: " . count($uniqueWheelsets) . "</div>";
            } catch (Exception $e) {
                echo "<div class='error'>Fehler: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
            
            // Test 3: Inventory
            echo "<h4>3. Inventory</h4>";
            try {
                $inventory = $api->getInventory();
                $inventoryCount = is_array($inventory) ? count($inventory) : 0;
                $totalInventory += $inventoryCount;
                echo "<div class='info'>Anzahl Inventory Items: {$inventoryCount}</div>";
                
                // Zähle alle Radsätze in Inventory
                $inventoryWheelsets = 0;
                foreach ($inventory as $item) {
                    if (isset($item['inventoryNumbers']) && is_array($item['inventoryNumbers'])) {
                        $inventoryWheelsets += count($item['inventoryNumbers']);
                    }
                }
                echo "<div class='info'>Gesamt Radsätze in Inventory: {$inventoryWheelsets}</div>";
            } catch (Exception $e) {
                echo "<div class='error'>Fehler: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
            
            // Test 4: Return Orders
            echo "<h4>4. Return Orders</h4>";
            try {
                $returnOrders = $api->getReturnOrders();
                $returnCount = is_array($returnOrders) ? count($returnOrders) : 0;
                $totalReturnOrders += $returnCount;
                echo "<div class='info'>Anzahl Return Orders: {$returnCount}</div>";
                
                // Zähle eindeutige Radsätze in Return Orders
                $uniqueWheelsets = [];
                foreach ($returnOrders as $order) {
                    if (isset($order['wheelSetNumber'])) {
                        $uniqueWheelsets[$order['wheelSetNumber']] = true;
                    }
                }
                echo "<div class='info'>Eindeutige Radsätze in Return Orders: " . count($uniqueWheelsets) . "</div>";
            } catch (Exception $e) {
                echo "<div class='error'>Fehler: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
            
            // Test 5: Global Inventory
            echo "<h4>5. Global Inventory</h4>";
            try {
                $globalInventory = $api->getGlobalInventory();
                $globalCount = is_array($globalInventory) ? count($globalInventory) : 0;
                echo "<div class='info'>Anzahl Global Inventory Items: {$globalCount}</div>";
                
                // Zähle alle Radsätze in Global Inventory
                $globalWheelsets = 0;
                foreach ($globalInventory as $item) {
                    if (isset($item['inventoryNumbers']) && is_array($item['inventoryNumbers'])) {
                        $globalWheelsets += count($item['inventoryNumbers']);
                    }
                }
                echo "<div class='info'>Gesamt Radsätze in Global Inventory: {$globalWheelsets}</div>";
            } catch (Exception $e) {
                echo "<div class='error'>Fehler: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
            
            // Zeige Details der ersten 10 Radsätze
            if (is_array($wheelsets) && count($wheelsets) > 0) {
                echo "<h4>📋 Erste 10 Radsätze (Details)</h4>";
                echo "<table>";
                echo "<tr><th>Satznummer</th><th>Kennzeichen</th><th>VIN</th><th>Reifenmodell</th><th>Reifengröße</th><th>Status</th></tr>";
                
                $displayCount = min(10, count($wheelsets));
                for ($i = 0; $i < $displayCount; $i++) {
                    $ws = $wheelsets[$i];
                    $satznummer = $ws['satznummer'] ?? $ws['wheelSetNumber'] ?? '-';
                    $kennzeichen = $ws['kennzeichen'] ?? $ws['licenseNo'] ?? '-';
                    $vin = $ws['fin'] ?? $ws['VIN'] ?? '-';
                    $reifenmodell = $ws['reifenmodell'] ?? '-';
                    $reifengroesse = $ws['reifengroesse'] ?? '-';
                    $status = $ws['status'] ?? '-';
                    
                    echo "<tr>";
                    echo "<td>{$satznummer}</td>";
                    echo "<td>{$kennzeichen}</td>";
                    echo "<td>{$vin}</td>";
                    echo "<td>{$reifenmodell}</td>";
                    echo "<td>{$reifengroesse}</td>";
                    echo "<td>{$status}</td>";
                    echo "</tr>";
                }
                echo "</table>";
                
                if (count($wheelsets) > 10) {
                    echo "<div class='info'>... und " . (count($wheelsets) - 10) . " weitere Radsätze</div>";
                }
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>Fehler beim Laden der Daten: " . htmlspecialchars($e->getMessage()) . "</div>";
            echo "<div class='info'>Stack Trace: <pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre></div>";
        }
        
        echo "</div>";
    }
    
    // Zusammenfassung
    echo "<div class='section'>";
    echo "<h2>📊 Zusammenfassung (alle Autohäuser)</h2>";
    echo "<table>";
    echo "<tr><th>Methode</th><th>Anzahl</th></tr>";
    echo "<tr><td><strong>getAllWheelsetsSmart() - Gesamt Radsätze</strong></td><td class='count'>{$totalWheelsets}</td></tr>";
    echo "<tr><td>Pickup Orders</td><td>{$totalPickupOrders}</td></tr>";
    echo "<tr><td>Inventory Items</td><td>{$totalInventory}</td></tr>";
    echo "<tr><td>Return Orders</td><td>{$totalReturnOrders}</td></tr>";
    echo "</table>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='error'>Kritischer Fehler: " . htmlspecialchars($e->getMessage()) . "</div>";
    echo "<div class='info'>Stack Trace: <pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre></div>";
}

echo "</div></body></html>";
?>

