<?php
/**
 * Setup-Script für Push-Subscriptions Tabelle
 * Prüft und erstellt/aktualisiert die Tabelle
 */

require_once 'config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    echo "=== PUSH-SUBSCRIPTIONS TABELLE SETUP ===\n\n";
    
    // Prüfe ob Tabelle existiert
    $stmt = $conn->query("SHOW TABLES LIKE 'push_subscriptions'");
    $tableExists = $stmt->rowCount() > 0;
    
    if (!$tableExists) {
        echo "📋 Erstelle Tabelle push_subscriptions...\n";
        
        $createTableSQL = "
        CREATE TABLE push_subscriptions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            endpoint VARCHAR(500) NOT NULL,
            p256dh VARCHAR(200) NOT NULL,
            auth VARCHAR(50) NOT NULL,
            user_agent TEXT,
            device_info TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            is_active TINYINT(1) DEFAULT 1,
            UNIQUE KEY unique_endpoint (endpoint),
            KEY idx_user_id (user_id),
            KEY idx_is_active (is_active),
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";
        
        $conn->exec($createTableSQL);
        echo "✅ Tabelle erfolgreich erstellt!\n\n";
    } else {
        echo "✅ Tabelle existiert bereits\n";
        echo "📋 Prüfe Spalten...\n\n";
        
        // Prüfe welche Spalten existieren
        $stmt = $conn->query("DESCRIBE push_subscriptions");
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $requiredColumns = [
            'id' => 'INT AUTO_INCREMENT PRIMARY KEY',
            'user_id' => 'INT NOT NULL',
            'endpoint' => 'VARCHAR(500) NOT NULL',
            'p256dh' => 'VARCHAR(200) NOT NULL',
            'auth' => 'VARCHAR(50) NOT NULL',
            'user_agent' => 'TEXT',
            'device_info' => 'TEXT',
            'created_at' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP',
            'updated_at' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',
            'is_active' => 'TINYINT(1) DEFAULT 1'
        ];
        
        $missingColumns = [];
        foreach ($requiredColumns as $column => $definition) {
            if (!in_array($column, $columns)) {
                $missingColumns[$column] = $definition;
            }
        }
        
        if (empty($missingColumns)) {
            echo "✅ Alle Spalten vorhanden!\n\n";
        } else {
            echo "⚠️  Fehlende Spalten gefunden:\n";
            foreach ($missingColumns as $column => $definition) {
                echo "   - $column\n";
            }
            echo "\n📋 Füge fehlende Spalten hinzu...\n";
            
            foreach ($missingColumns as $column => $definition) {
                try {
                    // Für AUTO_INCREMENT und PRIMARY KEY müssen wir speziell vorgehen
                    if (strpos($definition, 'AUTO_INCREMENT') !== false || strpos($definition, 'PRIMARY KEY') !== false) {
                        // Diese Spalten sollten bereits existieren, überspringen
                        continue;
                    }
                    
                    $alterSQL = "ALTER TABLE push_subscriptions ADD COLUMN $column $definition";
                    $conn->exec($alterSQL);
                    echo "   ✅ Spalte '$column' hinzugefügt\n";
                } catch (PDOException $e) {
                    if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
                        echo "   ℹ️  Spalte '$column' existiert bereits\n";
                    } else {
                        echo "   ❌ Fehler bei Spalte '$column': " . $e->getMessage() . "\n";
                    }
                }
            }
            
            // Prüfe Indizes
            echo "\n📋 Prüfe Indizes...\n";
            
            $stmt = $conn->query("SHOW INDEXES FROM push_subscriptions");
            $indexes = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            // Prüfe unique_endpoint Index
            $hasUniqueEndpoint = false;
            foreach ($indexes as $index) {
                if (strpos($index, 'unique_endpoint') !== false) {
                    $hasUniqueEndpoint = true;
                    break;
                }
            }
            
            if (!$hasUniqueEndpoint) {
                try {
                    $conn->exec("ALTER TABLE push_subscriptions ADD UNIQUE KEY unique_endpoint (endpoint)");
                    echo "   ✅ Index 'unique_endpoint' hinzugefügt\n";
                } catch (PDOException $e) {
                    echo "   ⚠️  Index 'unique_endpoint': " . $e->getMessage() . "\n";
                }
            }
            
            // Prüfe Foreign Key
            $stmt = $conn->query("
                SELECT CONSTRAINT_NAME 
                FROM information_schema.KEY_COLUMN_USAGE 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'push_subscriptions' 
                AND CONSTRAINT_NAME LIKE '%user_id%'
            ");
            $hasForeignKey = $stmt->rowCount() > 0;
            
            if (!$hasForeignKey) {
                try {
                    $conn->exec("ALTER TABLE push_subscriptions ADD FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE");
                    echo "   ✅ Foreign Key hinzugefügt\n";
                } catch (PDOException $e) {
                    echo "   ⚠️  Foreign Key: " . $e->getMessage() . "\n";
                }
            }
        }
    }
    
    echo "\n=== SETUP ABGESCHLOSSEN ===\n";
    echo "✅ Die Tabelle push_subscriptions ist bereit!\n\n";
    
    // Zeige Tabelle-Struktur
    echo "📋 Aktuelle Tabellen-Struktur:\n";
    $stmt = $conn->query("DESCRIBE push_subscriptions");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    printf("%-20s %-30s %-10s %-10s %-10s\n", "Spalte", "Typ", "Null", "Key", "Default");
    echo str_repeat("-", 80) . "\n";
    foreach ($columns as $col) {
        printf("%-20s %-30s %-10s %-10s %-10s\n", 
            $col['Field'], 
            $col['Type'], 
            $col['Null'], 
            $col['Key'], 
            $col['Default'] ?? 'NULL'
        );
    }
    
} catch (Exception $e) {
    echo "❌ Fehler: " . $e->getMessage() . "\n";
    echo "Stack Trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}
?>

