<?php
/**
 * Script zum Synchronisieren von P2 Teilenummern aus CSV-Dateien
 * 
 * Prüft alle Reifen aus der Reifen.csv und Komplettrad.csv,
 * ob sie bereits in P2 vorhanden sind und speichert die P2 Teilenummer
 * in der p2_teilenr_mapping Datenbank.
 * 
 * Verwendung: php setup/sync_p2_teilenr_from_csv.php
 */

require_once __DIR__ . '/../config/database.php';

// P2 API Konfiguration
$p2ApiBaseUrl = 'http://100.82.207.72:5001/api/teile';
$timeout = 15;

// Statistik
$stats = [
    'total' => 0,
    'found' => 0,
    'not_found' => 0,
    'saved' => 0,
    'errors' => 0,
    'skipped' => 0
];

/**
 * Normalisiert eine Teilenummer
 */
function normalizePartNumber($partNumber) {
    if (empty($partNumber)) {
        return null;
    }
    return strtoupper(trim(str_replace(' ', '', $partNumber)));
}

/**
 * Generiert Suchvarianten für eine Teilenummer
 */
function generateSearchVariants($partNumber) {
    if (empty($partNumber)) {
        return [];
    }
    
    $normalized = normalizePartNumber($partNumber);
    $variants = [$normalized];
    
    // Variante mit RE-Präfix
    if (substr($normalized, 0, 2) !== 'RE') {
        $variants[] = 'RE' . $normalized;
    }
    
    // Variante ohne RE-Präfix
    if (substr($normalized, 0, 2) === 'RE' && strlen($normalized) > 2) {
        $variants[] = substr($normalized, 2);
    }
    
    return array_unique($variants);
}

/**
 * Prüft ob ein Teil in P2 existiert
 */
function checkPartInP2($partNumber, $apiBaseUrl, $timeout) {
    if (empty($partNumber)) {
        return null;
    }
    
    $variants = generateSearchVariants($partNumber);
    
    foreach ($variants as $variant) {
        $url = $apiBaseUrl . '/' . urlencode($variant);
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json'
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            continue; // Versuche nächste Variante
        }
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            if ($data && isset($data['success']) && $data['success'] === true && isset($data['data'])) {
                // Teil gefunden!
                return $data['data']['Teilenr'] ?? $data['data']['Originalteilenr'] ?? $variant;
            }
        }
    }
    
    return null; // Nicht gefunden
}

/**
 * Speichert P2 Teilenummer in der Datenbank
 */
function saveP2TeilenrMapping($conn, $p2Teilenr, $ean, $oemPartNumber, $manufacturerNumber, $partNumber) {
    // Normalisiere alle Werte
    $p2Teilenr = normalizePartNumber($p2Teilenr);
    $ean = $ean ? normalizePartNumber($ean) : null;
    $oemPartNumber = $oemPartNumber ? normalizePartNumber($oemPartNumber) : null;
    $manufacturerNumber = $manufacturerNumber ? normalizePartNumber($manufacturerNumber) : null;
    $partNumber = $partNumber ? normalizePartNumber($partNumber) : null;
    
    if (empty($p2Teilenr)) {
        return false;
    }
    
    if (!$ean && !$oemPartNumber && !$manufacturerNumber && !$partNumber) {
        return false;
    }
    
    try {
        // Prüfe ob bereits vorhanden
        $checkVariants = [];
        if ($ean) {
            $checkVariants[] = ['field' => 'ean', 'value' => $ean];
        }
        if ($oemPartNumber) {
            $variants = generateSearchVariants($oemPartNumber);
            foreach ($variants as $variant) {
                $checkVariants[] = ['field' => 'oem_part_number', 'value' => $variant];
            }
        }
        if ($manufacturerNumber) {
            $variants = generateSearchVariants($manufacturerNumber);
            foreach ($variants as $variant) {
                $checkVariants[] = ['field' => 'manufacturer_number', 'value' => $variant];
            }
        }
        if ($partNumber) {
            $variants = generateSearchVariants($partNumber);
            foreach ($variants as $variant) {
                $checkVariants[] = ['field' => 'part_number', 'value' => $variant];
            }
        }
        
        $conditions = [];
        $params = [];
        foreach ($checkVariants as $variant) {
            $conditions[] = $variant['field'] . ' = ?';
            $params[] = $variant['value'];
        }
        
        $checkSql = "SELECT id, p2_teilenr FROM p2_teilenr_mapping WHERE " . implode(' OR ', $conditions) . " LIMIT 1";
        $checkStmt = $conn->prepare($checkSql);
        $checkStmt->execute($params);
        $existing = $checkStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing) {
            // Bereits vorhanden
            if ($existing['p2_teilenr'] !== $p2Teilenr) {
                // Update wenn P2 Teilenummer sich geändert hat
                $updateSql = "UPDATE p2_teilenr_mapping SET p2_teilenr = ?, updated_at = NOW() WHERE id = ?";
                $updateStmt = $conn->prepare($updateSql);
                $updateStmt->execute([$p2Teilenr, $existing['id']]);
                return true;
            }
            return false; // Bereits vorhanden, keine Änderung
        }
        
        // Neues Mapping erstellen
        $insertSql = "INSERT INTO p2_teilenr_mapping (ean, oem_part_number, manufacturer_number, part_number, p2_teilenr) VALUES (?, ?, ?, ?, ?)";
        $insertStmt = $conn->prepare($insertSql);
        $insertStmt->execute([$ean, $oemPartNumber, $manufacturerNumber, $partNumber, $p2Teilenr]);
        
        // Speichere auch Varianten
        $p2TeilenrUpper = strtoupper($p2Teilenr);
        if (substr($p2TeilenrUpper, 0, 2) === 'RE' && strlen($p2TeilenrUpper) > 2) {
            $p2TeilenrWithoutRE = substr($p2TeilenrUpper, 2);
            
            // Speichere Varianten für oem_part_number
            if ($oemPartNumber) {
                $variants = generateSearchVariants($oemPartNumber);
                foreach ($variants as $variant) {
                    if ($variant !== $oemPartNumber) {
                        try {
                            $insertStmt->execute([$ean, $variant, null, null, $p2Teilenr]);
                        } catch (PDOException $e) {
                            // Ignoriere Duplikat-Fehler
                            if ($e->getCode() !== '23000') {
                                throw $e;
                            }
                        }
                    }
                }
            }
            
            // Speichere Varianten für manufacturer_number
            if ($manufacturerNumber) {
                $variants = generateSearchVariants($manufacturerNumber);
                foreach ($variants as $variant) {
                    if ($variant !== $manufacturerNumber) {
                        try {
                            $insertStmt->execute([$ean, null, $variant, null, $p2Teilenr]);
                        } catch (PDOException $e) {
                            // Ignoriere Duplikat-Fehler
                            if ($e->getCode() !== '23000') {
                                throw $e;
                            }
                        }
                    }
                }
            }
            
            // Speichere Varianten für part_number
            if ($partNumber) {
                $variants = generateSearchVariants($partNumber);
                foreach ($variants as $variant) {
                    if ($variant !== $partNumber) {
                        try {
                            $insertStmt->execute([$ean, null, null, $variant, $p2Teilenr]);
                        } catch (PDOException $e) {
                            // Ignoriere Duplikat-Fehler
                            if ($e->getCode() !== '23000') {
                                throw $e;
                            }
                        }
                    }
                }
            }
        }
        
        return true;
    } catch (PDOException $e) {
        if ($e->getCode() === '23000') {
            // Duplikat - ignoriere
            return false;
        }
        throw $e;
    }
}

/**
 * Parst CSV-Datei und gibt Array zurück
 */
function parseCsvFile($filePath) {
    $data = [];
    $handle = fopen($filePath, 'r');
    
    if ($handle === false) {
        return $data;
    }
    
    // Lese Header-Zeile
    $headers = fgetcsv($handle, 0, ';');
    if ($headers === false) {
        fclose($handle);
        return $data;
    }
    
    // Normalisiere Header-Namen
    $normalizedHeaders = [];
    foreach ($headers as $header) {
        $normalizedHeaders[] = trim($header);
    }
    
    // Lese Daten
    while (($row = fgetcsv($handle, 0, ';')) !== false) {
        if (count($row) < count($normalizedHeaders)) {
            $row = array_pad($row, count($normalizedHeaders), '');
        }
        $data[] = array_combine($normalizedHeaders, array_slice($row, 0, count($normalizedHeaders)));
    }
    
    fclose($handle);
    return $data;
}

/**
 * Findet Spaltenwert unabhängig von Groß-/Kleinschreibung
 */
function findColumnValue($tire, $possibleNames) {
    foreach ($possibleNames as $name) {
        // Direkter Zugriff
        if (isset($tire[$name])) {
            return $tire[$name];
        }
        // Case-insensitive Suche
        foreach ($tire as $key => $value) {
            if (strcasecmp(trim($key), trim($name)) === 0) {
                return $value;
            }
        }
    }
    return null;
}

/**
 * Verarbeitet Reifen aus CSV
 */
function processTireFromCsv($tire, $conn, $apiBaseUrl, $timeout, &$stats) {
    // Extrahiere Teilenummern (verschiedene mögliche Spaltennamen)
    $ean = findColumnValue($tire, [
        'EAN', 'ean', 'EAN-Code', 'EAN Code', 'EANCode',
        'EAN-Nr', 'EAN Nr', 'EANNr'
    ]);
    
    $oemPartNumber = findColumnValue($tire, [
        'Volvo-Ersatzteil-Nr', 'Volvo Ersatzteil Nr', 'VolvoErsatzteilNr',
        'OEM Part Number', 'OEMPartNumber', 'oem_part_number',
        'Volvo Teilenummer', 'Volvo-Teilenummer', 'VolvoTeilenummer',
        'Volvo-Ersatzteilnummer', 'Volvo Ersatzteilnummer'
    ]);
    
    $manufacturerNumber = findColumnValue($tire, [
        'Hersteller-Artikel-Nr', 'Hersteller Artikel Nr', 'HerstellerArtikelNr',
        'Manufacturer Number', 'ManufacturerNumber', 'manufacturer_number',
        'Hersteller-Artikelnummer', 'Hersteller Artikelnummer',
        'Hersteller-Nr', 'Hersteller Nr', 'HerstellerNr',
        'Artikel-Nr', 'Artikel Nr', 'ArtikelNr'
    ]);
    
    $partNumber = findColumnValue($tire, [
        'Ersatzteil-Nr', 'Ersatzteil Nr', 'ErsatzteilNr',
        'Part Number', 'PartNumber', 'part_number',
        'Teilenummer', 'Teile-Nummer', 'TeileNummer',
        'Ersatzteilnummer', 'Ersatzteil-Nummer'
    ]);
    
    // Wenn keine Teilenummer vorhanden, überspringe
    if (!$ean && !$oemPartNumber && !$manufacturerNumber && !$partNumber) {
        $stats['skipped']++;
        return;
    }
    
    $stats['total']++;
    
    // Prüfe ob bereits in Datenbank vorhanden
    $mappingUrl = __DIR__ . '/../api/p2_teilenr_mapping.php';
    $params = [];
    if ($ean) $params[] = 'ean=' . urlencode($ean);
    if ($oemPartNumber) $params[] = 'oem_part_number=' . urlencode($oemPartNumber);
    if ($manufacturerNumber) $params[] = 'manufacturer_number=' . urlencode($manufacturerNumber);
    if ($partNumber) $params[] = 'part_number=' . urlencode($partNumber);
    
    if (!empty($params)) {
        $checkUrl = $mappingUrl . '?' . implode('&', $params);
        $checkResponse = @file_get_contents($checkUrl);
        if ($checkResponse) {
            $checkData = json_decode($checkResponse, true);
            if ($checkData && isset($checkData['success']) && $checkData['success'] === true) {
                // Bereits in Datenbank vorhanden
                $stats['skipped']++;
                echo "  ⏭️  Übersprungen (bereits in DB): " . ($oemPartNumber ?: $manufacturerNumber ?: $partNumber ?: $ean) . "\n";
                return;
            }
        }
    }
    
    // Prüfe in P2 API
    $searchPartNumber = $oemPartNumber ?: $manufacturerNumber ?: $partNumber ?: $ean;
    
    echo "  🔍 Prüfe: " . $searchPartNumber . "... ";
    
    $p2Teilenr = checkPartInP2($searchPartNumber, $apiBaseUrl, $timeout);
    
    if ($p2Teilenr) {
        $stats['found']++;
        echo "✅ Gefunden: " . $p2Teilenr . " ";
        
        // Speichere in Datenbank
        if (saveP2TeilenrMapping($conn, $p2Teilenr, $ean, $oemPartNumber, $manufacturerNumber, $partNumber)) {
            $stats['saved']++;
            echo "💾 Gespeichert\n";
        } else {
            echo "⏭️  Bereits vorhanden\n";
        }
    } else {
        $stats['not_found']++;
        echo "❌ Nicht gefunden\n";
    }
}

// Hauptprogramm
try {
    $database = new Database();
    $conn = $database->getConnection();
    
    echo "=== P2 Teilenummer Synchronisation aus CSV ===\n\n";
    
    $basePath = dirname(__DIR__);
    $reifenCsvPath = $basePath . '/assets/ftp/volvo_tyre_engine/Reifen.csv';
    $komplettradCsvPath = $basePath . '/assets/ftp/volvo_tyre_engine/Komplettrad.csv';
    
    // Verarbeite Reifen.csv
    if (file_exists($reifenCsvPath)) {
        echo "📂 Lade Reifen.csv...\n";
        $reifenData = parseCsvFile($reifenCsvPath);
        echo "   Gefunden: " . count($reifenData) . " Einträge\n\n";
        
        foreach ($reifenData as $index => $tire) {
            echo "[" . ($index + 1) . "/" . count($reifenData) . "] ";
            try {
                processTireFromCsv($tire, $conn, $p2ApiBaseUrl, $timeout, $stats);
            } catch (Exception $e) {
                $stats['errors']++;
                echo "❌ Fehler: " . $e->getMessage() . "\n";
            }
            
            // Kurze Pause um API nicht zu überlasten
            usleep(100000); // 0.1 Sekunden
        }
    } else {
        echo "⚠️  Reifen.csv nicht gefunden: " . $reifenCsvPath . "\n";
    }
    
    echo "\n";
    
    // Verarbeite Komplettrad.csv
    if (file_exists($komplettradCsvPath)) {
        echo "📂 Lade Komplettrad.csv...\n";
        $komplettradData = parseCsvFile($komplettradCsvPath);
        echo "   Gefunden: " . count($komplettradData) . " Einträge\n\n";
        
        foreach ($komplettradData as $index => $tire) {
            echo "[" . ($index + 1) . "/" . count($komplettradData) . "] ";
            try {
                processTireFromCsv($tire, $conn, $p2ApiBaseUrl, $timeout, $stats);
            } catch (Exception $e) {
                $stats['errors']++;
                echo "❌ Fehler: " . $e->getMessage() . "\n";
            }
            
            // Kurze Pause um API nicht zu überlasten
            usleep(100000); // 0.1 Sekunden
        }
    } else {
        echo "⚠️  Komplettrad.csv nicht gefunden: " . $komplettradCsvPath . "\n";
    }
    
    // Statistik ausgeben
    echo "\n=== Statistik ===\n";
    echo "Gesamt geprüft: " . $stats['total'] . "\n";
    echo "In P2 gefunden: " . $stats['found'] . "\n";
    echo "Nicht gefunden: " . $stats['not_found'] . "\n";
    echo "In DB gespeichert: " . $stats['saved'] . "\n";
    echo "Übersprungen (bereits in DB): " . $stats['skipped'] . "\n";
    echo "Fehler: " . $stats['errors'] . "\n";
    echo "\n✅ Synchronisation abgeschlossen!\n";
    
} catch (Exception $e) {
    echo "❌ Fehler: " . $e->getMessage() . "\n";
    echo "Stack Trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}

