<?php
/**
 * Migrationsskript für p2_teilenr_mapping Tabelle
 * Fügt Unique Constraints hinzu und normalisiert bestehende Daten
 */

require_once __DIR__ . '/../config/database.php';

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    echo "Starte Migration von p2_teilenr_mapping...\n";
    
    // Prüfe ob Tabelle existiert
    $stmt = $conn->query("SHOW TABLES LIKE 'p2_teilenr_mapping'");
    $tableExists = $stmt->rowCount() > 0;
    
    if (!$tableExists) {
        echo "Tabelle existiert nicht. Wird beim ersten API-Aufruf erstellt.\n";
        exit(0);
    }
    
    // 1. Normalisiere alle bestehenden Daten (Großbuchstaben, keine Leerzeichen)
    echo "Normalisiere bestehende Daten...\n";
    
    // Beginne Transaktion nur für Daten-Updates
    $conn->beginTransaction();
    try {
        $updateSql = "
            UPDATE p2_teilenr_mapping SET
                ean = CASE WHEN ean IS NOT NULL THEN UPPER(TRIM(REPLACE(ean, ' ', ''))) ELSE NULL END,
                oem_part_number = CASE WHEN oem_part_number IS NOT NULL THEN UPPER(TRIM(REPLACE(oem_part_number, ' ', ''))) ELSE NULL END,
                manufacturer_number = CASE WHEN manufacturer_number IS NOT NULL THEN UPPER(TRIM(REPLACE(manufacturer_number, ' ', ''))) ELSE NULL END,
                part_number = CASE WHEN part_number IS NOT NULL THEN UPPER(TRIM(REPLACE(part_number, ' ', ''))) ELSE NULL END,
                p2_teilenr = UPPER(TRIM(REPLACE(p2_teilenr, ' ', '')))
        ";
        $conn->exec($updateSql);
        
        // 2. Entferne Duplikate (behalte die neueste Version)
        // Duplikate nach oem_part_number
        $deleteDuplicatesSql = "
            DELETE t1 FROM p2_teilenr_mapping t1
            INNER JOIN p2_teilenr_mapping t2 
            WHERE t1.id < t2.id 
            AND t1.oem_part_number = t2.oem_part_number 
            AND t1.oem_part_number IS NOT NULL
        ";
        $conn->exec($deleteDuplicatesSql);
        
        // Duplikate nach manufacturer_number
        $deleteDuplicatesSql = "
            DELETE t1 FROM p2_teilenr_mapping t1
            INNER JOIN p2_teilenr_mapping t2 
            WHERE t1.id < t2.id 
            AND t1.manufacturer_number = t2.manufacturer_number 
            AND t1.manufacturer_number IS NOT NULL
        ";
        $conn->exec($deleteDuplicatesSql);
        
        // Duplikate nach part_number
        $deleteDuplicatesSql = "
            DELETE t1 FROM p2_teilenr_mapping t1
            INNER JOIN p2_teilenr_mapping t2 
            WHERE t1.id < t2.id 
            AND t1.part_number = t2.part_number 
            AND t1.part_number IS NOT NULL
        ";
        $conn->exec($deleteDuplicatesSql);
        
        // Duplikate nach ean
        $deleteDuplicatesSql = "
            DELETE t1 FROM p2_teilenr_mapping t1
            INNER JOIN p2_teilenr_mapping t2 
            WHERE t1.id < t2.id 
            AND t1.ean = t2.ean 
            AND t1.ean IS NOT NULL
        ";
        $conn->exec($deleteDuplicatesSql);
        
        $conn->commit();
        echo "Daten normalisiert und Duplikate entfernt.\n";
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
    // 3. Entferne alte Indizes falls vorhanden
    // HINWEIS: ALTER TABLE Statements committen automatisch, daher außerhalb der Transaktion
    echo "Entferne alte Indizes...\n";
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping DROP INDEX IF EXISTS idx_ean");
        $conn->exec("ALTER TABLE p2_teilenr_mapping DROP INDEX IF EXISTS idx_oem_part_number");
        $conn->exec("ALTER TABLE p2_teilenr_mapping DROP INDEX IF EXISTS idx_manufacturer_number");
        $conn->exec("ALTER TABLE p2_teilenr_mapping DROP INDEX IF EXISTS idx_part_number");
    } catch (PDOException $e) {
        // Ignoriere Fehler wenn Index nicht existiert
    }
    
    // 4. Füge Unique Constraints hinzu (falls nicht vorhanden)
    // HINWEIS: ALTER TABLE Statements committen automatisch
    echo "Füge Unique Constraints hinzu...\n";
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD UNIQUE KEY unique_ean (ean)");
        echo "  - unique_ean hinzugefügt\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key') === false && strpos($e->getMessage(), 'already exists') === false) {
            echo "  - unique_ean: " . $e->getMessage() . "\n";
        }
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD UNIQUE KEY unique_oem_part_number (oem_part_number)");
        echo "  - unique_oem_part_number hinzugefügt\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key') === false && strpos($e->getMessage(), 'already exists') === false) {
            echo "  - unique_oem_part_number: " . $e->getMessage() . "\n";
        }
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD UNIQUE KEY unique_manufacturer_number (manufacturer_number)");
        echo "  - unique_manufacturer_number hinzugefügt\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key') === false && strpos($e->getMessage(), 'already exists') === false) {
            echo "  - unique_manufacturer_number: " . $e->getMessage() . "\n";
        }
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD UNIQUE KEY unique_part_number (part_number)");
        echo "  - unique_part_number hinzugefügt\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key') === false && strpos($e->getMessage(), 'already exists') === false) {
            echo "  - unique_part_number: " . $e->getMessage() . "\n";
        }
    }
    
    // 5. Füge Indizes hinzu (falls nicht vorhanden)
    // HINWEIS: ALTER TABLE Statements committen automatisch
    echo "Füge Indizes hinzu...\n";
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD INDEX idx_p2_teilenr (p2_teilenr)");
        echo "  - idx_p2_teilenr hinzugefügt\n";
    } catch (PDOException $e) {
        // Index existiert bereits
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD INDEX idx_ean (ean)");
        echo "  - idx_ean hinzugefügt\n";
    } catch (PDOException $e) {
        // Index existiert bereits
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD INDEX idx_oem_part_number (oem_part_number)");
        echo "  - idx_oem_part_number hinzugefügt\n";
    } catch (PDOException $e) {
        // Index existiert bereits
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD INDEX idx_manufacturer_number (manufacturer_number)");
        echo "  - idx_manufacturer_number hinzugefügt\n";
    } catch (PDOException $e) {
        // Index existiert bereits
    }
    
    try {
        $conn->exec("ALTER TABLE p2_teilenr_mapping ADD INDEX idx_part_number (part_number)");
        echo "  - idx_part_number hinzugefügt\n";
    } catch (PDOException $e) {
        // Index existiert bereits
    }
    
    echo "\n✅ Migration erfolgreich abgeschlossen!\n";
    
} catch (Exception $e) {
    echo "❌ Fehler bei der Migration: " . $e->getMessage() . "\n";
    exit(1);
}

