<?php
/**
 * RESY Auto-Import Script (PHP-Version)
 * Lädt Credentials aus der Datenbank und führt Import aus
 */

// WICHTIG: resy_auto_import.php liegt im Root-Verzeichnis (boxenstop), nicht in api/
require_once __DIR__ . '/config/database.php';

$projectDir = __DIR__; // Projekt-Root ist das aktuelle Verzeichnis
$logDir = $projectDir . '/logs';

if (!is_dir($logDir)) {
    mkdir($logDir, 0755, true);
}

$timestamp = date('Y-m-d_H-i-s');

echo "============================================\n";
echo "RESY Auto-Import - Start\n";
echo "Zeit: " . date('Y-m-d H:i:s') . "\n";
echo "============================================\n\n";

try {
    // Verbinde mit Datenbank
    $db = new Database();
    $pdo = $db->getConnection();
    
    // Lade ALLE Autohäuser aus der Datenbank
    $stmt = $pdo->query("SELECT id, name, resy_username, resy_password, resy_dealer_id FROM autohaus WHERE resy_username IS NOT NULL AND resy_username != '' ORDER BY id");
    $autohaeuser = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($autohaeuser)) {
        echo "❌ FEHLER: Keine Autohäuser mit RESY-Credentials gefunden!\n";
        exit(1);
    }
    
    echo "📋 Gefundene Autohäuser: " . count($autohaeuser) . "\n\n";
    
    foreach ($autohaeuser as $autohaus) {
        $autohausId = $autohaus['id'];
        $autohausName = $autohaus['name'];
        $username = $autohaus['resy_username'] ?? '';
        $password = $autohaus['resy_password'] ?? '';
        $dealerIdRaw = $autohaus['resy_dealer_id'] ?? '';
        
        // Konvertiere resy_dealer_id zu numerischer ID
        $dealerId = convertDealerId($autohausId, $dealerIdRaw);
        
        echo "[Import] Starte Import für {$autohausName} (ID: {$autohausId}, RESY: {$dealerId})...\n";
        echo "   Username: {$username}\n";
        echo "   Password: " . str_repeat('*', strlen($password)) . " (Länge: " . strlen($password) . ")\n";
        echo "   Password (erste/last Zeichen): " . substr($password, 0, 1) . "..." . substr($password, -1) . "\n";
        echo "   Dealer ID (DB): '{$dealerIdRaw}' → RESY ID: '{$dealerId}'\n";
        
        // DEBUG: Zeige kompletten Password (nur für Debug!)
        if (strlen($password) < 20) {
            echo "   DEBUG Password (vollständig): '{$password}'\n";
        }
        
        if (empty($username) || empty($password)) {
            echo "   ⚠️  WARNUNG: Username oder Password fehlt, überspringe...\n\n";
            continue;
        }
        
        // Temp-Dateien
        $jsonFile = "{$logDir}/resy_data_{$autohausId}_{$timestamp}.json";
        $logFile = "{$logDir}/resy_auto_{$autohausId}_{$timestamp}.log";
        
        // Starte Playwright (mit Timeout!)
        $scriptPath = str_replace('/', '\\', $projectDir . '/resy_playwright.js');
        
        // WICHTIG: Setze Timeout (max. 3 Minuten pro Autohaus - optimiert!)
        set_time_limit(180);
        
        echo "   🔄 Starte Playwright-Import...\n";
        echo "   ⏱️  Dies kann 2-5 Minuten dauern...\n";
        flush();
        
        // WICHTIG: Windows Batch entfernt "!" Zeichen!
        // Lösung: Verwende JSON-Datei für Parameter statt Command-Line
        $paramsFile = "{$logDir}/resy_params_{$autohausId}_{$timestamp}.json";
        $params = [
            'autohausId' => $autohausId,
            'username' => $username,
            'password' => $password, // WICHTIG: Wird als JSON gespeichert, "!" bleibt erhalten!
            'dealerId' => $dealerId
        ];
        file_put_contents($paramsFile, json_encode($params, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
        
        echo "   🔍 DEBUG Password aus DB: '{$password}' (Länge: " . strlen($password) . ")\n";
        echo "   🔍 DEBUG Password in JSON gespeichert\n";
        
        // Playwright liest Parameter aus JSON-Datei statt Command-Line
        $command = sprintf(
            'node "%s" "%s"',
            $scriptPath,
            escapeshellarg($paramsFile)
        );
        
        echo "   🔍 DEBUG Command: " . str_replace($password, '***', $command) . "\n";
        
        // Führe Playwright aus (mit Progress-Tracking)
        $output = [];
        $returnCode = 0;
        
        // Verwende proc_open für bessere Kontrolle
        $descriptorspec = [
            0 => ["pipe", "r"],
            1 => ["pipe", "w"],
            2 => ["pipe", "w"]
        ];
        
        $process = proc_open($command, $descriptorspec, $pipes, $projectDir);
        
        if (is_resource($process)) {
            // Setze Streams auf non-blocking
            stream_set_blocking($pipes[1], false);
            stream_set_blocking($pipes[2], false);
            
            $startTime = time();
            $timeout = 180; // 3 Minuten (optimiert)
            
            // Lese Output mit Progress
            $fullOutput = '';
            $stdout = '';
            $stderr = '';
            
            while (true) {
                // Lese stdout
                $read = stream_get_contents($pipes[1]);
                if ($read) {
                    $stdout .= $read;
                    $fullOutput .= $read;
                    // Zeige wichtige Zeilen an
                    $lines = explode("\n", $read);
                    foreach ($lines as $line) {
                        $line = trim($line);
                        if (!empty($line) && (
                            strpos($line, '[1/5]') !== false ||
                            strpos($line, '[2/5]') !== false ||
                            strpos($line, '[3/5]') !== false ||
                            strpos($line, '[4/5]') !== false ||
                            strpos($line, '✅') !== false ||
                            strpos($line, '❌') !== false ||
                            strpos($line, 'Reifensätze') !== false
                        )) {
                            echo "      " . substr($line, 0, 100) . "\n";
                            flush();
                        }
                    }
                }
                
                // Lese stderr
                $read = stream_get_contents($pipes[2]);
                if ($read) {
                    $stderr .= $read;
                    $fullOutput .= $read;
                }
                
                // Prüfe ob Prozess beendet
                $status = proc_get_status($process);
                if (!$status['running']) {
                    $returnCode = $status['exitcode'];
                    break;
                }
                
                // Prüfe Timeout
                if ((time() - $startTime) > $timeout) {
                    echo "      ⚠️  Timeout erreicht ({$timeout}s), beende Prozess...\n";
                    proc_terminate($process, 9); // SIGKILL
                    $returnCode = 124; // Timeout-Exit-Code
                    break;
                }
                
                usleep(100000); // 0.1 Sekunde warten
            }
            
            // Schließe Pipes
            fclose($pipes[0]);
            fclose($pipes[1]);
            fclose($pipes[2]);
            proc_close($process);
            
            // Füge verbleibenden Output hinzu
            if (!empty($stdout)) {
                $output = explode("\n", $stdout);
            }
        } else {
            // Fallback: exec falls proc_open nicht funktioniert
            echo "   ⚠️  proc_open nicht verfügbar, verwende exec...\n";
            exec($command . ' 2>&1', $output, $returnCode);
            $fullOutput = implode("\n", $output);
        }
        
        // Speichere kompletten Output als Log
        file_put_contents($logFile, $fullOutput);
        
        // Extrahiere JSON-Zeile
        $jsonLine = '';
        foreach ($output as $line) {
            if (preg_match('/^\{.*"success".*\}$/', trim($line))) {
                $jsonLine = trim($line);
                break;
            }
        }
        
        if (empty($jsonLine)) {
            echo "   ❌ FEHLER: Keine JSON-Daten gefunden (Log: {$logFile})\n";
            echo "   Letzte 5 Zeilen:\n";
            $lines = array_slice($output, -5);
            foreach ($lines as $line) {
                echo "      " . substr($line, 0, 100) . "\n";
            }
            echo "\n";
            continue;
        }
        
        // Speichere JSON
        file_put_contents($jsonFile, $jsonLine);
        
        // Verarbeite JSON und speichere in DB
        echo "   ✅ Daten geladen, speichere in Datenbank...\n";
        flush();
        
        $phpScript = $projectDir . '/api/process_resy_data.php';
        $processCommand = sprintf(
            '"%s" "%s" "%s" "%s"',
            'C:\\xampp\\php\\php.exe',
            $phpScript,
            escapeshellarg($autohausId),
            escapeshellarg($jsonFile)
        );
        
        $processOutput = [];
        $processReturn = 0;
        
        // DEBUG: Zeige den auszuführenden Befehl
        echo "   🔧 Execute: {$processCommand}\n";
        flush();
        
        exec($processCommand, $processOutput, $processReturn);
        
        echo "   📊 Process Return Code: {$processReturn}\n";
        echo "   📊 Process Output Lines: " . count($processOutput) . "\n";
        if (!empty($processOutput)) {
            echo "   📊 First 3 lines: " . implode(" | ", array_slice($processOutput, 0, 3)) . "\n";
        }
        flush();
        
        if ($processReturn === 0) {
            $resultJson = implode("\n", $processOutput);
            $result = json_decode($resultJson, true);
            
            if ($result && isset($result['count'])) {
                $count = $result['count'];
                echo "   ✅ {$autohausName} Import erfolgreich: {$count} Reifensätze gespeichert\n\n";
            } else {
                echo "   ⚠️  WARNUNG: Kein 'count' im Resultat\n";
                echo "   Output: " . substr($resultJson, 0, 200) . "\n";
                if (isset($result['error'])) {
                    echo "   Error: " . $result['error'] . "\n";
                }
                echo "\n";
            }
        } else {
            echo "   ❌ FEHLER: Speichern fehlgeschlagen (Exit Code: {$processReturn})\n";
            if (!empty($processOutput)) {
                echo "   Output: " . implode("\n", array_slice($processOutput, 0, 10)) . "\n";
            }
            echo "   💡 Tipp: Prüfe logs/process_resy_data_" . date('Y-m-d') . ".log\n";
            echo "\n";
        }
        
        // Reset timeout für nächsten Durchlauf
        set_time_limit(180);
    }
    
    echo "============================================\n";
    echo "RESY Auto-Import - Fertig\n";
    echo "Zeit: " . date('Y-m-d H:i:s') . "\n";
    echo "============================================\n";
    
} catch (Exception $e) {
    echo "❌ KRITISCHER FEHLER: " . $e->getMessage() . "\n";
    echo "Stack Trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}

/**
 * Konvertiert resy_dealer_id aus DB zu numerischer RESY-ID
 */
function convertDealerId($autohausId, $dealerIdRaw) {
    // Direkte Mapping basierend auf Autohaus-ID (sicherste Methode!)
    if ($autohausId == 3 || $autohausId == '3') {
        return '454'; // Glinde
    } elseif ($autohausId == 1 || $autohausId == '1') {
        return '453'; // Kiel
    } elseif ($autohausId == 2 || $autohausId == '2') {
        return '452'; // Norderstedt
    }
    
    // Fallback: Text-basiert
    if (stripos($dealerIdRaw, 'Norderstedt') !== false || preg_match('/\b112\b/', $dealerIdRaw)) {
        return '452';
    } elseif (stripos($dealerIdRaw, 'Kiel') !== false || preg_match('/\b113\b/', $dealerIdRaw)) {
        return '453';
    } elseif (stripos($dealerIdRaw, 'Glinde') !== false || preg_match('/\b114\b/', $dealerIdRaw)) {
        return '454';
    }
    
    // Falls bereits numerisch
    if (in_array($dealerIdRaw, ['452', '453', '454'])) {
        return $dealerIdRaw;
    }
    
    // Default: Kiel
    return '453';
}

