<?php
/**
 * Boxenstop - Volvo Autohaus Event Management
 * Hauptseite der Webapp
 */

session_start();
require_once 'config/database.php';

// Prüfen ob Einrichtung abgeschlossen ist
$db = new Database();
$conn = $db->getConnection();

$stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'setup_completed'");
$stmt->execute();
$setup_completed = $stmt->fetchColumn();

if (!$setup_completed) {
    header('Location: setup/');
    exit;
}

// Prüfe ob Multi-Autohaus-System aktiviert ist
$stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'multi_autohaus_enabled'");
$stmt->execute();
$multiAutohausEnabled = $stmt->fetchColumn();

// Autohäuser mit Event-Anzahl laden
$stmt = $conn->prepare("
    SELECT a.*, 
           COUNT(e.id) as event_count,
           COUNT(CASE WHEN e.event_date >= CURDATE() AND e.is_active = 1 THEN 1 END) as upcoming_events
    FROM autohaus a
    LEFT JOIN events e ON a.id = e.autohaus_id
    WHERE a.is_active = 1
    GROUP BY a.id
    ORDER BY a.name ASC
");
$stmt->execute();
$autohauser = $stmt->fetchAll();

// Aktive Events laden (mit Autohaus-Information)
$stmt = $conn->prepare("SELECT e.*, a.name as autohaus_name, a.short_name as autohaus_short_name,
    (SELECT COUNT(*) FROM appointments a WHERE a.event_id=e.id AND a.status IN ('booked','vorbereitet')) AS booked_count
  FROM events e 
  LEFT JOIN autohaus a ON e.autohaus_id = a.id
  WHERE e.is_active = 1 AND e.event_date >= CURDATE() 
  ORDER BY e.event_date ASC");
$stmt->execute();
$events = $stmt->fetchAll();

// Prüfe ob Autohaus-Auswahl erforderlich ist
$autohausSelectionRequired = false;
if ($multiAutohausEnabled) {
    $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'autohaus_selection_required'");
    $stmt->execute();
    $autohausSelectionRequired = $stmt->fetchColumn() == '1';
}

// Unternehmensdaten aus den Einstellungen laden
$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_phone', 'company_email', 'company_address', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Standardwerte setzen falls keine Einstellungen vorhanden
$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyPhone = $companySettings['company_phone'] ?? '+49 (0) 123 456 789';
$companyEmail = $companySettings['company_email'] ?? 'info@volvo-autohaus.de';
$companyAddress = $companySettings['company_address'] ?? 'Musterstraße 123, 12345 Musterstadt';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/volvo-logo.png';
$appName = $companySettings['app_name'] ?? 'B&E Autocentrum Boxenstop ReifenConnect';
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="googlebot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="bingbot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <title><?php echo htmlspecialchars($appName); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    
    <style>
        /* Autohaus-Auswahl Styles */
        .autohaus-selection-section {
            padding: 4rem 0;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        }
        
        .autohaus-selection-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 2rem;
            margin-top: 2rem;
        }
        
        .autohaus-selection-card {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid transparent;
            position: relative;
            overflow: hidden;
        }
        
        .autohaus-selection-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
            border-color: #007bff;
        }
        
        .autohaus-selection-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #007bff, #0056b3);
            transform: scaleX(0);
            transition: transform 0.3s ease;
        }
        
        .autohaus-selection-card:hover::before {
            transform: scaleX(1);
        }
        
        .autohaus-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1.5rem;
        }
        
        .autohaus-card-header h3 {
            margin: 0;
            color: #495057;
            font-size: 1.25rem;
            font-weight: 600;
        }
        
        .autohaus-short {
            background: #007bff;
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .autohaus-card-info {
            margin-bottom: 1.5rem;
        }
        
        .autohaus-card-info .info-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 0.75rem;
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .autohaus-card-info .info-item i {
            width: 16px;
            color: #007bff;
        }
        
        .autohaus-card-action {
            text-align: center;
            color: #007bff;
            font-size: 1.2rem;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .autohaus-selection-card:hover .autohaus-card-action {
            opacity: 1;
        }
        
        /* Autohaus-Filter */
        .autohaus-filter {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-top: 1rem;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #dee2e6;
        }
        
        .autohaus-filter label {
            font-weight: 500;
            color: #495057;
            margin: 0;
        }
        
        .autohaus-filter select {
            padding: 0.5rem 1rem;
            border: 1px solid #ced4da;
            border-radius: 6px;
            background: white;
            font-size: 0.9rem;
            min-width: 200px;
        }
        
        /* Event-Autohaus-Info */
        .event-autohaus {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin: 0.5rem 0;
            padding: 0.5rem 0.75rem;
            background: #e3f2fd;
            border-radius: 6px;
            color: #1976d2;
            font-size: 0.9rem;
            font-weight: 500;
        }
        
        .event-autohaus i {
            color: #1976d2;
        }
        
        /* Ausgewähltes Autohaus Info */
        .selected-autohaus-info {
            margin-top: 1rem;
        }
        
        .autohaus-info-badge {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            background: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 8px;
            color: #1976d2;
        }
        
        .autohaus-info-badge i {
            font-size: 1.2rem;
        }
        
        .autohaus-info-badge span {
            flex: 1;
            font-weight: 500;
        }
        
        .btn-change-autohaus {
            background: #1976d2;
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.9rem;
            transition: background 0.3s ease;
        }
        
        .btn-change-autohaus:hover {
            background: #1565c0;
        }
        
        .btn-change-autohaus i {
            margin-right: 0.5rem;
        }
        
        /* Keine Events Nachricht */
        .no-events-message {
            grid-column: 1 / -1;
            text-align: center;
            padding: 3rem 2rem;
            background: #f8f9fa;
            border-radius: 12px;
            border: 2px dashed #dee2e6;
            margin: 2rem 0;
        }
        
        .no-events-message .no-events-icon {
            font-size: 3rem;
            color: #6c757d;
            margin-bottom: 1rem;
        }
        
        .no-events-message h3 {
            color: #495057;
            margin-bottom: 1rem;
        }
        
        .no-events-message p {
            color: #6c757d;
            margin-bottom: 0.5rem;
        }
        
        @media (max-width: 768px) {
            .autohaus-selection-grid {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }
            
            .autohaus-selection-card {
                padding: 1.5rem;
            }
            
            .autohaus-filter {
                flex-direction: column;
                align-items: stretch;
            }
            
            .autohaus-filter select {
                min-width: auto;
            }
        }
    </style>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=Space+Grotesk:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo">
                    <div class="brand-text">
                        <h1><?php echo htmlspecialchars($appName); ?></h1>
                        <span class="brand-subtitle">Volvo Service</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="admin/" class="nav-link admin-link">
                    <i class="fas fa-cog"></i>
                    <span>Admin</span>
                </a>
            </div>
        </div>
    </nav>

    <!-- Hero Section -->
    <section class="hero">
        <div class="hero-background">
            <div class="hero-pattern"></div>
            <div class="hero-gradient"></div>
        </div>
        <div class="hero-container">
            <div class="hero-content">
                <div class="hero-badge">
                    <i class="fas fa-shield-check"></i>
                    <span>Professioneller Volvo Service</span>
                </div>
                <h1 class="hero-title">
                    <span class="title-line"><?php echo htmlspecialchars($appName); ?></span>
                    <span class="title-line">Service</span>
                </h1>
                <p class="hero-description">
                    Professioneller Reifenwechsel-Service für Ihren Volvo. 
                    Präzise, zuverlässig und mit höchster Qualität.
                </p>
                <div class="hero-stats">
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-stopwatch"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-number">30</span>
                            <span class="stat-label">Minuten</span>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-calendar-check"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-number">2x</span>
                            <span class="stat-label">Jährlich</span>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="fas fa-shield-check"></i>
                        </div>
                        <div class="stat-content">
                            <span class="stat-number">100%</span>
                            <span class="stat-label">Sicherheit</span>
                        </div>
                    </div>
                </div>
            </div>
            <div class="hero-visual">
                <div class="car-illustration">
                    <div class="car-body"></div>
                    <div class="car-wheels">
                        <div class="wheel wheel-front"></div>
                        <div class="wheel wheel-rear"></div>
                    </div>
                    <div class="speed-lines"></div>
                </div>
            </div>
        </div>
    </section>

    <!-- Autohaus-Auswahl Section (nur wenn Multi-Autohaus aktiviert) -->
    <?php if ($multiAutohausEnabled && $autohausSelectionRequired): ?>
    <section class="autohaus-selection-section">
        <div class="section-container">
            <div class="section-header">
                <h2 class="section-title">Wählen Sie Ihr Autohaus</h2>
                <p class="section-subtitle">Bitte wählen Sie das Autohaus aus, für das Sie einen Termin buchen möchten</p>
            </div>
            
            <div class="autohaus-selection-grid">
                <?php foreach ($autohauser as $autohaus): ?>
                    <div class="autohaus-selection-card" onclick="selectAutohaus(<?php echo $autohaus['id']; ?>, '<?php echo htmlspecialchars($autohaus['name']); ?>')">
                        <div class="autohaus-card-header">
                            <h3><?php echo htmlspecialchars($autohaus['name']); ?></h3>
                            <span class="autohaus-short"><?php echo htmlspecialchars($autohaus['short_name']); ?></span>
                        </div>
                        <div class="autohaus-card-info">
                            <div class="info-item">
                                <i class="fas fa-map-marker-alt"></i>
                                <span><?php echo htmlspecialchars($autohaus['address']); ?></span>
                            </div>
                            <div class="info-item">
                                <i class="fas fa-phone"></i>
                                <span><?php echo htmlspecialchars($autohaus['phone']); ?></span>
                            </div>
                            <div class="info-item">
                                <i class="fas fa-calendar-alt"></i>
                                <span>
                                    <?php if ($autohaus['upcoming_events'] > 0): ?>
                                        <strong><?php echo $autohaus['upcoming_events']; ?></strong> Event<?php echo $autohaus['upcoming_events'] > 1 ? 's' : ''; ?> geplant
                                    <?php else: ?>
                                        Keine Events geplant
                                    <?php endif; ?>
                                </span>
                            </div>
                        </div>
                        <div class="autohaus-card-action">
                            <i class="fas fa-arrow-right"></i>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php endif; ?>

    <!-- Events Section -->
    <section class="events-section" id="events-section" <?php echo ($multiAutohausEnabled && $autohausSelectionRequired) ? 'style="display: none;"' : ''; ?>>
        <div class="section-container">
            <div class="section-header">
                <h2 class="section-title">Verfügbare Events</h2>
                <p class="section-subtitle">Buchen Sie Ihren Termin für den nächsten Boxenstop</p>
                <?php if ($multiAutohausEnabled && !$autohausSelectionRequired): ?>
                    <div class="autohaus-filter">
                        <label for="autohaus-filter">Autohaus filtern:</label>
                        <select id="autohaus-filter" onchange="filterEventsByAutohaus()">
                            <option value="">Alle Autohäuser</option>
                            <?php foreach ($autohauser as $autohaus): ?>
                                <option value="<?php echo $autohaus['id']; ?>"><?php echo htmlspecialchars($autohaus['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                <?php endif; ?>
            </div>
            
            <?php if (empty($events)): ?>
                <div class="no-events">
                    <div class="no-events-icon">
                        <i class="fas fa-calendar-times"></i>
                    </div>
                    <h3>Keine Events verfügbar</h3>
                    <p>Derzeit sind keine Boxenstop Events geplant.</p>
                    <p>Kontaktieren Sie uns für weitere Informationen.</p>
                </div>
            <?php else: ?>
                <div class="events-grid">
                    <?php foreach ($events as $event): ?>
                        <?php
                        $totalSlots = floor((strtotime($event['end_time'])-strtotime($event['start_time']))/($event['slot_duration']*60)) * (int)$event['max_monteure'];
                        $booked = (int)($event['booked_count'] ?? 0);
                        $ratio = $totalSlots>0 ? round($booked/$totalSlots*100) : 0;
                        
                        // Nächster freier Slot ermitteln
                        $nextFree = null;
                        $countsStmt = $conn->prepare("SELECT slot_time, COUNT(*) as c FROM appointments WHERE event_id = ? AND slot_date = ? AND status IN ('booked','vorbereitet') GROUP BY slot_time");
                        $countsStmt->execute([$event['id'], $event['event_date']]);
                        $counts = [];
                        foreach ($countsStmt->fetchAll() as $row) { $counts[$row['slot_time']] = (int)$row['c']; }
                        $slotDur = ((int)$event['slot_duration']) * 60;
                        for ($t = strtotime($event['start_time']); $t < strtotime($event['end_time']); $t += $slotDur) {
                            $tStr = date('H:i:s', $t);
                            $b = $counts[$tStr] ?? 0;
                            if ($b < (int)$event['max_monteure']) { $nextFree = date('H:i', $t); break; }
                        }
                        ?>
                        <div class="event-card" data-autohaus-id="<?php echo $event['autohaus_id'] ?? ''; ?>">
                            <div class="event-card-header">
                                <div class="event-status">
                                    <span class="status-badge <?php echo $ratio >= 90 ? 'status-full' : ($ratio >= 70 ? 'status-busy' : 'status-available'); ?>">
                                        <?php echo $ratio >= 90 ? 'Ausgebucht' : ($ratio >= 70 ? 'Fast voll' : 'Verfügbar'); ?>
                                    </span>
                                </div>
                                <div class="event-date-badge">
                                    <i class="fas fa-calendar"></i>
                                    <span><?php echo date('d.m', strtotime($event['event_date'])); ?></span>
                                </div>
                            </div>
                            
                            <div class="event-card-content">
                                <h3 class="event-title"><?php echo htmlspecialchars($event['name']); ?></h3>
                                
                                <?php if (!empty($event['autohaus_name'])): ?>
                                <div class="event-autohaus">
                                    <i class="fas fa-building"></i>
                                    <span><?php echo htmlspecialchars($event['autohaus_name']); ?></span>
                                </div>
                                <?php endif; ?>
                                
                                <div class="event-info-grid">
                                    <div class="info-item">
                                        <div class="info-icon">
                                            <i class="fas fa-clock"></i>
                                        </div>
                                        <div class="info-content">
                                            <span class="info-label">Zeit</span>
                                            <span class="info-value"><?php echo date('H:i', strtotime($event['start_time'])); ?> - <?php echo date('H:i', strtotime($event['end_time'])); ?></span>
                                        </div>
                                    </div>
                                    
                                    <div class="info-item">
                                        <div class="info-icon">
                                            <i class="fas fa-users"></i>
                                        </div>
                                        <div class="info-content">
                                            <span class="info-label">Monteure</span>
                                            <span class="info-value"><?php echo $event['max_monteure']; ?></span>
                                        </div>
                                    </div>
                                    
                                    <div class="info-item">
                                        <div class="info-icon">
                                            <i class="fas fa-stopwatch"></i>
                                        </div>
                                        <div class="info-content">
                                            <span class="info-label">Slot-Dauer</span>
                                            <span class="info-value"><?php echo $event['slot_duration']; ?> Min</span>
                                        </div>
                                    </div>
                                    
                                    <div class="info-item">
                                        <div class="info-icon">
                                            <i class="fas fa-chart-pie"></i>
                                        </div>
                                        <div class="info-content">
                                            <span class="info-label">Auslastung</span>
                                            <span class="info-value"><?php echo $booked; ?>/<?php echo $totalSlots; ?></span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="event-progress">
                                    <div class="progress-header">
                                        <span class="progress-label">Belegung</span>
                                        <span class="progress-percentage"><?php echo $ratio; ?>%</span>
                                    </div>
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?php echo $ratio; ?>%"></div>
                                    </div>
                                </div>
                                
                                <?php if ($nextFree): ?>
                                <div class="next-slot">
                                    <i class="fas fa-clock-rotate-left"></i>
                                    <span>Nächster freier Slot: <strong><?php echo $nextFree; ?></strong></span>
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="event-card-footer">
                                <a href="booking.php?event=<?php echo $event['id']; ?>" class="btn btn-primary event-btn">
                                    <i class="fas fa-calendar-plus"></i>
                                    <span>Termin buchen</span>
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Features Section -->
    <section class="features-section">
        <div class="section-container">
            <div class="section-header">
                <h2 class="section-title">Warum Boxenstop?</h2>
                <p class="section-subtitle">Professioneller Service für Ihren Volvo</p>
            </div>
            
            <div class="features-grid">
                <div class="feature-card">
                    <div class="feature-icon">
                        <i class="fas fa-tools"></i>
                    </div>
                    <h3 class="feature-title">Professioneller Service</h3>
                    <p class="feature-description">
                        Schneller, präziser Reifenwechsel in nur 30 Minuten. 
                        Zertifizierte Volvo-Techniker sorgen für höchste Qualität.
                    </p>
                </div>
                
                <div class="feature-card">
                    <div class="feature-icon">
                        <i class="fas fa-shield-check"></i>
                    </div>
                    <h3 class="feature-title">Volvo Qualität</h3>
                    <p class="feature-description">
                        Original Volvo Teile und zertifizierte Techniker für maximale Sicherheit. 
                        Vertrauen Sie auf bewährte Volvo-Standards.
                    </p>
                </div>
                
                <div class="feature-card">
                    <div class="feature-icon">
                        <i class="fas fa-calendar-check"></i>
                    </div>
                    <h3 class="feature-title">Terminvereinbarung</h3>
                    <p class="feature-description">
                        Einfache Online-Terminvereinbarung für Ihren Reifenwechsel. 
                        Komfortabel und zeitsparend.
                    </p>
                </div>
                
                <div class="feature-card">
                    <div class="feature-icon">
                        <i class="fas fa-warehouse"></i>
                    </div>
                    <h3 class="feature-title">Reifenlagerung</h3>
                    <p class="feature-description">
                        Sichere Einlagerung Ihrer Sommer- oder Winterreifen. 
                        Professionelle Lagerung in unserem klimatisierten Lager.
                    </p>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="cta-container">
            <div class="cta-content">
                <h2 class="cta-title">Bereit für Ihren Reifenwechsel?</h2>
                <p class="cta-description">
                    Buchen Sie jetzt Ihren Termin für einen professionellen Reifenwechsel-Service.
                </p>
                <a href="#events" class="btn btn-cta">
                    <i class="fas fa-calendar-plus"></i>
                    <span>Jetzt buchen</span>
                </a>
            </div>
        </div>
    </section>

    <!-- Footer -->
    <footer class="footer">
        <div class="footer-container">
            <div class="footer-content">
                <div class="footer-brand">
                    <div class="footer-logo">
                        <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="footer-logo-img">
                        <div class="footer-brand-text">
                            <h3><?php echo htmlspecialchars($appName); ?></h3>
                            <span>Volvo Service</span>
                        </div>
                    </div>
                    <p class="footer-description">
                        Ihr vertrauensvoller Partner für professionelle Reifenwechsel-Services. 
                        Qualität und Sicherheit für Ihren Volvo.
                    </p>
                </div>
                
                <div class="footer-links">
                    <div class="footer-column">
                        <h4 class="footer-title">Service</h4>
                        <ul class="footer-list">
                            <li><a href="#events">Events</a></li>
                            <li><a href="#booking">Termin buchen</a></li>
                            <li><a href="#services">Services</a></li>
                            <li><a href="#contact">Kontakt</a></li>
                        </ul>
                    </div>
                    
                    <div class="footer-column">
                        <h4 class="footer-title">Kontakt</h4>
                        <ul class="footer-list">
                            <li>
                                <i class="fas fa-phone"></i>
                                <span><?php echo htmlspecialchars($companyPhone); ?></span>
                            </li>
                            <li>
                                <i class="fas fa-envelope"></i>
                                <span><?php echo htmlspecialchars($companyEmail); ?></span>
                            </li>
                            <li>
                                <i class="fas fa-map-marker-alt"></i>
                                <span><?php echo nl2br(htmlspecialchars($companyAddress)); ?></span>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <div class="footer-bottom">
                <div class="footer-bottom-content">
                    <p class="footer-copyright">
                        &copy; 2024 <?php echo htmlspecialchars($companyName); ?>. Alle Rechte vorbehalten.
                    </p>
                    <div class="footer-social">
                        <a href="#" class="social-link">
                            <i class="fab fa-facebook"></i>
                        </a>
                        <a href="#" class="social-link">
                            <i class="fab fa-instagram"></i>
                        </a>
                        <a href="#" class="social-link">
                            <i class="fab fa-linkedin"></i>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="assets/js/main.js"></script>
    
    <script>
        // Autohaus-Auswahl Funktionen
        function selectAutohaus(autohausId, autohausName) {
            // Autohaus in Session speichern
            fetch('api/autohaus_selection.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'select_autohaus',
                    autohaus_id: autohausId,
                    autohaus_name: autohausName
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Events-Sektion anzeigen und nach Autohaus filtern
                    const eventsSection = document.getElementById('events-section');
                    eventsSection.style.display = 'block';
                    
                    // Warte kurz, dann filtere die Events
                    setTimeout(() => {
                        filterEventsByAutohaus(autohausId);
                    }, 100);
                    
                    // Autohaus-Auswahl-Sektion ausblenden
                    document.querySelector('.autohaus-selection-section').style.display = 'none';
                    
                    // Autohaus-Info in der Events-Sektion anzeigen
                    const eventsHeader = document.querySelector('#events-section .section-header');
                    const autohausInfo = document.createElement('div');
                    autohausInfo.className = 'selected-autohaus-info';
                    autohausInfo.innerHTML = `
                        <div class="autohaus-info-badge">
                            <i class="fas fa-building"></i>
                            <span>Ausgewähltes Autohaus: <strong>${autohausName}</strong></span>
                            <button type="button" onclick="changeAutohaus()" class="btn-change-autohaus">
                                <i class="fas fa-exchange-alt"></i> Wechseln
                            </button>
                        </div>
                    `;
                    eventsHeader.appendChild(autohausInfo);
                } else {
                    alert('Fehler bei der Autohaus-Auswahl: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Fehler:', error);
                alert('Fehler bei der Autohaus-Auswahl');
            });
        }
        
        function filterEventsByAutohaus(autohausId = null) {
            const eventCards = document.querySelectorAll('.event-card');
            const filterSelect = document.getElementById('autohaus-filter');
            
            if (autohausId === null && filterSelect) {
                autohausId = filterSelect.value;
            }
            
            // Konvertiere autohausId zu String für Vergleich
            const targetAutohausId = String(autohausId);
            
            let visibleCount = 0;
            
            eventCards.forEach(card => {
                const cardAutohausId = card.getAttribute('data-autohaus-id');
                
                // Zeige alle Events wenn kein Filter gesetzt ist, oder zeige nur Events des gewählten Autohauses
                if (!targetAutohausId || targetAutohausId === '' || targetAutohausId === 'null' || cardAutohausId === targetAutohausId) {
                    card.style.display = 'block';
                    visibleCount++;
                } else {
                    card.style.display = 'none';
                }
            });
            
            // Debug-Information
            console.log('Filtering events for autohaus ID:', targetAutohausId);
            console.log('Found event cards:', eventCards.length);
            console.log('Visible event cards:', visibleCount);
            
            // Prüfe ob Events sichtbar sind
            if (visibleCount === 0) {
                console.log('No events visible for autohaus ID:', targetAutohausId);
                // Zeige eine Meldung wenn keine Events gefunden werden
                showNoEventsMessage(targetAutohausId);
            } else {
                hideNoEventsMessage();
            }
        }
        
        function showNoEventsMessage(autohausId) {
            // Entferne existierende Nachricht
            hideNoEventsMessage();
            
            const eventsGrid = document.querySelector('.events-grid');
            if (eventsGrid) {
                const noEventsDiv = document.createElement('div');
                noEventsDiv.className = 'no-events-message';
                noEventsDiv.innerHTML = `
                    <div class="no-events-icon">
                        <i class="fas fa-calendar-times"></i>
                    </div>
                    <h3>Keine Events verfügbar</h3>
                    <p>Für das ausgewählte Autohaus sind derzeit keine Events geplant.</p>
                    <p>Kontaktieren Sie uns für weitere Informationen.</p>
                `;
                eventsGrid.appendChild(noEventsDiv);
            }
        }
        
        function hideNoEventsMessage() {
            const existingMessage = document.querySelector('.no-events-message');
            if (existingMessage) {
                existingMessage.remove();
            }
        }
        
        function changeAutohaus() {
            // Zurück zur Autohaus-Auswahl
            document.querySelector('.autohaus-selection-section').style.display = 'block';
            document.getElementById('events-section').style.display = 'none';
            
            // Autohaus-Info entfernen
            const autohausInfo = document.querySelector('.selected-autohaus-info');
            if (autohausInfo) {
                autohausInfo.remove();
            }
        }
        
        // Event-Listener für Autohaus-Filter
        document.addEventListener('DOMContentLoaded', function() {
            const filterSelect = document.getElementById('autohaus-filter');
            if (filterSelect) {
                filterSelect.addEventListener('change', function() {
                    filterEventsByAutohaus();
                });
            }
        });
    </script>
</body>
</html>
