<?php
/**
 * Profil-Seite für Benutzer - Modern Design
 */

session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$user = getCurrentUser();

if (!$user) {
    header('Location: login.php');
    exit;
}

// Fahrzeuge des Users laden
$db = new Database();
$conn = $db->getConnection();

// Prüfe und füge Spalte für Banner-View-Count hinzu (falls nicht vorhanden)
try {
    $conn->exec("ALTER TABLE users ADD COLUMN verification_banner_viewed_count INT DEFAULT 0");
} catch (PDOException $e) {
    // Spalte existiert bereits, das ist OK
}

// Lade den aktuellen View-Count für den Verifizierungs-Banner
$stmt = $conn->prepare("SELECT verification_banner_viewed_count FROM users WHERE id = ?");
$stmt->execute([$user['id']]);
$banner_view_count = $stmt->fetchColumn() ?? 0;

// Prüfe ob der Verifizierungs-Banner angezeigt werden soll (nur approved Status und max. 5x)
$show_verification_banner = false;
if (($user['verification_status'] ?? 'pending') === 'approved' && $banner_view_count < 5) {
    $show_verification_banner = true;
    // Erhöhe den Zähler
    $new_count = $banner_view_count + 1;
    $stmt = $conn->prepare("UPDATE users SET verification_banner_viewed_count = ? WHERE id = ?");
    $stmt->execute([$new_count, $user['id']]);
}

$stmt = $conn->prepare("
    SELECT * FROM user_vehicles 
    WHERE user_id = ? AND is_active = 1 
    ORDER BY is_default DESC, created_at DESC
");
$stmt->execute([$user['id']]);
$vehicles = $stmt->fetchAll();

/**
 * Normalisiert Einlagerungsnummern für Vergleich
 */
function normalizeStorageNumber($number) {
    if (empty($number)) {
        return '';
    }
    $normalized = preg_replace('/[^\w.]/', '', $number);
    $normalized = strtoupper($normalized);
    if (preg_match('/^[\d.]+$/', $normalized)) {
        $normalized = preg_replace('/\.+/', '.', trim($normalized, '.'));
    }
    return $normalized;
}

/**
 * Normalisiert Kennzeichen für Vergleich
 */
function normalizeLicensePlate($plate) {
    if (empty($plate)) {
        return '';
    }
    return strtoupper(preg_replace('/[\s-]+/', '', $plate));
}

/**
 * Findet Angebote für einen User basierend auf Fahrzeugen und Terminen
 */
function getUserWheelsetOffers($conn, $user_id, $vehicles, $appointments) {
    $offers = [];
    $all_license_plates = [];
    $all_storage_numbers = [];
    
    // Sammle alle Kennzeichen aus Fahrzeugen
    foreach ($vehicles as $vehicle) {
        if (!empty($vehicle['license_plate'])) {
            $all_license_plates[] = $vehicle['license_plate'];
        }
    }
    
    // Sammle alle Einlagerungsnummern aus Terminen
    foreach ($appointments as $apt) {
        if (!empty($apt['storage_number'])) {
            $all_storage_numbers[] = $apt['storage_number'];
        }
        if (!empty($apt['license_plate'])) {
            $all_license_plates[] = $apt['license_plate'];
        }
    }
    
    // Finde Angebote über Einlagerungsnummern
    foreach ($all_storage_numbers as $storage_number) {
        $normalizedStorage = normalizeStorageNumber($storage_number);
        $variants = [
            $storage_number,
            $normalizedStorage,
            str_replace('.', '', $normalizedStorage)
        ];
        
        foreach ($variants as $variant) {
            $normalizedVariant = normalizeStorageNumber($variant);
            $normalizedForLike = '%' . $normalizedVariant . '%';
            
            $stmt = $conn->prepare("
                SELECT wo.*, ah.name as autohaus_name
                FROM wheelset_offers wo
                LEFT JOIN autohaus ah ON wo.autohaus_id = ah.id
                WHERE (
                    REPLACE(REPLACE(REPLACE(UPPER(wo.wheelset_number), ' ', ''), '-', ''), '.', '') = ?
                    OR wo.wheelset_number LIKE ?
                    OR wo.wheelset_number = ?
                )
                ORDER BY wo.created_at DESC
            ");
            $stmt->execute([$normalizedVariant, $normalizedForLike, $variant]);
            $foundOffers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($foundOffers as $offer) {
                $offer['matched_by'] = 'storage_number';
                $offer['matched_storage_number'] = $storage_number;
                $offers[$offer['id']] = $offer;
            }
        }
    }
    
    // Finde Angebote über Kennzeichen (als zusätzliche Quelle)
    foreach (array_unique($all_license_plates) as $license_plate) {
        $normalizedLicense = normalizeLicensePlate($license_plate);
        
        $stmt = $conn->prepare("
            SELECT wo.*, ah.name as autohaus_name
            FROM wheelset_offers wo
            LEFT JOIN autohaus ah ON wo.autohaus_id = ah.id
            WHERE REPLACE(REPLACE(UPPER(wo.license_plate), ' ', ''), '-', '') = ?
            ORDER BY wo.created_at DESC
        ");
        $stmt->execute([$normalizedLicense]);
        $foundOffers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($foundOffers as $offer) {
            if (!isset($offers[$offer['id']])) {
                $offer['matched_by'] = 'license_plate';
                $offer['matched_license_plate'] = $license_plate;
                $offers[$offer['id']] = $offer;
            }
        }
    }
    
    return array_values($offers);
}

/**
 * Findet letzte Einlagerungsnummern für einen User
 */
function getUserStorageNumbers($conn, $user_id, $appointments) {
    $storage_numbers = [];
    
    foreach ($appointments as $apt) {
        if (!empty($apt['storage_number'])) {
            $storage_numbers[] = [
                'storage_number' => $apt['storage_number'],
                'license_plate' => $apt['license_plate'] ?? '',
                'appointment_date' => $apt['event_date'] ?? '',
                'appointment_id' => $apt['id'],
                'autohaus_name' => $apt['autohaus_name'] ?? ''
            ];
        }
    }
    
    // Sortiere nach Datum (neueste zuerst)
    usort($storage_numbers, function($a, $b) {
        return strtotime($b['appointment_date']) - strtotime($a['appointment_date']);
    });
    
    return $storage_numbers;
}

// Termine des Users laden
$stmt = $conn->prepare("
    SELECT 
        a.*,
        e.name as event_name,
        e.event_date,
        COALESCE(a.slot_time, e.event_time, e.start_time) as appointment_time,
        COALESCE(e.event_time, e.start_time) as event_time,
        e.description as event_description,
        ah.name as autohaus_name,
        ah.address as autohaus_address,
        ah.phone as autohaus_phone,
        ah.id as autohaus_id
    FROM appointments a
    LEFT JOIN events e ON a.event_id = e.id
    LEFT JOIN autohaus ah ON e.autohaus_id = ah.id
    WHERE a.user_id = ? AND a.is_active = 1
    ORDER BY COALESCE(a.slot_date, e.event_date) DESC, COALESCE(a.slot_time, e.event_time, e.start_time) DESC
");
$stmt->execute([$user['id']]);
$appointments = $stmt->fetchAll();

// Lade Angebote für diesen User
$user_offers = getUserWheelsetOffers($conn, $user['id'], $vehicles, $appointments);

// Lade Einlagerungsnummern
$storage_numbers = getUserStorageNumbers($conn, $user['id'], $appointments);

/**
 * Lädt das Fahrzeugbild basierend auf dem Modellnamen
 */
function getVehicleImage($model) {
    if (empty($model)) {
        // Fallback auf defaultcar.png
        $defaultPath = 'assets/images/defaultcar.png';
        if (file_exists($defaultPath)) {
            return $defaultPath;
        }
        // Wenn defaultcar.png nicht existiert, verwende Volvo-Logo
        return file_exists('assets/images/volvo-logo.png') ? 'assets/images/volvo-logo.png' : '';
    }
    
    // Normalisiere Modellnamen (Kleinschreibung, Sonderzeichen entfernen)
    $normalizedModel = strtolower(trim($model));
    $normalizedModel = preg_replace('/[^a-z0-9]/', '', $normalizedModel);
    
    // Mögliche Dateierweiterungen
    $extensions = ['png', 'jpg', 'jpeg', 'webp'];
    
    // Prüfe ob Bild existiert
    foreach ($extensions as $ext) {
        $imagePath = "assets/images/fahrzeugmodellbilder/{$normalizedModel}.{$ext}";
        if (file_exists($imagePath)) {
            return $imagePath;
        }
    }
    
    // Fallback: Versuche mit Original-Modellnamen (mit Sonderzeichen)
    $originalModel = strtolower(trim($model));
    foreach ($extensions as $ext) {
        $imagePath = "assets/images/fahrzeugmodellbilder/{$originalModel}.{$ext}";
        if (file_exists($imagePath)) {
            return $imagePath;
        }
    }
    
    // Letzter Fallback: defaultcar.png
    $defaultPath = 'assets/images/defaultcar.png';
    if (file_exists($defaultPath)) {
        return $defaultPath;
    }
    
    // Wenn auch das nicht existiert, Volvo-Logo
    return file_exists('assets/images/volvo-logo.png') ? 'assets/images/volvo-logo.png' : '';
}

/**
 * Zählt die Anzahl der Termine für ein Fahrzeug basierend auf Kennzeichen
 */
function countVehicleAppointments($conn, $license_plate, $user_id) {
    if (empty($license_plate)) {
        return 0;
    }
    
    $normalizedLicense = normalizeLicensePlate($license_plate);
    
    // Suche in appointments Tabelle (license_plate ist nur in appointments, nicht in events)
    // Prüfe ob is_active Spalte existiert, sonst verwende status
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count
        FROM appointments a
        WHERE a.user_id = ? 
        AND (
            a.status NOT IN ('cancelled', 'rejected')
            OR a.status IS NULL
        )
        AND REPLACE(REPLACE(UPPER(COALESCE(a.license_plate, '')), ' ', ''), '-', '') = ?
    ");
    $stmt->execute([$user_id, $normalizedLicense]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return intval($result['count'] ?? 0);
}

/**
 * Zählt die Anzahl der Reifenangebote (tire_offers) für ein Fahrzeug
 */
function countVehicleTireOffers($conn, $license_plate) {
    if (empty($license_plate)) {
        return 0;
    }
    
    $normalizedLicense = normalizeLicensePlate($license_plate);
    
    // Suche in tire_offers Tabelle
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count
        FROM tire_offers
        WHERE REPLACE(REPLACE(UPPER(COALESCE(license_plate, '')), ' ', ''), '-', '') = ?
    ");
    $stmt->execute([$normalizedLicense]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return intval($result['count'] ?? 0);
}

// Erweitere Fahrzeuge mit Statistiken und Bildern
foreach ($vehicles as &$vehicle) {
    $vehicle['appointment_count'] = countVehicleAppointments($conn, $vehicle['license_plate'], $user['id']);
    $vehicle['tire_offer_count'] = countVehicleTireOffers($conn, $vehicle['license_plate']);
    $vehicle['image_path'] = getVehicleImage($vehicle['vehicle_model']);
}
unset($vehicle);

// Statistiken berechnen
$total_appointments = count($appointments);
$upcoming_appointments = array_filter($appointments, function($apt) {
    $date = $apt['slot_date'] ?? $apt['event_date'] ?? '';
    $time = $apt['appointment_time'] ?? $apt['slot_time'] ?? $apt['event_time'] ?? '00:00:00';
    return strtotime($date . ' ' . $time) >= time();
});
$upcoming_count = count($upcoming_appointments);
$past_appointments = array_filter($appointments, function($apt) {
    $date = $apt['slot_date'] ?? $apt['event_date'] ?? '';
    $time = $apt['appointment_time'] ?? $apt['slot_time'] ?? $apt['event_time'] ?? '00:00:00';
    return strtotime($date . ' ' . $time) < time();
});
$past_count = count($past_appointments);

// Verifizierungsdokument laden falls vorhanden
$verification_document = !empty($user['verification_document']) ? $user['verification_document'] : null;

// Activity Log laden
$stmt = $conn->prepare("
    SELECT * FROM user_activity_log 
    WHERE user_id = ? 
    ORDER BY created_at DESC 
    LIMIT 20
");
$stmt->execute([$user['id']]);
$activity_log = $stmt->fetchAll();

// Benachrichtigungen laden
$stmt = $conn->prepare("
    SELECT * FROM user_notifications 
    WHERE user_id = ? 
    AND is_read = 0
    ORDER BY created_at DESC 
    LIMIT 10
");
$stmt->execute([$user['id']]);
$unread_notifications = $stmt->fetchAll();

$notifications_count = count($unread_notifications);

// Unternehmensdaten laden
$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/pwa_logo.png';
$appName = $companySettings['app_name'] ?? 'B&E Gruppe';

// Aktive Autohäuser laden (mit allen Daten für Cards)
$stmt = $conn->prepare("SELECT id, name, short_name, address, phone, footer_image, footer_image_alt FROM autohaus WHERE is_active = 1 ORDER BY name ASC");
$stmt->execute();
$autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Aktuelles Autohaus des Users (falls vorhanden)
$user_autohaus_id = $user['preferred_dealership_id'] ?? null;

// Verifizierungsstatus bestimmen - Volvo Design Guidelines
$verification_status = $user['verification_status'] ?? 'pending';
$status_info = [
    'pending' => [
        'icon' => 'fas fa-hourglass-half', 
        'text' => 'Wartend auf Verifizierung', 
        'color' => '#ffffff', 
        'bg' => '#495057', // Dunkelgrau
        'type' => 'warning'
    ],
    'approved' => [
        'icon' => 'fas fa-check-circle', 
        'text' => 'Verifiziert und freigeschaltet', 
        'color' => '#ffffff', 
        'bg' => '#2e7d32', // Dunkelgrün
        'type' => 'success'
    ],
    'rejected' => [
        'icon' => 'fas fa-times-circle', 
        'text' => 'Abgelehnt', 
        'color' => '#ffffff', 
        'bg' => '#c62828', // Dunkelrot
        'type' => 'error'
    ]
];

$current_status = $status_info[$verification_status] ?? $status_info['pending'];

// Prüfe ob alle Pflichtfelder ausgefüllt sind
$required_fields = ['phone_number', 'address', 'postal_code', 'city'];
$missing_fields = [];

foreach ($required_fields as $field) {
    if (empty($user[$field])) {
        $missing_fields[] = $field;
    }
}

$has_missing_fields = count($missing_fields) > 0;

// Prüfe ob User von Push-Benachrichtigung kommt
$from_push_notification = isset($_GET['complete_profile']) && $_GET['complete_profile'] == '1';

// Prüfe ob User gesperrt ist und lade Suspension-Details
$is_suspended = ($user['status'] ?? 'pending') === 'suspended' || ($user['is_active'] ?? 1) == 0;
$suspension_details = null;

if ($is_suspended) {
    // Lade die neueste Suspension-Aktion aus admin_actions
    try {
        $stmt = $conn->prepare("
            SELECT 
                aa.*,
                au.username as admin_username,
                au.email as admin_email
            FROM admin_actions aa
            LEFT JOIN admin_users au ON aa.admin_id = au.id
            WHERE aa.user_id = ? 
            AND aa.action = 'suspend'
            ORDER BY aa.created_at DESC
            LIMIT 1
        ");
        $stmt->execute([$user['id']]);
        $suspension_details = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Tabelle existiert möglicherweise nicht, das ist OK
        error_log("Could not load suspension details: " . $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Mein Profil - <?php echo htmlspecialchars($appName); ?></title>
    
    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#0056b3">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="default">
    <meta name="apple-mobile-web-app-title" content="<?php echo htmlspecialchars($appName); ?>">
    <link rel="manifest" href="/manifest.json">
    
    <!-- Apple Touch Icons -->
    <!-- Apple Touch Icons - iOS (180x180 ist Standard für moderne iOS-Geräte) -->
    <link rel="apple-touch-icon" sizes="180x180" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon-precomposed" sizes="180x180" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="152x152" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="144x144" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="120x120" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="114x114" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="76x76" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="72x72" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="60x60" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="57x57" href="assets/images/new-pwa_icon.png">
    
    <!-- Favicons - Alle Browser -->
    <link rel="icon" type="image/png" sizes="512x512" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="384x384" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="192x192" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="152x152" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="144x144" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="128x128" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="96x96" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="72x72" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="48x48" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="32x32" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="16x16" href="assets/images/new-pwa_icon.png">
    <link rel="shortcut icon" href="assets/images/new-pwa_icon.png">
    <link rel="icon" href="assets/images/new-pwa_icon.png">
    
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            /* Volvo Shadow Guidelines - 5 Elevation Levels */
            --shadow-1: 0 1px 2px rgba(0, 0, 0, 0.1); /* Content switcher, Tabs */
            --shadow-2: 0 2px 4px rgba(0, 0, 0, 0.12); /* Buttons, Slider, Tooltip */
            --shadow-4: 0 4px 8px rgba(0, 0, 0, 0.15); /* Cards, Dropdown, Search fields, Snackbar */
            --shadow-8: 0 8px 16px rgba(0, 0, 0, 0.2); /* Dialog, Date picker, Time picker */
            --shadow-16: 0 16px 32px rgba(0, 0, 0, 0.25); /* Modal dialog */
            --card-shadow: var(--shadow-4);
            --card-shadow-hover: var(--shadow-8);
        }
        
        /* Button Styles - Volvo Default (4px) - Überschreibt externe Styles */
        .btn,
        button.btn,
        .btn-primary,
        .btn-secondary,
        .btn-action,
        .btn-action-modern,
        .btn-action-edit,
        .btn-action-delete,
        .appointment-download-btn,
        .appointment-cancel-btn,
        .sidebar-nav-item,
        .nav-link {
            border-radius: 4px !important;
        }
        
        /* Volvo Design System Button Styles - Volvo Guidelines */
        .btn,
        button.btn {
            padding: 0.625rem 1.25rem;
            border: none;
            border-radius: 4px;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            text-decoration: none;
            line-height: 1.5;
            font-family: inherit;
        }
        
        /* Button Style 1: Solid Dark (Primary) - Volvo Guidelines */
        .btn-primary {
            background: #1a1a1a;
            color: #ffffff;
            box-shadow: none;
        }
        
        .btn-primary:hover {
            background: #2d2d2d;
            color: #ffffff;
        }
        
        .btn-primary:active {
            background: #0d0d0d;
            color: #ffffff;
        }
        
        .btn-primary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .btn-primary:disabled {
            background: #e9ecef;
            color: #adb5bd;
            cursor: not-allowed;
            opacity: 0.6;
        }
        
        /* Button Style 2: Outlined (Secondary) - Volvo Guidelines */
        .btn-secondary {
            background: #ffffff;
            color: #1a1a1a;
            border: 1px solid #dee2e6;
            box-shadow: none;
        }
        
        .btn-secondary:hover {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .btn-secondary:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .btn-secondary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .btn-secondary:disabled {
            background: #ffffff;
            color: #adb5bd;
            border-color: #e9ecef;
            cursor: not-allowed;
            opacity: 0.6;
        }
        
        /* Button Style 4: Solid Blue (Accent) - Volvo Guidelines */
        .btn-accent {
            background: #0056b3;
            color: #ffffff;
            box-shadow: none;
        }
        
        .btn-accent:hover {
            background: #0066cc;
            color: #ffffff;
        }
        
        .btn-accent:active {
            background: #004c99;
            color: #ffffff;
        }
        
        .btn-accent:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .btn-accent:disabled {
            background: #cfe2ff;
            color: #adb5bd;
            cursor: not-allowed;
            opacity: 0.6;
        }
        
        /* Action Buttons - Edit (Solid Blue) - Volvo Guidelines */
        .btn-action-edit,
        .btn-action-modern.btn-edit {
            background: #0056b3;
            color: #ffffff;
            box-shadow: none;
        }
        
        .btn-action-edit:hover,
        .btn-action-edit:focus,
        .btn-action-modern.btn-edit:hover,
        .btn-action-modern.btn-edit:focus {
            background: #0066cc;
            color: #ffffff;
        }
        
        .btn-action-edit:active,
        .btn-action-modern.btn-edit:active {
            background: #004c99;
            color: #ffffff;
        }
        
        .btn-action-edit:focus,
        .btn-action-modern.btn-edit:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Action Buttons - Delete (Solid Dark) - Volvo Guidelines */
        .btn-action-delete,
        .btn-action-modern.btn-delete {
            background: #1a1a1a;
            color: #ffffff;
            box-shadow: none;
        }
        
        .btn-action-delete:hover,
        .btn-action-delete:focus,
        .btn-action-modern.btn-delete:hover,
        .btn-action-modern.btn-delete:focus {
            background: #2d2d2d;
            color: #ffffff;
        }
        
        .btn-action-delete:active,
        .btn-action-modern.btn-delete:active {
            background: #0d0d0d;
            color: #ffffff;
        }
        
        .btn-action-delete:focus,
        .btn-action-modern.btn-delete:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Appointment Download Button (Solid Blue) - Volvo Guidelines */
        .appointment-download-btn {
            background: #0056b3;
            color: #ffffff;
            box-shadow: none;
        }
        
        .appointment-download-btn:hover,
        .appointment-download-btn:focus {
            background: #0066cc;
            color: #ffffff;
        }
        
        .appointment-download-btn:active {
            background: #004c99;
            color: #ffffff;
        }
        
        .appointment-download-btn:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Appointment Cancel Button (Outlined) - Volvo Guidelines */
        .appointment-cancel-btn {
            background: #ffffff;
            color: #1a1a1a;
            border: 1px solid #dee2e6;
            box-shadow: none;
        }
        
        .appointment-cancel-btn:hover,
        .appointment-cancel-btn:focus {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .appointment-cancel-btn:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .appointment-cancel-btn:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Disabled State - Volvo Guidelines */
        .btn:disabled,
        button.btn:disabled,
        .btn-action:disabled,
        .btn-action-modern:disabled,
        .btn-primary:disabled,
        .btn-secondary:disabled,
        .btn-accent:disabled,
        .btn-action-edit:disabled,
        .btn-action-delete:disabled,
        .appointment-download-btn:disabled,
        .appointment-cancel-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
        }
        
        .profile-container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 2rem;
        }
        
        /* Status Banner - Volvo Design Guidelines - Kompakt & Dezent */
        .status-banner {
            background: <?php echo $current_status['bg']; ?>;
            padding: 1rem 1.5rem;
            border-radius: 4px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            box-shadow: var(--shadow-2);
            color: #ffffff;
        }
        
        .status-banner i {
            font-size: 1.25rem;
            color: #ffffff;
            flex-shrink: 0;
        }
        
        .status-banner-content {
            flex: 1;
            color: #ffffff;
            min-width: 0;
        }
        
        .status-banner-title {
            font-weight: 600;
            font-size: 0.9rem;
            color: #ffffff;
            margin-bottom: 0.25rem;
            line-height: 1.3;
        }
        
        .status-banner-text {
            color: rgba(255, 255, 255, 0.9);
            font-size: 0.85rem;
            line-height: 1.4;
        }
        
        .status-banner-text strong {
            color: #ffffff;
            font-weight: 600;
        }
        
        /* Status Action Buttons - Volvo Design */
        .status-action {
            display: flex;
            gap: 0.5rem;
            align-items: center;
            flex-shrink: 0;
        }
        
        /* Secondary Action Button - Nur Text, kein Border */
        .status-banner .btn-secondary-action {
            background: transparent;
            border: none;
            color: #ffffff;
            padding: 0.5rem 0.875rem;
            border-radius: 4px;
            font-size: 0.85rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.4rem;
            white-space: nowrap;
        }
        
        .status-banner .btn-secondary-action:hover {
            background: rgba(255, 255, 255, 0.1);
            color: #ffffff;
        }
        
        /* Main Action Button - Text mit weißem Border */
        .status-banner .btn-main-action,
        .status-banner .btn-primary {
            background: transparent;
            border: 1px solid #ffffff;
            color: #ffffff;
            padding: 0.5rem 1rem;
            border-radius: 4px;
            font-size: 0.85rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.4rem;
            white-space: nowrap;
        }
        
        .status-banner .btn-main-action:hover,
        .status-banner .btn-primary:hover {
            background: rgba(255, 255, 255, 0.15);
            border-color: #ffffff;
            color: #ffffff;
        }
        
        .status-banner .btn-main-action i,
        .status-banner .btn-primary i {
            font-size: 0.8rem;
        }
        
        /* Warning Banner (Orange) */
        .status-banner.status-warning {
            background: #f57c00;
        }
        
        /* Success Banner (Grün) */
        .status-banner.status-success {
            background: #2e7d32;
        }
        
        /* Error Banner (Rot) */
        .status-banner.status-error {
            background: #c62828;
        }
        
        /* Suspension Banner - Volvo Design Guidelines */
        .status-banner.status-suspended {
            background: #c62828;
            border-left: 4px solid #b71c1c;
            box-shadow: 0 4px 12px rgba(198, 40, 40, 0.2);
        }
        
        .status-banner.status-suspended i {
            color: #ffffff;
            font-size: 1.5rem;
        }
        
        .status-banner.status-suspended .status-banner-title {
            color: #ffffff;
            font-weight: 700;
        }
        
        .status-banner.status-suspended .status-banner-text {
            color: rgba(255, 255, 255, 0.95);
        }
        
        .status-banner.status-suspended .status-banner-text strong {
            color: #ffffff;
        }
        
        /* Info Banner (Dunkelgrau) */
        .status-banner.status-info {
            background: #495057;
        }
        
        /* Feedback Banner (für Push-Benachrichtigungen) */
        .feedback-banner {
            background: linear-gradient(135deg, #2196F3 0%, #1976D2 100%);
            padding: 1.25rem 1.5rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 12px rgba(33, 150, 243, 0.2);
            animation: slideInFromTop 0.4s ease-out;
        }
        
        @keyframes slideInFromTop {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes slideOutToTop {
            from {
                opacity: 1;
                transform: translateY(0);
            }
            to {
                opacity: 0;
                transform: translateY(-20px);
            }
        }
        
        .feedback-banner-content {
            display: flex;
            align-items: center;
            gap: 1rem;
            color: #ffffff;
        }
        
        .feedback-banner-icon {
            font-size: 1.5rem;
            color: #ffffff;
            flex-shrink: 0;
        }
        
        .feedback-banner-text {
            flex: 1;
            color: #ffffff;
        }
        
        .feedback-banner-text strong {
            display: block;
            font-size: 1.05rem;
            font-weight: 700;
            margin-bottom: 0.25rem;
            color: #ffffff;
        }
        
        .feedback-banner-text p {
            margin: 0;
            font-size: 0.9rem;
            color: rgba(255, 255, 255, 0.95);
            line-height: 1.5;
        }
        
        .feedback-banner-close {
            background: rgba(255, 255, 255, 0.2);
            border: none;
            color: #ffffff;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s;
            flex-shrink: 0;
        }
        
        .feedback-banner-close:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: scale(1.1);
        }
        
        .feedback-banner-close:active {
            transform: scale(0.95);
        }
        
        /* Profile Header - Volvo Design Guidelines - Kompakt & Dezent */
        .profile-header-card {
            background: #ffffff;
            border-radius: 4px;
            padding: 1.5rem 2rem;
            margin-bottom: 2rem;
            color: #003057;
            box-shadow: var(--shadow-4);
            border: 1px solid #e9ecef;
            position: relative;
        }
        
        .profile-header-content {
            display: grid;
            grid-template-columns: auto 1fr auto;
            gap: 1.5rem;
            align-items: center;
            position: relative;
        }
        
        .profile-avatar {
            width: 64px;
            height: 64px;
            border-radius: 50%;
            background: #f8f9fa;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.75rem;
            font-weight: 600;
            border: 1px solid #e9ecef;
            color: #003057;
            box-shadow: var(--shadow-1);
            flex-shrink: 0;
        }
        
        .profile-info {
            color: #003057;
            min-width: 0;
        }
        
        .profile-name {
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 0.375rem;
            color: #003057;
            letter-spacing: -0.01em;
            line-height: 1.3;
        }
        
        .profile-email {
            color: #6c757d;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-weight: 400;
        }
        
        .profile-email i {
            color: #0056b3;
            font-size: 0.85rem;
        }
        
        .profile-stats {
            display: flex;
            flex-direction: row;
            gap: 1rem;
            flex-shrink: 0;
        }
        
        .stat-item {
            background: #f8f9fa;
            padding: 0.625rem 1rem;
            border-radius: 4px;
            border: 1px solid #e9ecef;
            text-align: center;
            min-width: 70px;
        }
        
        .stat-value {
            font-size: 1.25rem;
            font-weight: 600;
            color: #003057;
            line-height: 1.2;
        }
        
        .stat-label {
            font-size: 0.75rem;
            color: #6c757d;
            font-weight: 400;
            margin-top: 0.25rem;
        }
        
        /* Section Cards */
        .section-card {
            background: white;
            border-radius: 8px;
            padding: 2.5rem;
            margin-bottom: 2rem;
            box-shadow: var(--card-shadow);
            transition: all 0.3s;
        }
        
        .section-card:hover {
            box-shadow: var(--card-shadow-hover);
            transform: translateY(-2px);
        }
        
        .section-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #003057;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .section-title i {
            color: #0056b3;
        }
        
        /* Vehicle Cards - Modern Volvo Design */
        .vehicles-grid-modern {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 2rem;
            margin-top: 1.5rem;
        }
        
        .vehicle-card-modern {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            overflow: hidden;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            background: white;
            box-shadow: var(--shadow-2);
            display: flex;
            flex-direction: column;
        }
        
        .vehicle-card-modern:hover {
            border-color: #003057;
            transform: translateY(-6px);
            box-shadow: var(--shadow-4);
        }
        
        .vehicle-card-modern.default {
            border-color: #003057;
            border-width: 3px;
            box-shadow: var(--shadow-4);
        }
        
        .vehicle-card-modern.default::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #003057 0%, #0056b3 100%);
            z-index: 1;
        }
        
        /* Standardfahrzeug Badge */
        .vehicle-badge-modern {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 4px;
            font-size: 0.8rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            box-shadow: var(--shadow-2);
            z-index: 2;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .vehicle-badge-modern i {
            font-size: 0.9rem;
        }
        
        /* Fahrzeugbild Container */
        .vehicle-image-container {
            width: 100%;
            height: 320px;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
            position: relative;
            padding: 0;
        }
        
        .vehicle-image {
            width: 100%;
            height: 100%;
            object-fit: cover;
            object-position: center;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            filter: drop-shadow(0 12px 40px rgba(0, 0, 0, 0.3)) drop-shadow(0 4px 12px rgba(0, 0, 0, 0.2));
            display: block;
        }
        
        .vehicle-card-modern:hover .vehicle-image {
            transform: scale(1.05);
            filter: drop-shadow(0 20px 60px rgba(0, 48, 87, 0.5)) drop-shadow(0 8px 20px rgba(0, 0, 0, 0.4));
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .vehicle-image-placeholder {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            background: white;
            color: #6c757d;
            font-size: 4rem;
        }
        
        /* Fahrzeuginformationen */
        .vehicle-info-modern {
            padding: 1.5rem;
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }
        
        .vehicle-brand-model {
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
        }
        
        .vehicle-brand {
            font-size: 1.1rem;
            font-weight: 600;
            color: #003057;
            letter-spacing: 0.3px;
        }
        
        .vehicle-model {
            font-size: 1.5rem;
            font-weight: 700;
            color: #003057;
            letter-spacing: -0.5px;
        }
        
        /* Kennzeichen - Volvo Design Guidelines */
        .vehicle-license-plate {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
            border: 1px solid #e9ecef;
            border-radius: 4px;
            margin-bottom: 0.5rem;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .vehicle-card-modern:hover .vehicle-license-plate {
            background: linear-gradient(135deg, #e3f2fd 0%, #f8f9fa 100%);
            border-color: #0056b3;
        }
        
        .vehicle-card-modern.default .vehicle-license-plate {
            background: linear-gradient(135deg, #e3f2fd 0%, #f0f7ff 100%);
            border-color: #0056b3;
            border-width: 1.5px;
        }
        
        .vehicle-license-plate i {
            color: #0056b3;
            font-size: 1rem;
            flex-shrink: 0;
        }
        
        .license-plate-text {
            font-size: 1.1rem;
            font-weight: 700;
            color: #003057;
            letter-spacing: 0.5px;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            text-transform: uppercase;
        }
        
        /* Statistiken */
        .vehicle-stats {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            margin-top: 0.5rem;
        }
        
        .vehicle-stat-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            color: #495057;
            font-size: 0.95rem;
            font-weight: 500;
        }
        
        .vehicle-stat-item i {
            width: 20px;
            color: #0056b3;
            font-size: 1rem;
        }
        
        /* Aktionen */
        .vehicle-actions-modern {
            display: flex;
            gap: 0.75rem;
            padding: 1rem 1.5rem;
            border-top: 1px solid #e9ecef;
            background: #f8f9fa;
        }
        
        .btn-action-modern {
            flex: 1;
            padding: 0.625rem 1.25rem;
            border: none;
            border-radius: 4px;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            box-shadow: none;
        }
        
        /* Button-Styles bereits oben in Volvo Design System Button Styles definiert */
        
        /* Legacy Styles für Kompatibilität */
        .vehicles-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 1.5rem;
        }
        
        .vehicle-card {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 1.75rem;
            transition: all 0.3s;
            position: relative;
            background: white;
        }
        
        .vehicle-card:hover {
            border-color: #0056b3;
            transform: translateY(-4px);
            box-shadow: 0 8px 24px rgba(0, 86, 179, 0.15);
        }
        
        .vehicle-card.default {
            border-color: #28a745;
            background: linear-gradient(135deg, #d4edda 0%, #e8f5e9 100%);
            box-shadow: var(--shadow-4);
        }
        
        .vehicle-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: #28a745;
            color: white;
            padding: 0.35rem 0.85rem;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            box-shadow: 0 2px 8px rgba(40, 167, 69, 0.3);
        }
        
        .vehicle-license {
            font-size: 1.75rem;
            font-weight: 700;
            color: #003057;
            margin-bottom: 0.75rem;
        }
        
        .vehicle-info {
            color: #6c757d;
            font-size: 0.95rem;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .vehicle-info i {
            width: 20px;
            color: #0056b3;
        }
        
        .vehicle-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid #e9ecef;
        }
        
        .btn-action {
            flex: 1;
            padding: 0.625rem 1.25rem;
            border: none;
            border-radius: 4px;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            box-shadow: none;
        }
        
        /* Button-Styles bereits oben in Volvo Design System Button Styles definiert */
        
        .add-vehicle-card {
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            padding: 2rem;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s;
            min-height: 250px;
            background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
        }
        
        .add-vehicle-card:hover {
            border-color: #0056b3;
            background: linear-gradient(135deg, #e3f2fd 0%, #f8f9fa 100%);
            transform: translateY(-4px);
            box-shadow: 0 8px 24px rgba(0, 86, 179, 0.15);
        }
        
        .add-vehicle-icon {
            font-size: 4rem;
            color: #0056b3;
            margin-bottom: 1rem;
        }
        
        .add-vehicle-text {
            color: #6c757d;
            font-weight: 600;
            font-size: 1.1rem;
        }
        
        /* Modal */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.6);
            backdrop-filter: blur(4px);
            animation: fadeIn 0.3s;
        }
        
        .modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .modal-content {
            background: white;
            border-radius: 8px;
            padding: 2.5rem;
            max-width: 550px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-16);
            animation: slideUp 0.3s;
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .modal-title {
            font-size: 1.75rem;
            font-weight: 700;
            color: #003057;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            transition: all 0.3s;
        }
        
        .close-modal:hover {
            color: #dc3545;
            transform: rotate(90deg);
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            color: #495057;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.875rem 1.25rem;
            border: 2px solid #dee2e6;
            border-radius: 4px;
            font-size: 1rem;
            transition: all 0.3s;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #0056b3;
            box-shadow: 0 0 0 4px rgba(0, 86, 179, 0.1);
        }
        
        .form-actions {
            display: flex;
            gap: 1rem;
        }
        
        .btn {
            flex: 1;
            padding: 1rem;
            border: none;
            border-radius: 4px;
            font-size: 1.05rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        /* Override für spezifische Button-Styles - bereits oben definiert */
        
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: none;
            font-weight: 500;
        }
        
        .alert.show {
            display: block;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        /* Notification Banner */
        .notification-banner {
            position: fixed;
            top: 80px;
            left: 50%;
            transform: translateX(-50%);
            background: white;
            padding: 1.25rem 2rem;
            border-radius: 8px;
            box-shadow: var(--shadow-8);
            z-index: 9999;
            display: none;
            align-items: center;
            gap: 1rem;
            animation: slideDown 0.3s ease-out;
            max-width: 90%;
        }
        
        .notification-banner.show {
            display: flex;
        }
        
        .notification-banner i {
            font-size: 1.5rem;
        }
        
        .notification-banner.success {
            border-left: 4px solid #28a745;
        }
        
        .notification-banner.success i {
            color: #28a745;
        }
        
        .notification-banner.error {
            border-left: 4px solid #dc3545;
        }
        
        .notification-banner.error i {
            color: #dc3545;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateX(-50%) translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateX(-50%) translateY(0);
            }
        }
        
        /* Responsive Styles für Fahrzeugkarten */
        @media (max-width: 992px) {
            .vehicles-grid-modern {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }
        }
        
        @media (max-width: 768px) {
            .vehicles-grid-modern {
                grid-template-columns: 1fr;
                gap: 1.25rem;
            }
            
            .vehicle-card-modern {
                border-radius: 8px;
            }
            
            .vehicle-image-container {
                height: 260px;
                padding: 0;
            }
            
            .vehicle-image {
                width: 100%;
                height: 100%;
            }
            
            .vehicle-info-modern {
                padding: 1.25rem;
            }
            
            .vehicle-brand {
                font-size: 1rem;
            }
            
            .vehicle-model {
                font-size: 1.3rem;
            }
            
            .vehicle-actions-modern {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .btn-action-modern {
                width: 100%;
            }
            
            .vehicle-badge-modern {
                top: 0.75rem;
                right: 0.75rem;
                padding: 0.4rem 0.8rem;
                font-size: 0.75rem;
            }
        }
        
        /* Mobile Optimierung - Umfassend */
        @media (max-width: 768px) {
            /* Profile Container */
            .profile-container {
                padding: 0.75rem;
                margin: 0.5rem auto;
                max-width: 100%;
                overflow-x: hidden;
            }
            
            /* Verhindere horizontales Scrollen */
            body {
                overflow-x: hidden;
                max-width: 100vw;
            }
            
            html {
                overflow-x: hidden;
                max-width: 100vw;
            }
            
            /* Status Banner */
            .status-banner {
                flex-direction: column;
                align-items: flex-start;
                padding: 0.875rem 1rem;
                gap: 0.75rem;
                margin-bottom: 1rem;
            }
            
            .status-banner i {
                font-size: 1.1rem;
            }
            
            .status-banner-title {
                font-size: 0.85rem;
            }
            
            .status-banner-text {
                font-size: 0.8rem;
            }
            
            .status-action {
                width: 100%;
                flex-direction: column;
            }
            
            .status-banner .btn-main-action,
            .status-banner .btn-primary,
            .status-banner .btn-secondary-action {
                width: 100%;
                justify-content: center;
                padding: 0.5rem 0.875rem;
                font-size: 0.8rem;
            }
            
            /* Profile Header Card */
            .profile-header-card {
                padding: 1rem 1.25rem;
                margin-bottom: 1.5rem;
            }
            
            .profile-header-content {
                grid-template-columns: 1fr;
                text-align: center;
                gap: 1rem;
            }
            
            .profile-avatar {
                width: 56px;
                height: 56px;
                font-size: 1.5rem;
                margin: 0 auto;
            }
            
            .profile-name {
                font-size: 1.1rem;
                justify-content: center;
                margin-bottom: 0.5rem;
            }
            
            .profile-email {
                font-size: 0.85rem;
                justify-content: center;
            }
            
            .profile-stats {
                flex-direction: row;
                justify-content: center;
                gap: 0.5rem;
                width: 100%;
            }
            
            .stat-item {
                flex: 1;
                text-align: center;
                padding: 0.5rem 0.5rem;
                min-width: 0;
            }
            
            .stat-value {
                font-size: 1.1rem;
            }
            
            .stat-label {
                font-size: 0.7rem;
            }
            
            /* Section Cards */
            .section-card {
                padding: 1rem 0.875rem;
                margin-bottom: 1rem;
                border-radius: 4px;
            }
            
            .section-title {
                font-size: 1.1rem;
                margin-bottom: 1rem;
                gap: 0.5rem;
            }
            
            .section-title i {
                font-size: 1rem;
            }
            
            /* Vehicles Grid */
            .vehicles-grid-modern {
                grid-template-columns: 1fr;
                gap: 0.75rem;
                margin-top: 1rem;
            }
            
            .vehicles-grid {
                grid-template-columns: 1fr;
                gap: 0.75rem;
            }
            
            .vehicle-card-modern {
                border-radius: 4px;
            }
            
            .vehicle-image-container {
                height: 180px;
            }
            
            .vehicle-info-modern {
                padding: 0.875rem;
                gap: 0.75rem;
            }
            
            .vehicle-brand {
                font-size: 0.85rem;
            }
            
            .vehicle-model {
                font-size: 1.1rem;
            }
            
            .vehicle-license-plate {
                padding: 0.625rem 0.875rem;
                gap: 0.625rem;
                margin-bottom: 0.5rem;
            }
            
            .vehicle-license-plate i {
                font-size: 0.9rem;
            }
            
            .license-plate-text {
                font-size: 0.95rem;
                letter-spacing: 0.3px;
            }
            
            .vehicle-stats {
                gap: 0.5rem;
                margin-top: 0.375rem;
            }
            
            .vehicle-stat-item {
                font-size: 0.85rem;
                gap: 0.5rem;
            }
            
            .vehicle-stat-item i {
                width: 16px;
                font-size: 0.9rem;
            }
            
            .vehicle-actions-modern {
                flex-direction: column;
                gap: 0.5rem;
                padding: 0.875rem;
            }
            
            .btn-action-modern {
                width: 100%;
                padding: 0.5rem 1rem;
                font-size: 0.85rem;
            }
            
            .vehicle-badge-modern {
                top: 0.75rem;
                right: 0.75rem;
                padding: 0.4rem 0.8rem;
                font-size: 0.75rem;
            }
            
            .add-vehicle-card {
                padding: 1rem;
                min-height: 160px;
            }
            
            .add-vehicle-icon {
                font-size: 2.5rem;
                margin-bottom: 0.75rem;
            }
            
            .add-vehicle-text {
                font-size: 0.9rem;
            }
            
            /* Appointment Cards */
            .appointment-card {
                padding: 1.25rem;
                margin-bottom: 1rem;
            }
            
            .appointment-actions {
                flex-direction: column;
                width: 100%;
                gap: 0.5rem;
            }
            
            .appointment-download-btn,
            .appointment-cancel-btn {
                width: 100%;
                justify-content: center;
            }
            
            /* Storage & Offer Cards */
            .storage-card,
            .offer-card {
                padding: 1rem;
            }
            
            /* Modals */
            .modal-content {
                padding: 1.5rem 1.25rem;
                width: 95%;
                max-width: 100%;
                margin: 0.5rem;
            }
            
            #profile-modal .modal-content {
                width: 95%;
                max-width: 100%;
                margin: 0.5rem;
                padding: 1.25rem 1rem;
            }
            
            .modal-header {
                padding: 1rem 1.25rem;
                margin-bottom: 1.5rem;
            }
            
            .modal-title {
                font-size: 1.25rem;
            }
            
            .form-group {
                margin-bottom: 1.25rem;
            }
            
            .form-group label {
                font-size: 0.9rem;
                margin-bottom: 0.5rem;
            }
            
            .form-group input,
            .form-group select,
            .form-group textarea {
                padding: 0.75rem 1rem;
                font-size: 0.9rem;
            }
            
            #profile-modal .form-group input,
            #profile-modal .form-group select {
                padding: 0.875rem 1rem;
                font-size: 0.9rem;
            }
            
            #profile-modal .form-group label {
                font-size: 0.9rem;
                margin-bottom: 0.5rem;
            }
            
            #profile-modal .form-group label i {
                font-size: 0.85rem;
                width: 16px;
            }
            
            .form-actions {
                flex-direction: column;
                gap: 0.75rem;
            }
            
            .form-actions .btn {
                width: 100%;
            }
            
            /* Autohaus Selection */
            .autohaus-selection-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            #profile-modal .autohaus-selection-grid {
                grid-template-columns: 1fr;
            }
            
            .autohaus-selection-card {
                padding: 1.25rem;
            }
            
            .autohaus-card-header h3 {
                font-size: 1rem;
            }
            
            /* Notification Banner */
            .notification-banner {
                top: 60px;
                padding: 1rem 1.25rem;
                left: 5%;
                right: 5%;
                transform: none;
                max-width: 90%;
            }
        }
        
        /* Sehr kleine Geräte (480px und kleiner) */
        @media (max-width: 480px) {
            .profile-container {
                padding: 0.5rem;
                margin: 0.25rem auto;
            }
            
            .profile-header-card {
                padding: 0.75rem 0.875rem;
                margin-bottom: 1rem;
            }
            
            .profile-avatar {
                width: 44px;
                height: 44px;
                font-size: 1.1rem;
            }
            
            .profile-name {
                font-size: 0.95rem;
            }
            
            .profile-email {
                font-size: 0.75rem;
            }
            
            .profile-stats {
                gap: 0.25rem;
            }
            
            .stat-item {
                padding: 0.375rem 0.375rem;
            }
            
            .stat-value {
                font-size: 0.95rem;
            }
            
            .stat-label {
                font-size: 0.6rem;
            }
            
            .section-card {
                padding: 0.875rem 0.75rem;
                margin-bottom: 0.875rem;
            }
            
            .section-title {
                font-size: 1rem;
                margin-bottom: 0.875rem;
            }
            
            .section-title i {
                font-size: 0.9rem;
            }
            
            .vehicle-image-container {
                height: 150px;
            }
            
            .vehicle-info-modern {
                padding: 0.75rem;
            }
            
            .vehicle-brand {
                font-size: 0.8rem;
            }
            
            .vehicle-model {
                font-size: 1rem;
            }
            
            .vehicle-license-plate {
                padding: 0.5rem 0.75rem;
                gap: 0.5rem;
                margin-bottom: 0.375rem;
            }
            
            .vehicle-license-plate i {
                font-size: 0.85rem;
            }
            
            .license-plate-text {
                font-size: 0.875rem;
                letter-spacing: 0.2px;
            }
            
            .vehicle-actions-modern {
                padding: 0.75rem;
            }
            
            .btn-action-modern {
                padding: 0.5rem 0.875rem;
                font-size: 0.8rem;
            }
            
            .add-vehicle-card {
                padding: 0.875rem;
                min-height: 140px;
            }
            
            .add-vehicle-icon {
                font-size: 2rem;
            }
            
            .add-vehicle-text {
                font-size: 0.85rem;
            }
            
            .modal-content {
                padding: 1.25rem 1rem;
                margin: 0.25rem;
            }
            
            .modal-title {
                font-size: 1.1rem;
            }
            
            .form-group input,
            .form-group select,
            .form-group textarea {
                padding: 0.625rem 0.875rem;
                font-size: 0.875rem;
            }
            
            .status-banner {
                padding: 0.75rem 1rem;
            }
            
            .status-banner-title {
                font-size: 0.8rem;
            }
            
            .status-banner-text {
                font-size: 0.75rem;
            }
        }
        
        /* Verbessertes Navigation Design - Volvo Style */
        .nav-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 4px;
            text-decoration: none;
            font-weight: 500;
            font-size: 0.9rem;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            border: 1px solid #dee2e6;
            box-shadow: none;
        }
        
        .nav-link i {
            font-size: 0.9rem;
        }
        
        /* Navigation Links - Volvo Design System */
        .back-link {
            background: #ffffff;
            color: #1a1a1a;
            border-color: #dee2e6;
        }
        
        .back-link:hover,
        .back-link:focus {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .back-link:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .back-link:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .logout-link {
            background: #1a1a1a;
            color: #ffffff;
            border-color: #1a1a1a;
        }
        
        .logout-link:hover,
        .logout-link:focus {
            background: #2d2d2d;
            color: #ffffff;
            border-color: #2d2d2d;
        }
        
        .logout-link:active {
            background: #0d0d0d;
            color: #ffffff;
            border-color: #0d0d0d;
        }
        
        .logout-link:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* User Profile Dropdown - Volvo Design System */
        .user-profile-dropdown {
            position: relative;
        }
        
        .user-profile-trigger {
            background: transparent;
            color: #1a1a1a;
            border: none;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.5rem 1rem;
            cursor: pointer;
            border-radius: 4px;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            font-family: inherit;
            font-size: 0.9rem;
            font-weight: 500;
        }
        
        .user-profile-trigger:hover,
        .user-profile-trigger:focus {
            background: #f8f9fa;
            color: #1a1a1a;
        }
        
        .user-profile-trigger:active {
            background: #e9ecef;
            color: #1a1a1a;
        }
        
        .user-profile-trigger:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .user-profile-trigger.active {
            background: #f8f9fa;
        }
        
        .user-dropdown-icon {
            font-size: 0.75rem;
            color: #6c757d;
            transition: transform 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .user-profile-trigger.active .user-dropdown-icon {
            transform: rotate(180deg);
        }
        
        .user-dropdown-menu {
            position: absolute;
            top: calc(100% + 0.5rem);
            right: 0;
            background: #ffffff;
            border-radius: 4px;
            box-shadow: var(--shadow-8);
            min-width: 200px;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 1000;
            border: 1px solid #e9ecef;
            overflow: hidden;
        }
        
        .user-dropdown-menu.show {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }
        
        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: #1a1a1a;
            text-decoration: none;
            font-size: 0.9rem;
            font-weight: 500;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            border: none;
            background: none;
            width: 100%;
            text-align: left;
        }
        
        .dropdown-item:hover {
            background: #f8f9fa;
            color: #1a1a1a;
        }
        
        .dropdown-item:active {
            background: #e9ecef;
        }
        
        .dropdown-item i {
            width: 18px;
            text-align: center;
            color: #6c757d;
        }
        
        .dropdown-item:hover i {
            color: #0056b3;
        }
        
        /* Logout Item - Rot */
        .logout-item {
            color: #dc3545 !important;
        }
        
        .logout-item i {
            color: #dc3545 !important;
        }
        
        .logout-item:hover {
            background-color: #f8f9fa;
            color: #c82333 !important;
        }
        
        .logout-item:hover i {
            color: #c82333 !important;
        }
        
        /* User Avatar in Navbar - Volvo Design System */
        .user-avatar-nav {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            overflow: hidden;
            background: linear-gradient(135deg, #0056b3 0%, #003057 100%);
            position: relative;
        }
        
        .user-avatar-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 50%;
        }
        
        .user-avatar-initials {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #ffffff;
            font-weight: 600;
            font-size: 0.875rem;
            letter-spacing: 0.01em;
            background: linear-gradient(135deg, #0056b3 0%, #003057 100%);
        }
        
        .user-name-nav {
            font-weight: 500;
            font-size: 0.9rem;
            white-space: nowrap;
        }
        
        /* Mobile Menu Button (Burger Menu) - ENTFERNT, nicht mehr verwendet */
        .mobile-menu-btn {
            display: none !important;
        }
        
        @media (max-width: 768px) {
            .user-name-nav {
                display: none;
            }
            
            /* User-Profile-Dropdown ist immer sichtbar */
            .user-profile-dropdown {
                display: flex;
            }
            
            .user-profile-trigger {
                padding: 0.375rem 0.5rem;
                min-width: auto;
                justify-content: center;
            }
            
            /* User Avatar kleiner auf mobilen Geräten */
            .user-avatar-nav {
                width: 32px;
                height: 32px;
            }
            
            .user-dropdown-menu {
                right: 0;
                left: auto;
            }
            
            /* Navigation */
            .navbar {
                height: auto;
                min-height: 60px;
            }
            
            .nav-container {
                padding: 0.5rem 0.75rem;
                height: auto;
                min-height: 60px;
                flex-wrap: wrap;
            }
            
            .nav-brand {
                flex: 1;
                min-width: 0;
            }
            
            .logo-wrapper {
                gap: 0.5rem;
            }
            
            .nav-logo {
                height: 24px;
            }
            
            .brand-text h1 {
                font-size: 1rem;
            }
            
            .brand-subtitle {
                font-size: 0.65rem;
            }
            
            /* Nav-Menu ist immer sichtbar auf mobilen Geräten */
            .nav-menu {
                position: static;
                flex-direction: row;
                flex-wrap: wrap;
                gap: 0.5rem;
                padding: 0;
                box-shadow: none;
                border: none;
                transform: none;
                opacity: 1;
                visibility: visible;
            }
            
            .nav-link {
                width: auto;
                height: auto;
                padding: 0.5rem 0.75rem;
                justify-content: center;
                font-size: 0.85rem;
            }
            
            .nav-link span {
                display: none;
            }
            
            /* Padding für Body wegen fixed Navbar */
            body {
                padding-top: 60px;
            }
            
            /* Profile Edit Button */
            .profile-edit-btn {
                width: 36px;
                height: 36px;
                font-size: 0.8rem;
            }
            
            /* Tab Buttons */
            .tab-btn {
                padding: 0.5rem 1rem;
                font-size: 0.85rem;
            }
            
            .appointments-tabs {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            /* Appointment Cards Details */
            .appointment-card {
                flex-direction: column;
            }
            
            .appointment-status {
                width: 100%;
                flex-direction: row;
                justify-content: flex-start;
                padding: 0.75rem;
                min-width: auto;
            }
            
            .appointment-status i {
                font-size: 1.25rem;
            }
            
            .appointment-main {
                width: 100%;
            }
            
            .appointment-main h3 {
                font-size: 1.1rem;
                margin-bottom: 0.75rem;
            }
            
            .appointment-details {
                grid-template-columns: 1fr;
                gap: 0.5rem;
                margin-bottom: 0.75rem;
            }
            
            .detail-item {
                font-size: 0.875rem;
                padding: 0.5rem 0;
            }
            
            .appointment-notes {
                padding: 0.75rem;
                font-size: 0.875rem;
                margin-bottom: 0.75rem;
            }
            
            /* Empty States */
            .empty-state {
                padding: 2rem 1rem;
            }
            
            .empty-state i {
                font-size: 3rem;
            }
            
            .empty-state p {
                font-size: 1rem;
            }
            
            /* Storage Tab Empty State */
            #storage-tab .empty-state i[style*="font-size: 4rem"] {
                font-size: 3rem !important;
                margin-bottom: 0.75rem !important;
            }
            
            #storage-tab .empty-state p {
                font-size: 0.95rem;
            }
            
            #storage-tab .empty-state small {
                font-size: 0.85rem;
            }
            
            /* Storage Tab - Reifenlager Mobile Optimierung */
            #storage-tab .section-title {
                font-size: 1.1rem;
                margin-bottom: 1rem;
            }
            
            #storage-tab h3 {
                font-size: 1rem !important;
                margin-bottom: 1rem !important;
                gap: 0.5rem !important;
            }
            
            #storage-tab h3 i {
                font-size: 0.9rem !important;
            }
            
            /* Einlagerungsnummern Cards */
            .storage-card {
                padding: 1rem !important;
                border-radius: 4px !important;
            }
            
            .storage-card > div > div:first-child {
                flex-direction: column !important;
                align-items: flex-start !important;
                gap: 0.75rem !important;
            }
            
            .storage-card > div > div:first-child > div:first-child {
                width: 100%;
            }
            
            .storage-card > div > div:first-child > div:last-child {
                width: 100%;
                text-align: left !important;
                margin-top: 0.5rem;
            }
            
            .storage-card [style*="font-size: 1.5rem"] {
                font-size: 1.25rem !important;
            }
            
            .storage-card [style*="font-size: 0.875rem"] {
                font-size: 0.8rem !important;
            }
            
            .storage-card [style*="font-size: 0.95rem"] {
                font-size: 0.85rem !important;
            }
            
            .storage-card [style*="position: absolute"] {
                position: relative !important;
                top: auto !important;
                right: auto !important;
                margin-bottom: 0.75rem;
            }
            
            .storage-card [style*="display: flex"][style*="align-items: center"][style*="padding: 0.75rem"] {
                padding: 0.625rem !important;
                gap: 0.5rem !important;
            }
            
            .storage-card [style*="font-size: 1rem"] {
                font-size: 0.9rem !important;
            }
            
            .storage-card [style*="font-size: 0.75rem"] {
                font-size: 0.7rem !important;
            }
            
            /* Angebote Cards */
            .offer-card {
                padding: 1rem !important;
                border-radius: 4px !important;
            }
            
            .offer-card > div:first-child {
                flex-direction: column !important;
                gap: 0.75rem !important;
                margin-bottom: 0.75rem !important;
            }
            
            .offer-card [style*="font-size: 1.25rem"] {
                font-size: 1.1rem !important;
            }
            
            .offer-card [style*="font-size: 0.95rem"] {
                font-size: 0.85rem !important;
            }
            
            .offer-card [style*="font-size: 0.875rem"] {
                font-size: 0.8rem !important;
            }
            
            .offer-card [style*="font-size: 1.5rem"][style*="color: #28a745"] {
                font-size: 1.25rem !important;
            }
            
            .offer-card [style*="grid-template-columns: repeat(auto-fit, minmax(200px, 1fr))"] {
                grid-template-columns: 1fr !important;
                gap: 0.75rem !important;
            }
            
            .offer-card [style*="font-size: 0.95rem"][style*="font-weight: 600"] {
                font-size: 0.85rem !important;
            }
            
            .offer-card a[style*="padding: 0.75rem 1.25rem"] {
                width: 100% !important;
                padding: 0.625rem 1rem !important;
                font-size: 0.85rem !important;
                justify-content: center !important;
            }
            
            .offer-card [style*="display: inline-flex"][style*="cursor: not-allowed"] {
                width: 100% !important;
                padding: 0.625rem 1rem !important;
                font-size: 0.85rem !important;
                justify-content: center !important;
            }
            
            /* Security Tab - Sicherheit Mobile Optimierung */
            #security-tab .section-title {
                font-size: 1.1rem;
                margin-bottom: 1rem;
            }
            
            #security-tab .security-section {
                padding: 1rem !important;
            }
            
            #security-tab .security-section .btn {
                width: 100%;
                padding: 0.625rem 1rem;
                font-size: 0.9rem;
            }
            
            #security-tab [style*="margin-top: 2rem"] {
                margin-top: 1.5rem !important;
                padding: 1rem !important;
            }
            
            #security-tab [style*="margin: 0 0 1rem 0"] {
                font-size: 1rem !important;
                margin-bottom: 0.75rem !important;
            }
            
            #security-tab [style*="display: grid; gap: 1rem"] {
                gap: 0.75rem !important;
            }
            
            #security-tab [style*="display: flex; justify-content: space-between"] {
                flex-direction: column !important;
                gap: 0.5rem !important;
                padding: 0.625rem !important;
            }
            
            #security-tab [style*="color: #6c757d"] {
                font-size: 0.85rem !important;
            }
            
            #security-tab strong {
                font-size: 0.9rem !important;
                word-break: break-word;
            }
            
            /* Document Cards */
            .document-card {
                flex-direction: column;
                gap: 1rem;
                padding: 1.25rem;
            }
            
            .document-icon {
                width: 50px;
                height: 50px;
            }
            
            .document-icon i {
                font-size: 1.25rem;
            }
            
            .document-info h4 {
                font-size: 1rem;
            }
            
            .document-info p {
                font-size: 0.85rem;
            }
            
            .document-actions {
                width: 100%;
            }
            
            .document-actions .btn {
                width: 100%;
            }
            
            /* Notification Items */
            .notification-item {
                flex-direction: column;
                gap: 0.75rem;
                padding: 1rem;
            }
            
            .notification-icon {
                width: 40px;
                height: 40px;
            }
            
            .notification-icon i {
                font-size: 1.25rem;
            }
            
            .notification-content h4 {
                font-size: 1rem;
            }
            
            .notification-content p {
                font-size: 0.875rem;
            }
            
            /* Activity Items */
            .activity-item {
                flex-direction: column;
                gap: 0.75rem;
                padding: 1rem;
            }
            
            .activity-icon {
                width: 40px;
                height: 40px;
            }
            
            .activity-icon i {
                font-size: 1.25rem;
            }
            
            .activity-content p {
                font-size: 0.875rem;
            }
            
            /* Security Section */
            .security-section {
                flex-direction: column;
            }
            
            .security-section .btn {
                width: 100%;
            }
        }
        
        @media (max-width: 480px) {
            /* Navigation - Sehr kleine Geräte */
            .navbar {
                min-height: 56px;
            }
            
            .nav-container {
                padding: 0.5rem 0.625rem;
                min-height: 56px;
            }
            
            .nav-logo {
                height: 20px;
            }
            
            .brand-text h1 {
                font-size: 0.9rem;
            }
            
            .brand-subtitle {
                font-size: 0.6rem;
            }
            
            .nav-link {
                width: 36px;
                height: 36px;
                font-size: 0.8rem;
            }
            
            body {
                padding-top: 56px;
            }
            
            /* Tab Buttons - Sehr kleine Geräte */
            .tab-btn {
                padding: 0.5rem 0.75rem;
                font-size: 0.8rem;
            }
            
            .tab-btn i {
                font-size: 0.75rem;
            }
            
            /* Appointment Cards - Sehr kleine Geräte */
            .appointment-card {
                padding: 1rem;
            }
            
            .appointment-status {
                padding: 0.625rem;
            }
            
            .appointment-status i {
                font-size: 1.1rem;
            }
            
            .appointment-main h3 {
                font-size: 1rem;
            }
            
            .appointment-notes {
                padding: 0.625rem;
                font-size: 0.8rem;
            }
            
            .detail-item {
                font-size: 0.8rem;
            }
            
            /* Storage Cards - Sehr kleine Geräte */
            .storage-card,
            .offer-card {
                padding: 0.875rem !important;
            }
            
            #storage-tab .empty-state i[style*="font-size: 4rem"] {
                font-size: 2.5rem !important;
            }
            
            #storage-tab .empty-state p {
                font-size: 0.9rem;
            }
            
            #storage-tab .empty-state small {
                font-size: 0.8rem;
            }
            
            #storage-tab h3 {
                font-size: 0.9rem !important;
            }
            
            .storage-card [style*="font-size: 1.5rem"] {
                font-size: 1rem !important;
            }
            
            .storage-card [style*="font-size: 1rem"] {
                font-size: 0.85rem !important;
            }
            
            .offer-card [style*="font-size: 1.25rem"] {
                font-size: 0.95rem !important;
            }
            
            .offer-card [style*="font-size: 1.5rem"][style*="color: #28a745"] {
                font-size: 1.1rem !important;
            }
            
            /* Security Tab - Sehr kleine Geräte */
            #security-tab [style*="margin-top: 2rem"] {
                margin-top: 1rem !important;
                padding: 0.75rem !important;
            }
            
            #security-tab [style*="margin: 0 0 1rem 0"] {
                font-size: 0.9rem !important;
            }
            
            #security-tab [style*="display: grid; gap: 1rem"] {
                gap: 0.5rem !important;
            }
            
            #security-tab [style*="display: flex; justify-content: space-between"] {
                padding: 0.5rem !important;
            }
            
            #security-tab [style*="color: #6c757d"] {
                font-size: 0.8rem !important;
            }
            
            #security-tab strong {
                font-size: 0.85rem !important;
            }
            
            /* Profile Edit Button - Sehr kleine Geräte */
            .profile-edit-btn {
                width: 32px;
                height: 32px;
                font-size: 0.75rem;
            }
        }
        
        /* Profile Details View */
        .profile-details-view {
            display: grid;
            gap: 1.5rem;
        }
        
        .profile-detail-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.25rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e9ecef;
            transition: all 0.3s;
        }
        
        .profile-detail-item:hover {
            background: #e9ecef;
            border-color: #dee2e6;
        }
        
        .profile-detail-label {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-weight: 500;
            color: #495057;
        }
        
        .profile-detail-label i {
            color: #0056b3;
            font-size: 1.1rem;
        }
        
        .profile-detail-value {
            font-weight: 600;
            color: #003057;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #495057;
        }
        
        .form-group input {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #0056b3;
            box-shadow: 0 0 0 3px rgba(0, 86, 179, 0.1);
        }
        
        /* Profile Edit Button - Volvo Design */
        .profile-edit-btn {
            margin-left: 0.5rem;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            color: #003057;
            border-radius: 4px;
            width: 32px;
            height: 32px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: var(--shadow-1);
            font-size: 0.85rem;
        }
        
        .profile-edit-btn:hover {
            background: #e9ecef;
            border-color: #dee2e6;
            color: #0056b3;
            box-shadow: var(--shadow-2);
            transform: translateY(-1px);
        }
        
        .profile-edit-btn:active {
            transform: translateY(0);
            box-shadow: var(--shadow-1);
        }
        
        .profile-name {
            display: flex;
            align-items: center;
        }
        
        /* Profile Modal */
        #profile-modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            align-items: center;
            justify-content: center;
        }
        
        #profile-modal.active {
            display: flex;
        }
        
        #profile-modal .modal-content {
            background: white;
            border-radius: 8px;
            max-width: 1000px;
            width: 95%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-16);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem 2rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .modal-header h3 {
            margin: 0;
            color: #003057;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #6c757d;
            transition: color 0.3s;
        }
        
        .close-modal:hover {
            color: #003057;
        }
        
        .profile-image-upload {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .profile-image-preview {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            object-fit: cover;
            border: 4px solid #e9ecef;
        }
        
        /* Verbessertes Form-Design für Profil-Modal */
        #profile-modal .form-group {
            margin-bottom: 1.75rem;
        }
        
        #profile-modal .form-group label {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0.75rem;
            color: #003057;
            font-weight: 600;
            font-size: 0.95rem;
            letter-spacing: 0.01em;
        }
        
        #profile-modal .form-group label i {
            color: #0056b3;
            font-size: 0.9rem;
            width: 18px;
            text-align: center;
        }
        
        #profile-modal .form-group input,
        #profile-modal .form-group select {
            width: 100%;
            padding: 1rem 1.25rem;
            border: 2px solid #e9ecef;
            border-radius: 4px;
            font-size: 0.95rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            background-color: #ffffff;
            font-family: inherit;
            color: #495057;
        }
        
        #profile-modal .form-group input:hover,
        #profile-modal .form-group select:hover {
            border-color: #dee2e6;
            background-color: #fafbfc;
        }
        
        #profile-modal .form-group input:focus,
        #profile-modal .form-group select:focus {
            outline: none;
            border-color: #003057;
            background-color: #ffffff;
            box-shadow: 0 0 0 4px rgba(0, 48, 87, 0.1);
            transform: translateY(-1px);
        }
        
        #profile-modal .form-group input:disabled,
        #profile-modal .form-group select:disabled {
            background-color: #f8f9fa;
            color: #6c757d;
            cursor: not-allowed;
            opacity: 0.7;
        }
        
        #profile-modal .form-group input::placeholder,
        #profile-modal .form-group select::placeholder {
            color: #adb5bd;
            opacity: 0.8;
        }
        
        #profile-modal .form-group select {
            cursor: pointer;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 16 16'%3E%3Cpath fill='%23003057' d='M8 11L3 6h10z'/%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 1rem center;
            padding-right: 3rem;
        }
        
        /* Input mit Icon (falls benötigt) */
        #profile-modal .input-with-icon {
            position: relative;
        }
        
        #profile-modal .input-with-icon i {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: #6c757d;
            pointer-events: none;
            z-index: 1;
        }
        
        #profile-modal .input-with-icon input {
            padding-left: 2.75rem;
        }
        
        /* Required Field Indicator */
        #profile-modal .form-group label:has(+ input[required]):after,
        #profile-modal .form-group label:has(+ select[required]):after {
            content: '*';
            color: #dc3545;
            margin-left: 0.25rem;
            font-weight: 700;
        }
        
        /* Label ohne Icon-Stil */
        #profile-modal .form-group label:not(:has(i)) {
            padding-left: 0;
        }
        
        /* Passwort-Modal Styles - Konsistent mit anderen Modals */
        #password-modal form {
            padding: 2rem;
        }
        
        #password-modal .form-group {
            margin-bottom: 1.75rem;
        }
        
        #password-modal .form-group label {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0.75rem;
            color: #003057;
            font-weight: 600;
            font-size: 0.95rem;
            letter-spacing: 0.01em;
        }
        
        #password-modal .form-group label i {
            color: #0056b3;
            font-size: 0.9rem;
            width: 18px;
            text-align: center;
        }
        
        #password-modal .form-group input {
            width: 100%;
            padding: 1rem 1.25rem;
            border: 2px solid #e9ecef;
            border-radius: 4px;
            font-size: 0.95rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            background-color: #ffffff;
            font-family: inherit;
            color: #495057;
        }
        
        #password-modal .form-group input:hover {
            border-color: #dee2e6;
            background-color: #fafbfc;
        }
        
        #password-modal .form-group input:focus {
            outline: none;
            border-color: #003057;
            background-color: #ffffff;
            box-shadow: 0 0 0 4px rgba(0, 48, 87, 0.1);
            transform: translateY(-1px);
        }
        
        #password-modal .form-group input::placeholder {
            color: #adb5bd;
            opacity: 0.8;
        }
        
        #password-modal .form-group small {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-top: 0.5rem;
        }
        
        #password-modal .form-group small i {
            color: #0056b3;
            font-size: 0.875rem;
        }
        
        /* Tab Buttons (Content Switcher) - Volvo Guidelines */
        .tab-btn {
            padding: 0.625rem 1.25rem;
            border: 1px solid #dee2e6;
            background: #ffffff;
            color: #6c757d;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            border-radius: 4px;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            position: relative;
            box-shadow: none;
        }
        
        .tab-btn:hover {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .tab-btn.active {
            background: #1a1a1a;
            color: #ffffff;
            border-color: #1a1a1a;
            font-weight: 500;
        }
        
        .tab-btn:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Appointment Actions Buttons */
        .appointment-actions {
            display: flex;
            gap: 0.75rem;
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid #e9ecef;
            flex-wrap: wrap;
        }
        
        /* Button-Styles bereits oben in Volvo Design System Button Styles definiert */
        .appointment-download-btn,
        .appointment-cancel-btn {
            padding: 0.5rem 1rem;
            font-size: 0.85rem;
            font-weight: 500;
        }
        
        @media (max-width: 768px) {
            .appointment-actions {
                flex-direction: column;
                width: 100%;
            }
            
            .appointment-download-btn,
            .appointment-cancel-btn {
                width: 100%;
                justify-content: center;
            }
        }
        
        /* Storage & Offers Styles */
        .storage-card {
            transition: all 0.3s ease;
        }
        
        .storage-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-4);
        }
        
        .offer-card {
            position: relative;
        }
        
        .offer-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #0056b3 0%, #003057 100%);
            border-radius: 8px 8px 0 0;
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem 2rem;
            color: #6c757d;
        }
        
        .empty-state i {
            display: block;
            margin-bottom: 1rem;
        }
        
        .empty-state p {
            font-size: 1.1rem;
            font-weight: 600;
            color: #495057;
            margin-bottom: 0.5rem;
        }
        
        .empty-state small {
            font-size: 0.9rem;
            display: block;
        }
        
        /* Responsive für Storage Tab - bereits in @media (max-width: 768px) oben definiert */
        
        /* Autohaus Cards wie in login.php */
        .autohaus-selection-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }
        
        /* Im Profil-Modal: Mindestens 2 Spalten für Autohaus-Cards */
        #profile-modal .autohaus-selection-grid {
            grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
        }
        
        @media (min-width: 900px) {
            #profile-modal .autohaus-selection-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }
        
        @media (min-width: 1200px) {
            #profile-modal .autohaus-selection-grid {
                grid-template-columns: repeat(3, 1fr);
            }
        }
        
        .autohaus-selection-card {
            background: white;
            border-radius: 8px;
            padding: 1.5rem;
            box-shadow: var(--shadow-4);
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid transparent;
            position: relative;
            overflow: hidden;
        }
        
        .autohaus-selection-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-8);
            border-color: #003057;
        }
        
        .autohaus-selection-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #003057, #0056b3);
            transform: scaleX(0);
            transition: transform 0.3s ease;
        }
        
        .autohaus-selection-card:hover::before {
            transform: scaleX(1);
        }
        
        .autohaus-selection-card.selected {
            border-color: #003057;
            background: #f8f9ff;
        }
        
        .autohaus-selection-card.selected::before {
            transform: scaleX(1);
        }
        
        .autohaus-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }
        
        .autohaus-card-header h3 {
            margin: 0;
            color: #003057;
            font-size: 1.1rem;
            font-weight: 700;
        }
        
        .autohaus-short {
            background: #003057;
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .autohaus-card-info {
            margin-bottom: 0.5rem;
        }
        
        .autohaus-card-info .info-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 0.5rem;
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .autohaus-card-info .info-item i {
            width: 16px;
            color: #003057;
        }
        
        .autohaus-card-image {
            margin-top: 1rem;
            margin-bottom: 0.5rem;
            border-radius: 8px;
            overflow: hidden;
        }
        
        .autohaus-card-image img {
            width: 100%;
            max-height: 150px;
            object-fit: cover;
            border-radius: 8px;
        }
        
        .autohaus-card-action {
            text-align: center;
            color: #003057;
            font-size: 1.2rem;
            opacity: 0;
            transition: opacity 0.3s ease;
            margin-top: 0.5rem;
        }
        
        .autohaus-selection-card:hover .autohaus-card-action {
            opacity: 1;
        }
        
        .autohaus-selection-card.selected .autohaus-card-action {
            opacity: 1;
            color: #28a745;
        }
        
        /* Autohaus Selection Mobile - bereits in @media (max-width: 768px) oben definiert */
        
        /* Profile Layout - Sidebar + Content */
        .profile-layout {
            display: grid;
            grid-template-columns: 280px 1fr;
            gap: 2rem;
            margin-top: 2rem;
        }
        
        /* Sidebar Navigation - Volvo Design Guidelines */
        .profile-sidebar {
            background: #ffffff;
            border-radius: 4px;
            padding: 1.25rem;
            box-shadow: var(--shadow-4);
            border: 1px solid #e9ecef;
            height: fit-content;
            position: sticky;
            top: 2rem;
        }
        
        .sidebar-nav {
            display: flex;
            flex-direction: column;
            gap: 0.375rem;
        }
        
        .sidebar-nav-item {
            display: flex;
            align-items: center;
            gap: 0.875rem;
            padding: 0.875rem 1rem;
            border: none;
            background: transparent;
            border-radius: 4px;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            text-align: left;
            width: 100%;
            font-family: inherit;
        }
        
        .sidebar-nav-item:hover {
            background: #f8f9fa;
        }
        
        .sidebar-nav-item.active {
            background: #f8f9fa;
            color: #003057;
        }
        
        .sidebar-nav-item.active .sidebar-nav-icon {
            color: #003057;
        }
        
        .sidebar-nav-item.active .sidebar-nav-label {
            color: #003057;
            font-weight: 600;
        }
        
        /* Hover-Effekt für aktive Items - Hintergrund wird etwas dunkler, Text bleibt sichtbar */
        .sidebar-nav-item.active:hover {
            background: #e9ecef;
        }
        
        .sidebar-nav-item.active:hover .sidebar-nav-icon {
            color: #003057;
        }
        
        .sidebar-nav-item.active:hover .sidebar-nav-label {
            color: #003057;
        }
        
        .sidebar-nav-icon {
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
            color: #6c757d;
            flex-shrink: 0;
            transition: color 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .sidebar-nav-item:hover .sidebar-nav-icon {
            color: #003057;
        }
        
        .sidebar-nav-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
            min-width: 0;
        }
        
        .sidebar-nav-label {
            font-size: 0.9rem;
            font-weight: 500;
            color: #495057;
            white-space: nowrap;
            transition: color 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .sidebar-nav-item:hover .sidebar-nav-label {
            color: #003057;
        }
        
        /* Badge System - Volvo Design Guidelines - Dezent & Minimalistisch */
        .sidebar-nav-badge {
            font-size: 0.65rem;
            padding: 0.15rem 0.4rem;
            border-radius: 4px;
            background: #f1f3f5;
            color: #868e96;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            min-width: 16px;
            height: 16px;
            line-height: 1;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        /* Badge mit Wert 0 ausblenden */
        .sidebar-nav-badge[data-count="0"]:not(.badge-alert):not(.badge-warning) {
            display: none;
        }
        
        /* Badge Varianten - Volvo-konform - Dezenter */
        .sidebar-nav-badge.badge-default {
            background: #f1f3f5;
            color: #868e96;
        }
        
        .sidebar-nav-badge.badge-primary {
            background: #e7f3ff;
            color: #0056b3;
        }
        
        .sidebar-nav-badge.badge-highlight {
            background: #e7f3ff;
            color: #0056b3;
        }
        
        .sidebar-nav-badge.badge-warning {
            background: #fff8e1;
            color: #f57c00;
        }
        
        .sidebar-nav-badge.badge-alert {
            background: #ffebee;
            color: #c62828;
        }
        
        .sidebar-nav-badge.badge-success {
            background: #e8f5e9;
            color: #2e7d32;
        }
        
        /* Badge in aktiven Items - Dezenter */
        .sidebar-nav-item.active .sidebar-nav-badge {
            background: #e9ecef;
            color: #6c757d;
        }
        
        .sidebar-nav-item.active .sidebar-nav-badge.badge-primary,
        .sidebar-nav-item.active .sidebar-nav-badge.badge-highlight {
            background: #d0e7ff;
            color: #0056b3;
        }
        
        .sidebar-nav-item.active .sidebar-nav-badge.badge-warning {
            background: #ffe0b2;
            color: #f57c00;
        }
        
        .sidebar-nav-item.active .sidebar-nav-badge.badge-alert {
            background: #ffcdd2;
            color: #c62828;
        }
        
        .sidebar-nav-item.active .sidebar-nav-badge.badge-success {
            background: #c8e6c9;
            color: #2e7d32;
        }
        
        /* Hover-Effekt für Badges in aktiven Items - Minimal */
        .sidebar-nav-item.active:hover .sidebar-nav-badge {
            background: #dee2e6;
            color: #495057;
        }
        
        .sidebar-nav-item.active:hover .sidebar-nav-badge.badge-primary,
        .sidebar-nav-item.active:hover .sidebar-nav-badge.badge-highlight {
            background: #b3d9ff;
            color: #0056b3;
        }
        
        .sidebar-nav-item.active:hover .sidebar-nav-badge.badge-warning {
            background: #ffcc80;
            color: #f57c00;
        }
        
        .sidebar-nav-item.active:hover .sidebar-nav-badge.badge-alert {
            background: #ef9a9a;
            color: #c62828;
        }
        
        .sidebar-nav-item.active:hover .sidebar-nav-badge.badge-success {
            background: #a5d6a7;
            color: #2e7d32;
        }
        
        /* Hover-Effekt für Badges in inaktiven Items - Minimal */
        .sidebar-nav-item:hover .sidebar-nav-badge.badge-default {
            background: #e9ecef;
            color: #6c757d;
        }
        
        .sidebar-nav-item:hover .sidebar-nav-badge.badge-primary,
        .sidebar-nav-item:hover .sidebar-nav-badge.badge-highlight {
            background: #d0e7ff;
            color: #0056b3;
        }
        
        .sidebar-nav-item:hover .sidebar-nav-badge.badge-warning {
            background: #ffe0b2;
            color: #f57c00;
        }
        
        .sidebar-nav-item:hover .sidebar-nav-badge.badge-alert {
            background: #ffcdd2;
            color: #c62828;
        }
        
        .sidebar-nav-item:hover .sidebar-nav-badge.badge-success {
            background: #c8e6c9;
            color: #2e7d32;
        }
        
        /* Profile Content Area */
        .profile-content {
            min-width: 0; /* Prevents grid overflow */
        }
        
        /* Responsive: Mobile Sidebar */
        @media (max-width: 1024px) {
            .profile-layout {
                grid-template-columns: 240px 1fr;
                gap: 1.5rem;
            }
            
            .profile-sidebar {
                padding: 1.25rem;
            }
            
            .sidebar-nav-item {
                padding: 0.875rem 1rem;
            }
            
            .sidebar-nav-icon {
                width: 36px;
                height: 36px;
                font-size: 1.1rem;
            }
        }
        
        @media (max-width: 768px) {
            .profile-layout {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .profile-sidebar {
                position: relative;
                top: 0;
                padding: 0.75rem;
                margin-bottom: 1rem;
            }
            
            .sidebar-nav {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(80px, 1fr));
                gap: 0.5rem;
                overflow-x: visible;
                overflow-y: visible;
                padding-bottom: 0;
            }
            
            .sidebar-nav-item {
                flex-direction: column;
                min-width: 0;
                padding: 0.625rem 0.5rem;
                text-align: center;
                gap: 0.375rem;
            }
            
            .sidebar-nav-item:hover {
                background: #f8f9fa;
            }
            
            .sidebar-nav-icon {
                width: 28px;
                height: 28px;
                font-size: 0.9rem;
                margin: 0 auto;
            }
            
            .sidebar-nav-content {
                align-items: center;
                gap: 0.25rem;
            }
            
            .sidebar-nav-label {
                font-size: 0.7rem;
                white-space: normal;
                text-align: center;
                line-height: 1.1;
                word-break: break-word;
            }
            
            .sidebar-nav-badge {
                font-size: 0.55rem;
                padding: 0.1rem 0.3rem;
                min-width: 12px;
                height: 12px;
                line-height: 1;
            }
        }
        
        @media (max-width: 480px) {
            .profile-sidebar {
                padding: 0.625rem;
            }
            
            .sidebar-nav {
                grid-template-columns: repeat(3, 1fr);
                gap: 0.375rem;
            }
            
            .sidebar-nav-item {
                padding: 0.5rem 0.375rem;
            }
            
            .sidebar-nav-icon {
                width: 24px;
                height: 24px;
                font-size: 0.8rem;
            }
            
            .sidebar-nav-label {
                font-size: 0.65rem;
            }
            
            .sidebar-nav-badge {
                font-size: 0.5rem;
                padding: 0.05rem 0.25rem;
                min-width: 10px;
                height: 10px;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo">
                    <div class="brand-text">
                        <h1><?php echo htmlspecialchars($appName); ?></h1>
                        <span class="brand-subtitle">B&E Gruppe - Mein Profil</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="index.php" class="nav-link back-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Zurück</span>
                </a>
                <?php 
                    $user_profile_image = getUserProfileImage($user);
                    $user_initials = getUserInitials($user);
                    $user_display_name = getUserDisplayName($user);
                    
                    // Prüfe ob der eingeloggte User ein Admin ist
                    $is_admin = false;
                    if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] && isset($_SESSION['user_id'])) {
                        // Lade User-Daten
                        $stmt = $conn->prepare("SELECT id, email, admin_user_id FROM users WHERE id = ? AND is_active = 1");
                        $stmt->execute([$_SESSION['user_id']]);
                        $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($user_data) {
                            // Prüfe ob User direkt mit Admin verknüpft ist
                            if (!empty($user_data['admin_user_id'])) {
                                // Prüfe ob verknüpfter Admin aktiv ist
                                $stmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ? AND is_active = 1");
                                $stmt->execute([$user_data['admin_user_id']]);
                                $admin_user = $stmt->fetch();
                                $is_admin = (bool)$admin_user;
                            } else {
                                // Fallback: Suche nach E-Mail in admin_users
                                $user_email = $user_data['email'] ?? '';
                                if (!empty($user_email)) {
                                    $stmt = $conn->prepare("SELECT id FROM admin_users WHERE email = ? AND is_active = 1");
                                    $stmt->execute([$user_email]);
                                    $admin_user = $stmt->fetch();
                                    $is_admin = (bool)$admin_user;
                                }
                            }
                        }
                    }
                    // Oder prüfe ob der User bereits als Admin eingeloggt ist
                    if (!$is_admin && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
                        $is_admin = true;
                    }
                ?>
                <div class="user-profile-dropdown">
                    <button type="button" class="user-profile-trigger" onclick="toggleUserDropdown(event)">
                        <div class="user-avatar-nav">
                            <?php if ($user_profile_image): ?>
                                <img src="<?php echo htmlspecialchars($user_profile_image); ?>" 
                                     alt="<?php echo htmlspecialchars($user_display_name); ?>" 
                                     class="user-avatar-img"
                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="user-avatar-initials" style="display: none;">
                                    <?php echo htmlspecialchars($user_initials); ?>
                                </div>
                            <?php else: ?>
                                <div class="user-avatar-initials">
                                    <?php echo htmlspecialchars($user_initials); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <span class="user-name-nav"><?php echo htmlspecialchars($user_display_name); ?></span>
                        <i class="fas fa-chevron-down user-dropdown-icon"></i>
                    </button>
                    <div class="user-dropdown-menu" id="userDropdownMenu">
                        <a href="profile.php" class="dropdown-item" style="opacity: 0.6; cursor: default;">
                            <i class="fas fa-user"></i>
                            <span>Profil</span>
                        </a>
                        <?php if ($is_admin): ?>
                        <a href="admin/" class="dropdown-item">
                            <i class="fas fa-cog"></i>
                            <span>Admin</span>
                        </a>
                        <?php endif; ?>
                        <a href="api/logout.php" class="dropdown-item logout-item">
                            <i class="fas fa-sign-out-alt"></i>
                            <span>Abmelden</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <!-- Profile Container -->
    <div class="profile-container">
        <!-- Profile Header -->
        <div class="profile-header-card">
            <div class="profile-header-content">
                <div class="profile-avatar">
                    <?php if (!empty($user['profile_image'])): ?>
                        <img src="<?php echo htmlspecialchars($user['profile_image']); ?>" alt="Profilbild" style="width: 100%; height: 100%; object-fit: cover; border-radius: 50%;">
                    <?php else: ?>
                        <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                    <?php endif; ?>
                </div>
                <div class="profile-info">
                    <div class="profile-name">
                        <?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>
                        <?php if (!$is_suspended): ?>
                        <button class="profile-edit-btn" onclick="openProfileModal()" title="Profil bearbeiten">
                            <i class="fas fa-edit"></i>
                        </button>
                        <?php endif; ?>
                    </div>
                    <div class="profile-email">
                        <i class="fas fa-envelope"></i>
                        <?php echo htmlspecialchars($user['email']); ?>
                    </div>
                </div>
                <div class="profile-stats">
                    <div class="stat-item">
                        <div class="stat-value"><?php echo count($vehicles); ?></div>
                        <div class="stat-label">Fahrzeuge</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value"><?php echo $total_appointments; ?></div>
                        <div class="stat-label">Termine</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value"><?php echo $upcoming_count; ?></div>
                        <div class="stat-label">Bevorstehend</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Suspension Banner - Volvo Design (Höchste Priorität) -->
        <?php if ($is_suspended): ?>
        <div class="status-banner status-suspended" style="background: #c62828; border-left: 4px solid #b71c1c;">
            <i class="fas fa-ban" style="font-size: 1.5rem;"></i>
            <div class="status-banner-content" style="flex: 1;">
                <div class="status-banner-title" style="font-size: 1rem; font-weight: 700; margin-bottom: 0.5rem;">
                    Konto gesperrt
                </div>
                <div class="status-banner-text" style="font-size: 0.9rem; line-height: 1.6;">
                    <?php if ($suspension_details): ?>
                        <div style="margin-bottom: 0.5rem;">
                            <strong>Grund:</strong> <?php echo htmlspecialchars($suspension_details['reason'] ?? 'Nicht angegeben'); ?>
                        </div>
                        <div style="margin-bottom: 0.5rem; opacity: 0.9;">
                            <i class="fas fa-calendar-alt" style="margin-right: 0.5rem;"></i>
                            <strong>Gesperrt am:</strong> <?php echo date('d.m.Y H:i', strtotime($suspension_details['created_at'])); ?>
                        </div>
                        <?php if ($suspension_details['admin_username'] || $suspension_details['admin_email']): ?>
                        <div style="opacity: 0.9;">
                            <i class="fas fa-user-shield" style="margin-right: 0.5rem;"></i>
                            <strong>Von:</strong> <?php echo htmlspecialchars($suspension_details['admin_username'] ?? $suspension_details['admin_email'] ?? 'Administrator'); ?>
                        </div>
                        <?php endif; ?>
                    <?php else: ?>
                        <div>Ihr Konto wurde von einem Administrator gesperrt. Bitte kontaktieren Sie den Support für weitere Informationen.</div>
                    <?php endif; ?>
                </div>
            </div>
            <div class="status-action">
                <a href="index.php?contact=support" class="btn-main-action" style="white-space: nowrap;">
                    <i class="fas fa-envelope"></i>
                    Support kontaktieren
                </a>
            </div>
        </div>
        <?php endif; ?>

        <!-- Status Banner - Volvo Design -->
        <?php if (!$is_suspended && ($verification_status === 'pending' || $verification_status === 'rejected' || $show_verification_banner)): ?>
        <div class="status-banner status-<?php echo $current_status['type']; ?>">
            <i class="<?php echo $current_status['icon']; ?>"></i>
            <div class="status-banner-content">
                <div class="status-banner-title"><?php echo $current_status['text']; ?></div>
                <div class="status-banner-text">
                    <?php if ($verification_status === 'pending'): ?>
                        Bitte laden Sie ein Identitätsdokument hoch, um Ihr Konto zu verifizieren.
                    <?php elseif ($verification_status === 'rejected'): ?>
                        Grund: <?php echo htmlspecialchars($user['verification_reason'] ?? 'Nicht angegeben'); ?>
                    <?php else: ?>
                        Ihr Konto ist verifiziert und aktiv. Sie können alle Funktionen nutzen.
                    <?php endif; ?>
                </div>
            </div>
            <?php if ($verification_status === 'pending' || $verification_status === 'rejected'): ?>
            <div class="status-action">
                <a href="verification_pending.php" class="btn-main-action">
                    <i class="fas fa-file-upload"></i>
                    <?php echo $verification_status === 'rejected' ? 'Neues Dokument hochladen' : 'Dokument hochladen'; ?>
                </a>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        
        <?php if (!$is_suspended && $has_missing_fields): ?>
        <div class="status-banner status-warning">
            <i class="fas fa-exclamation-triangle"></i>
            <div class="status-banner-content">
                <div class="status-banner-title">Profil unvollständig</div>
                <div class="status-banner-text">
                    Bitte vervollständigen Sie Ihr Profil. Fehlende Pflichtangaben: 
                    <strong><?php 
                        $field_labels = [
                            'phone_number' => 'Telefonnummer',
                            'address' => 'Adresse',
                            'postal_code' => 'Postleitzahl',
                            'city' => 'Stadt'
                        ];
                        $missing_labels = array_map(function($field) use ($field_labels) {
                            return $field_labels[$field] ?? $field;
                        }, $missing_fields);
                        echo htmlspecialchars(implode(', ', $missing_labels));
                    ?></strong>
                </div>
            </div>
            <?php if (!$is_suspended): ?>
            <div class="status-action">
                <a href="#profile-modal" onclick="openProfileModal(); return false;" class="btn-main-action">
                    <i class="fas fa-edit"></i>
                    Profil bearbeiten
                </a>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <!-- Alerts -->
        <div id="alert-message" class="alert"></div>
        
        <!-- Feedback-Banner wenn von Push-Benachrichtigung -->
        <?php if ($from_push_notification): ?>
        <div class="feedback-banner" id="feedback-banner">
            <div class="feedback-banner-content">
                <div class="feedback-banner-icon">
                    <i class="fas fa-info-circle"></i>
                </div>
                <div class="feedback-banner-text">
                    <strong>Willkommen zurück!</strong>
                    <?php if ($has_missing_fields): ?>
                        <p>Bitte vervollständigen Sie Ihr Profil, um alle Funktionen nutzen zu können.</p>
                    <?php else: ?>
                        <p>Ihr Profil ist vollständig. Sie können jetzt alle Funktionen nutzen!</p>
                    <?php endif; ?>
                </div>
                <button class="feedback-banner-close" onclick="closeFeedbackBanner()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
        <?php endif; ?>

        <!-- Profile Layout: Sidebar + Content -->
        <div class="profile-layout">
            <!-- Sidebar Navigation -->
            <aside class="profile-sidebar">
                <nav class="sidebar-nav">
                    <button class="sidebar-nav-item active" onclick="showProfileTab('vehicles')" data-tab="vehicles">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-car"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Fahrzeuge</span>
                            <small class="sidebar-nav-badge badge-default" data-count="<?php echo count($vehicles); ?>"><?php echo count($vehicles); ?></small>
                        </div>
                    </button>
                    <button class="sidebar-nav-item" onclick="showProfileTab('appointments')" data-tab="appointments">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-calendar-alt"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Termine</span>
                            <small class="sidebar-nav-badge badge-default" data-count="<?php echo $total_appointments; ?>"><?php echo $total_appointments; ?></small>
                        </div>
                    </button>
                    <button class="sidebar-nav-item" onclick="showProfileTab('storage')" data-tab="storage">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-warehouse"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Reifenlager</span>
                            <?php 
                            $storage_count = count($user_offers) + count($storage_numbers);
                            $storage_badge_class = $storage_count > 0 ? 'badge-primary' : 'badge-default';
                            ?>
                            <small class="sidebar-nav-badge <?php echo $storage_badge_class; ?>" data-count="<?php echo $storage_count; ?>"><?php echo $storage_count; ?></small>
                        </div>
                    </button>
                    <button class="sidebar-nav-item" onclick="showProfileTab('notifications')" data-tab="notifications">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-bell"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Benachrichtigungen</span>
                            <?php 
                            $notif_badge_class = $notifications_count > 0 ? 'badge-alert' : 'badge-default';
                            ?>
                            <small class="sidebar-nav-badge <?php echo $notif_badge_class; ?>" data-count="<?php echo $notifications_count; ?>"><?php echo $notifications_count; ?></small>
                        </div>
                    </button>
                    <button class="sidebar-nav-item" onclick="showProfileTab('activity')" data-tab="activity">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-history"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Aktivität</span>
                        </div>
                    </button>
                    <?php if ($verification_document): ?>
                    <button class="sidebar-nav-item" onclick="showProfileTab('documents')" data-tab="documents">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-file-alt"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Dokumente</span>
                        </div>
                    </button>
                    <?php endif; ?>
                    <button class="sidebar-nav-item" onclick="showProfileTab('security')" data-tab="security">
                        <div class="sidebar-nav-icon">
                            <i class="fas fa-shield-alt"></i>
                        </div>
                        <div class="sidebar-nav-content">
                            <span class="sidebar-nav-label">Sicherheit</span>
                        </div>
                    </button>
                </nav>
            </aside>

            <!-- Main Content Area -->
            <main class="profile-content">

        <!-- Fahrzeuge Tab Content -->
        <div id="vehicles-tab" class="tab-content active">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-car"></i>
                Meine Fahrzeuge
            </h2>
            
            <div class="vehicles-grid-modern">
                <?php if (empty($vehicles)): ?>
                    <?php if (!$is_suspended): ?>
                    <div class="add-vehicle-card" onclick="openVehicleModal()">
                        <div class="add-vehicle-icon">
                            <i class="fas fa-plus-circle"></i>
                        </div>
                        <div class="add-vehicle-text">Fahrzeug hinzufügen</div>
                    </div>
                    <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-lock" style="font-size: 3rem; color: #c62828; margin-bottom: 1rem;"></i>
                        <p>Ihr Konto ist gesperrt. Sie können keine Fahrzeuge hinzufügen.</p>
                    </div>
                    <?php endif; ?>
                <?php else: ?>
                    <?php foreach ($vehicles as $vehicle): ?>
                        <div class="vehicle-card-modern <?php echo $vehicle['is_default'] ? 'default' : ''; ?>">
                            <!-- Standardfahrzeug Badge -->
                            <?php if ($vehicle['is_default']): ?>
                                <div class="vehicle-badge-modern">
                                    <i class="fas fa-check-circle"></i>
                                    <span>Standardfahrzeug</span>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Fahrzeugbild -->
                            <div class="vehicle-image-container">
                                <?php if (!empty($vehicle['image_path'])): ?>
                                    <img src="<?php echo htmlspecialchars($vehicle['image_path']); ?>" 
                                         alt="<?php echo htmlspecialchars(($vehicle['vehicle_brand'] ?? '') . ' ' . ($vehicle['vehicle_model'] ?? '')); ?>"
                                         class="vehicle-image"
                                         onerror="this.onerror=null; this.src='assets/images/defaultcar.png';">
                                <?php else: ?>
                                    <img src="assets/images/defaultcar.png" 
                                         alt="Fahrzeugbild"
                                         class="vehicle-image"
                                         onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                    <div class="vehicle-image-placeholder" style="display: none;">
                                        <i class="fas fa-car"></i>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Fahrzeuginformationen -->
                            <div class="vehicle-info-modern">
                                <?php if (!empty($vehicle['license_plate'])): ?>
                                <div class="vehicle-license-plate">
                                    <i class="fas fa-car"></i>
                                    <span class="license-plate-text"><?php echo htmlspecialchars($vehicle['license_plate']); ?></span>
                                </div>
                                <?php endif; ?>
                                <div class="vehicle-brand-model">
                                    <?php if ($vehicle['vehicle_brand']): ?>
                                        <span class="vehicle-brand"><?php echo htmlspecialchars($vehicle['vehicle_brand']); ?></span>
                                    <?php endif; ?>
                                    <?php if ($vehicle['vehicle_model']): ?>
                                        <span class="vehicle-model"><?php echo htmlspecialchars($vehicle['vehicle_model']); ?></span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <!-- Aktionen -->
                            <div class="vehicle-actions-modern">
                                <?php if (!$is_suspended): ?>
                                <button class="btn-action-modern btn-edit" onclick="openVehicleModal(<?php echo $vehicle['id']; ?>)" title="Fahrzeug bearbeiten">
                                    <i class="fas fa-edit"></i>
                                    <span>Bearbeiten</span>
                                </button>
                                <?php if (!$vehicle['is_default']): ?>
                                    <button class="btn-action-modern btn-delete" onclick="deleteVehicle(<?php echo $vehicle['id']; ?>)" title="Fahrzeug löschen">
                                        <i class="fas fa-trash"></i>
                                        <span>Löschen</span>
                                    </button>
                                <?php endif; ?>
                            <?php else: ?>
                                <div style="padding: 0.625rem 1.25rem; background: #f8f9fa; border-radius: 4px; color: #6c757d; font-size: 0.85rem; text-align: center;">
                                    <i class="fas fa-lock"></i> Bearbeitung gesperrt
                                </div>
                            <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    <?php if (!$is_suspended): ?>
                    <div class="add-vehicle-card" onclick="openVehicleModal()">
                        <div class="add-vehicle-icon">
                            <i class="fas fa-plus-circle"></i>
                        </div>
                        <div class="add-vehicle-text">Weiteres Fahrzeug hinzufügen</div>
                    </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Appointments Tab Content -->
        <div id="appointments-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-calendar-alt"></i>
                Meine Termine
            </h2>
            
            <div class="appointments-list">
                <?php if (empty($appointments)): ?>
                    <div class="empty-state">
                        <i class="fas fa-calendar-times"></i>
                        <p>Sie haben noch keine Termine gebucht.</p>
                        <a href="index.php" class="btn btn-primary">
                            <i class="fas fa-plus"></i>
                            Termin buchen
                        </a>
                    </div>
                <?php else: ?>
                    <div class="appointments-tabs" style="display: flex; gap: 1rem; margin-bottom: 1.5rem; border-bottom: 2px solid #e9ecef;">
                        <button class="tab-btn active" onclick="showAppointmentTab('all')" data-tab="all">
                            <i class="fas fa-list"></i> Alle (<?php echo $total_appointments; ?>)
                        </button>
                        <button class="tab-btn" onclick="showAppointmentTab('upcoming')" data-tab="upcoming">
                            <i class="fas fa-clock"></i> Bevorstehend (<?php echo $upcoming_count; ?>)
                        </button>
                        <button class="tab-btn" onclick="showAppointmentTab('past')" data-tab="past">
                            <i class="fas fa-history"></i> Vergangene (<?php echo $past_count; ?>)
                        </button>
                    </div>
                    
                    <div id="appointments-content">
                        <?php foreach ($appointments as $apt): ?>
                            <?php 
                            // Zeit korrekt formatieren - Priorität: slot_time > appointment_time > event_time
                            $appointment_time = $apt['slot_time'] ?? $apt['appointment_time'] ?? $apt['event_time'] ?? '';
                            
                            if (!empty($appointment_time)) {
                                // Wenn die Zeit im Format HH:MM:SS ist, nur HH:MM nehmen
                                if (strlen($appointment_time) >= 5) {
                                    $time_formatted = substr($appointment_time, 0, 5); // Erste 5 Zeichen (HH:MM)
                                } else {
                                    $time_formatted = $appointment_time;
                                }
                            } else {
                                $time_formatted = '';
                            }
                            
                            // Datum: slot_date hat Priorität über event_date
                            $appointment_date = $apt['slot_date'] ?? $apt['event_date'] ?? '';
                            $date_formatted = !empty($appointment_date) ? date('d.m.Y', strtotime($appointment_date)) : '';
                            
                            // Für is_past Vergleich: Datum und Zeit kombinieren
                            $datetime_string = $appointment_date . ' ' . ($appointment_time ?: '00:00:00');
                            $is_past = strtotime($datetime_string) < time();
                            ?>
                            <div class="appointment-card <?php echo $is_past ? 'past' : ''; ?>" data-date="<?php echo $appointment_date; ?>">
                                <div class="appointment-status">
                                    <?php if ($is_past): ?>
                                        <i class="fas fa-check-circle" style="color: #6c757d;"></i>
                                        <span>Abgeschlossen</span>
                                    <?php else: ?>
                                        <i class="fas fa-clock" style="color: #28a745;"></i>
                                        <span>Bevorstehend</span>
                                    <?php endif; ?>
                                </div>
                                <div class="appointment-main">
                                    <h3><?php echo htmlspecialchars($apt['event_name']); ?></h3>
                                    <div class="appointment-details">
                                        <div class="detail-item">
                                            <i class="fas fa-calendar"></i>
                                            <span><?php echo $date_formatted; ?></span>
                                        </div>
                                        <?php if (!empty($time_formatted)): ?>
                                        <div class="detail-item">
                                            <i class="fas fa-clock"></i>
                                            <span><?php echo htmlspecialchars($time_formatted); ?> Uhr</span>
                                        </div>
                                        <?php endif; ?>
                                        <div class="detail-item">
                                            <i class="fas fa-building"></i>
                                            <span><?php echo htmlspecialchars($apt['autohaus_name']); ?></span>
                                        </div>
                                        <?php if ($apt['autohaus_phone']): ?>
                                        <div class="detail-item">
                                            <i class="fas fa-phone"></i>
                                            <span><?php echo htmlspecialchars($apt['autohaus_phone']); ?></span>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ($apt['notes']): ?>
                                        <div class="appointment-notes">
                                            <i class="fas fa-sticky-note"></i>
                                            <?php echo htmlspecialchars($apt['notes']); ?>
                                        </div>
                                    <?php endif; ?>
                                    <div class="appointment-actions">
                                        <a href="download_appointment_ics.php?id=<?php echo $apt['id']; ?>&_session=<?php echo session_id(); ?>" class="appointment-download-btn">
                                            <i class="fas fa-download"></i>
                                            In Kalender eintragen
                                        </a>
                                        <?php if (!$is_past): ?>
                                            <button class="appointment-cancel-btn" onclick="cancelAppointment(<?php echo $apt['id']; ?>)">
                                                <i class="fas fa-times"></i>
                                                Termin stornieren
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Storage Tab Content - Reifenlager & Angebote -->
        <div id="storage-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-warehouse"></i>
                Reifenlager & Angebote
            </h2>
            
            <?php if (empty($user_offers) && empty($storage_numbers)): ?>
                <div class="empty-state">
                    <i class="fas fa-warehouse" style="font-size: 4rem; color: #dee2e6; margin-bottom: 1rem;"></i>
                    <p>Noch keine Reifen eingelagert oder Angebote vorhanden</p>
                    <small style="color: #6c757d;">Ihre Einlagerungsinformationen und Angebote werden hier angezeigt, sobald Sie einen Termin mit Einlagerung gebucht haben.</small>
                </div>
            <?php else: ?>
                <!-- Einlagerungsnummern Section -->
                <?php if (!empty($storage_numbers)): ?>
                <div style="margin-bottom: 2.5rem;">
                    <h3 style="font-size: 1.25rem; font-weight: 600; color: #003057; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 0.75rem;">
                        <i class="fas fa-box" style="color: #0056b3;"></i>
                        Einlagerungsnummern
                    </h3>
                    
                    <div style="display: grid; gap: 1rem;">
                        <?php foreach ($storage_numbers as $index => $storage): ?>
                            <?php 
                            $is_latest = $index === 0;
                            ?>
                            <div class="storage-card" style="background: <?php echo $is_latest ? 'linear-gradient(135deg, #e3f2fd 0%, #f8f9fa 100%)' : '#f8f9fa'; ?>; border: 2px solid <?php echo $is_latest ? '#0056b3' : '#dee2e6'; ?>; border-radius: 8px; padding: 1.5rem; position: relative; transition: all 0.3s;">
                                <?php if ($is_latest): ?>
                                    <div style="position: absolute; top: 1rem; right: 1rem; background: #0056b3; color: white; padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.75rem; font-weight: 600;">
                                        <i class="fas fa-star"></i> Aktuellste
                                    </div>
                                <?php endif; ?>
                                
                                <div style="display: grid; gap: 0.75rem;">
                                    <div style="display: flex; justify-content: space-between; align-items: center;">
                                        <div>
                                            <div style="font-size: 0.875rem; color: #6c757d; margin-bottom: 0.25rem;">Einlagerungsnummer</div>
                                            <div style="font-size: 1.5rem; font-weight: 700; color: #003057;"><?php echo htmlspecialchars($storage['storage_number']); ?></div>
                                        </div>
                                        <?php if (!empty($storage['autohaus_name'])): ?>
                                            <div style="text-align: right;">
                                                <div style="font-size: 0.875rem; color: #6c757d; margin-bottom: 0.25rem;">Autohaus</div>
                                                <div style="font-size: 0.95rem; font-weight: 600; color: #0056b3;">
                                                    <i class="fas fa-building"></i> <?php echo htmlspecialchars($storage['autohaus_name']); ?>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <?php if (!empty($storage['license_plate'])): ?>
                                    <div style="display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; background: white; border-radius: 8px;">
                                        <i class="fas fa-car" style="color: #0056b3;"></i>
                                        <div>
                                            <div style="font-size: 0.75rem; color: #6c757d;">Kennzeichen</div>
                                            <div style="font-size: 1rem; font-weight: 600; color: #495057;"><?php echo htmlspecialchars($storage['license_plate']); ?></div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($storage['appointment_date'])): ?>
                                    <div style="display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; background: white; border-radius: 8px;">
                                        <i class="fas fa-calendar" style="color: #0056b3;"></i>
                                        <div>
                                            <div style="font-size: 0.75rem; color: #6c757d;">Termindatum</div>
                                            <div style="font-size: 1rem; font-weight: 600; color: #495057;"><?php echo date('d.m.Y', strtotime($storage['appointment_date'])); ?></div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Angebote Section -->
                <?php if (!empty($user_offers)): ?>
                <div>
                    <h3 style="font-size: 1.25rem; font-weight: 600; color: #003057; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 0.75rem;">
                        <i class="fas fa-file-invoice-dollar" style="color: #0056b3;"></i>
                        Reifenangebote (<?php echo count($user_offers); ?>)
                    </h3>
                    
                    <div style="display: grid; gap: 1.25rem;">
                        <?php foreach ($user_offers as $offer): ?>
                            <?php
                            $status = $offer['status'] ?? 'draft';
                            $statusConfig = [
                                'draft' => ['label' => 'Entwurf', 'color' => '#6c757d', 'bg' => '#f8f9fa', 'icon' => 'fa-file'],
                                'sent' => ['label' => 'Gesendet', 'color' => '#0d6efd', 'bg' => '#cfe2ff', 'icon' => 'fa-paper-plane'],
                                'viewed' => ['label' => 'Angesehen', 'color' => '#0dcaf0', 'bg' => '#cff4fc', 'icon' => 'fa-eye'],
                                'accepted' => ['label' => 'Angenommen', 'color' => '#198754', 'bg' => '#d1e7dd', 'icon' => 'fa-check-circle'],
                                'declined' => ['label' => 'Abgelehnt', 'color' => '#dc3545', 'bg' => '#f8d7da', 'icon' => 'fa-times-circle'],
                                'new_request' => ['label' => 'Neue Anfrage', 'color' => '#ffc107', 'bg' => '#fff3cd', 'icon' => 'fa-exclamation-triangle'],
                                'processed' => ['label' => 'Bearbeitet', 'color' => '#6610f2', 'bg' => '#e7d6ff', 'icon' => 'fa-check-double']
                            ];
                            $config = $statusConfig[$status] ?? ['label' => ucfirst($status), 'color' => '#6c757d', 'bg' => '#f8f9fa', 'icon' => 'fa-file'];
                            
                            $offerData = !empty($offer['offer_data']) ? json_decode($offer['offer_data'], true) : [];
                            $totalAmount = $offer['total_amount'] ?? 0;
                            ?>
                            
                            <?php 
                            // Prüfe ob Angebot deaktiviert ist
                            $disabledStatuses = ['accepted', 'declined', 'new_request'];
                            $isOfferDisabled = in_array($status, $disabledStatuses);
                            ?>
                            
                            <div class="offer-card" style="background: white; border: 2px solid <?php echo $config['color']; ?>; border-radius: 8px; padding: 1.75rem; box-shadow: var(--shadow-4); transition: all 0.3s; cursor: <?php echo $isOfferDisabled ? 'default' : 'pointer'; ?>;" 
                                 <?php if (!$isOfferDisabled && !empty($offer['token'])): ?>
                                 onclick="window.open('wheelset_offer_selection.php?token=<?php echo urlencode($offer['token']); ?>', '_blank')"
                                 <?php endif; ?>
                                 onmouseover="<?php if (!$isOfferDisabled): ?>this.style.transform='translateY(-4px)'; this.style.boxShadow='0 8px 20px rgba(0,0,0,0.12)';<?php endif; ?>"
                                 onmouseout="<?php if (!$isOfferDisabled): ?>this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(0,0,0,0.08)';<?php endif; ?>">
                                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                                    <div style="flex: 1;">
                                        <div style="display: flex; align-items: center; gap: 0.75rem; margin-bottom: 0.5rem;">
                                            <div style="padding: 0.5rem 1rem; background: <?php echo $config['bg']; ?>; color: <?php echo $config['color']; ?>; border-radius: 4px; font-weight: 600; font-size: 0.875rem; display: flex; align-items: center; gap: 0.5rem;">
                                                <i class="fas <?php echo $config['icon']; ?>"></i>
                                                <?php echo htmlspecialchars($config['label']); ?>
                                            </div>
                                            <?php if ($offer['matched_by']): ?>
                                                <span style="font-size: 0.75rem; color: #6c757d; padding: 0.25rem 0.5rem; background: #f8f9fa; border-radius: 4px;">
                                                    <i class="fas fa-<?php echo $offer['matched_by'] === 'storage_number' ? 'box' : 'car'; ?>"></i>
                                                    Gefunden via <?php echo $offer['matched_by'] === 'storage_number' ? 'Einlagerungsnummer' : 'Kennzeichen'; ?>
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div style="font-size: 1.25rem; font-weight: 700; color: #003057; margin-bottom: 0.5rem;">
                                            Satznummer: <?php echo htmlspecialchars($offer['wheelset_number'] ?? '-'); ?>
                                        </div>
                                        
                                        <?php if (!empty($offer['autohaus_name'])): ?>
                                        <div style="font-size: 0.95rem; color: #6c757d; display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                            <i class="fas fa-building"></i>
                                            <?php echo htmlspecialchars($offer['autohaus_name']); ?>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <?php if ($totalAmount > 0): ?>
                                    <div style="text-align: right;">
                                        <div style="font-size: 0.75rem; color: #6c757d; margin-bottom: 0.25rem;">Gesamtbetrag</div>
                                        <div style="font-size: 1.5rem; font-weight: 700; color: #28a745;">
                                            <?php echo number_format($totalAmount, 2, ',', '.'); ?> €
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                                
                                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-top: 1rem; padding-top: 1rem; border-top: 1px solid #e9ecef;">
                                    <?php if (!empty($offer['license_plate'])): ?>
                                    <div>
                                        <div style="font-size: 0.75rem; color: #6c757d; margin-bottom: 0.25rem;">Kennzeichen</div>
                                        <div style="font-size: 0.95rem; font-weight: 600; color: #495057;">
                                            <i class="fas fa-car"></i> <?php echo htmlspecialchars($offer['license_plate']); ?>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($offer['created_at'])): ?>
                                    <div>
                                        <div style="font-size: 0.75rem; color: #6c757d; margin-bottom: 0.25rem;">Erstellt am</div>
                                        <div style="font-size: 0.95rem; font-weight: 600; color: #495057;">
                                            <i class="fas fa-calendar"></i> <?php echo date('d.m.Y', strtotime($offer['created_at'])); ?>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($offer['email_sent_at'])): ?>
                                    <div>
                                        <div style="font-size: 0.75rem; color: #6c757d; margin-bottom: 0.25rem;">E-Mail gesendet</div>
                                        <div style="font-size: 0.95rem; font-weight: 600; color: #495057;">
                                            <i class="fas fa-envelope"></i> <?php echo date('d.m.Y H:i', strtotime($offer['email_sent_at'])); ?>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php 
                                    // Nur Link anzeigen wenn Status nicht accepted, declined oder new_request ist
                                    $canViewOffer = !empty($offer['token']) && !$isOfferDisabled;
                                    ?>
                                    
                                    <?php if ($canViewOffer): ?>
                                    <div style="grid-column: 1 / -1; margin-top: 0.5rem;">
                                        <a href="wheelset_offer_selection.php?token=<?php echo urlencode($offer['token']); ?>" 
                                           target="_blank"
                                           style="display: inline-flex; align-items: center; gap: 0.5rem; padding: 0.75rem 1.25rem; background: #0056b3; color: white; border-radius: 4px; text-decoration: none; font-weight: 600; font-size: 0.95rem; transition: all 0.3s;"
                                           onmouseover="this.style.background='#003057'; this.style.transform='translateY(-2px)';"
                                           onmouseout="this.style.background='#0056b3'; this.style.transform='translateY(0)';">
                                            <i class="fas fa-external-link-alt"></i>
                                            Angebot ansehen & auswählen
                                        </a>
                                    </div>
                                    <?php elseif (!empty($offer['token']) && $isOfferDisabled): ?>
                                    <div style="grid-column: 1 / -1; margin-top: 0.5rem;">
                                        <div style="display: inline-flex; align-items: center; gap: 0.5rem; padding: 0.75rem 1.25rem; background: #e9ecef; color: #6c757d; border-radius: 4px; font-weight: 600; font-size: 0.95rem; cursor: not-allowed;">
                                            <i class="fas fa-lock"></i>
                                            <?php 
                                            $statusMessages = [
                                                'accepted' => 'Angebot wurde bereits angenommen',
                                                'declined' => 'Angebot wurde abgelehnt',
                                                'new_request' => 'Neue Anfrage wurde gestellt'
                                            ];
                                            echo $statusMessages[$status] ?? 'Angebot nicht mehr verfügbar';
                                            ?>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (empty($storage_numbers) && !empty($user_offers)): ?>
                    <div style="margin-top: 2rem; padding: 1.5rem; background: #e3f2fd; border-radius: 8px; border-left: 4px solid #0056b3;">
                        <div style="display: flex; align-items: start; gap: 1rem;">
                            <i class="fas fa-info-circle" style="color: #0056b3; font-size: 1.5rem; margin-top: 0.25rem;"></i>
                            <div>
                                <div style="font-weight: 600; color: #003057; margin-bottom: 0.5rem;">Information</div>
                                <div style="color: #495057; font-size: 0.95rem;">
                                    Es wurden Angebote für Ihre Fahrzeuge gefunden. Ihre Einlagerungsnummern werden hier angezeigt, sobald Sie einen Termin mit Einlagerung gebucht haben.
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        </div>

        <!-- Documents Tab Content -->
        <div id="documents-tab" class="tab-content">
        <?php if ($verification_document): ?>
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-file-alt"></i>
                Meine Dokumente
            </h2>
            
            <div class="document-list">
                <div class="document-card">
                    <div class="document-icon">
                        <i class="fas fa-id-card"></i>
                    </div>
                    <div class="document-info">
                        <h4>Verifizierungsdokument</h4>
                        <p>Hochgeladen am <?php echo date('d.m.Y', strtotime($user['created_at'])); ?></p>
                        <small style="color: #6c757d;">Status: <?php echo $verification_status === 'approved' ? '✓ Verifiziert' : ($verification_status === 'rejected' ? '✗ Abgelehnt' : '🔄 Wird bearbeitet'); ?></small>
                    </div>
                    <div class="document-actions">
                        <a href="<?php echo htmlspecialchars($verification_document); ?>" target="_blank" class="btn-action btn-action-edit">
                            <i class="fas fa-eye"></i>
                            <span>Ansehen</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php else: ?>
        <div class="section-card">
            <div class="empty-state">
                <i class="fas fa-file-alt"></i>
                <p>Keine Dokumente vorhanden</p>
            </div>
        </div>
        <?php endif; ?>
        </div>

        <!-- Notifications Tab Content -->
        <div id="notifications-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-bell"></i>
                Benachrichtigungen
            </h2>
            
            <!-- Push-Benachrichtigungen Einstellungen - Dezenter -->
            <div style="margin-bottom: 2rem; padding: 1rem; background: #ffffff; border-radius: 4px; border: 1px solid #e9ecef;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.75rem;">
                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <i class="fas fa-bell" style="color: #6c757d; font-size: 0.9rem;"></i>
                        <span style="font-weight: 500; color: #495057; font-size: 0.9rem;">Push-Benachrichtigungen</span>
                    </div>
                    <div id="push-status-icon" style="font-size: 1rem;">
                        <i class="fas fa-spinner fa-spin" style="color: #6c757d;"></i>
                    </div>
                </div>
                <div id="push-status-text" style="color: #6c757d; font-size: 0.85rem; margin-bottom: 0.75rem;">Wird geladen...</div>
                <div id="push-actions" style="display: flex; gap: 0.5rem;">
                    <button id="push-subscribe-btn" class="btn btn-accent" onclick="subscribeToPush()" style="display: none; padding: 0.5rem 1rem; font-size: 0.85rem;">
                        <i class="fas fa-bell"></i>
                        Aktivieren
                    </button>
                    <button id="push-unsubscribe-btn" class="btn btn-secondary" onclick="unsubscribeFromPush()" style="display: none; padding: 0.5rem 1rem; font-size: 0.85rem;">
                        <i class="fas fa-bell-slash"></i>
                        Deaktivieren
                    </button>
                </div>
            </div>
            
            <h3 style="font-size: 1.25rem; font-weight: 600; color: #003057; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 0.75rem;">
                <i class="fas fa-inbox"></i>
                Meine Benachrichtigungen
            </h3>
            
            <div class="notifications-list">
                <?php if (empty($unread_notifications)): ?>
                    <div class="empty-state">
                        <i class="fas fa-bell-slash"></i>
                        <p>Keine neuen Benachrichtigungen</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($unread_notifications as $notif): ?>
                        <div class="notification-item">
                            <div class="notification-icon">
                                <?php
                                $icons = [
                                    'system' => 'fas fa-info-circle',
                                    'appointment' => 'fas fa-calendar-check',
                                    'verification' => 'fas fa-check-circle',
                                    'warning' => 'fas fa-exclamation-triangle',
                                    'success' => 'fas fa-check-circle'
                                ];
                                $icon = $icons[$notif['type']] ?? 'fas fa-bell';
                                ?>
                                <i class="<?php echo $icon; ?>"></i>
                            </div>
                            <div class="notification-content">
                                <h4><?php echo htmlspecialchars($notif['title'] ?? 'Benachrichtigung'); ?></h4>
                                <p><?php echo htmlspecialchars($notif['message']); ?></p>
                                <small><?php echo date('d.m.Y H:i', strtotime($notif['created_at'])); ?></small>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Activity Tab Content -->
        <div id="activity-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-history"></i>
                Aktivitätsprotokoll
            </h2>
            
            <div class="activity-list">
                <?php if (empty($activity_log)): ?>
                    <div class="empty-state">
                        <i class="fas fa-history"></i>
                        <p>Noch keine Aktivitäten</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($activity_log as $activity): ?>
                        <div class="activity-item">
                            <div class="activity-icon">
                                <?php
                                $action_icons = [
                                    'profile_updated' => 'fas fa-user-edit',
                                    'vehicle_added' => 'fas fa-plus-circle',
                                    'vehicle_updated' => 'fas fa-edit',
                                    'vehicle_deleted' => 'fas fa-trash',
                                    'appointment_created' => 'fas fa-calendar-plus',
                                    'appointment_cancelled' => 'fas fa-calendar-times',
                                    'appointment_rescheduled' => 'fas fa-calendar-alt',
                                    'document_uploaded' => 'fas fa-file-upload',
                                    'password_changed' => 'fas fa-key',
                                    'login' => 'fas fa-sign-in-alt',
                                    'logout' => 'fas fa-sign-out-alt'
                                ];
                                $icon = $action_icons[$activity['action_type']] ?? 'fas fa-circle';
                                ?>
                                <i class="<?php echo $icon; ?>"></i>
                            </div>
                            <div class="activity-content">
                                <p><strong><?php echo htmlspecialchars($activity['description']); ?></strong></p>
                                <small><?php echo date('d.m.Y H:i', strtotime($activity['created_at'])); ?></small>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Security Tab Content -->
        <div id="security-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-shield-alt"></i>
                Sicherheit & Privatsphäre
            </h2>
            
            <div class="security-section">
                <?php if (!$is_suspended): ?>
                <button class="btn btn-secondary" onclick="openPasswordModal()">
                    <i class="fas fa-key"></i>
                    Passwort ändern
                </button>
                <?php else: ?>
                <div style="padding: 1rem; background: #f8f9fa; border-radius: 4px; border: 1px solid #e9ecef; color: #6c757d; text-align: center;">
                    <i class="fas fa-lock"></i> Passwort-Änderung ist gesperrt
                </div>
                <?php endif; ?>
            </div>
            
            <div style="margin-top: 2rem; padding: 1.5rem; background: #f8f9fa; border-radius: 8px;">
                <h3 style="margin: 0 0 1rem 0; color: #003057;">
                    <i class="fas fa-info-circle"></i>
                    Konto-Informationen
                </h3>
                <div style="display: grid; gap: 1rem;">
                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; background: #fff; border-radius: 8px;">
                        <span style="color: #6c757d;">E-Mail-Adresse:</span>
                        <strong><?php echo htmlspecialchars($user['email']); ?></strong>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; background: #fff; border-radius: 8px;">
                        <span style="color: #6c757d;">Registriert am:</span>
                        <strong><?php echo date('d.m.Y', strtotime($user['created_at'])); ?></strong>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; background: #fff; border-radius: 8px;">
                        <span style="color: #6c757d;">Registrierung via:</span>
                        <strong><?php 
                            $reg_source = $user['registration_source'] ?? 'unknown';
                            $source_labels = [
                                'web' => 'Web',
                                'ios' => 'iOS App',
                                'android' => 'Android App',
                                'apple_signin' => 'Apple Sign-In',
                                'google_signin' => 'Google Sign-In',
                                'unknown' => 'Unbekannt'
                            ];
                            echo $source_labels[$reg_source] ?? 'Unbekannt';
                        ?></strong>
                    </div>
                </div>
            </div>
        </div>
        </div>
            </main>
        </div>
    </div>

    <!-- Notification Banner -->
    <div id="notification-banner" class="notification-banner">
        <i class="fas"></i>
        <span id="notification-text"></span>
    </div>

    <!-- Modal für Fahrzeug hinzufügen/bearbeiten -->
    <div id="vehicle-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title" id="modal-title">Fahrzeug hinzufügen</h3>
                <button class="close-modal" onclick="closeVehicleModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="vehicle-form">
                <input type="hidden" id="vehicle-id" name="vehicle_id">
                
                <div class="form-group">
                    <label for="license-plate">Kennzeichen *</label>
                    <input type="text" id="license-plate" name="license_plate" required 
                           placeholder="z. B. M-AB 123">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-brand">Marke</label>
                    <input type="text" id="vehicle-brand" name="vehicle_brand" 
                           placeholder="z. B. Volvo">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-model">Modell</label>
                    <input type="text" id="vehicle-model" name="vehicle_model" 
                           placeholder="z. B. XC60">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-year">Baujahr</label>
                    <input type="number" id="vehicle-year" name="vehicle_year" 
                           placeholder="z. B. 2020" min="1900" max="<?php echo date('Y'); ?>">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-notes">Notizen</label>
                    <textarea id="vehicle-notes" name="vehicle_notes" rows="3" 
                              placeholder="Zusätzliche Informationen..."></textarea>
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 0.5rem; cursor: pointer;">
                        <input type="checkbox" id="is-default" name="is_default" style="width: auto;">
                        Als Standardfahrzeug setzen
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeVehicleModal()">
                        Abbrechen
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Speichern
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal für Fahrzeug löschen -->
    <div id="delete-vehicle-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Fahrzeug löschen</h3>
                <button class="close-modal" onclick="closeDeleteVehicleModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div style="padding: 2rem;">
                <div style="text-align: center; margin-bottom: 2rem;">
                    <div style="width: 80px; height: 80px; margin: 0 auto 1.5rem; background: #f8d7da; border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                        <i class="fas fa-exclamation-triangle" style="font-size: 2.5rem; color: #dc3545;"></i>
                    </div>
                    <h3 style="color: #003057; margin-bottom: 1rem;">Möchten Sie dieses Fahrzeug wirklich löschen?</h3>
                    <p style="color: #6c757d; font-size: 1rem; margin-bottom: 0.5rem;">Diese Aktion kann nicht rückgängig gemacht werden.</p>
                </div>
                
                <div style="background: #f8f9fa; border-radius: 8px; padding: 1.5rem; margin-bottom: 2rem; border: 2px solid #e9ecef;">
                    <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                        <i class="fas fa-car" style="font-size: 1.5rem; color: #0056b3;"></i>
                        <div style="flex: 1;">
                            <div style="font-size: 0.875rem; color: #6c757d; margin-bottom: 0.25rem;">Kennzeichen</div>
                            <div style="font-size: 1.25rem; font-weight: 700; color: #003057;" id="delete-vehicle-license"></div>
                        </div>
                    </div>
                    <div style="display: flex; align-items: center; gap: 1rem;">
                        <i class="fas fa-info-circle" style="font-size: 1.5rem; color: #0056b3;"></i>
                        <div style="flex: 1;">
                            <div style="font-size: 0.875rem; color: #6c757d; margin-bottom: 0.25rem;">Fahrzeug</div>
                            <div style="font-size: 1.1rem; font-weight: 600; color: #495057;" id="delete-vehicle-info"></div>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeDeleteVehicleModal()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button type="button" class="btn btn-action-delete" onclick="confirmDeleteVehicle()">
                        <i class="fas fa-trash"></i> Fahrzeug löschen
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal für Profil bearbeiten -->
    <div id="profile-modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Profil bearbeiten</h3>
                <button class="close-modal" onclick="closeProfileModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="profile-edit-form" style="padding: 2rem;">
                <div class="profile-image-upload">
                    <img id="profile-image-preview" src="<?php echo !empty($user['profile_image']) ? htmlspecialchars($user['profile_image']) : 'assets/images/default-avatar.png'; ?>" alt="Profilbild" class="profile-image-preview">
                    <input type="file" id="profile-image-input" name="profile_image" accept="image/*" style="display: none;" onchange="previewImage(event)">
                    <button type="button" class="btn-action btn-action-edit" style="margin-top: 1rem;" onclick="document.getElementById('profile-image-input').click()">
                        <i class="fas fa-camera"></i>
                        <span>Bild ändern</span>
                    </button>
                </div>
                

                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 1.5rem;">
                    <!-- Autohaus Auswahl als Cards -->
                    <div class="form-group" style="grid-column: 1 / -1;">
                        <label style="margin-bottom: 1rem; display: flex; align-items: center; gap: 0.5rem; color: #003057; font-weight: 600; font-size: 0.95rem;">
                            <i class="fas fa-building" style="color: #0056b3; font-size: 0.9rem; width: 18px; text-align: center;"></i>
                            <span>Autohaus <span style="color: #dc3545; margin-left: 0.25rem;">*</span></span>
                        </label>
                        <div class="autohaus-selection-grid">
                            <?php foreach ($autohauser as $autohaus) { ?>
                                <div class="autohaus-selection-card <?php echo ($user_autohaus_id == $autohaus['id']) ? 'selected' : ''; ?>" 
                                     data-autohaus-id="<?php echo $autohaus['id']; ?>">
                                    <div class="autohaus-card-header">
                                        <h3><?php echo htmlspecialchars($autohaus['name']); ?></h3>
                                        <?php if (!empty($autohaus['short_name'])): ?>
                                            <span class="autohaus-short"><?php echo htmlspecialchars($autohaus['short_name']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="autohaus-card-info">
                                        <?php if (!empty($autohaus['address'])): ?>
                                            <div class="info-item">
                                                <i class="fas fa-map-marker-alt"></i>
                                                <span><?php echo htmlspecialchars($autohaus['address']); ?></span>
                                            </div>
                                        <?php endif; ?>
                                        <?php if (!empty($autohaus['phone'])): ?>
                                            <div class="info-item">
                                                <i class="fas fa-phone"></i>
                                                <span><?php echo htmlspecialchars($autohaus['phone']); ?></span>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <?php if (!empty($autohaus['footer_image'])): ?>
                                        <div class="autohaus-card-image">
                                            <?php 
                                            // Extrahiere Dateinamen aus dem Pfad
                                            $footer_filename = basename($autohaus['footer_image']); 
                                            ?>
                                            <img src="api/view_footer_image.php?file=<?php echo urlencode($footer_filename); ?>" 
                                                 alt="<?php echo htmlspecialchars($autohaus['footer_image_alt'] ?? $autohaus['name']); ?>" 
                                                 style="width: 100%; max-height: 150px; object-fit: cover; border-radius: 8px; margin-top: 1rem;">
                                        </div>
                                    <?php endif; ?>
                                    <div class="autohaus-card-action">
                                        <i class="fas fa-check-circle"></i>
                                    </div>
                                </div>
                            <?php } ?>
                        </div>
                        <input type="hidden" id="edit-autohaus" name="autohaus_id" required value="<?php echo $user_autohaus_id ?? ''; ?>">
                    </div>
                    <div class="form-group">
                        <label for="edit-first-name">
                            <i class="fas fa-user"></i>
                            <span>Vorname *</span>
                        </label>
                        <input type="text" id="edit-first-name" name="first_name" required 
                               value="<?php echo htmlspecialchars($user['first_name']); ?>"
                               placeholder="Ihr Vorname">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-last-name">
                            <i class="fas fa-user"></i>
                            <span>Nachname *</span>
                        </label>
                        <input type="text" id="edit-last-name" name="last_name" required 
                               value="<?php echo htmlspecialchars($user['last_name']); ?>"
                               placeholder="Ihr Nachname">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-phone">
                            <i class="fas fa-phone"></i>
                            <span>Telefonnummer</span>
                        </label>
                        <input type="tel" id="edit-phone" name="phone_number" 
                               value="<?php echo htmlspecialchars($user['phone_number'] ?? ''); ?>"
                               placeholder="z. B. +49 123 456 789">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-address">
                            <i class="fas fa-map-marker-alt"></i>
                            <span>Adresse</span>
                        </label>
                        <input type="text" id="edit-address" name="address" 
                               value="<?php echo htmlspecialchars($user['address'] ?? ''); ?>"
                               placeholder="z. B. Musterstraße 123">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-postal-code">
                            <i class="fas fa-mail-bulk"></i>
                            <span>Postleitzahl</span>
                        </label>
                        <input type="text" id="edit-postal-code" name="postal_code" 
                               value="<?php echo htmlspecialchars($user['postal_code'] ?? ''); ?>"
                               placeholder="z. B. 12345">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-city">
                            <i class="fas fa-city"></i>
                            <span>Stadt</span>
                        </label>
                        <input type="text" id="edit-city" name="city" 
                               value="<?php echo htmlspecialchars($user['city'] ?? ''); ?>"
                               placeholder="z. B. München">
                    </div>
                </div>
                
                <div style="margin-top: 2rem; display: flex; gap: 1rem; justify-content: flex-end;">
                    <button type="button" class="btn-action" onclick="closeProfileModal()" style="background: #6c757d;">
                        <i class="fas fa-times"></i>
                        <span>Abbrechen</span>
                    </button>
                    <button type="submit" class="btn-action btn-action-edit">
                        <i class="fas fa-save"></i>
                        <span>Speichern</span>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal für Passwort ändern -->
    <div id="password-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">Passwort ändern</h3>
                <button class="close-modal" onclick="closePasswordModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="password-change-form">
                <div class="form-group">
                    <label for="current-password">
                        <i class="fas fa-lock"></i>
                        <span>Aktuelles Passwort *</span>
                    </label>
                    <input type="password" id="current-password" name="current_password" required
                           placeholder="Ihr aktuelles Passwort eingeben">
                </div>
                
                <div class="form-group">
                    <label for="new-password">
                        <i class="fas fa-key"></i>
                        <span>Neues Passwort *</span>
                    </label>
                    <input type="password" id="new-password" name="new_password" required minlength="6"
                           placeholder="Mindestens 6 Zeichen">
                    <small style="color: #6c757d; font-size: 0.875rem; margin-top: 0.5rem; display: block;">
                        <i class="fas fa-info-circle"></i> Mindestens 6 Zeichen erforderlich
                    </small>
                </div>
                
                <div class="form-group">
                    <label for="confirm-password">
                        <i class="fas fa-check-circle"></i>
                        <span>Neues Passwort bestätigen *</span>
                    </label>
                    <input type="password" id="confirm-password" name="confirm_password" required
                           placeholder="Neues Passwort wiederholen">
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closePasswordModal()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-key"></i> Passwort ändern
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script src="assets/js/main.js"></script>
    <script src="assets/js/push-notifications.js"></script>
    <script src="assets/js/pwa-install.js"></script>
    <script>
        // User Dropdown Toggle Funktion
        function toggleUserDropdown(event) {
            event.stopPropagation();
            const menu = document.getElementById('userDropdownMenu');
            const trigger = event.currentTarget;
            
            if (menu && trigger) {
                const isOpen = menu.classList.contains('show');
                
                // Schließe alle anderen Dropdowns
                document.querySelectorAll('.user-dropdown-menu').forEach(m => {
                    m.classList.remove('show');
                });
                document.querySelectorAll('.user-profile-trigger').forEach(t => {
                    t.classList.remove('active');
                });
                
                // Toggle aktuelles Dropdown
                if (!isOpen) {
                    menu.classList.add('show');
                    trigger.classList.add('active');
                } else {
                    menu.classList.remove('show');
                    trigger.classList.remove('active');
                }
            }
        }
        
        // Schließe Dropdown beim Klick außerhalb
        document.addEventListener('click', function(event) {
            if (!event.target.closest('.user-profile-dropdown')) {
                document.querySelectorAll('.user-dropdown-menu').forEach(menu => {
                    menu.classList.remove('show');
                });
                document.querySelectorAll('.user-profile-trigger').forEach(trigger => {
                    trigger.classList.remove('active');
                });
            }
        });
    </script>
    <script>
        let currentVehicleId = null;
        
        // Modal öffnen
        function openVehicleModal(vehicleId = null) {
            <?php if ($is_suspended): ?>
            showNotification('Ihr Konto ist gesperrt. Sie können keine Änderungen vornehmen.', 'error');
            return;
            <?php endif; ?>
            
            currentVehicleId = vehicleId;
            const modal = document.getElementById('vehicle-modal');
            const form = document.getElementById('vehicle-form');
            const title = document.getElementById('modal-title');
            
            if (vehicleId) {
                title.textContent = 'Fahrzeug bearbeiten';
                loadVehicleData(vehicleId);
            } else {
                title.textContent = 'Fahrzeug hinzufügen';
                form.reset();
            }
            
            modal.classList.add('active');
        }
        
        // Modal schließen
        function closeVehicleModal() {
            document.getElementById('vehicle-modal').classList.remove('active');
            document.getElementById('vehicle-form').reset();
            currentVehicleId = null;
        }
        
        // Fahrzeugdaten laden
        async function loadVehicleData(vehicleId) {
            try {
                const response = await fetch(`api/get_vehicle.php?id=${vehicleId}`);
                const data = await response.json();
                
                if (data.success) {
                    const vehicle = data.vehicle;
                    document.getElementById('vehicle-id').value = vehicle.id;
                    document.getElementById('license-plate').value = vehicle.license_plate;
                    document.getElementById('vehicle-brand').value = vehicle.vehicle_brand || '';
                    document.getElementById('vehicle-model').value = vehicle.vehicle_model || '';
                    document.getElementById('vehicle-year').value = vehicle.vehicle_year || '';
                    document.getElementById('vehicle-notes').value = vehicle.notes || '';
                    document.getElementById('is-default').checked = vehicle.is_default;
                }
            } catch (error) {
                showAlert('Fehler beim Laden der Fahrzeugdaten', 'error');
            }
        }
        
        // Fahrzeug speichern
        document.getElementById('vehicle-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {};
            formData.forEach((value, key) => {
                data[key] = value;
            });
            
            try {
                const response = await fetch(currentVehicleId ? 'api/update_vehicle.php' : 'api/add_vehicle.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showAlert('Fahrzeug erfolgreich gespeichert!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showAlert(result.message || 'Fehler beim Speichern', 'error');
                }
            } catch (error) {
                showAlert('Verbindungsfehler', 'error');
            }
        });
        
        // Fahrzeug löschen - Modal öffnen
        let currentDeleteVehicleId = null;
        
        async function deleteVehicle(vehicleId) {
            currentDeleteVehicleId = vehicleId;
            await openDeleteVehicleModal(vehicleId);
        }
        
        // Lösch-Modal öffnen
        async function openDeleteVehicleModal(vehicleId) {
            try {
                const response = await fetch(`api/get_vehicle.php?id=${vehicleId}`);
                const data = await response.json();
                
                if (data.success) {
                    const vehicle = data.vehicle;
                    
                    // Fahrzeugdaten im Modal anzeigen
                    document.getElementById('delete-vehicle-license').textContent = vehicle.license_plate || 'Nicht angegeben';
                    
                    // Fahrzeuginfo zusammenstellen
                    let vehicleInfo = '';
                    if (vehicle.vehicle_brand && vehicle.vehicle_model) {
                        vehicleInfo = `${vehicle.vehicle_brand} ${vehicle.vehicle_model}`;
                    } else if (vehicle.vehicle_brand) {
                        vehicleInfo = vehicle.vehicle_brand;
                    } else if (vehicle.vehicle_model) {
                        vehicleInfo = vehicle.vehicle_model;
                    } else {
                        vehicleInfo = 'Keine Angaben';
                    }
                    
                    if (vehicle.vehicle_year) {
                        vehicleInfo += ` (${vehicle.vehicle_year})`;
                    }
                    
                    document.getElementById('delete-vehicle-info').textContent = vehicleInfo || 'Keine Angaben';
                    
                    // Modal anzeigen
                    document.getElementById('delete-vehicle-modal').classList.add('active');
                } else {
                    showAlert('Fehler beim Laden der Fahrzeugdaten', 'error');
                }
            } catch (error) {
                showAlert('Verbindungsfehler beim Laden der Fahrzeugdaten', 'error');
            }
        }
        
        // Lösch-Modal schließen
        function closeDeleteVehicleModal() {
            document.getElementById('delete-vehicle-modal').classList.remove('active');
            currentDeleteVehicleId = null;
        }
        
        // Fahrzeug löschen bestätigen
        async function confirmDeleteVehicle() {
            if (!currentDeleteVehicleId) {
                return;
            }
            
            try {
                const response = await fetch('api/delete_vehicle.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ vehicle_id: currentDeleteVehicleId })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Fahrzeug erfolgreich gelöscht', 'success');
                    closeDeleteVehicleModal();
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Löschen', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        }
        
        // Alert anzeigen
        function showAlert(message, type = 'error') {
            const alert = document.getElementById('alert-message');
            alert.textContent = message;
            alert.className = `alert alert-${type} show`;
            setTimeout(() => {
                alert.classList.remove('show');
            }, 5000);
        }
        
        // Feedback-Banner schließen
        function closeFeedbackBanner() {
            const banner = document.getElementById('feedback-banner');
            if (banner) {
                banner.style.animation = 'slideOutToTop 0.3s ease-out';
                setTimeout(() => {
                    banner.style.display = 'none';
                    // Entferne complete_profile Parameter aus URL
                    const url = new URL(window.location);
                    url.searchParams.delete('complete_profile');
                    window.history.replaceState({}, '', url);
                }, 300);
            }
        }
        
        // Auto-öffne Profil-Modal wenn Profil unvollständig und von Push-Benachrichtigung
        <?php if ($from_push_notification && $has_missing_fields): ?>
        window.addEventListener('load', function() {
            setTimeout(() => {
                if (typeof openProfileModal === 'function') {
                    openProfileModal();
                }
            }, 1000);
        });
        <?php endif; ?>
        
        // Notification Banner anzeigen (für E-Mail-Bestätigungen etc.)
        function showNotification(message, type = 'success') {
            const banner = document.getElementById('notification-banner');
            const icon = banner.querySelector('i');
            const text = document.getElementById('notification-text');
            
            // Icon setzen
            if (type === 'success') {
                icon.className = 'fas fa-check-circle';
                banner.className = 'notification-banner success show';
            } else {
                icon.className = 'fas fa-exclamation-circle';
                banner.className = 'notification-banner error show';
            }
            
            text.textContent = message;
            
            // Auto-hide nach 5 Sekunden
            setTimeout(() => {
                banner.classList.remove('show');
            }, 5000);
        }
        
        // Modal schließen bei Klick außerhalb
        document.getElementById('vehicle-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeVehicleModal();
            }
        });
        
        // Lösch-Modal schließen bei Klick außerhalb
        document.getElementById('delete-vehicle-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeDeleteVehicleModal();
            }
        });
        
        // Profil-Modal
        async function openProfileModal() {
            <?php if ($is_suspended): ?>
            showNotification('Ihr Konto ist gesperrt. Sie können keine Änderungen vornehmen.', 'error');
            return;
            <?php endif; ?>
            
            document.getElementById('profile-modal').classList.add('active');
            
            // Prüfe ob bereits ein Autohaus ausgewählt ist
            const currentAutohaus = document.getElementById('edit-autohaus')?.value;
            if (!currentAutohaus || currentAutohaus === '') {
                // Kein Autohaus ausgewählt - versuche automatisch zu finden
                await getBrowserLocationProfile();
            }
        }
        
        function closeProfileModal() {
            document.getElementById('profile-modal').classList.remove('active');
        }
        
        function previewImage(event) {
            const reader = new FileReader();
            reader.onload = function() {
                document.getElementById('profile-image-preview').src = reader.result;
            };
            reader.readAsDataURL(event.target.files[0]);
        }
        
        // Profil speichern
        document.getElementById('profile-edit-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            
            try {
                const response = await fetch('api/update_profile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Profil erfolgreich aktualisiert!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Speichern', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        });
        
        // Modal schließen bei Klick außerhalb
        document.getElementById('profile-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeProfileModal();
            }
        });
        
        // Passwort-Modal
        function openPasswordModal() {
            <?php if ($is_suspended): ?>
            showNotification('Ihr Konto ist gesperrt. Sie können keine Änderungen vornehmen.', 'error');
            return;
            <?php endif; ?>
            
            document.getElementById('password-modal').classList.add('active');
        }
        
        function closePasswordModal() {
            document.getElementById('password-modal').classList.remove('active');
            document.getElementById('password-change-form').reset();
        }
        
        // Passwort-Änderungsformular
        document.getElementById('password-change-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const currentPassword = document.getElementById('current-password').value;
            const newPassword = document.getElementById('new-password').value;
            const confirmPassword = document.getElementById('confirm-password').value;
            
            // Validierung
            if (newPassword !== confirmPassword) {
                showNotification('Die Passwörter stimmen nicht überein', 'error');
                return;
            }
            
            if (newPassword.length < 6) {
                showNotification('Das Passwort muss mindestens 6 Zeichen lang sein', 'error');
                return;
            }
            
            try {
                const response = await fetch('api/change_password.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        current_password: currentPassword,
                        new_password: newPassword
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Passwort erfolgreich geändert!', 'success');
                    setTimeout(() => {
                        closePasswordModal();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Ändern des Passworts', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        });
        
        // Modal schließen bei Klick außerhalb
        document.getElementById('password-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closePasswordModal();
            }
        });
        
        // Profile Tabs
        function showProfileTab(tab) {
            const tabs = document.querySelectorAll('.sidebar-nav-item[data-tab], .tab-btn[data-tab]');
            const contents = document.querySelectorAll('.tab-content');
            
            // Alle Tabs deaktivieren
            tabs.forEach(btn => {
                if (btn.dataset.tab === tab) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            });
            
            // Alle Inhalte ausblenden
            contents.forEach(content => {
                if (content.id === tab + '-tab') {
                    content.classList.add('active');
                } else {
                    content.classList.remove('active');
                }
            });
        }
        
        // Appointment Cancellation
        async function cancelAppointment(appointmentId) {
            if (!confirm('Möchten Sie diesen Termin wirklich stornieren?')) {
                return;
            }
            
            try {
                const response = await fetch('api/cancel_appointment.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        appointment_id: appointmentId
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Termin erfolgreich storniert', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Stornieren', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        }
        
        // Appointment Tabs
        function showAppointmentTab(tab) {
            const cards = document.querySelectorAll('.appointment-card');
            const buttons = document.querySelectorAll('.tab-btn');
            
            // Buttons aktivieren/deaktivieren
            buttons.forEach(btn => {
                if (btn.dataset.tab === tab) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            });
            
            // Cards filtern
            cards.forEach(card => {
                const isPast = card.classList.contains('past');
                const cardDate = card.dataset.date;
                const currentTime = new Date().getTime();
                const appointmentTime = new Date(cardDate + 'T00:00:00').getTime();
                
                switch(tab) {
                    case 'all':
                        card.style.display = 'flex';
                        break;
                    case 'upcoming':
                        card.style.display = appointmentTime >= currentTime ? 'flex' : 'none';
                        break;
                    case 'past':
                        card.style.display = appointmentTime < currentTime ? 'flex' : 'none';
                        break;
                }
            });
        }
        
        // Finde nächstes Autohaus basierend auf Koordinaten (für Profile)
        async function findNearestAutohausProfile(latitude, longitude) {
            try {
                const response = await fetch('api/find_nearest_autohaus.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        latitude: latitude,
                        longitude: longitude
                    })
                });
                
                const data = await response.json();
                if (data.success && data.nearest_autohaus) {
                    return data.nearest_autohaus;
                }
                return null;
            } catch (error) {
                console.warn('Fehler beim Finden des nächsten Autohauses:', error);
                return null;
            }
        }
        
        // Automatische Autohaus-Auswahl im Profil basierend auf Standort
        async function autoSelectAutohausProfile(latitude, longitude) {
            const nearestAutohaus = await findNearestAutohausProfile(latitude, longitude);
            
            if (nearestAutohaus) {
                const autohausId = nearestAutohaus.id;
                const autohausName = nearestAutohaus.name;
                const distance = nearestAutohaus.distance_km;
                
                // Prüfe ob bereits ein Autohaus ausgewählt ist
                const currentAutohaus = document.getElementById('edit-autohaus')?.value;
                if (currentAutohaus && currentAutohaus !== '') {
                    // Bereits ausgewählt - keine automatische Änderung
                    return false;
                }
                
                // Wähle Autohaus automatisch aus
                const input = document.getElementById('edit-autohaus');
                if (input) {
                    input.value = autohausId;
                }
                
                // Visuell auswählen
                const cards = document.querySelectorAll('#profile-modal .autohaus-selection-card');
                cards.forEach(card => {
                    card.classList.remove('selected');
                    if (parseInt(card.getAttribute('data-autohaus-id')) === autohausId) {
                        card.classList.add('selected');
                        // Smooth scroll zu ausgewähltem Autohaus
                        setTimeout(() => {
                            card.scrollIntoView({ behavior: 'smooth', block: 'center' });
                        }, 300);
                    }
                });
                
                // Professionelle Notification anzeigen
                if (typeof showNotification === 'function') {
                    // Verwende showNotification mit HTML-Escape für Sicherheit
                    const message = `Sind Sie Kunde bei ${autohausName} oder haben Sie Ihre Räder dort eingelagert? Anhand Ihres Standorts haben wir dieses Autohaus (${distance} km entfernt) automatisch für Sie ausgewählt.`;
                    showNotification(message, 'success');
                    
                    // Zusätzlich: Visuelles Feedback mit einem schönen Banner
                    const notificationHtml = `
                        <div style="position: fixed; top: 100px; right: 20px; background: linear-gradient(135deg, #007bff 0%, #0056b3 100%); color: white; padding: 1.25rem 1.5rem; border-radius: 12px; box-shadow: 0 8px 24px rgba(0,123,255,0.3); z-index: 10001; max-width: 400px; animation: slideInRight 0.4s ease;">
                            <div style="display: flex; align-items: start; gap: 1rem;">
                                <div style="width: 40px; height: 40px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center; flex-shrink: 0;">
                                    <i class="fas fa-map-marker-alt" style="font-size: 1.25rem;"></i>
                                </div>
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; margin-bottom: 0.5rem; font-size: 1rem;">Autohaus automatisch zugeordnet</div>
                                    <div style="font-size: 0.9rem; line-height: 1.5; opacity: 0.95;">
                                        Sind Sie Kunde bei "${autohausName}" oder haben Sie Ihre Räder dort eingelagert? Anhand Ihres Standorts haben wir dieses Autohaus (${distance} km entfernt) automatisch für Sie ausgewählt.
                                    </div>
                                </div>
                                <button onclick="this.parentElement.parentElement.remove()" style="background: none; border: none; color: white; font-size: 1.25rem; cursor: pointer; opacity: 0.8; padding: 0.25rem; line-height: 1;">&times;</button>
                            </div>
                        </div>
                    `;
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = notificationHtml;
                    document.body.appendChild(tempDiv.firstElementChild);
                    
                    // Auto-Entfernen nach 8 Sekunden
                    setTimeout(() => {
                        const banner = document.querySelector('[style*="position: fixed"][style*="top: 100px"]');
                        if (banner && banner.textContent.includes('Autohaus automatisch')) {
                            banner.style.animation = 'slideOutRight 0.3s ease';
                            setTimeout(() => banner.remove(), 300);
                        }
                    }, 8000);
                } else {
                    // Fallback: Alert
                    alert(`Anhand Ihres Standorts haben wir festgestellt, dass das nächste Autohaus ${autohausName} ist (${distance} km entfernt). Das Autohaus wurde automatisch für Sie ausgewählt.`);
                }
                
                return true;
            }
            return false;
        }
        
        // Browser Geolocation für Profil
        async function getBrowserLocationProfile() {
            return new Promise(async (resolve) => {
                if (!navigator.geolocation) {
                    resolve(null);
                    return;
                }
                
                // Zeige erst Modal für Erklärung (nur wenn noch nicht gefragt wurde)
                const locationPermissionAsked = sessionStorage.getItem('location_permission_asked');
                let userAllowed = locationPermissionAsked === 'granted';
                
                if (!locationPermissionAsked) {
                    // Zeige Modal nur wenn noch nie gefragt wurde
                    if (typeof showLocationPermissionModal === 'function') {
                        userAllowed = await showLocationPermissionModal();
                    } else {
                        // Fallback: Direkt fragen
                        userAllowed = confirm('Möchten Sie Ihren Standort freigeben, um das nächste Autohaus automatisch zu finden?');
                        if (userAllowed) {
                            sessionStorage.setItem('location_permission_asked', 'granted');
                        } else {
                            sessionStorage.setItem('location_permission_asked', 'denied');
                        }
                    }
                }
                
                if (!userAllowed) {
                    resolve(null);
                    return;
                }
                
                // Timeout nach 5 Sekunden
                const timeout = setTimeout(() => {
                    resolve(null);
                }, 5000);
                
                navigator.geolocation.getCurrentPosition(
                    async (position) => {
                        clearTimeout(timeout);
                        const lat = position.coords.latitude;
                        const lng = position.coords.longitude;
                        
                        // Automatische Autohaus-Auswahl im Profil
                        await autoSelectAutohausProfile(lat, lng);
                        
                        resolve({
                            latitude: lat,
                            longitude: lng,
                            accuracy: position.coords.accuracy,
                            source: 'browser_geolocation'
                        });
                    },
                    (error) => {
                        clearTimeout(timeout);
                        resolve(null);
                    },
                    {
                        enableHighAccuracy: true,
                        timeout: 5000,
                        maximumAge: 0
                    }
                );
            });
        }
        
        // Autohaus Card Auswahl im Profil-Modal
        function selectAutohausCardProfile(autohausId, event) {
            // Event-Objekt bekommen falls nicht übergeben
            if (!event) {
                event = window.event;
            }
            
            // Alle Cards deselektieren
            document.querySelectorAll('#profile-modal .autohaus-selection-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            // Gewählte Card selektieren
            const selectedCard = event.currentTarget || event.target.closest('.autohaus-selection-card');
            if (selectedCard) {
                selectedCard.classList.add('selected');
            }
            
            // Hidden input setzen
            const input = document.getElementById('edit-autohaus');
            if (input) {
                input.value = autohausId;
            }
        }
        
        // Event Listener für Autohaus Cards im Profil-Modal
        document.addEventListener('DOMContentLoaded', function() {
            // Warte bis Modal geladen ist
            const profileModal = document.getElementById('profile-modal');
            if (profileModal) {
                // Event Listener für Cards im Modal
                profileModal.addEventListener('click', function(e) {
                    const card = e.target.closest('.autohaus-selection-card');
                    if (card) {
                        const autohausId = card.getAttribute('data-autohaus-id');
                        if (autohausId) {
                            selectAutohausCardProfile(parseInt(autohausId), e);
                        }
                    }
                });
            }
            
            // Initialisiere Badge-System
            initializeBadges();
        });
        
        /**
         * Badge-System - Volvo-konform
         * Aktualisiert automatisch Badge-Klassen basierend auf dem Wert
         */
        function initializeBadges() {
            const badges = document.querySelectorAll('.sidebar-nav-badge');
            
            badges.forEach(function(badge) {
                updateBadgeClass(badge);
            });
        }
        
        /**
         * Aktualisiert die Badge-Klasse basierend auf dem Wert
         * @param {HTMLElement} badge - Das Badge-Element
         */
        function updateBadgeClass(badge) {
            const count = parseInt(badge.getAttribute('data-count') || badge.textContent.trim()) || 0;
            
            // Entferne alle Badge-Varianten-Klassen
            badge.classList.remove('badge-default', 'badge-primary', 'badge-highlight', 'badge-warning', 'badge-alert', 'badge-success');
            
            // Setze die entsprechende Klasse basierend auf dem Wert
            if (count === 0) {
                badge.classList.add('badge-default');
            } else if (count > 0 && count <= 5) {
                // Kleine Werte: Primary/Highlight (Volvo Blue)
                badge.classList.add('badge-primary');
            } else if (count > 5 && count <= 10) {
                // Mittlere Werte: Warning (Gelb)
                badge.classList.add('badge-warning');
            } else {
                // Große Werte: Alert (Rot)
                badge.classList.add('badge-alert');
            }
        }
        
        /**
         * Aktualisiert ein Badge mit neuem Wert und Klasse
         * @param {string} selector - CSS-Selektor für das Badge
         * @param {number} count - Neuer Wert
         * @param {string} variant - Optional: 'default', 'primary', 'warning', 'alert', 'success'
         */
        function updateBadge(selector, count, variant = null) {
            const badge = document.querySelector(selector);
            if (!badge) return;
            
            badge.textContent = count;
            badge.setAttribute('data-count', count);
            
            if (variant) {
                // Entferne alle Badge-Varianten-Klassen
                badge.classList.remove('badge-default', 'badge-primary', 'badge-highlight', 'badge-warning', 'badge-alert', 'badge-success');
                badge.classList.add('badge-' + variant);
            } else {
                // Automatische Klassenzuweisung basierend auf Wert
                updateBadgeClass(badge);
            }
        }
        
        // Push-Benachrichtigungen Funktionen
        async function subscribeToPush() {
            const btn = document.getElementById('push-subscribe-btn');
            const statusText = document.getElementById('push-status-text');
            const statusIcon = document.getElementById('push-status-icon');
            
            if (!btn) return;
            
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird aktiviert...';
            if (statusText) statusText.textContent = 'Wird aktiviert...';
            
            try {
                await window.PushNotifications.subscribe();
                updatePushStatus();
                showNotification('Push-Benachrichtigungen erfolgreich aktiviert!', 'success');
            } catch (error) {
                console.error('Push subscription error:', error);
                if (statusText) statusText.textContent = 'Fehler: ' + error.message;
                if (statusIcon) statusIcon.innerHTML = '<i class="fas fa-exclamation-circle" style="color: #dc3545;"></i>';
                showNotification('Fehler beim Aktivieren: ' + error.message, 'error');
            } finally {
                if (btn) btn.disabled = false;
            }
        }
        
        async function unsubscribeFromPush() {
            const btn = document.getElementById('push-unsubscribe-btn');
            const statusText = document.getElementById('push-status-text');
            
            if (!btn) return;
            
            if (!confirm('Möchten Sie Push-Benachrichtigungen wirklich deaktivieren?')) {
                return;
            }
            
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird deaktiviert...';
            if (statusText) statusText.textContent = 'Wird deaktiviert...';
            
            try {
                await window.PushNotifications.unsubscribe();
                updatePushStatus();
                showNotification('Push-Benachrichtigungen erfolgreich deaktiviert!', 'success');
            } catch (error) {
                console.error('Push unsubscription error:', error);
                if (statusText) statusText.textContent = 'Fehler: ' + error.message;
                showNotification('Fehler beim Deaktivieren: ' + error.message, 'error');
            } finally {
                if (btn) btn.disabled = false;
            }
        }
        
        async function updatePushStatus() {
            const statusText = document.getElementById('push-status-text');
            const statusIcon = document.getElementById('push-status-icon');
            const subscribeBtn = document.getElementById('push-subscribe-btn');
            const unsubscribeBtn = document.getElementById('push-unsubscribe-btn');
            
            if (!statusText || !window.PushNotifications) return;
            
            if (!window.PushNotifications.isSupported()) {
                statusText.textContent = 'Push-Benachrichtigungen werden von diesem Browser nicht unterstützt';
                if (statusIcon) statusIcon.innerHTML = '<i class="fas fa-times-circle" style="color: #6c757d;"></i>';
                if (subscribeBtn) subscribeBtn.style.display = 'none';
                if (unsubscribeBtn) unsubscribeBtn.style.display = 'none';
                return;
            }
            
            try {
                const permission = await window.PushNotifications.checkPermission();
                const isSubscribed = await window.PushNotifications.checkStatus();
                
                if (permission === 'denied') {
                    statusText.textContent = 'Benachrichtigungen wurden blockiert. Bitte in den Browser-Einstellungen aktivieren.';
                    if (statusIcon) statusIcon.innerHTML = '<i class="fas fa-ban" style="color: #dc3545;"></i>';
                    if (subscribeBtn) subscribeBtn.style.display = 'none';
                    if (unsubscribeBtn) unsubscribeBtn.style.display = 'none';
                } else if (isSubscribed) {
                    statusText.textContent = 'Push-Benachrichtigungen sind aktiviert';
                    if (statusIcon) statusIcon.innerHTML = '<i class="fas fa-check-circle" style="color: #28a745;"></i>';
                    if (subscribeBtn) subscribeBtn.style.display = 'none';
                    if (unsubscribeBtn) unsubscribeBtn.style.display = 'block';
                } else {
                    statusText.textContent = 'Push-Benachrichtigungen sind deaktiviert';
                    if (statusIcon) statusIcon.innerHTML = '<i class="fas fa-bell-slash" style="color: #6c757d;"></i>';
                    if (subscribeBtn) subscribeBtn.style.display = 'block';
                    if (unsubscribeBtn) unsubscribeBtn.style.display = 'none';
                }
            } catch (error) {
                console.error('Error updating push status:', error);
                statusText.textContent = 'Fehler beim Laden des Status';
                if (statusIcon) statusIcon.innerHTML = '<i class="fas fa-exclamation-circle" style="color: #dc3545;"></i>';
            }
        }
        
        // Initialisiere Push-Status
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(() => {
                if (window.PushNotifications) {
                    updatePushStatus();
                }
            }, 1000);
        });
        
        // Update Status wenn Notifications Tab geöffnet wird
        const originalShowProfileTab = showProfileTab;
        showProfileTab = function(tab) {
            originalShowProfileTab(tab);
            if (tab === 'notifications' && window.PushNotifications) {
                setTimeout(updatePushStatus, 100);
            }
        };
        
        // Prüfe ob PWA installiert ist und biete Benachrichtigungen an
        async function checkPWAAndOfferNotifications() {
            // Prüfe ob PWA installiert ist
            const isPWAInstalled = window.matchMedia('(display-mode: standalone)').matches || 
                                  window.navigator.standalone === true ||
                                  document.referrer.includes('android-app://');
            
            if (!isPWAInstalled) {
                return; // PWA nicht installiert, keine Anfrage
            }
            
            // Prüfe ob Push-Benachrichtigungen unterstützt werden
            if (!window.PushNotifications || !window.PushNotifications.isSupported()) {
                return; // Push nicht unterstützt
            }
            
            // Prüfe ob bereits abonniert
            try {
                const isSubscribed = await window.PushNotifications.checkStatus();
                if (isSubscribed) {
                    return; // Bereits abonniert
                }
            } catch (error) {
                console.error('Error checking push status:', error);
                return;
            }
            
            // Prüfe Berechtigung
            try {
                const permission = await window.PushNotifications.checkPermission();
                if (permission === 'denied') {
                    return; // Bereits abgelehnt
                }
                
                // Prüfe ob bereits einmal angefragt wurde (in dieser Session)
                const notificationAskedKey = 'pwa_notification_asked_' + new Date().toDateString();
                if (sessionStorage.getItem(notificationAskedKey)) {
                    return; // Bereits in dieser Session angefragt
                }
                
                // Warte kurz, dann zeige Modal
                setTimeout(() => {
                    showPWANotificationRequest();
                    sessionStorage.setItem(notificationAskedKey, 'true');
                }, 2000); // Warte 2 Sekunden nach Seitenladung
            } catch (error) {
                console.error('Error checking notification permission:', error);
            }
        }
        
        // Zeige Benachrichtigungsanfrage-Modal für PWA
        function showPWANotificationRequest() {
            // Prüfe ob Modal bereits existiert
            if (document.getElementById('pwa-notification-request-modal')) {
                return;
            }
            
            const modal = document.createElement('div');
            modal.id = 'pwa-notification-request-modal';
            modal.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.7);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10003;
                animation: fadeIn 0.3s ease;
            `;
            
            const modalContent = document.createElement('div');
            modalContent.style.cssText = `
                background: white;
                border-radius: 16px;
                padding: 2.5rem;
                max-width: 500px;
                width: 90%;
                box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
                position: relative;
                text-align: center;
            `;
            
            modalContent.innerHTML = `
                <div style="font-size: 4rem; color: #0056b3; margin-bottom: 1.5rem;">
                    <i class="fas fa-bell"></i>
                </div>
                <h3 style="color: #003057; margin-bottom: 1rem; font-size: 1.75rem; font-weight: 600;">
                    Benachrichtigungen aktivieren?
                </h3>
                <p style="color: #6c757d; margin-bottom: 2rem; line-height: 1.7; font-size: 1rem;">
                    Sie haben die App installiert! Möchten Sie Benachrichtigungen erhalten, wenn neue Nachrichten oder wichtige Updates in Ihrem Profil erscheinen?
                </p>
                <div style="display: flex; gap: 1rem; justify-content: center; flex-wrap: wrap;">
                    <button id="pwa-notification-later" style="
                        padding: 0.875rem 2rem;
                        background: #f8f9fa;
                        color: #495057;
                        border: 1px solid #dee2e6;
                        border-radius: 8px;
                        font-weight: 600;
                        cursor: pointer;
                        transition: all 0.2s;
                        font-size: 1rem;
                        font-family: 'Inter', sans-serif;
                    ">
                        Später
                    </button>
                    <button id="pwa-notification-activate" style="
                        padding: 0.875rem 2rem;
                        background: #0056b3;
                        color: white;
                        border: none;
                        border-radius: 8px;
                        font-weight: 600;
                        cursor: pointer;
                        transition: all 0.2s;
                        font-size: 1rem;
                        font-family: 'Inter', sans-serif;
                        box-shadow: 0 4px 12px rgba(0, 86, 179, 0.3);
                    ">
                        <i class="fas fa-bell"></i> Aktivieren
                    </button>
                </div>
            `;
            
            // Event Listeners
            const laterBtn = modalContent.querySelector('#pwa-notification-later');
            const activateBtn = modalContent.querySelector('#pwa-notification-activate');
            
            laterBtn.addEventListener('click', () => {
                modal.style.animation = 'fadeOut 0.3s ease';
                setTimeout(() => {
                    if (modal.parentNode) {
                        document.body.removeChild(modal);
                    }
                }, 300);
            });
            
            laterBtn.addEventListener('mouseenter', () => {
                laterBtn.style.background = '#e9ecef';
                laterBtn.style.transform = 'translateY(-1px)';
            });
            laterBtn.addEventListener('mouseleave', () => {
                laterBtn.style.background = '#f8f9fa';
                laterBtn.style.transform = 'translateY(0)';
            });
            
            activateBtn.addEventListener('click', async () => {
                activateBtn.disabled = true;
                activateBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird aktiviert...';
                
                try {
                    await window.PushNotifications.subscribe();
                    
                    // Erfolg-Meldung
                    modalContent.innerHTML = `
                        <div style="font-size: 4rem; color: #28a745; margin-bottom: 1.5rem;">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <h3 style="color: #003057; margin-bottom: 1rem; font-size: 1.75rem; font-weight: 600;">
                            Erfolgreich aktiviert!
                        </h3>
                        <p style="color: #6c757d; margin-bottom: 2rem; line-height: 1.7; font-size: 1rem;">
                            Sie erhalten jetzt Benachrichtigungen auf diesem Gerät, wenn neue Nachrichten in Ihrem Profil erscheinen.
                        </p>
                        <button onclick="this.closest('[id=pwa-notification-request-modal]').remove()" style="
                            padding: 0.875rem 2rem;
                            background: #0056b3;
                            color: white;
                            border: none;
                            border-radius: 8px;
                            font-weight: 600;
                            cursor: pointer;
                            transition: all 0.2s;
                            font-size: 1rem;
                            font-family: 'Inter', sans-serif;
                        ">
                            Schließen
                        </button>
                    `;
                    
                    // Update Push Status im Profil
                    if (window.updatePushStatus) {
                        setTimeout(updatePushStatus, 500);
                    }
                    
                    // Zeige Erfolgs-Benachrichtigung
                    if (window.showNotification) {
                        showNotification('Benachrichtigungen erfolgreich aktiviert!', 'success');
                    }
                    
                    // Schließe Modal nach 3 Sekunden
                    setTimeout(() => {
                        if (modal.parentNode) {
                            modal.style.animation = 'fadeOut 0.3s ease';
                            setTimeout(() => {
                                if (modal.parentNode) {
                                    document.body.removeChild(modal);
                                }
                            }, 300);
                        }
                    }, 3000);
                } catch (error) {
                    console.error('Push subscription error:', error);
                    activateBtn.disabled = false;
                    activateBtn.innerHTML = '<i class="fas fa-bell"></i> Aktivieren';
                    
                    // Zeige Fehler-Meldung
                    const errorMsg = document.createElement('div');
                    errorMsg.style.cssText = 'color: #dc3545; margin-top: 1rem; font-size: 0.9rem;';
                    errorMsg.textContent = 'Fehler: ' + (error.message || 'Unbekannter Fehler');
                    modalContent.appendChild(errorMsg);
                    
                    setTimeout(() => {
                        if (errorMsg.parentNode) {
                            errorMsg.remove();
                        }
                    }, 5000);
                }
            });
            
            activateBtn.addEventListener('mouseenter', () => {
                activateBtn.style.background = '#003057';
                activateBtn.style.transform = 'translateY(-1px)';
                activateBtn.style.boxShadow = '0 6px 16px rgba(0, 86, 179, 0.4)';
            });
            activateBtn.addEventListener('mouseleave', () => {
                activateBtn.style.background = '#0056b3';
                activateBtn.style.transform = 'translateY(0)';
                activateBtn.style.boxShadow = '0 4px 12px rgba(0, 86, 179, 0.3)';
            });
            
            // Klicke außerhalb schließen
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.style.animation = 'fadeOut 0.3s ease';
                    setTimeout(() => {
                        if (modal.parentNode) {
                            document.body.removeChild(modal);
                        }
                    }, 300);
                }
            });
            
            // CSS Animationen hinzufügen
            if (!document.getElementById('pwa-notification-modal-styles')) {
                const style = document.createElement('style');
                style.id = 'pwa-notification-modal-styles';
                style.textContent = `
                    @keyframes fadeIn {
                        from {
                            opacity: 0;
                            transform: scale(0.95);
                        }
                        to {
                            opacity: 1;
                            transform: scale(1);
                        }
                    }
                    @keyframes fadeOut {
                        from {
                            opacity: 1;
                            transform: scale(1);
                        }
                        to {
                            opacity: 0;
                            transform: scale(0.95);
                        }
                    }
                `;
                document.head.appendChild(style);
            }
            
            document.body.appendChild(modal);
        }
        
        // Initialisiere PWA-Benachrichtigungsprüfung beim Laden
        document.addEventListener('DOMContentLoaded', function() {
            // Warte bis PushNotifications geladen ist
            setTimeout(() => {
                checkPWAAndOfferNotifications();
            }, 1500);
        });
    </script>
    
    <?php if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in']): ?>
        <?php include 'includes/feedback_form.php'; ?>
    <?php endif; ?>
</body>
</html>

