#!/usr/bin/env node
/**
 * GLB-Modell-Optimierung mit glTF-Transform
 * 
 * Verwendung:
 *   node optimize_models.js input.glb output.glb
 *   oder
 *   node optimize_models.js (optimiert alle .glb Dateien im assets/3dmodel Ordner)
 */

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

// Prüfe ob gltf-transform installiert ist
function checkGltfTransform() {
    try {
        execSync('gltf-transform --version', { stdio: 'ignore' });
        return true;
    } catch (error) {
        console.error('❌ gltf-transform CLI ist nicht installiert!');
        console.log('\n📦 Installiere mit:');
        console.log('   npm install -g @gltf-transform/cli');
        return false;
    }
}

// Optimiere eine einzelne GLB-Datei
function optimizeModel(inputPath, outputPath) {
    console.log(`\n🔧 Optimiere: ${inputPath}`);
    console.log(`   → ${outputPath}`);
    
    try {
        // Draco-Kompression + Texturoptimierung
        const command = `gltf-transform optimize "${inputPath}" "${outputPath}" ` +
            `--compress draco ` +
            `--texture-compress ktx2 ` +
            `--texture-size 2048 ` +
            `--simplify ` +
            `--simplify-ratio 0.98`;
        
        console.log(`   Ausführung: ${command}`);
        execSync(command, { stdio: 'inherit' });
        
        // Dateigrößen vergleichen
        const inputSize = fs.statSync(inputPath).size;
        const outputSize = fs.statSync(outputPath).size;
        const reduction = ((1 - outputSize / inputSize) * 100).toFixed(1);
        
        console.log(`\n✅ Optimierung abgeschlossen!`);
        console.log(`   Original: ${(inputSize / 1024 / 1024).toFixed(2)} MB`);
        console.log(`   Optimiert: ${(outputSize / 1024 / 1024).toFixed(2)} MB`);
        console.log(`   Reduktion: ${reduction}%`);
        
        return true;
    } catch (error) {
        console.error(`❌ Fehler beim Optimieren von ${inputPath}:`, error.message);
        return false;
    }
}

// Optimiere alle GLB-Dateien in einem Ordner
function optimizeAllInFolder(folderPath) {
    const files = fs.readdirSync(folderPath);
    const glbFiles = files.filter(f => f.endsWith('.glb') && !f.includes('_optimized'));
    
    if (glbFiles.length === 0) {
        console.log('⚠️ Keine GLB-Dateien gefunden!');
        return;
    }
    
    console.log(`\n📁 Gefundene GLB-Dateien: ${glbFiles.length}`);
    
    glbFiles.forEach(file => {
        const inputPath = path.join(folderPath, file);
        const outputPath = path.join(folderPath, file.replace('.glb', '_optimized.glb'));
        optimizeModel(inputPath, outputPath);
    });
}

// Hauptfunktion
function main() {
    console.log('🚀 GLB-Modell-Optimierung');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    
    if (!checkGltfTransform()) {
        process.exit(1);
    }
    
    const args = process.argv.slice(2);
    
    if (args.length === 0) {
        // Keine Argumente: Optimiere alle Dateien im assets/3dmodel Ordner
        const defaultFolder = path.join(__dirname, 'assets', '3dmodel');
        if (fs.existsSync(defaultFolder)) {
            optimizeAllInFolder(defaultFolder);
        } else {
            console.error('❌ Ordner assets/3dmodel nicht gefunden!');
            console.log('\nVerwendung:');
            console.log('  node optimize_models.js input.glb output.glb');
            console.log('  node optimize_models.js (optimiert alle .glb im assets/3dmodel Ordner)');
        }
    } else if (args.length === 2) {
        // Zwei Argumente: Optimiere einzelne Datei
        const inputPath = path.resolve(args[0]);
        const outputPath = path.resolve(args[1]);
        
        if (!fs.existsSync(inputPath)) {
            console.error(`❌ Datei nicht gefunden: ${inputPath}`);
            process.exit(1);
        }
        
        optimizeModel(inputPath, outputPath);
    } else {
        console.error('❌ Ungültige Argumente!');
        console.log('\nVerwendung:');
        console.log('  node optimize_models.js input.glb output.glb');
        console.log('  node optimize_models.js (optimiert alle .glb im assets/3dmodel Ordner)');
        process.exit(1);
    }
    
    console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log('✅ Alle Optimierungen abgeschlossen!');
}

main();

