import { GraphEdge } from './graph-edge.js';
import { GraphNode } from './graph-node.js';
export type Ref<T extends GraphNode = GraphNode> = GraphEdge<GraphNode, T>;
/**
 * An ordered collection of {@link Ref Refs}, allowing duplicates. Removing
 * a Ref is an O(n) operation — use {@link RefSet} for faster removal, if
 * duplicates are not required.
 */
export declare class RefList<T extends GraphNode = GraphNode> {
    list: Ref<T>[];
    constructor(refs?: Ref<T>[]);
    add(ref: Ref<T>): void;
    remove(ref: Ref<T>): void;
    removeChild(child: T): Ref<T>[];
    listRefsByChild(child: T): Ref<T>[];
    values(): Ref<T>[];
}
/**
 * An ordered collection of {@link Ref Refs}, without duplicates. Adding or
 * removing a Ref is typically O(1) or O(log(n)), and faster than
 * {@link RefList}. If support for duplicates is required, use {@link RefList}.
 */
export declare class RefSet<T extends GraphNode = GraphNode> {
    set: Set<Ref<T>>;
    map: Map<T, Ref<T>>;
    constructor(refs?: Ref<T>[]);
    add(ref: Ref<T>): void;
    remove(ref: Ref<T>): void;
    removeChild(child: T): Ref<T> | null;
    getRefByChild(child: T): Ref<T> | null;
    values(): Ref<T>[];
}
/**
 * Map (or dictionary) from string keys to {@link Ref Refs}.
 */
export declare class RefMap<T extends GraphNode = GraphNode> {
    map: {
        [key: string]: Ref<T>;
    };
    constructor(map?: Record<string, Ref<T>>);
    set(key: string, child: Ref<T>): void;
    delete(key: string): void;
    get(key: string): Ref<T> | null;
    keys(): string[];
    values(): Ref<T>[];
}
