var InterpolationInternal;

(function (InterpolationInternal) {
  InterpolationInternal[InterpolationInternal["STEP"] = 0] = "STEP";
  InterpolationInternal[InterpolationInternal["LERP"] = 1] = "LERP";
  InterpolationInternal[InterpolationInternal["SLERP"] = 2] = "SLERP";
})(InterpolationInternal || (InterpolationInternal = {}));

const TO_INTERPOLATION_INTERNAL = {
  step: InterpolationInternal.STEP,
  lerp: InterpolationInternal.LERP,
  slerp: InterpolationInternal.SLERP
};
const EPSILON = 0.000001;

/* Implementation */

function resampleDebug(input, output, interpolation, tolerance) {
  if (tolerance === void 0) {
    tolerance = 1e-4;
  }

  const elementSize = output.length / input.length;
  const tmp = new Array(elementSize).fill(0);
  const value = new Array(elementSize).fill(0);
  const valueNext = new Array(elementSize).fill(0);
  const valuePrev = new Array(elementSize).fill(0);
  const lastIndex = input.length - 1;
  let writeIndex = 1;

  for (let i = 1; i < lastIndex; ++i) {
    const timePrev = input[writeIndex - 1];
    const time = input[i];
    const timeNext = input[i + 1];
    const t = (time - timePrev) / (timeNext - timePrev);
    let keep = false; // Remove unnecessary adjacent keyframes.

    if (time !== timeNext && (i !== 1 || time !== input[0])) {
      getElement(output, writeIndex - 1, valuePrev);
      getElement(output, i, value);
      getElement(output, i + 1, valueNext);

      if (interpolation === 'slerp') {
        // Prune keyframes colinear with prev/next keyframes.
        const sample = slerp(tmp, valuePrev, valueNext, t);
        const angle = getAngle(valuePrev, value) + getAngle(value, valueNext);
        keep = !eq(value, sample, tolerance) || angle + Number.EPSILON >= Math.PI;
      } else if (interpolation === 'lerp') {
        // Prune keyframes colinear with prev/next keyframes.
        const sample = vlerp(tmp, valuePrev, valueNext, t);
        keep = !eq(value, sample, tolerance);
      } else if (interpolation === 'step') {
        // Prune keyframes identical to prev/next keyframes.
        keep = !eq(value, valuePrev) || !eq(value, valueNext);
      }
    } // In-place compaction.


    if (keep) {
      if (i !== writeIndex) {
        input[writeIndex] = input[i];
        setElement(output, writeIndex, getElement(output, i, tmp));
      }

      writeIndex++;
    }
  } // Flush last keyframe (compaction looks ahead).


  if (lastIndex > 0) {
    input[writeIndex] = input[lastIndex];
    setElement(output, writeIndex, getElement(output, lastIndex, tmp));
    writeIndex++;
  }

  return writeIndex;
}
/* Utilities */

function getElement(array, index, target) {
  for (let i = 0, elementSize = target.length; i < elementSize; i++) {
    target[i] = array[index * elementSize + i];
  }

  return target;
}

function setElement(array, index, value) {
  for (let i = 0, elementSize = value.length; i < elementSize; i++) {
    array[index * elementSize + i] = value[i];
  }
}

function eq(a, b, tolerance) {
  if (tolerance === void 0) {
    tolerance = 0;
  }

  if (a.length !== b.length) {
    return false;
  }

  for (let i = 0; i < a.length; i++) {
    if (Math.abs(a[i] - b[i]) > tolerance) {
      return false;
    }
  }

  return true;
}

function lerp(v0, v1, t) {
  return v0 * (1 - t) + v1 * t;
}

function vlerp(out, a, b, t) {
  for (let i = 0; i < a.length; i++) out[i] = lerp(a[i], b[i], t);

  return out;
} // From gl-matrix.


function slerp(out, a, b, t) {
  // benchmarks:
  //    http://jsperf.com/quaternion-slerp-implementations
  let ax = a[0],
      ay = a[1],
      az = a[2],
      aw = a[3];
  let bx = b[0],
      by = b[1],
      bz = b[2],
      bw = b[3];
  let omega, cosom, sinom, scale0, scale1; // calc cosine

  cosom = ax * bx + ay * by + az * bz + aw * bw; // adjust signs (if necessary)

  if (cosom < 0.0) {
    cosom = -cosom;
    bx = -bx;
    by = -by;
    bz = -bz;
    bw = -bw;
  } // calculate coefficients


  if (1.0 - cosom > EPSILON) {
    // standard case (slerp)
    omega = Math.acos(cosom);
    sinom = Math.sin(omega);
    scale0 = Math.sin((1.0 - t) * omega) / sinom;
    scale1 = Math.sin(t * omega) / sinom;
  } else {
    // "from" and "to" quaternions are very close
    //  ... so we can do a linear interpolation
    scale0 = 1.0 - t;
    scale1 = t;
  } // calculate final values


  out[0] = scale0 * ax + scale1 * bx;
  out[1] = scale0 * ay + scale1 * by;
  out[2] = scale0 * az + scale1 * bz;
  out[3] = scale0 * aw + scale1 * bw;
  return out;
}

function getAngle(a, b) {
  const dotproduct = dot(a, b);
  return Math.acos(2 * dotproduct * dotproduct - 1);
}

function dot(a, b) {
  return a[0] * b[0] + a[1] * b[1] + a[2] * b[2] + a[3] * b[3];
}

/** WASM loader using an inline Data URI. */
const wasm = /* #__PURE__ */fetch('data:application/wasm;base64' + "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").then(response => response.arrayBuffer()).then(buffer => new Uint8Array(buffer));

// SETUP
///////////////////////////////////////////////////////////////////////////////

function _catch(body, recover) {
  try {
    var result = body();
  } catch (e) {
    return recover(e);
  }

  if (result && result.then) {
    return result.then(void 0, recover);
  }

  return result;
}

const instantiate = function (module, imports) {

  try {
    return Promise.resolve(WebAssembly.instantiate(module, {
      env: Object.assign(Object.create(globalThis), {}, {
        abort: __abort
      })
    })).then(function (instance) {
      return instance.exports;
    });
  } catch (e) {
    return Promise.reject(e);
  }
}; ///////////////////////////////////////////////////////////////////////////////
// PUBLIC API
///////////////////////////////////////////////////////////////////////////////


let exports$1;
const ready = /* #__PURE__ */new Promise(function (resolve, reject) {
  try {
    const _temp = _catch(function () {
      const _WebAssembly = WebAssembly,
            _compile = _WebAssembly.compile;
      return Promise.resolve(wasm).then(function (_wasm) {
        return Promise.resolve(_compile.call(_WebAssembly, _wasm)).then(function (module) {
          return Promise.resolve(instantiate(module, {})).then(function (_instantiate) {
            exports$1 = _instantiate;
            resolve();
          });
        });
      });
    }, function (e) {
      reject(e);
    });

    return Promise.resolve(_temp && _temp.then ? _temp.then(function () {}) : void 0);
  } catch (e) {
    return Promise.reject(e);
  }
});
const CHUNK_SIZE = 1024; // The first and last keyframes cannot be removed in any given step, but we need to
// somehow remove keyframes on chunk boundaries. So after processing each chunk,
// we copy its last two keyframes in front of the next chunk, and run from there.
//
// 🟩 ⬜️ ⬜️ ⬜️ ⬜️ ⬜️                  // chunk 1, original
// 🟩 ⬜️ 🟨 🟥                       // chunk 1, resampled
//            🟨 🟥 🟩 ⬜️ ⬜️ ⬜️       // chunk 2, original
//            🟨 🟩 ⬜️ ⬜️            // chunk 2, resampled
// ...

function resample(input, output, interpolation, tolerance) {
  if (tolerance === void 0) {
    tolerance = 1e-4;
  }

  __assert(!!exports$1, 'Await "ready" before using module.');

  __assert(input instanceof Float32Array, 'Missing Float32Array input.');

  __assert(output instanceof Float32Array, 'Missing Float32Array output.');

  const outputSize = output.length / input.length;

  __assert(Number.isInteger(outputSize), 'Invalid input/output counts.');

  __assert(interpolation in TO_INTERPOLATION_INTERNAL, 'Invalid interpolation.');

  __assert(Number.isFinite(tolerance), 'Invalid tolerance');

  const interpVal = TO_INTERPOLATION_INTERNAL[interpolation];
  const srcCount = input.length;
  let dstCount = 0;

  for (let chunkStart = 0; chunkStart < input.length; chunkStart += CHUNK_SIZE) {
    const chunkCount = Math.min(srcCount - chunkStart, CHUNK_SIZE); // Allocate a two-keyframe prefix for all chunks after the first.

    const prefixCount = chunkStart > 0 ? 2 : 0;
    const chunkInput = new Float32Array(input.buffer, input.byteOffset + (chunkStart - prefixCount) * Float32Array.BYTES_PER_ELEMENT, chunkCount + prefixCount);
    const chunkOutput = new Float32Array(output.buffer, output.byteOffset + (chunkStart - prefixCount) * outputSize * Float32Array.BYTES_PER_ELEMENT, (chunkCount + prefixCount) * outputSize); // Copy prefix to start of next chunk.

    if (prefixCount > 0) {
      input.copyWithin(chunkStart - prefixCount, dstCount - prefixCount, dstCount);
      output.copyWithin((chunkStart - prefixCount) * outputSize, (dstCount - prefixCount) * outputSize, dstCount * outputSize);
    }

    const inputPtr = __retain(__lowerStaticArray(chunkInput, 4, 2));

    const outputPtr = __retain(__lowerStaticArray(chunkOutput, 4, 2));

    try {
      exports$1.__setArgumentsLength(4);

      const count = exports$1.resample(inputPtr, outputPtr, interpVal, tolerance) >>> 0;
      dstCount -= prefixCount;

      __liftStaticArray(inputPtr, input, dstCount, count);

      __liftStaticArray(outputPtr, output, dstCount * outputSize, count * outputSize);

      dstCount += count;
    } finally {
      __release(inputPtr);

      __release(outputPtr);

      exports$1.__collect();
    }
  } // console.log(`Memory: ${exports.memory.buffer.byteLength} bytes`);


  return dstCount;
} ///////////////////////////////////////////////////////////////////////////////
// INTERNAL
///////////////////////////////////////////////////////////////////////////////

function __assert(cond, msg) {
  if (!cond) throw new Error(msg);
}

function __retain(ptr) {
  exports$1.__pin(ptr);

  return ptr;
}

function __release(ptr) {
  exports$1.__unpin(ptr);

  return ptr;
}

function __liftString(ptr) {
  if (!ptr) return null;
  const end = ptr + new Uint32Array(exports$1.memory.buffer)[ptr - 4 >>> 2] >>> 1,
        memoryU16 = new Uint16Array(exports$1.memory.buffer);
  let start = ptr >>> 1,
      string = '';

  while (end - start > 1024) string += String.fromCharCode(...memoryU16.subarray(start, start += 1024));

  return string + String.fromCharCode(...memoryU16.subarray(start, end));
}

function __lowerStaticArray(values, id, align) {
  const ptr = exports$1.__pin(exports$1.__new(values.length << align, id)) >>> 0;
  new Float32Array(exports$1.memory.buffer, ptr, values.length).set(values);

  exports$1.__unpin(ptr);

  return ptr;
}

function __liftStaticArray(ptr, values, offset, count) {
  values.set(new Float32Array(exports$1.memory.buffer, ptr, count), offset);
}

function __abort(messagePtr, fileNamePtr, lineNumber, columnNumber) {
  const message = __liftString(messagePtr >>> 0);

  const fileName = __liftString(fileNamePtr >>> 0);

  lineNumber = lineNumber >>> 0;
  columnNumber = columnNumber >>> 0;

  (() => {
    throw Error(`${message} in ${fileName}:${lineNumber}:${columnNumber}`);
  })();
}

exports.ready = ready;
exports.resample = resample;
exports.resampleDebug = resampleDebug;
