# gltf-validator

This is an npm package for the official [glTF Validator](https://github.com/KhronosGroup/glTF-Validator/) compiled from Dart to JS.

## Installation

    npm install --save gltf-validator

## Examples

### Basic usage (Node.js)

```javascript
const fs = require('fs');
const validator = require('gltf-validator');

const asset = fs.readFileSync('./Box.gltf');

validator.validateBytes(new Uint8Array(asset))
    .then((report) => console.info('Validation succeeded: ', report))
    .catch((error) => console.error('Validation failed: ', error));
```

### Basic usage (Browser)

```javascript
const validator = require('gltf-validator');

fetch('Box.gltf')
    .then((response) => response.arrayBuffer())
    .then((asset) => validator.validateBytes(new Uint8Array(asset)))
    .then((report) => console.info('Validation succeeded: ', report))
    .catch((error) => console.error('Validation failed: ', error));
```

### Full usage

```javascript
const fs = require("fs");
const path = require("path");
const validator = require('gltf-validator');

const filename = 'Box.gltf';
const fullpath = __dirname + '/' + filename;
const asset = fs.readFileSync(fullpath);

validator.validateBytes(new Uint8Array(asset), {
    uri: filename,
    format: 'gltf', // skip auto-detection and parse the input as glTF JSON
    maxIssues: 10, // limit max number of output issues to 10
    ignoredIssues: ['UNSUPPORTED_EXTENSION'], // mute UNSUPPORTED_EXTENSION issue
    onlyIssues: ['ACCESSOR_INVALID_FLOAT'], // only consider ACCESSOR_INVALID_FLOAT an issue. Cannot be used along with ignoredIssues.
    severityOverrides: { 'ACCESSOR_INDEX_TRIANGLE_DEGENERATE': 0 }, // treat degenerate triangles as errors
    externalResourceFunction: (uri) =>
        new Promise((resolve, reject) => {
            uri = path.resolve(path.dirname(fullpath), decodeURIComponent(uri));
            console.info("Loading external file: " + uri);
            fs.readFile(uri, (err, data) => {
                if (err) {
                    console.error(err.toString());
                    reject(err.toString());
                    return;
                }
                resolve(data);
            });
        })
}).then((result) => {
    // [result] will contain validation report in object form.
    // You can convert it to JSON to see its internal structure. 
    console.log(JSON.stringify(result, null, '  '));
}, (result) => {
    // Promise rejection means that arguments were invalid or validator was unable 
    // to detect file format (glTF or GLB). 
    // [result] will contain exception string.
    console.error(result);
});
```

## API

<!-- Generated by documentation.js. Update this documentation by updating the source code. -->

#### Table of Contents

*   [version](#version)
*   [supportedExtensions](#supportedextensions)
*   [validateBytes](#validatebytes)
    *   [Parameters](#parameters)
*   [validateString](#validatestring)
    *   [Parameters](#parameters-1)
*   [ValidationOptions](#validationoptions)
    *   [Properties](#properties)
*   [ExternalResourceFunction](#externalresourcefunction)
    *   [Parameters](#parameters-2)

### version

Returns a version string.

Returns **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** 

### supportedExtensions

Returns an array of supported extensions names.

Returns **[Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array)<[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)>** 

### validateBytes

Validates an asset from bytes.

#### Parameters

*   `data` **[Uint8Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Uint8Array)** Byte array containing glTF or GLB data.
*   `options` **[ValidationOptions](#validationoptions)** Object with validation options.

Returns **[Promise](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Promise)** Promise with validation result in object form.

### validateString

Validates an asset from JSON string.

#### Parameters

*   `json` **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** String containing glTF JSON.
*   `options` **[ValidationOptions](#validationoptions)** Object with validation options.

Returns **[Promise](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Promise)** Promise with validation result in object form.

### ValidationOptions

Type: [Object](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object)

#### Properties

*   `uri` **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** Absolute or relative asset URI that will be copied to validation report.
*   `format` **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** Set to `glb` or `gltf` to skip auto-detection of the asset format based on the first byte; any other value will be ignored. This option has no effect on `validateString`.
*   `externalResourceFunction` **[ExternalResourceFunction](#externalresourcefunction)** Function for loading external resources. If omitted, external resources are not validated.
*   `writeTimestamp` **[boolean](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Boolean)** Set to `false` to omit timestamp from the validation report. Default is `true`.
*   `maxIssues` **[number](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Number)** Max number of reported issues. Use `0` for unlimited output.
*   `ignoredIssues` **[Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array)<[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)>** Array of ignored issue codes.
*   `onlyIssues` **[Array](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array)<[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)>** Array of only issues to consider. Cannot be used along with ignoredIssues.
*   `severityOverrides` **[Object](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object)** Object with overridden severities for issue codes.

### ExternalResourceFunction

Type: [Function](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Statements/function)

#### Parameters

*   `uri` **[string](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String)** Relative URI of the external resource.

Returns **[Promise](https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Promise)** Promise with Uint8Array data.
