import { Primitive, type Transform } from '@gltf-transform/core';
/** Options for the {@link simplify} function. */
export interface SimplifyOptions {
    /** MeshoptSimplifier instance. */
    simplifier: unknown;
    /** Target ratio (0–1) of vertices to keep. Default: 0.0 (0%). */
    ratio?: number;
    /** Limit on error, as a fraction of mesh radius. Default: 0.0001 (0.01%). */
    error?: number;
    /**
     * Whether to lock topological borders of the mesh. May be necessary when
     * adjacent 'chunks' of a large mesh (e.g. terrain) share a border, helping
     * to ensure no seams appear.
     */
    lockBorder?: boolean;
}
export declare const SIMPLIFY_DEFAULTS: Required<Omit<SimplifyOptions, 'simplifier'>>;
/**
 * Simplification algorithm, based on meshoptimizer, producing meshes with fewer
 * triangles and vertices. Simplification is lossy, but the algorithm aims to
 * preserve visual quality as much as possible for given parameters.
 *
 * The algorithm aims to reach the target 'ratio', while minimizing error. If
 * error exceeds the specified 'error' threshold, the algorithm will quit
 * before reaching the target ratio. Examples:
 *
 * - ratio=0.0, error=0.0001: Aims for maximum simplification, constrained to 0.01% error.
 * - ratio=0.5, error=0.0001: Aims for 50% simplification, constrained to 0.01% error.
 * - ratio=0.5, error=1: Aims for 50% simplification, unconstrained by error.
 *
 * Topology, particularly split vertices, will also limit the simplifier. For
 * best results, apply a {@link weld} operation before simplification.
 *
 * Example:
 *
 * ```javascript
 * import { simplify, weld } from '@gltf-transform/functions';
 * import { MeshoptSimplifier } from 'meshoptimizer';
 *
 * await document.transform(
 *   weld({}),
 *   simplify({ simplifier: MeshoptSimplifier, ratio: 0.75, error: 0.001 })
 * );
 * ```
 *
 * References:
 * - https://github.com/zeux/meshoptimizer/blob/master/js/README.md#simplifier
 *
 * @category Transforms
 */
export declare function simplify(_options: SimplifyOptions): Transform;
/** @hidden */
export declare function simplifyPrimitive(prim: Primitive, _options: SimplifyOptions): Primitive;
