import type { Program } from "../program/index.js";
import { Action, ParserOptions, ParserResult, Option, Argument, CreateArgumentOpts, CommandConfig, CreateOptionCommandOpts } from "../types.js";
import { CustomizedHelpOpts } from "../help/types.js";
/**
 * @ignore
 */
export declare const PROG_CMD = "__self_cmd";
/**
 * @ignore
 */
export declare const HELP_CMD = "help";
/**
 * Command class
 *
 */
export declare class Command {
    private program;
    private _action?;
    private _lastAddedArgOrOpt?;
    private _aliases;
    private _name;
    private _config;
    /**
     * Command description
     *
     * @internal
     */
    readonly description: string;
    /**
     * Command options array
     *
     * @internal
     */
    readonly options: Option[];
    /**
     * Command arguments array
     *
     * @internal
     */
    readonly args: Argument[];
    /**
     *
     * @param program
     * @param name
     * @param description
     * @internal
     */
    constructor(program: Program, name: string, description: string, config?: Partial<CommandConfig>);
    /**
     * Add one or more aliases so the command can be called by different names.
     *
     * @param aliases Command aliases
     */
    alias(...aliases: string[]): Command;
    /**
     * Name getter. Will return an empty string in the program-command context
     *
     * @internal
     */
    get name(): string;
    /**
     * Add an argument to the command.
     * Synopsis is a string like `<my-argument>` or `[my-argument]`.
     * Angled brackets (e.g. `<item>`) indicate required input. Square brackets (e.g. `[env]`) indicate optional input.
     *
     * Returns the {@link Command} object to facilitate chaining of methods.
     *
     * @param synopsis Argument synopsis.
     * @param description - Argument description.
     * @param [options] - Optional parameters including validator and default value.
     */
    argument(synopsis: string, description: string, options?: CreateArgumentOpts): Command;
    /**
     * Set the corresponding action to execute for this command
     *
     * @param action Action to execute
     */
    action(action: Action): Command;
    /**
     * Allow chaining command() calls. See {@link Program.command}.
     *
     */
    command(name: string, description: string, config?: Partial<CommandConfig>): Command;
    /**
     * Makes the command the default one for the program.
     */
    default(): Command;
    /**
     * Checks if the command has the given alias registered.
     *
     * @param alias
     * @internal
     */
    hasAlias(alias: string): boolean;
    /**
     * Get command aliases.
     * @internal
     */
    getAliases(): string[];
    /**
     * @internal
     */
    isProgramCommand(): boolean;
    /**
     * @internal
     */
    isHelpCommand(): boolean;
    /**
     * Hide the command from help.
     * Shortcut to calling `.configure({ visible: false })`.
     */
    hide(): Command;
    /**
     * Add an option to the current command.
     *
     * @param synopsis Option synopsis like '-f, --force', or '-f, --file \<file\>', or '--with-openssl [path]'
     * @param description Option description
     * @param options Additional parameters
     */
    option(synopsis: string, description: string, options?: CreateOptionCommandOpts): Command;
    /**
     * @internal
     */
    getParserConfig(): Partial<ParserOptions>;
    /**
     * Return a reformated synopsis string
     * @internal
     */
    get synopsis(): string;
    /**
     * Customize command help. Can be called multiple times to add more paragraphs and/or sections.
     *
     * @param text Help contents
     * @param options Display options
     */
    help(text: string, options?: Partial<CustomizedHelpOpts>): Command;
    /**
     * Configure some behavioral properties.
     *
     * @param props properties to set/update
     */
    configure(props: Partial<CommandConfig>): Command;
    /**
     * Get a configuration property value.
     *
     * @internal
     * @param key Property key to get value for. See {@link CommandConfig}.
     */
    getConfigProperty<K extends keyof CommandConfig>(key: K): CommandConfig[K];
    /**
     * Get the auto-casting flag.
     *
     * @internal
     */
    get autoCast(): boolean;
    /**
     * Auto-complete
     */
    complete(completer: unknown): Command;
    /**
     * Toggle strict mode.
     * Shortcut to calling: `.configure({ strictArgsCount: strict, strictOptions: strict }).
     * By default, strict settings are not defined for commands, and inherit from the
     * program settings. Calling `.strict(value)` on a command will override the program
     * settings.
     *
     * @param strict boolean enabled flag
     */
    strict(strict?: boolean): Command;
    /**
     * Computed strictOptions flag.
     *
     * @internal
     */
    get strictOptions(): boolean;
    /**
     * Computed strictArgsCount flag.
     *
     * @internal
     */
    get strictArgsCount(): boolean;
    /**
     * Enable or disable auto casting of arguments & options for the command.
     * This is basically a shortcut to calling `command.configure({ autoCast: enabled })`.
     * By default, auto-casting is inherited from the program configuration.
     * This method allows overriding what's been set on the program level.
     *
     * @param enabled
     */
    cast(enabled: boolean): Command;
    /**
     * Visible flag
     *
     * @internal
     */
    get visible(): boolean;
    /**
     * Run the action associated with the command
     *
     * @internal
     */
    run(parsed: Partial<ParserResult>): Promise<unknown>;
}
/**
 * Create a new command
 *
 * @internal
 */
export declare function createCommand(...args: ConstructorParameters<typeof Command>): InstanceType<typeof Command>;
