<?php
/**
 * Rate Limiter - Verhindert zu viele Requests von derselben IP
 * 
 * Verwendung:
 * require_once 'includes/rate_limiter.php';
 * $limiter = new RateLimiter();
 * $limiter->checkLimit($_SERVER['REMOTE_ADDR'], 60, 60); // 60 Requests pro Minute
 */

class RateLimiter {
    private $cacheDir;
    private $cleanupInterval = 300; // 5 Minuten
    
    public function __construct() {
        $this->cacheDir = __DIR__ . '/../cache/rate_limit/';
        if (!is_dir($this->cacheDir)) {
            mkdir($this->cacheDir, 0755, true);
        }
        
        // Zufaellige Cleanup (nicht bei jedem Request)
        if (rand(1, 100) === 1) {
            $this->cleanup();
        }
    }
    
    /**
     * Prueft ob die IP das Limit ueberschreitet
     * 
     * @param string $ip IP-Adresse
     * @param int $maxRequests Maximale Anzahl Requests
     * @param int $timeWindow Zeitfenster in Sekunden
     * @return bool true wenn erlaubt, false wenn blockiert
     */
    public function checkLimit($ip, $maxRequests = 60, $timeWindow = 60) {
        $file = $this->cacheDir . md5($ip) . '.txt';
        $now = time();
        
        if (file_exists($file)) {
            $data = json_decode(file_get_contents($file), true);
            
            if (!$data || !isset($data['requests'])) {
                $data = ['requests' => []];
            }
            
            // Alte Eintraege entfernen (aelters als timeWindow)
            $data['requests'] = array_filter($data['requests'], function($timestamp) use ($now, $timeWindow) {
                return ($now - $timestamp) < $timeWindow;
            });
            
            // Pruefe Limit
            if (count($data['requests']) >= $maxRequests) {
                $this->logBlocked($ip, count($data['requests']));
                http_response_code(429);
                header('Retry-After: ' . $timeWindow);
                die(json_encode([
                    'error' => 'Rate limit exceeded',
                    'message' => 'Zu viele Anfragen. Bitte versuchen Sie es spaeter erneut.',
                    'retry_after' => $timeWindow
                ]));
            }
        } else {
            $data = ['requests' => []];
        }
        
        // Neuen Request hinzufuegen
        $data['requests'][] = $now;
        file_put_contents($file, json_encode($data));
        
        return true;
    }
    
    /**
     * Loggt blockierte Requests
     */
    private function logBlocked($ip, $requestCount) {
        $logFile = $this->cacheDir . 'blocked.log';
        $logEntry = date('Y-m-d H:i:s') . " - IP: $ip - Requests: $requestCount\n";
        file_put_contents($logFile, $logEntry, FILE_APPEND);
    }
    
    /**
     * Bereinigt alte Cache-Dateien
     */
    private function cleanup() {
        $files = glob($this->cacheDir . '*.txt');
        $now = time();
        
        foreach ($files as $file) {
            if (basename($file) === 'blocked.log') {
                continue;
            }
            
            $mtime = filemtime($file);
            if ($now - $mtime > $this->cleanupInterval) {
                unlink($file);
            }
        }
    }
    
    /**
     * Setzt das Limit fuer eine IP zurueck
     */
    public function resetLimit($ip) {
        $file = $this->cacheDir . md5($ip) . '.txt';
        if (file_exists($file)) {
            unlink($file);
        }
    }
}



