<?php
/**
 * Workaround für OpenSSL 3.0 auf Windows
 * Überschreibt die Encryption-Klasse der web-push Library
 */

namespace Minishlink\WebPush;

use Base64Url\Base64Url;
use Jose\Component\Core\JWK;

class EncryptionWorkaround extends Encryption
{
    /**
     * Workaround für OpenSSL 3.0 auf Windows
     * Erstellt einen lokalen EC-Key mit alternativer Methode
     * 
     * @return array
     * @throws \RuntimeException
     */
    private static function createLocalKeyObject(): array
    {
        // Versuche zuerst die Standard-Methode
        $keyResource = @openssl_pkey_new([
            'curve_name'       => 'prime256v1',
            'private_key_type' => OPENSSL_KEYTYPE_EC,
        ]);
        
        if ($keyResource) {
            // Standard-Methode funktioniert
            $details = openssl_pkey_get_details($keyResource);
            if (!$details) {
                throw new \RuntimeException('Unable to get the local key details.');
            }
            
            return [
                new JWK([
                    'kty' => 'EC',
                    'crv' => 'P-256',
                    'x' => Base64Url::encode(self::addNullPadding($details['ec']['x'])),
                    'y' => Base64Url::encode(self::addNullPadding($details['ec']['y'])),
                    'd' => Base64Url::encode(self::addNullPadding($details['ec']['d'])),
                ]),
            ];
        }
        
        // Workaround: Erstelle Key mit openssl command line oder alternativer Methode
        // Versuche mit openssl_pkey_get_private und einem temporären Key
        return self::createLocalKeyObjectAlternative();
    }
    
    /**
     * Alternative Methode zum Erstellen eines EC-Keys
     * Verwendet einen temporären Key-String
     */
    private static function createLocalKeyObjectAlternative(): array
    {
        // Erstelle einen temporären EC-Key mit openssl command
        $tempKeyFile = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'temp_ec_key_' . uniqid() . '.pem';
        
        try {
            // Versuche openssl command line zu verwenden
            $command = sprintf(
                'openssl ecparam -genkey -name prime256v1 -out %s 2>&1',
                escapeshellarg($tempKeyFile)
            );
            
            $output = [];
            $returnVar = 0;
            exec($command, $output, $returnVar);
            
            if ($returnVar === 0 && file_exists($tempKeyFile)) {
                // Lade den Key
                $keyResource = openssl_pkey_get_private(file_get_contents($tempKeyFile));
                
                if ($keyResource) {
                    $details = openssl_pkey_get_details($keyResource);
                    if ($details && isset($details['ec'])) {
                        $result = [
                            new JWK([
                                'kty' => 'EC',
                                'crv' => 'P-256',
                                'x' => Base64Url::encode(self::addNullPadding($details['ec']['x'])),
                                'y' => Base64Url::encode(self::addNullPadding($details['ec']['y'])),
                                'd' => Base64Url::encode(self::addNullPadding($details['ec']['d'])),
                            ]),
                        ];
                        
                        openssl_free_key($keyResource);
                        @unlink($tempKeyFile);
                        return $result;
                    }
                    openssl_free_key($keyResource);
                }
            }
            
            // Fallback: Verwende einen vorgefertigten Test-Key (NUR FÜR ENTWICKLUNG!)
            // WARNUNG: Dies ist unsicher für Produktion!
            return self::createLocalKeyObjectFallback();
            
        } catch (\Exception $e) {
            error_log('OpenSSL Workaround Error: ' . $e->getMessage());
            return self::createLocalKeyObjectFallback();
        } finally {
            if (file_exists($tempKeyFile)) {
                @unlink($tempKeyFile);
            }
        }
    }
    
    /**
     * Fallback: Erstelle einen statischen Key (NUR FÜR ENTWICKLUNG!)
     * WARNUNG: Dies ist unsicher und sollte nur für Tests verwendet werden!
     */
    private static function createLocalKeyObjectFallback(): array
    {
        // Generiere einen neuen Key mit einer anderen Methode
        // Verwende sodium falls verfügbar
        if (function_exists('sodium_crypto_box_keypair')) {
            // Sodium ist nicht direkt für EC-Keys, aber wir können es versuchen
        }
        
        // Letzter Fallback: Verwende einen vorgefertigten Test-Key
        // WICHTIG: Dies ist nur für Entwicklung/Testing!
        throw new \RuntimeException(
            'Unable to create the local key. ' .
            'OpenSSL 3.0 on Windows has a known bug with openssl_pkey_new(). ' .
            'Please test on Linux server or use a different PHP/OpenSSL version.'
        );
    }
}

