<?php
/**
 * Modulares Benachrichtigungs-Trigger-System
 * 
 * Dieses System erlaubt es, für neue Features einfach Benachrichtigungs-Trigger zu definieren.
 * 
 * Verwendung:
 * 1. Erstelle einen neuen Trigger in $notification_triggers
 * 2. Verwende sendNotificationForTrigger() in deinem Feature-Code
 * 
 * Beispiel:
 *   sendNotificationForTrigger('appointment_confirmed', $user_id, [
 *     'appointment_id' => 123,
 *     'event_name' => 'Winterreifenwechsel'
 *   ]);
 */

require_once __DIR__ . '/push_notification_helper.php';

/**
 * Definiert alle verfügbaren Benachrichtigungs-Trigger
 * 
 * Struktur:
 * 'trigger_name' => [
 *   'enabled' => true/false,           // Ob dieser Trigger aktiviert ist
 *   'title_template' => '...',        // Titel-Template (kann Platzhalter enthalten)
 *   'message_template' => '...',       // Nachricht-Template (kann Platzhalter enthalten)
 *   'type' => 'info|success|warning|error',
 *   'url' => '/path/to/page',          // URL zum Öffnen
 *   'send_push' => true/false,         // Ob Push-Benachrichtigung gesendet werden soll
 *   'conditions' => function($data) {  // Optional: Bedingungen prüfen
 *     return true; // oder false
 *   }
 * ]
 */
$notification_triggers = [
    // Beispiel: Termin bestätigt
    'appointment_confirmed' => [
        'enabled' => true,
        'title_template' => 'Termin bestätigt',
        'message_template' => 'Ihr Termin für "{event_name}" am {date} um {time} wurde bestätigt.',
        'type' => 'success',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Termin storniert
    'appointment_cancelled' => [
        'enabled' => true,
        'title_template' => 'Termin storniert',
        'message_template' => 'Ihr Termin für "{event_name}" wurde storniert.',
        'type' => 'info',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Dokument genehmigt
    'document_approved' => [
        'enabled' => true,
        'title_template' => 'Dokument genehmigt',
        'message_template' => 'Ihr Verifikationsdokument wurde genehmigt.',
        'type' => 'success',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Dokument abgelehnt
    'document_rejected' => [
        'enabled' => true,
        'title_template' => 'Dokument abgelehnt',
        'message_template' => 'Ihr Verifikationsdokument wurde abgelehnt. Grund: {reason}',
        'type' => 'error',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Feedback-Antwort
    'feedback_response' => [
        'enabled' => true,
        'title_template' => 'Antwort auf Ihr Feedback',
        'message_template' => 'Sie haben eine Antwort auf Ihr Feedback erhalten.',
        'type' => 'info',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Account freigeschaltet
    'account_approved' => [
        'enabled' => true,
        'title_template' => 'Account freigeschaltet',
        'message_template' => 'Ihr Account wurde erfolgreich freigeschaltet. Sie können sich jetzt anmelden.',
        'type' => 'success',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Account abgelehnt
    'account_rejected' => [
        'enabled' => true,
        'title_template' => 'Account abgelehnt',
        'message_template' => 'Ihr Account wurde abgelehnt. Grund: {reason}',
        'type' => 'error',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Account gesperrt
    'account_suspended' => [
        'enabled' => true,
        'title_template' => 'Account gesperrt',
        'message_template' => 'Ihr Account wurde von einem Administrator gesperrt. {reason}',
        'type' => 'warning',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Account entsperrt
    'account_unsuspended' => [
        'enabled' => true,
        'title_template' => 'Account entsperrt',
        'message_template' => 'Ihr Account wurde von einem Administrator entsperrt.',
        'type' => 'success',
        'url' => '/profile.php',
        'send_push' => true,
    ],
    
    // Beispiel: Neues Feature - Reifenangebot
    'tire_offer_available' => [
        'enabled' => false, // Deaktiviert bis Feature fertig ist
        'title_template' => 'Neues Reifenangebot',
        'message_template' => 'Ein neues Reifenangebot ist für Sie verfügbar.',
        'type' => 'info',
        'url' => '/profile.php',
        'send_push' => true,
        'conditions' => function($data) {
            // Beispiel: Nur senden wenn bestimmte Bedingungen erfüllt sind
            return isset($data['offer_id']) && !empty($data['offer_id']);
        }
    ],
    
    // Verifizierung: Konto wurde freigeschaltet
    'verification_approved' => [
        'enabled' => true,
        'title_template' => 'Verifizierung erfolgreich',
        'message_template' => 'Ihr Konto wurde erfolgreich verifiziert und freigeschaltet. Bitte vervollständigen Sie Ihr Profil!',
        'type' => 'success',
        'url' => '/profile.php?complete_profile=1',
        'send_push' => true,
    ],
    
    // Verifizierung: Konto wurde abgelehnt
    'verification_rejected' => [
        'enabled' => true,
        'title_template' => 'Verifizierung abgelehnt',
        'message_template' => 'Ihr Verifizierungsantrag wurde abgelehnt. Grund: {reason}',
        'type' => 'error',
        'url' => '/profile.php?complete_profile=1',
        'send_push' => true,
    ],
    
    // Verifizierung: Dokument wurde hochgeladen
    'verification_document_uploaded' => [
        'enabled' => true,
        'title_template' => 'Dokument hochgeladen',
        'message_template' => 'Ihr Verifikationsdokument wurde erfolgreich hochgeladen und wird nun geprüft. Sie erhalten eine Benachrichtigung, sobald die Prüfung abgeschlossen ist.',
        'type' => 'info',
        'url' => '/profile.php?complete_profile=1',
        'send_push' => true,
    ],
];

/**
 * Sendet eine Benachrichtigung für einen bestimmten Trigger
 * 
 * @param string $trigger_name Name des Triggers (muss in $notification_triggers definiert sein)
 * @param int $user_id Benutzer-ID
 * @param array $data Daten für Platzhalter im Template (z.B. ['event_name' => 'Winterreifenwechsel'])
 * @return array Ergebnis mit 'success', 'notification_id', 'push_sent'
 */
function sendNotificationForTrigger($trigger_name, $user_id, $data = []) {
    global $notification_triggers;
    
    // Prüfe ob Trigger existiert
    if (!isset($notification_triggers[$trigger_name])) {
        error_log("Notification Trigger '$trigger_name' nicht gefunden");
        return [
            'success' => false,
            'message' => "Trigger '$trigger_name' nicht gefunden"
        ];
    }
    
    $trigger = $notification_triggers[$trigger_name];
    
    // Prüfe ob Trigger aktiviert ist
    if (!($trigger['enabled'] ?? true)) {
        return [
            'success' => false,
            'message' => "Trigger '$trigger_name' ist deaktiviert"
        ];
    }
    
    // Prüfe Bedingungen falls vorhanden
    if (isset($trigger['conditions']) && is_callable($trigger['conditions'])) {
        if (!$trigger['conditions']($data)) {
            return [
                'success' => false,
                'message' => "Bedingungen für Trigger '$trigger_name' nicht erfüllt"
            ];
        }
    }
    
    // Ersetze Platzhalter in Templates
    $title = replacePlaceholders($trigger['title_template'], $data);
    $message = replacePlaceholders($trigger['message_template'], $data);
    $type = $trigger['type'] ?? 'info';
    $url = replacePlaceholders($trigger['url'] ?? '/profile.php', $data);
    
    // Sende Benachrichtigung
    require_once __DIR__ . '/../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    if ($trigger['send_push'] ?? true) {
        // Mit Push-Benachrichtigung
        return createNotificationWithPush($conn, $user_id, $title, $message, $type, $url);
    } else {
        // Nur Datenbank-Benachrichtigung
        try {
            $stmt = $conn->prepare("
                INSERT INTO user_notifications (user_id, type, title, message, is_read, created_at)
                VALUES (?, ?, ?, ?, 0, NOW())
            ");
            $stmt->execute([$user_id, $type, $title, $message]);
            $notification_id = $conn->lastInsertId();
            
            return [
                'success' => true,
                'notification_id' => $notification_id,
                'push_sent' => false
            ];
        } catch (Exception $e) {
            error_log('Error creating notification: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
}

/**
 * Ersetzt Platzhalter in einem Template-String
 * 
 * @param string $template Template-String mit Platzhaltern wie {key} oder "{key}"
 * @param array $data Daten-Array
 * @return string Ersetzter String
 */
function replacePlaceholders($template, $data) {
    $result = $template;
    
    foreach ($data as $key => $value) {
        // Ersetze {key} und "{key}"
        $result = str_replace(['{' . $key . '}', '"{' . $key . '}"'], [$value, '"' . $value . '"'], $result);
    }
    
    return $result;
}

/**
 * Aktiviert oder deaktiviert einen Trigger
 * 
 * @param string $trigger_name Name des Triggers
 * @param bool $enabled true = aktiviert, false = deaktiviert
 */
function setTriggerEnabled($trigger_name, $enabled) {
    global $notification_triggers;
    
    if (isset($notification_triggers[$trigger_name])) {
        $notification_triggers[$trigger_name]['enabled'] = $enabled;
        
        // Speichere in Datenbank
        try {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                INSERT INTO settings (setting_key, setting_value)
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute(["trigger_enabled_{$trigger_name}", (int)$enabled]);
        } catch (Exception $e) {
            error_log("Error saving trigger state: " . $e->getMessage());
        }
    }
}

/**
 * Gibt alle verfügbaren Trigger zurück
 * 
 * @return array Array mit allen Triggern
 */
function getAllTriggers() {
    global $notification_triggers;
    
    // Lade Trigger-Status aus Datenbank
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        $stmt = $conn->query("SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE 'trigger_enabled_%'");
        $db_trigger_settings = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $trigger_name = str_replace('trigger_enabled_', '', $row['setting_key']);
            $db_trigger_settings[$trigger_name] = (bool)$row['setting_value'];
        }
        
        // Überschreibe die Standard-Trigger-Konfiguration mit den Datenbank-Einstellungen
        foreach ($notification_triggers as $name => &$config) {
            if (isset($db_trigger_settings[$name])) {
                $config['enabled'] = $db_trigger_settings[$name];
            }
        }
        unset($config); // Wichtig, um Referenz zu entfernen
    } catch (Exception $e) {
        error_log("Error loading trigger states: " . $e->getMessage());
    }
    
    return $notification_triggers;
}

/**
 * Gibt einen spezifischen Trigger zurück
 * 
 * @param string $trigger_name Name des Triggers
 * @return array|null Trigger-Daten oder null
 */
function getTrigger($trigger_name) {
    global $notification_triggers;
    return $notification_triggers[$trigger_name] ?? null;
}

?>

