<?php
/**
 * CSRF Protection Utilities
 * Verhindert Cross-Site Request Forgery Angriffe
 */

// Sicherstellen dass Session gestartet ist
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Generiert ein CSRF-Token und speichert es in der Session
 * @return string CSRF-Token
 */
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Validiert ein CSRF-Token
 * @param string $token Das zu validierende Token
 * @return bool True wenn Token gültig, sonst False
 */
function validate_csrf_token($token) {
    if (!isset($_SESSION['csrf_token'])) {
        return false;
    }
    return hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Validiert ein CSRF-Token und gibt bei Fehler eine Fehlermeldung aus
 * @param string $token Das zu validierende Token
 * @param bool $throw_exception Wenn true, wirft eine Exception bei ungültigem Token
 * @return bool True wenn Token gültig
 * @throws Exception Wenn Token ungültig und $throw_exception true ist
 */
function require_csrf_token($token, $throw_exception = true) {
    if (!validate_csrf_token($token)) {
        if ($throw_exception) {
            http_response_code(403);
            throw new Exception('CSRF token mismatch. Möglicherweise ist Ihre Session abgelaufen. Bitte laden Sie die Seite neu.');
        }
        return false;
    }
    return true;
}

/**
 * Generiert ein verstecktes Formular-Feld mit CSRF-Token
 * @return string HTML input-Feld
 */
function csrf_field() {
    $token = generate_csrf_token();
    return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
}

/**
 * Gibt das CSRF-Token zurück (für AJAX-Requests)
 * @return string CSRF-Token
 */
function csrf_token() {
    return generate_csrf_token();
}

/**
 * Validiert ein CSRF-Token aus POST-Request
 * @return bool True wenn Token gültig
 */
function validate_csrf_post() {
    $token = $_POST['csrf_token'] ?? '';
    return validate_csrf_token($token);
}

/**
 * Validiert ein CSRF-Token aus JSON-Request
 * @return bool True wenn Token gültig
 */
function validate_csrf_json() {
    $input = json_decode(file_get_contents('php://input'), true);
    $token = $input['csrf_token'] ?? '';
    return validate_csrf_token($token);
}

/**
 * Validiert ein CSRF-Token aus Header (z. B. für AJAX)
 * @return bool True wenn Token gültig
 */
function validate_csrf_header() {
    $token = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
    return validate_csrf_token($token);
}

/**
 * Regeneriert das CSRF-Token (z. B. nach erfolgreicher Form-Submission)
 * @return string Neues CSRF-Token
 */
function regenerate_csrf_token() {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    return $_SESSION['csrf_token'];
}

?>

