/**
 * B&E Boxenstop - Intelligente PWA Install-Logik
 * Überzeugt Kunden strategisch und indirekt, die App zu installieren
 */

// ============================================
// SOFORTIGE VERFÜGBARKEIT FÜR DEBUG - ABSOLUT ERSTE ZEILE
// ============================================
// KEINE Abhängigkeiten - muss IMMER funktionieren
try {
  window.PWA_DEBUG = window.PWA_DEBUG || {};
  console.log('%c✅ [PWA] Script START - PWA_DEBUG erstellt!', 'color: #28a745; font-weight: bold; font-size: 14px;');
  console.log('[PWA] PWA_DEBUG verfügbar:', typeof window.PWA_DEBUG, window.PWA_DEBUG);
} catch (e) {
  console.error('[PWA] KRITISCHER FEHLER bei PWA_DEBUG Erstellung:', e);
  // Letzter Versuch
  try {
    if (typeof window === 'undefined') {
      this.PWA_DEBUG = this.PWA_DEBUG || {};
    } else {
      window.PWA_DEBUG = window.PWA_DEBUG || {};
    }
  } catch (e2) {
    console.error('[PWA] PWA_DEBUG konnte NICHT erstellt werden:', e2);
  }
}

// ============================================
// KONFIGURATION
// ============================================
const PWA_CONFIG = {
  // Engagement-Schwellenwerte
  MIN_PAGE_VIEWS: 2,              // Mindestanzahl Seitenaufrufe
  MIN_SESSION_TIME: 30000,         // Mindestzeit auf Seite (30 Sekunden)
  MIN_ACTIONS: 1,                  // Mindestanzahl Aktionen (Klicks, Scrolls, etc.)
  
  // Timing-Konfiguration
  DELAY_AFTER_LOGIN: 5000,         // Verzögerung nach Login (5 Sekunden)
  DELAY_AFTER_ACTION: 3000,        // Verzögerung nach wichtiger Aktion (3 Sekunden)
  DELAY_INITIAL: 10000,            // Initiale Verzögerung (10 Sekunden)
  
  // Wiederholungs-Logik
  MAX_SHOW_ATTEMPTS: 3,            // Max. Anzeige-Versuche
  COOLDOWN_DAYS: 7,                // Tage bis zum nächsten Versuch nach Ablehnung
  
  // Auto-Install Option
  AUTO_INSTALL_DELAY: 3000,        // Verzögerung bis Auto-Install (3 Sekunden, 0 = deaktiviert)
  AUTO_INSTALL_ENABLED: true,      // Auto-Install aktivieren (nur wenn deferredPrompt verfügbar)
  
  // LocalStorage Keys
  STORAGE_KEY: 'pwa_install_data',
  INSTALLED_KEY: 'pwa_installed',
  DECLINED_KEY: 'pwa_declined'
};

// ============================================
// STATE MANAGEMENT
// ============================================
let deferredPrompt = null;
let installModal = null;
let engagementTracker = {
  pageViews: 0,
  sessionTime: 0,
  actions: 0,
  lastActionTime: null,
  startTime: Date.now()
};

// ============================================
// UTILITY FUNCTIONS
// ============================================

/**
 * Prüft ob PWA bereits installiert ist
 */
function isPWAInstalled() {
  // Prüfe verschiedene Indikatoren
  if (window.matchMedia('(display-mode: standalone)').matches) {
    return true;
  }
  if (window.navigator.standalone === true) {
    return true; // iOS Safari
  }
  if (document.referrer.includes('android-app://')) {
    return true; // Android
  }
  
  // Prüfe LocalStorage
  const installed = localStorage.getItem(PWA_CONFIG.INSTALLED_KEY);
  return installed === 'true';
}

/**
 * Lädt gespeicherte Install-Daten
 */
function loadInstallData() {
  try {
    const data = localStorage.getItem(PWA_CONFIG.STORAGE_KEY);
    if (data) {
      return JSON.parse(data);
    }
  } catch (e) {
    console.error('[PWA] Fehler beim Laden der Install-Daten:', e);
  }
  return {
    showCount: 0,
    lastShown: null,
    declined: false,
    declinedDate: null
  };
}

/**
 * Speichert Install-Daten
 */
function saveInstallData(data) {
  try {
    localStorage.setItem(PWA_CONFIG.STORAGE_KEY, JSON.stringify(data));
  } catch (e) {
    console.error('[PWA] Fehler beim Speichern der Install-Daten:', e);
  }
}

/**
 * Prüft ob Modal bereits abgelehnt wurde
 */
function isDeclined() {
  const declined = localStorage.getItem(PWA_CONFIG.DECLINED_KEY);
  if (!declined) return false;
  
  const declinedDate = new Date(declined);
  const now = new Date();
  const daysDiff = (now - declinedDate) / (1000 * 60 * 60 * 24);
  
  // Wenn Cooldown abgelaufen, erlaube erneute Anzeige
  if (daysDiff >= PWA_CONFIG.COOLDOWN_DAYS) {
    localStorage.removeItem(PWA_CONFIG.DECLINED_KEY);
    return false;
  }
  
  return true;
}

/**
 * Prüft ob Modal angezeigt werden sollte
 */
function shouldShowModal() {
  // Nicht anzeigen wenn bereits installiert
  if (isPWAInstalled()) {
    return false;
  }
  
  // Nicht anzeigen wenn abgelehnt (und Cooldown noch aktiv)
  if (isDeclined()) {
    return false;
  }
  
  const data = loadInstallData();
  
  // Nicht anzeigen wenn Max-Versuche erreicht
  if (data.showCount >= PWA_CONFIG.MAX_SHOW_ATTEMPTS) {
    return false;
  }
  
  // Prüfe Engagement-Kriterien
  const hasMinPageViews = engagementTracker.pageViews >= PWA_CONFIG.MIN_PAGE_VIEWS;
  const hasMinSessionTime = engagementTracker.sessionTime >= PWA_CONFIG.MIN_SESSION_TIME;
  const hasMinActions = engagementTracker.actions >= PWA_CONFIG.MIN_ACTIONS;
  
  return hasMinPageViews && hasMinSessionTime && hasMinActions;
}

/**
 * Markiert Installation als abgelehnt
 */
function markAsDeclined() {
  localStorage.setItem(PWA_CONFIG.DECLINED_KEY, new Date().toISOString());
  const data = loadInstallData();
  data.declined = true;
  data.declinedDate = new Date().toISOString();
  saveInstallData(data);
}

/**
 * Markiert Installation als erfolgreich
 */
function markAsInstalled() {
  localStorage.setItem(PWA_CONFIG.INSTALLED_KEY, 'true');
  const data = loadInstallData();
  data.installed = true;
  data.installedDate = new Date().toISOString();
  saveInstallData(data);
}

// ============================================
// ENGAGEMENT TRACKING
// ============================================

/**
 * Initialisiert Engagement-Tracking
 */
function initEngagementTracking() {
  // Seitenaufruf zählen
  engagementTracker.pageViews = parseInt(sessionStorage.getItem('pwa_page_views') || '0') + 1;
  sessionStorage.setItem('pwa_page_views', engagementTracker.pageViews.toString());
  
  // Session-Zeit tracken
  setInterval(() => {
    engagementTracker.sessionTime = Date.now() - engagementTracker.startTime;
  }, 1000);
  
  // Aktionen tracken (Klicks, Scrolls)
  let actionCount = 0;
  const trackAction = () => {
    actionCount++;
    engagementTracker.actions = actionCount;
    engagementTracker.lastActionTime = Date.now();
  };
  
  // Klick-Tracking
  document.addEventListener('click', trackAction, { passive: true });
  
  // Scroll-Tracking (nur bei signifikantem Scroll)
  let lastScrollTop = 0;
  document.addEventListener('scroll', () => {
    const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
    if (Math.abs(scrollTop - lastScrollTop) > 100) {
      trackAction();
      lastScrollTop = scrollTop;
    }
  }, { passive: true });
  
  // Form-Interaktionen tracken
  document.addEventListener('focus', (e) => {
    if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA' || e.target.tagName === 'SELECT') {
      trackAction();
    }
  }, { passive: true });
}

// ============================================
// MODAL UI
// ============================================

/**
 * Erstellt das Install-Modal mit überzeugendem Design
 */
function createInstallModal() {
  if (installModal) return;
  
  // Prüfe ob bereits installiert
  if (isPWAInstalled()) {
    return;
  }
  
  const data = loadInstallData();
  data.showCount++;
  data.lastShown = new Date().toISOString();
  saveInstallData(data);
  
  // Modal Container
  installModal = document.createElement('div');
  installModal.id = 'pwa-install-modal';
  installModal.className = 'pwa-install-modal';
  installModal.setAttribute('role', 'dialog');
  installModal.setAttribute('aria-labelledby', 'pwa-modal-title');
  installModal.setAttribute('aria-modal', 'true');
  
  // Modal Content
  const modalContent = document.createElement('div');
  modalContent.className = 'pwa-modal-content';
  
  // Header - Minimalistisch
  const header = document.createElement('div');
  header.className = 'pwa-modal-header';
  header.innerHTML = `
    <div class="pwa-icon-container">
      <img src="assets/images/new-pwa_icon.png" alt="B&E Gruppe App" class="pwa-app-icon">
    </div>
    <h2 id="pwa-modal-title" class="pwa-modal-title">
      App installieren
    </h2>
    <p class="pwa-modal-subtitle">Für schnelleren Zugriff und bessere Erfahrung</p>
  `;
  
  // Vorteile-Liste - Minimalistisch mit FontAwesome
  const benefits = document.createElement('div');
  benefits.className = 'pwa-benefits';
  benefits.innerHTML = `
    <div class="pwa-benefit-item">
      <i class="fas fa-bolt pwa-benefit-icon"></i>
      <div class="pwa-benefit-text">
        <strong>Schneller Zugriff</strong>
        <span>Direkt vom Startbildschirm</span>
      </div>
    </div>
    <div class="pwa-benefit-item">
      <i class="fas fa-mobile-alt pwa-benefit-icon"></i>
      <div class="pwa-benefit-text">
        <strong>Native App-Erfahrung</strong>
        <span>Ohne Browser</span>
      </div>
    </div>
    <div class="pwa-benefit-item">
      <i class="fas fa-bell pwa-benefit-icon"></i>
      <div class="pwa-benefit-text">
        <strong>Push-Benachrichtigungen</strong>
        <span>Wichtige Updates sofort</span>
      </div>
    </div>
  `;
  
  // Buttons - Minimalistisch
  const buttons = document.createElement('div');
  buttons.className = 'pwa-modal-buttons';
  
  // Auto-Install Info Text (wenn aktiviert)
  let autoInstallInfo = '';
  if (PWA_CONFIG.AUTO_INSTALL_ENABLED && 
      PWA_CONFIG.AUTO_INSTALL_DELAY > 0 && 
      deferredPrompt) {
    autoInstallInfo = `<p class="pwa-auto-install-info">Installation startet automatisch in <span id="pwa-countdown">${Math.ceil(PWA_CONFIG.AUTO_INSTALL_DELAY / 1000)}</span> Sekunden</p>`;
  }
  
  buttons.innerHTML = `
    ${autoInstallInfo}
    <div class="pwa-btn-group">
      <button class="pwa-btn pwa-btn-secondary" id="pwa-decline-btn">
        Später
      </button>
      <button class="pwa-btn pwa-btn-primary" id="pwa-install-btn">
        <i class="fas fa-download pwa-btn-icon"></i>
        Jetzt installieren
      </button>
    </div>
  `;
  
  // Zusammenbauen
  modalContent.appendChild(header);
  modalContent.appendChild(benefits);
  modalContent.appendChild(buttons);
  installModal.appendChild(modalContent);

  // Event Listeners
  const installBtn = installModal.querySelector('#pwa-install-btn');
  const declineBtn = installModal.querySelector('#pwa-decline-btn');
  const closeBtn = document.createElement('button');
  closeBtn.className = 'pwa-modal-close';
  closeBtn.innerHTML = '<i class="fas fa-times"></i>';
  closeBtn.setAttribute('aria-label', 'Schließen');
  modalContent.appendChild(closeBtn);
  
  // Countdown für Auto-Install
  let countdownInterval = null;
  if (PWA_CONFIG.AUTO_INSTALL_ENABLED && 
      PWA_CONFIG.AUTO_INSTALL_DELAY > 0 && 
      deferredPrompt) {
    const countdownElement = installModal.querySelector('#pwa-countdown');
    if (countdownElement) {
      let remaining = Math.ceil(PWA_CONFIG.AUTO_INSTALL_DELAY / 1000);
      countdownInterval = setInterval(() => {
        remaining--;
        if (countdownElement && remaining >= 0) {
          countdownElement.textContent = remaining;
        }
        if (remaining <= 0) {
          clearInterval(countdownInterval);
          countdownInterval = null;
        }
      }, 1000);
    }
  }
  
  // Helper: Stoppe Auto-Install
  const stopAutoInstall = () => {
    if (countdownInterval) {
      clearInterval(countdownInterval);
      countdownInterval = null;
    }
    if (installModal && installModal._autoInstallTimeout) {
      clearTimeout(installModal._autoInstallTimeout);
      installModal._autoInstallTimeout = null;
    }
    const countdownInfo = installModal?.querySelector('.pwa-auto-install-info');
    if (countdownInfo) {
      countdownInfo.remove();
    }
  };
  
  // Install Button - Stoppt Auto-Install
  installBtn.addEventListener('click', async () => {
    stopAutoInstall();
    await handleInstall();
  });
  
  // Decline Button - Stoppt Auto-Install
  declineBtn.addEventListener('click', () => {
    stopAutoInstall();
    markAsDeclined();
    hideModal();
  });
  
  // Close Button - Stoppt Auto-Install
  closeBtn.addEventListener('click', () => {
    stopAutoInstall();
    hideModal();
  });
  
  // Klick außerhalb schließen - Stoppt Auto-Install
  installModal.addEventListener('click', (e) => {
    if (e.target === installModal) {
      stopAutoInstall();
      hideModal();
    }
  });
  
  // ESC-Taste schließen - Stoppt Auto-Install
  const escapeHandler = (e) => {
    if (e.key === 'Escape' && installModal && installModal.parentNode) {
      stopAutoInstall();
      hideModal();
      document.removeEventListener('keydown', escapeHandler);
    }
  };
  document.addEventListener('keydown', escapeHandler);
  
  // Füge Modal zum Body hinzu
  document.body.appendChild(installModal);
  
  // Animation: Fade In
  requestAnimationFrame(() => {
    installModal.classList.add('pwa-modal-visible');
    
    // Auto-Install: Zeige Browser-Prompt automatisch nach Verzögerung
    if (PWA_CONFIG.AUTO_INSTALL_ENABLED && 
        PWA_CONFIG.AUTO_INSTALL_DELAY > 0 && 
        deferredPrompt) {
      setTimeout(() => {
        // Prüfe ob Modal noch sichtbar ist und nicht bereits interagiert wurde
        if (installModal && installModal.parentNode && !installBtn.disabled) {
          console.log('[PWA] Auto-Install: Zeige Browser-Prompt automatisch');
          handleInstall();
        }
      }, PWA_CONFIG.AUTO_INSTALL_DELAY);
    }
  });
}

/**
 * Versteckt das Modal
 */
function hideModal() {
  if (!installModal) return;
  
  installModal.classList.remove('pwa-modal-visible');
  setTimeout(() => {
    if (installModal && installModal.parentNode) {
      installModal.parentNode.removeChild(installModal);
      installModal = null;
    }
  }, 300);
}

/**
 * Zeigt Fallback-Anleitung für Geräte ohne beforeinstallprompt
 */
function showInstallInstructions() {
  const userAgent = navigator.userAgent.toLowerCase();
  const isIOS = /iphone|ipad|ipod/.test(userAgent);
  const isAndroid = /android/.test(userAgent);
  const isSafari = /safari/.test(userAgent) && !/chrome/.test(userAgent);
  const isChrome = /chrome/.test(userAgent);
  const isEdge = /edg/.test(userAgent);
  const isMac = /macintosh/.test(userAgent);
  const isWindows = /windows/.test(userAgent);

  let instructions = '';

  if (isIOS && isSafari) {
      instructions = `
      <div class="pwa-instructions">
        <h3><i class="fas fa-mobile-alt"></i> App installieren (iOS Safari)</h3>
        <ol>
          <li>Tippen Sie auf das <strong>Teilen-Symbol</strong> <i class="fas fa-share"></i> unten</li>
          <li>Scrollen Sie nach unten und wählen Sie <strong>"Zum Home-Bildschirm"</strong></li>
          <li>Tippen Sie auf <strong>"Hinzufügen"</strong></li>
        </ol>
      </div>
      `;
  } else if (isAndroid && isChrome) {
      instructions = `
      <div class="pwa-instructions">
        <h3><i class="fas fa-mobile-alt"></i> App installieren (Android Chrome)</h3>
        <ol>
          <li>Tippen Sie auf das <strong>Menü</strong> <i class="fas fa-ellipsis-v"></i> oben rechts</li>
          <li>Wählen Sie <strong>"Zum Startbildschirm hinzufügen"</strong> oder <strong>"App installieren"</strong></li>
          <li>Bestätigen Sie die Installation</li>
        </ol>
      </div>
      `;
  } else if ((isChrome || isEdge) && (isWindows || isMac)) {
      instructions = `
      <div class="pwa-instructions">
        <h3><i class="fas fa-desktop"></i> App installieren (Desktop)</h3>
        <ol>
          <li>Klicken Sie auf das <strong>Install-Symbol</strong> <i class="fas fa-plus"></i> in der Adressleiste</li>
          <li>Oder: Menü <i class="fas fa-ellipsis-v"></i> → <strong>"App installieren"</strong></li>
          <li>Bestätigen Sie die Installation</li>
        </ol>
      </div>
    `;
  } else {
    instructions = `
      <div class="pwa-instructions">
        <h3><i class="fas fa-mobile-alt"></i> App installieren</h3>
        <p>Bitte suchen Sie in Ihrem Browser-Menü nach der Option <strong>"Zum Startbildschirm hinzufügen"</strong> oder <strong>"App installieren"</strong>.</p>
      </div>
    `;
  }
  
  // Erstelle Anleitungs-Modal
  const instructionModal = document.createElement('div');
  instructionModal.className = 'pwa-instruction-modal';
  instructionModal.innerHTML = `
    <div class="pwa-modal-content">
      <button class="pwa-modal-close" onclick="this.closest('.pwa-instruction-modal').remove()"><i class="fas fa-times"></i></button>
      ${instructions}
      <button class="pwa-btn pwa-btn-primary" onclick="this.closest('.pwa-instruction-modal').remove()" style="margin-top: 20px; width: 100%;">
        <i class="fas fa-check"></i> Verstanden
      </button>
    </div>
  `;
  
  document.body.appendChild(instructionModal);
  requestAnimationFrame(() => {
    instructionModal.classList.add('pwa-modal-visible');
  });
}

// ============================================
// INSTALL HANDLING
// ============================================

/**
 * Behandelt die Installation
 */
async function handleInstall() {
  const installBtn = installModal?.querySelector('#pwa-install-btn');
  
  if (!deferredPrompt) {
    // Fallback: Zeige Anleitung
    hideModal();
    setTimeout(() => {
      showInstallInstructions();
    }, 300);
    return;
  }
  
  // Button deaktivieren und Loading-State
  if (installBtn) {
    installBtn.disabled = true;
      installBtn.innerHTML = '<i class="fas fa-spinner fa-spin pwa-btn-icon"></i> Wird installiert...';
  }
  
  try {
    // Zeige Install Prompt
    deferredPrompt.prompt();
    
    // Warte auf User-Entscheidung
    const { outcome } = await deferredPrompt.userChoice;
    
    console.log(`[PWA] User choice: ${outcome}`);
    
    if (outcome === 'accepted') {
      console.log('[PWA] App wurde installiert');
      markAsInstalled();
      
      // Erfolgs-Animation
      if (installBtn) {
        installBtn.innerHTML = '<i class="fas fa-check pwa-btn-icon"></i> Installiert!';
        installBtn.style.background = '#28a745';
      }
      
      // Modal nach kurzer Verzögerung schließen
      setTimeout(() => {
        hideModal();
        
        // Nach Installation: Push-Benachrichtigungen anbieten
        setTimeout(() => {
          offerPushNotificationsAfterInstall();
        }, 2000);
      }, 1500);
  } else {
      console.log('[PWA] Installation abgelehnt');
      markAsDeclined();
      
      if (installBtn) {
        installBtn.disabled = false;
        installBtn.innerHTML = '<i class="fas fa-download pwa-btn-icon"></i> Jetzt installieren';
      }
      
      hideModal();
    }
  } catch (error) {
    console.error('[PWA] Install-Fehler:', error);
    if (installBtn) {
      installBtn.disabled = false;
        installBtn.innerHTML = '<i class="fas fa-download pwa-btn-icon"></i> Jetzt installieren';
    }
    hideModal();
    showInstallInstructions();
  }
  
  // Prompt kann nur einmal verwendet werden
  deferredPrompt = null;
}

/**
 * Bietet Push-Benachrichtigungen nach Installation an
 */
async function offerPushNotificationsAfterInstall() {
  if (!window.PushNotifications || !window.PushNotifications.isSupported()) {
    return;
  }
  
  const isSubscribed = await window.PushNotifications.checkStatus();
  if (isSubscribed) {
    return;
  }
  
  const permission = await window.PushNotifications.checkPermission();
  if (permission === 'denied') {
    return;
  }
  
  // Zeige Push-Angebot (wenn vorhanden)
  if (typeof showPushNotificationOffer === 'function') {
    showPushNotificationOffer();
  }
}

// ============================================
// STRATEGIC TIMING LOGIC
// ============================================

/**
 * Prüft ob Modal angezeigt werden sollte und zeigt es gegebenenfalls
 */
function checkAndShowModal() {
  if (!shouldShowModal()) {
    return;
  }
  
  // Verzögerung basierend auf Kontext
  let delay = PWA_CONFIG.DELAY_INITIAL;
  
  // Prüfe ob User eingeloggt ist (aus PHP Session)
  const isLoggedIn = document.body.classList.contains('user-logged-in') || 
                     document.querySelector('[data-user-logged-in="true"]');
  
  if (isLoggedIn) {
    delay = PWA_CONFIG.DELAY_AFTER_LOGIN;
  } else if (engagementTracker.lastActionTime) {
    const timeSinceAction = Date.now() - engagementTracker.lastActionTime;
    if (timeSinceAction < 5000) {
      delay = PWA_CONFIG.DELAY_AFTER_ACTION;
    }
  }
  
  setTimeout(() => {
    if (shouldShowModal() && !installModal) {
      createInstallModal();
    }
  }, delay);
}

// ============================================
// EVENT LISTENERS
// ============================================

/**
 * Initialisiert die PWA Install-Logik
 */
function initPWAInstall() {
  // Prüfe ob bereits installiert
  if (isPWAInstalled()) {
    console.log('[PWA] App ist bereits installiert');
    return;
  }
  
  // Initialisiere Engagement-Tracking
  initEngagementTracking();
  
  // beforeinstallprompt Event
  window.addEventListener('beforeinstallprompt', (e) => {
    console.log('[PWA] beforeinstallprompt Event empfangen');
    e.preventDefault();
    deferredPrompt = e;
    
    // Prüfe ob Modal angezeigt werden sollte
    setTimeout(() => {
      checkAndShowModal();
    }, 2000);
  });
  
  // App wurde installiert
  window.addEventListener('appinstalled', () => {
    console.log('[PWA] App wurde installiert');
    markAsInstalled();
    deferredPrompt = null;
    
    if (installModal) {
      hideModal();
    }
    
    // Nach Installation: Push-Benachrichtigungen anbieten
    setTimeout(() => {
      offerPushNotificationsAfterInstall();
    }, 2000);
  });
  
  // Prüfe regelmäßig ob Modal angezeigt werden sollte
  // (für Fälle wo beforeinstallprompt nicht kommt)
  setInterval(() => {
    if (!installModal && !isPWAInstalled() && !isDeclined()) {
      checkAndShowModal();
    }
  }, 5000);
  
  // Initiale Prüfung nach Verzögerung
  setTimeout(() => {
    checkAndShowModal();
  }, PWA_CONFIG.DELAY_INITIAL);
}

// ============================================
// STYLES
// ============================================

/**
 * Fügt Styles hinzu
 */
function addStyles() {
  const style = document.createElement('style');
  style.textContent = `
    /* Modal Container - Minimalistisch Volvo Style */
    .pwa-install-modal {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.5);
      backdrop-filter: blur(2px);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10000;
      opacity: 0;
      transition: opacity 0.2s ease;
      padding: 20px;
    }
    
    .pwa-install-modal.pwa-modal-visible {
      opacity: 1;
    }
    
    /* Modal Content - Clean & Minimal */
    .pwa-modal-content {
      background: #ffffff;
      border-radius: 8px;
      padding: 32px;
      max-width: 440px;
      width: 100%;
      box-shadow: 0 4px 24px rgba(0, 0, 0, 0.12);
      position: relative;
      transform: translateY(10px);
      transition: transform 0.2s ease;
    }
    
    .pwa-install-modal.pwa-modal-visible .pwa-modal-content {
      transform: translateY(0);
    }
    
    /* Close Button - Minimal */
    .pwa-modal-close {
      position: absolute;
      top: 16px;
      right: 16px;
      background: none;
      border: none;
      font-size: 18px;
      color: #6c757d;
      cursor: pointer;
      width: 32px;
      height: 32px;
      display: flex;
      align-items: center;
      justify-content: center;
      border-radius: 4px;
      transition: all 0.15s;
      line-height: 1;
    }
    
    .pwa-modal-close:hover {
      background: #f5f5f5;
      color: #003057;
    }
    
    /* Header - Minimalistisch */
    .pwa-modal-header {
      text-align: center;
      margin-bottom: 24px;
    }
    
    .pwa-icon-container {
      margin-bottom: 16px;
    }
    
    .pwa-app-icon {
      width: 64px;
      height: 64px;
      border-radius: 8px;
      box-shadow: 0 2px 8px rgba(0, 48, 87, 0.1);
    }
    
    .pwa-modal-title {
      font-size: 20px;
      font-weight: 600;
      color: #003057;
      margin: 0 0 8px 0;
      line-height: 1.4;
      letter-spacing: -0.01em;
    }
    
    .pwa-modal-subtitle {
      font-size: 14px;
      color: #6c757d;
      margin: 0;
      line-height: 1.5;
    }
    
    /* Benefits - Clean List */
    .pwa-benefits {
      margin-bottom: 24px;
    }
    
    .pwa-benefit-item {
      display: flex;
      align-items: flex-start;
      gap: 12px;
      padding: 12px 0;
      border-bottom: 1px solid #f0f0f0;
      transition: padding-left 0.15s;
    }
    
    .pwa-benefit-item:last-child {
      border-bottom: none;
    }
    
    .pwa-benefit-item:hover {
      padding-left: 4px;
    }
    
    .pwa-benefit-icon {
      font-size: 18px;
      flex-shrink: 0;
      width: 32px;
      height: 32px;
      display: flex;
      align-items: center;
      justify-content: center;
      color: #003057;
      background: #f5f5f5;
      border-radius: 4px;
    }
    
    .pwa-benefit-text {
      flex: 1;
      padding-top: 2px;
    }
    
    .pwa-benefit-text strong {
      display: block;
      font-size: 14px;
      font-weight: 600;
      color: #003057;
      margin-bottom: 2px;
      line-height: 1.4;
    }
    
    .pwa-benefit-text span {
      display: block;
      font-size: 13px;
      color: #6c757d;
      line-height: 1.4;
    }
    
    /* Auto-Install Info */
    .pwa-auto-install-info {
      font-size: 12px;
      color: #6c757d;
      text-align: center;
      margin-bottom: 12px;
      padding: 8px;
      background: #f9f9f9;
      border-radius: 4px;
    }
    
    .pwa-auto-install-info #pwa-countdown {
      font-weight: 600;
      color: #003057;
    }
    
    /* Buttons - Minimalistisch */
    .pwa-modal-buttons {
      display: flex;
      flex-direction: column;
      gap: 8px;
    }
    
    .pwa-btn-group {
      display: flex;
      gap: 8px;
    }
    
    .pwa-btn {
      flex: 1;
      padding: 12px 20px;
      border: none;
      border-radius: 4px;
      font-size: 14px;
      font-weight: 500;
      cursor: pointer;
      transition: all 0.15s;
      display: flex;
      align-items: center;
      justify-content: center;
      gap: 8px;
      letter-spacing: 0.01em;
    }
    
    .pwa-btn-primary {
      background: #003057;
      color: white;
    }
    
    .pwa-btn-primary:hover:not(:disabled) {
      background: #004080;
    }
    
    .pwa-btn-primary:active:not(:disabled) {
      background: #002040;
    }
    
    .pwa-btn-primary:disabled {
      opacity: 0.6;
      cursor: not-allowed;
    }
    
    .pwa-btn-secondary {
      background: #ffffff;
      color: #6c757d;
      border: 1px solid #e0e0e0;
    }
    
    .pwa-btn-secondary:hover {
      background: #f9f9f9;
      border-color: #d0d0d0;
    }
    
    .pwa-btn-icon {
      font-size: 14px;
    }
    
    /* Instruction Modal - Minimalistisch */
    .pwa-instruction-modal {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.5);
      backdrop-filter: blur(2px);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10001;
      opacity: 0;
      transition: opacity 0.2s ease;
      padding: 20px;
    }
    
    .pwa-instruction-modal.pwa-modal-visible {
      opacity: 1;
    }
    
    .pwa-instructions {
      text-align: left;
    }
    
    .pwa-instructions h3 {
      font-size: 18px;
      font-weight: 600;
      color: #003057;
      margin-bottom: 16px;
      line-height: 1.4;
    }
    
    .pwa-instructions h3 i {
      margin-right: 8px;
      color: #003057;
    }
    
    .pwa-instructions ol {
      padding-left: 20px;
      margin: 0;
    }
    
    .pwa-instructions li {
      margin-bottom: 12px;
      color: #495057;
      line-height: 1.6;
      font-size: 14px;
    }
    
    .pwa-instructions li i {
      margin-right: 6px;
      color: #003057;
      font-size: 12px;
    }
    
    .pwa-instructions p {
      color: #6c757d;
      font-size: 14px;
      line-height: 1.6;
      margin: 0;
    }
    
    /* Responsive - Minimalistisch */
    @media (max-width: 768px) {
      .pwa-modal-content {
        padding: 24px 20px;
        border-radius: 8px;
        max-width: calc(100% - 40px);
      }
      
      .pwa-modal-title {
        font-size: 18px;
      }
      
      .pwa-modal-subtitle {
        font-size: 13px;
      }
      
      .pwa-app-icon {
        width: 56px;
        height: 56px;
      }
      
      .pwa-modal-buttons {
        flex-direction: column;
      }
      
      .pwa-btn {
        width: 100%;
      }
      
      .pwa-benefit-item {
        padding: 10px 0;
      }
      
      .pwa-benefit-icon {
        width: 28px;
        height: 28px;
        font-size: 16px;
      }
    }
  `;
  
document.head.appendChild(style);
}

// ============================================
// INITIALIZATION
// ============================================

// ============================================
// DEBUG FUNCTIONS (Nur für Entwicklung)
// ============================================

/**
 * Debug-Objekt für Console-Tests
 * Verwendung in der Browser-Konsole:
 * 
 * PWA_DEBUG.show()              - Zeigt Modal sofort an
 * PWA_DEBUG.status()            - Zeigt aktuellen Status
 * PWA_DEBUG.reset()             - Setzt alle Daten zurück
 * PWA_DEBUG.setEngagement()    - Setzt Engagement-Werte
 * PWA_DEBUG.clearAll()         - Löscht alle LocalStorage-Daten
 * PWA_DEBUG.simulatePrompt()   - Simuliert beforeinstallprompt Event
 * PWA_DEBUG.forceShow()        - Erzwingt Modal-Anzeige (ignoriert alle Checks)
 */
// Befülle das bereits existierende PWA_DEBUG Objekt
// Verwende try-catch um sicherzustellen, dass PWA_DEBUG immer verfügbar ist
try {
  Object.assign(window.PWA_DEBUG, {
  /**
   * Zeigt Modal sofort an (wenn möglich)
   */
  show: function() {
    if (typeof isPWAInstalled !== 'function') {
      console.error('[PWA DEBUG] isPWAInstalled() ist noch nicht verfügbar. Bitte warte einen Moment und versuche es erneut.');
      return;
    }
    
    if (isPWAInstalled()) {
      console.warn('[PWA DEBUG] App ist bereits installiert - Modal kann nicht angezeigt werden');
      return;
    }
    
    if (typeof installModal !== 'undefined' && installModal) {
      console.log('[PWA DEBUG] Modal ist bereits sichtbar');
      return;
    }
    
    if (typeof createInstallModal !== 'function') {
      console.error('[PWA DEBUG] createInstallModal() ist noch nicht verfügbar. Bitte warte einen Moment und versuche es erneut.');
      return;
    }
    
    console.log('[PWA DEBUG] Zeige Modal manuell...');
    createInstallModal();
  },
  
  /**
   * Zeigt aktuellen Status
   */
  status: function() {
    if (typeof loadInstallData !== 'function' || typeof isPWAInstalled !== 'function' || typeof isDeclined !== 'function' || typeof shouldShowModal !== 'function') {
      console.error('[PWA DEBUG] Einige Funktionen sind noch nicht verfügbar. Bitte warte einen Moment und versuche es erneut.');
      console.log('Verfügbare Funktionen:', {
        loadInstallData: typeof loadInstallData,
        isPWAInstalled: typeof isPWAInstalled,
        isDeclined: typeof isDeclined,
        shouldShowModal: typeof shouldShowModal
      });
      return;
    }
    
    const data = loadInstallData();
    const isInstalled = isPWAInstalled();
    const isDeclinedNow = isDeclined();
    const shouldShow = shouldShowModal();
    
    console.group('🔍 PWA Install Status');
    console.log('📱 Installiert:', isInstalled);
    console.log('❌ Abgelehnt:', isDeclinedNow);
    console.log('✅ Sollte angezeigt werden:', shouldShow);
    console.log('📊 Anzeige-Versuche:', data.showCount + '/' + PWA_CONFIG.MAX_SHOW_ATTEMPTS);
    console.log('📅 Letzte Anzeige:', data.lastShown ? new Date(data.lastShown).toLocaleString() : 'Nie');
    console.log('');
    console.group('📈 Engagement-Tracking:');
    console.log('Seitenaufrufe:', engagementTracker.pageViews, '(Min:', PWA_CONFIG.MIN_PAGE_VIEWS + ')');
    console.log('Session-Zeit:', Math.round(engagementTracker.sessionTime / 1000) + 's', '(Min:', PWA_CONFIG.MIN_SESSION_TIME / 1000 + 's)');
    console.log('Aktionen:', engagementTracker.actions, '(Min:', PWA_CONFIG.MIN_ACTIONS + ')');
    console.log('Letzte Aktion:', engagementTracker.lastActionTime ? new Date(engagementTracker.lastActionTime).toLocaleTimeString() : 'Keine');
    console.groupEnd();
    console.log('');
    console.group('⚙️ Konfiguration:');
    console.log('MIN_PAGE_VIEWS:', PWA_CONFIG.MIN_PAGE_VIEWS);
    console.log('MIN_SESSION_TIME:', PWA_CONFIG.MIN_SESSION_TIME / 1000 + 's');
    console.log('MIN_ACTIONS:', PWA_CONFIG.MIN_ACTIONS);
    console.log('MAX_SHOW_ATTEMPTS:', PWA_CONFIG.MAX_SHOW_ATTEMPTS);
    console.log('COOLDOWN_DAYS:', PWA_CONFIG.COOLDOWN_DAYS);
    console.groupEnd();
    console.log('');
    console.log('💾 LocalStorage-Daten:', data);
    console.log('🔔 deferredPrompt vorhanden:', deferredPrompt !== null);
    console.log('🎨 Modal vorhanden:', installModal !== null);
    console.groupEnd();
    
    return {
      installed: isInstalled,
      declined: isDeclinedNow,
      shouldShow: shouldShow,
      showCount: data.showCount,
      maxAttempts: PWA_CONFIG.MAX_SHOW_ATTEMPTS,
      engagement: {
        pageViews: engagementTracker.pageViews,
        sessionTime: engagementTracker.sessionTime,
        actions: engagementTracker.actions
      },
      config: PWA_CONFIG,
      data: data
    };
  },
  
  /**
   * Setzt alle Daten zurück
   */
  reset: function() {
    console.log('[PWA DEBUG] Setze alle Daten zurück...');
    
    // LocalStorage löschen
    localStorage.removeItem(PWA_CONFIG.STORAGE_KEY);
    localStorage.removeItem(PWA_CONFIG.INSTALLED_KEY);
    localStorage.removeItem(PWA_CONFIG.DECLINED_KEY);
    sessionStorage.removeItem('pwa_page_views');
    
    // Engagement-Tracker zurücksetzen
    engagementTracker = {
      pageViews: 1,
      sessionTime: 0,
      actions: 0,
      lastActionTime: null,
      startTime: Date.now()
    };
    
    // Modal schließen
    if (installModal) {
      hideModal();
    }
    
    console.log('✅ Alle Daten wurden zurückgesetzt');
    console.log('💡 Tipp: Verwende PWA_DEBUG.setEngagement() um Engagement-Werte zu setzen');
  },
  
  /**
   * Setzt Engagement-Werte manuell
   * @param {Object} values - {pageViews, sessionTime, actions}
   */
  setEngagement: function(values = {}) {
    if (values.pageViews !== undefined) {
      engagementTracker.pageViews = parseInt(values.pageViews);
      sessionStorage.setItem('pwa_page_views', engagementTracker.pageViews.toString());
      console.log('✅ Seitenaufrufe gesetzt auf:', engagementTracker.pageViews);
    }
    
    if (values.sessionTime !== undefined) {
      engagementTracker.sessionTime = parseInt(values.sessionTime) * 1000; // Sekunden zu Millisekunden
      console.log('✅ Session-Zeit gesetzt auf:', engagementTracker.sessionTime / 1000 + 's');
    }
    
    if (values.actions !== undefined) {
      engagementTracker.actions = parseInt(values.actions);
      console.log('✅ Aktionen gesetzt auf:', engagementTracker.actions);
    }
    
    if (values.lastActionTime !== undefined) {
      engagementTracker.lastActionTime = values.lastActionTime === 'now' ? Date.now() : parseInt(values.lastActionTime);
      console.log('✅ Letzte Aktion gesetzt auf:', new Date(engagementTracker.lastActionTime).toLocaleTimeString());
    }
    
    console.log('📊 Aktuelle Engagement-Werte:');
    console.log('  Seitenaufrufe:', engagementTracker.pageViews);
    console.log('  Session-Zeit:', Math.round(engagementTracker.sessionTime / 1000) + 's');
    console.log('  Aktionen:', engagementTracker.actions);
    
    // Prüfe ob Modal jetzt angezeigt werden sollte
    if (shouldShowModal() && !installModal) {
      console.log('💡 Engagement-Kriterien erfüllt! Modal sollte angezeigt werden.');
      console.log('💡 Verwende PWA_DEBUG.show() um Modal anzuzeigen');
    } else {
      console.log('⚠️ Engagement-Kriterien noch nicht erfüllt');
      console.log('  Benötigt:', {
        pageViews: PWA_CONFIG.MIN_PAGE_VIEWS,
        sessionTime: PWA_CONFIG.MIN_SESSION_TIME / 1000 + 's',
        actions: PWA_CONFIG.MIN_ACTIONS
      });
    }
  },
  
  /**
   * Löscht alle LocalStorage-Daten
   */
  clearAll: function() {
    console.log('[PWA DEBUG] Lösche alle LocalStorage-Daten...');
    localStorage.removeItem(PWA_CONFIG.STORAGE_KEY);
    localStorage.removeItem(PWA_CONFIG.INSTALLED_KEY);
    localStorage.removeItem(PWA_CONFIG.DECLINED_KEY);
    sessionStorage.removeItem('pwa_page_views');
    console.log('✅ Alle Daten gelöscht');
  },
  
  /**
   * Simuliert beforeinstallprompt Event
   */
  simulatePrompt: function() {
    console.log('[PWA DEBUG] Simuliere beforeinstallprompt Event...');
    
    // Erstelle Mock-Event
    const mockEvent = {
      preventDefault: function() {
        console.log('  preventDefault() aufgerufen');
      },
      prompt: function() {
        console.log('  prompt() aufgerufen - würde normalerweise Browser-Prompt zeigen');
        return Promise.resolve({ outcome: 'accepted' });
      },
      userChoice: Promise.resolve({ outcome: 'accepted' })
    };
    
    // Setze deferredPrompt
    deferredPrompt = mockEvent;
    console.log('✅ deferredPrompt gesetzt');
    console.log('💡 Verwende PWA_DEBUG.show() um Modal anzuzeigen');
  },
  
  /**
   * Erzwingt Modal-Anzeige (ignoriert alle Checks)
   */
  forceShow: function() {
    console.log('[PWA DEBUG] Erzwinge Modal-Anzeige (ignoriert alle Checks)...');
    
    if (installModal) {
      console.log('⚠️ Modal ist bereits sichtbar');
      return;
    }
    
    // Temporär alle Checks umgehen
    const originalShouldShow = shouldShowModal;
    window.shouldShowModal = function() { return true; };
    
    createInstallModal();
    
    // Original-Funktion wiederherstellen
    setTimeout(() => {
      window.shouldShowModal = originalShouldShow;
    }, 100);
    
    console.log('✅ Modal wird angezeigt');
  },
  
  /**
   * Setzt Konfiguration temporär (nur für aktuelle Session)
   */
  setConfig: function(newConfig) {
    console.log('[PWA DEBUG] Setze Konfiguration...');
    console.log('Alte Konfiguration:', PWA_CONFIG);
    
    Object.assign(PWA_CONFIG, newConfig);
    
    console.log('Neue Konfiguration:', PWA_CONFIG);
    console.log('⚠️ Änderungen sind nur für diese Session gültig');
  },
  
  /**
   * Zeigt Hilfe
   */
  help: function() {
    console.group('📚 PWA DEBUG - Verfügbare Befehle');
    console.log('PWA_DEBUG.show()              - Zeigt Modal sofort an');
    console.log('PWA_DEBUG.status()            - Zeigt aktuellen Status');
    console.log('PWA_DEBUG.reset()             - Setzt alle Daten zurück');
    console.log('PWA_DEBUG.setEngagement({     - Setzt Engagement-Werte');
    console.log('  pageViews: 3,');
    console.log('  sessionTime: 60,  // in Sekunden');
    console.log('  actions: 5');
    console.log('})');
    console.log('PWA_DEBUG.clearAll()         - Löscht alle LocalStorage-Daten');
    console.log('PWA_DEBUG.simulatePrompt()   - Simuliert beforeinstallprompt Event');
    console.log('PWA_DEBUG.forceShow()        - Erzwingt Modal-Anzeige');
    console.log('PWA_DEBUG.setConfig({        - Setzt Konfiguration');
    console.log('  MIN_PAGE_VIEWS: 1,');
    console.log('  MIN_SESSION_TIME: 10000');
    console.log('})');
    console.log('PWA_DEBUG.help()             - Zeigt diese Hilfe');
    console.groupEnd();
  }
  });
} catch (e) {
  console.error('[PWA] Fehler beim Befüllen von PWA_DEBUG:', e);
  // Stelle sicher, dass PWA_DEBUG mindestens eine help-Funktion hat
  if (!window.PWA_DEBUG.help) {
    window.PWA_DEBUG.help = function() {
      console.error('[PWA] PWA_DEBUG konnte nicht vollständig initialisiert werden:', e);
    };
  }
}

// Initialisiere wenn DOM bereit ist
try {
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      try {
        addStyles();
        initPWAInstall();
        
        // Debug-Hinweis in Console (immer verfügbar für Tests)
        console.log('%c🔧 PWA DEBUG verfügbar!', 'color: #0056b3; font-weight: bold; font-size: 14px;');
        console.log('%cTippe PWA_DEBUG.help() in die Konsole für alle Befehle', 'color: #6c757d; font-size: 12px;');
      } catch (error) {
        console.error('[PWA] Fehler bei Initialisierung:', error);
        console.log('%c⚠️ PWA DEBUG ist verfügbar, aber Initialisierung fehlgeschlagen', 'color: #dc3545; font-weight: bold;');
      }
    });
  } else {
    try {
      addStyles();
      initPWAInstall();
      
      // Debug-Hinweis in Console (immer verfügbar für Tests)
      console.log('%c🔧 PWA DEBUG verfügbar!', 'color: #0056b3; font-weight: bold; font-size: 14px;');
      console.log('%cTippe PWA_DEBUG.help() in die Konsole für alle Befehle', 'color: #6c757d; font-size: 12px;');
    } catch (error) {
      console.error('[PWA] Fehler bei Initialisierung:', error);
      console.log('%c⚠️ PWA DEBUG ist verfügbar, aber Initialisierung fehlgeschlagen', 'color: #dc3545; font-weight: bold;');
    }
  }
} catch (error) {
  console.error('[PWA] Kritischer Fehler:', error);
  console.log('%c⚠️ PWA DEBUG ist verfügbar, aber es gab einen Fehler', 'color: #dc3545; font-weight: bold;');
}
