<?php
/**
 * Script zum Anwenden der Smart API Datenbank-Updates
 * 
 * Dieses Script:
 * - Liest die SQL-Updates aus der Datei
 * - Wendet sie schrittweise an
 * - Zeigt Fortschritt und Fehler an
 * - Erstellt ein Backup vor den Änderungen
 */

require_once 'config/database.php';

class DatabaseUpdater {
    private $conn;
    private $logFile;
    
    public function __construct() {
        try {
            $db = new Database();
            $this->conn = $db->getConnection();
            $this->logFile = 'logs/smart_api_update_' . date('Y-m-d_H-i-s') . '.log';
            
            // Erstelle logs-Verzeichnis falls nicht vorhanden
            if (!is_dir('logs')) {
                mkdir('logs', 0755, true);
            }
            
        } catch (Exception $e) {
            die("Fehler beim Verbinden zur Datenbank: " . $e->getMessage());
        }
    }
    
    /**
     * Wendet die Datenbank-Updates an
     */
    public function applyUpdates() {
        echo "<h2>Smart API Datenbank-Updates</h2>\n";
        echo "<p>Starte Update-Prozess...</p>\n";
        
        // 1. Erstelle Backup
        $this->createBackup();
        
        // 2. Lade SQL-Updates
        $sqlUpdates = $this->loadSQLUpdates();
        
        // 3. Wende Updates an
        $this->applySQLUpdates($sqlUpdates);
        
        // 4. Validiere Updates
        $this->validateUpdates();
        
        echo "<p><strong>✅ Alle Updates erfolgreich angewendet!</strong></p>\n";
    }
    
    /**
     * Erstellt ein Backup der autohaus-Tabelle
     */
    private function createBackup() {
        echo "<p>📦 Erstelle Backup der autohaus-Tabelle...</p>\n";
        
        try {
            // Prüfe ob Backup-Tabelle bereits existiert
            $stmt = $this->conn->prepare("SHOW TABLES LIKE 'autohaus_backup_before_smart_api'");
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                echo "<p>⚠️ Backup-Tabelle existiert bereits, überspringe Backup</p>\n";
                return;
            }
            
            // Erstelle Backup
            $sql = "CREATE TABLE autohaus_backup_before_smart_api AS SELECT * FROM autohaus";
            $this->conn->exec($sql);
            
            echo "<p>✅ Backup erfolgreich erstellt</p>\n";
            $this->log("Backup erfolgreich erstellt");
            
        } catch (Exception $e) {
            echo "<p>❌ Fehler beim Erstellen des Backups: " . $e->getMessage() . "</p>\n";
            $this->log("Fehler beim Backup: " . $e->getMessage());
        }
    }
    
    /**
     * Lädt die SQL-Updates aus der Datei
     */
    private function loadSQLUpdates() {
        echo "<p>📄 Lade SQL-Updates...</p>\n";
        
        $sqlFile = 'database_updates_smart_api.sql';
        
        if (!file_exists($sqlFile)) {
            throw new Exception("SQL-Datei nicht gefunden: $sqlFile");
        }
        
        $sqlContent = file_get_contents($sqlFile);
        
        // Teile SQL in einzelne Statements
        $statements = $this->splitSQLStatements($sqlContent);
        
        echo "<p>✅ " . count($statements) . " SQL-Statements geladen</p>\n";
        return $statements;
    }
    
    /**
     * Teilt SQL-Content in einzelne Statements
     */
    private function splitSQLStatements($sqlContent) {
        // Entferne Kommentare und leere Zeilen
        $sqlContent = preg_replace('/--.*$/m', '', $sqlContent);
        $sqlContent = preg_replace('/\/\*.*?\*\//s', '', $sqlContent);
        
        // Teile an Semikolons
        $statements = array_filter(
            array_map('trim', explode(';', $sqlContent)),
            function($stmt) {
                return !empty($stmt) && !preg_match('/^\s*$/s', $stmt);
            }
        );
        
        return $statements;
    }
    
    /**
     * Wendet die SQL-Updates an
     */
    private function applySQLUpdates($statements) {
        echo "<p>🔧 Wende SQL-Updates an...</p>\n";
        
        $successCount = 0;
        $errorCount = 0;
        
        foreach ($statements as $index => $statement) {
            try {
                // Überspringe leere Statements
                if (empty(trim($statement))) {
                    continue;
                }
                
                echo "<p>📝 Statement " . ($index + 1) . "...</p>\n";
                
                // Führe Statement aus
                $this->conn->exec($statement);
                
                $successCount++;
                $this->log("Statement " . ($index + 1) . " erfolgreich");
                
            } catch (Exception $e) {
                $errorCount++;
                echo "<p>❌ Fehler bei Statement " . ($index + 1) . ": " . $e->getMessage() . "</p>\n";
                $this->log("Fehler bei Statement " . ($index + 1) . ": " . $e->getMessage());
                
                // Bei kritischen Fehlern, stoppe den Prozess
                if (strpos($e->getMessage(), 'Table') !== false && strpos($e->getMessage(), "doesn't exist") !== false) {
                    echo "<p>🛑 Kritischer Fehler: Tabelle existiert nicht. Stoppe Update-Prozess.</p>\n";
                    break;
                }
            }
        }
        
        echo "<p>📊 Ergebnis: $successCount erfolgreich, $errorCount Fehler</p>\n";
    }
    
    /**
     * Validiert die angewendeten Updates
     */
    private function validateUpdates() {
        echo "<p>🔍 Validiere Updates...</p>\n";
        
        $validations = [
            'Performance-Spalten' => "SHOW COLUMNS FROM autohaus LIKE 'last_wheelset_count'",
            'Performance-Log-Tabelle' => "SHOW TABLES LIKE 'api_performance_log'",
            'Cache-Statistiken-Tabelle' => "SHOW TABLES LIKE 'cache_statistics'",
            'Error-Log-Tabelle' => "SHOW TABLES LIKE 'api_error_log'",
            'Performance-View' => "SHOW TABLES LIKE 'v_autohaus_performance'",
            'Cache-View' => "SHOW TABLES LIKE 'v_cache_statistics'",
            'Error-View' => "SHOW TABLES LIKE 'v_error_overview'"
        ];
        
        $allValid = true;
        
        foreach ($validations as $name => $query) {
            try {
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    echo "<p>✅ $name: OK</p>\n";
                } else {
                    echo "<p>❌ $name: FEHLT</p>\n";
                    $allValid = false;
                }
                
            } catch (Exception $e) {
                echo "<p>❌ $name: FEHLER - " . $e->getMessage() . "</p>\n";
                $allValid = false;
            }
        }
        
        if ($allValid) {
            echo "<p>✅ Alle Validierungen erfolgreich</p>\n";
        } else {
            echo "<p>⚠️ Einige Validierungen fehlgeschlagen</p>\n";
        }
    }
    
    /**
     * Loggt Nachrichten
     */
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message\n";
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Zeigt den Log-Inhalt an
     */
    public function showLog() {
        if (file_exists($this->logFile)) {
            echo "<h3>Update-Log:</h3>\n";
            echo "<pre>" . htmlspecialchars(file_get_contents($this->logFile)) . "</pre>\n";
        }
    }
}

// Führe Updates aus
try {
    $updater = new DatabaseUpdater();
    $updater->applyUpdates();
    $updater->showLog();
    
} catch (Exception $e) {
    echo "<p>❌ <strong>Kritischer Fehler:</strong> " . $e->getMessage() . "</p>\n";
    echo "<p>Bitte überprüfen Sie die Datenbankverbindung und Berechtigungen.</p>\n";
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Smart API Datenbank-Updates</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        h2 { color: #2c3e50; }
        p { margin: 10px 0; }
        .success { color: #27ae60; }
        .error { color: #e74c3c; }
        .warning { color: #f39c12; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 5px; overflow-x: auto; }
    </style>
</head>
<body>
    <h1>Smart API Datenbank-Updates</h1>
    <p>Dieses Script hat die Datenbank für die neue Smart API vorbereitet.</p>
    
    <h2>Nächste Schritte:</h2>
    <ol>
        <li>Testen Sie die neue API in der <a href="admin/storage.php">Storage-Verwaltung</a></li>
        <li>Überwachen Sie die Performance-Metriken</li>
        <li>Bei Problemen können Sie das Backup wiederherstellen</li>
    </ol>
    
    <h2>Neue Features:</h2>
    <ul>
        <li>✅ Performance-Metriken für jedes Autohaus</li>
        <li>✅ Intelligente Cache-Strategien</li>
        <li>✅ Detaillierte Fehler-Logs</li>
        <li>✅ Automatische Performance-Tier-Anpassung</li>
        <li>✅ Monitoring-Views für bessere Übersicht</li>
    </ul>
</body>
</html>

