<?php
/**
 * update_user.php
 * Aktualisiert Benutzer-Informationen
 * Unterstützt sowohl Bearer Token als auch PHP Session
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Api-Key, X-Timestamp, X-Nonce, X-Signature');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Session starten
session_start();

// Log-Funktion (optional, kann später entfernt werden)
function logToFile($message, $data = null) {
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    $logFile = $logDir . '/update_user_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message";
    if ($data !== null) {
        $logMessage .= "\n" . (is_string($data) ? $data : json_encode($data, JSON_PRETTY_PRINT));
    }
    $logMessage .= "\n" . str_repeat('-', 80) . "\n";
    @file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// Database connection helper - verwendet config.php wie upload_verification_document.php
function getDbConnection() {
    static $pdo = null;
    if ($pdo !== null) return $pdo;
    
    logToFile("getDbConnection: Versuche DB-Verbindung herzustellen");
    
    try {
        // Methode 1: Versuche config.php (wie upload_verification_document.php es macht)
        $configPath = __DIR__ . '/../config.php';
        logToFile("getDbConnection: Prüfe config.php", $configPath);
        
        if (file_exists($configPath)) {
            logToFile("getDbConnection: config.php gefunden, lade...");
            require_once $configPath;
            
            // Nach require_once sollte $pdo verfügbar sein (globale Variable)
            if (isset($pdo) && $pdo instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$pdo)");
                return $pdo;
            }
            
            // Prüfe globale Variablen
            if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$GLOBALS['pdo'])");
                $pdo = $GLOBALS['pdo'];
                return $pdo;
            }
        } else {
            logToFile("getDbConnection WARNING", "config.php nicht gefunden: $configPath");
        }
        
        // Methode 2: Versuche config/database.php
        $dbConfigPath = __DIR__ . '/../config/database.php';
        logToFile("getDbConnection: Prüfe config/database.php", $dbConfigPath);
        
        if (file_exists($dbConfigPath)) {
            logToFile("getDbConnection: config/database.php gefunden, lade...");
            require_once $dbConfigPath;
            
            if (class_exists('Database')) {
                try {
                    $db = new Database();
                    $pdo = $db->getConnection();
                    if ($pdo instanceof PDO) {
                        logToFile("getDbConnection SUCCESS", "Verbindung via Database-Klasse hergestellt");
                        return $pdo;
                    }
                } catch (Exception $e) {
                    logToFile("getDbConnection WARNING", "Database-Klasse gefunden, aber getConnection() fehlgeschlagen: " . $e->getMessage());
                }
            }
        }
        
        // Methode 3: Fallback mit Constants
        if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER')) {
            $host = DB_HOST;
            $dbname = DB_NAME;
            $user = DB_USER;
            $pass = defined('DB_PASS') ? DB_PASS : '';
            logToFile("getDbConnection: Verwende Constants", "Host: $host, DB: $dbname, User: $user");
            try {
                $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $user, $pass);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                logToFile("getDbConnection SUCCESS", "Verbindung via Constants hergestellt");
                return $pdo;
            } catch (PDOException $e) {
                logToFile("getDbConnection ERROR", "Constants-Verbindung fehlgeschlagen: " . $e->getMessage());
            }
        }
        
        logToFile("getDbConnection ERROR", "Alle Verbindungsversuche fehlgeschlagen");
        return null;
        
    } catch (Exception $e) {
        logToFile("getDbConnection ERROR", "Exception: " . $e->getMessage());
        return null;
    }
}

// Bearer Token extrahieren
function getBearerToken(): ?string {
    $hdrs = getallheaders();
    if (!$hdrs) return null;
    foreach ($hdrs as $k => $v) {
        if (strtolower($k) === 'authorization' && stripos($v, 'bearer ') === 0) {
            return trim(substr($v, 7));
        }
    }
    return null;
}

// User-ID aus Token ermitteln
function getUserIdFromToken(string $token) {
    $pdo = getDbConnection();
    if (!$pdo) return null;
    try {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE auth_token = ? AND token_expires_at > NOW() AND is_active = 1 LIMIT 1");
        $stmt->execute([$token]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['id'] : null;
    } catch (Exception $e) {
        error_log("getUserIdFromToken error: " . $e->getMessage());
        return null;
    }
}

// User-ID ermitteln (aus Session oder Token)
$userId = null;

// 1. Prüfe Session
if (isset($_SESSION['user_id'])) {
    $userId = (int)$_SESSION['user_id'];
    logToFile("User-ID aus Session", $userId);
}

// 2. Prüfe Bearer Token (falls keine Session)
if (!$userId) {
    $token = getBearerToken();
    if ($token) {
        logToFile("Bearer Token gefunden", substr($token, 0, 20) . '...');
        $userId = getUserIdFromToken($token);
        if ($userId) {
            // Setze Session für zukünftige Requests
            $_SESSION['user_id'] = $userId;
            $_SESSION['user_logged_in'] = true;
            logToFile("User-ID aus Token", $userId);
        } else {
            logToFile("Token ungültig", "Token konnte nicht zu User-ID aufgelöst werden");
        }
    }
}

// Kein User gefunden
if (!$userId) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Nicht autorisiert']);
    exit;
}

// Input lesen
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Ungültige Eingabe']);
    exit;
}

logToFile("Update Request", [
    'user_id' => $userId,
    'input' => $input
]);

// Erlaubte Felder
$allowedFields = [
    'first_name',
    'last_name',
    'name', // Falls als kombinierter Name gesendet
    'email',
    'phone_number',
    'phone', // Alternative
    'address',
    'postal_code',
    'postalCode', // Alternative
    'city',
    'license_plate'
];

// Prepare UPDATE-Statement
$pdo = getDbConnection();
if (!$pdo) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Datenbankverbindung fehlgeschlagen']);
    exit;
}

$updates = [];
$values = [];

// Wenn 'name' gesendet wurde, splitte es in first_name und last_name
if (isset($input['name']) && !isset($input['first_name']) && !isset($input['last_name'])) {
    $nameParts = explode(' ', trim($input['name']), 2);
    $input['first_name'] = $nameParts[0] ?? '';
    $input['last_name'] = $nameParts[1] ?? '';
    unset($input['name']);
}

// Normalisiere Feldnamen
if (isset($input['phone']) && !isset($input['phone_number'])) {
    $input['phone_number'] = $input['phone'];
    unset($input['phone']);
}

if (isset($input['postalCode']) && !isset($input['postal_code'])) {
    $input['postal_code'] = $input['postalCode'];
    unset($input['postalCode']);
}

// Baue UPDATE-Statement
foreach ($allowedFields as $field) {
    if (isset($input[$field])) {
        $updates[] = "$field = ?";
        $values[] = trim($input[$field]);
    }
}

if (empty($updates)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Keine gültigen Felder zum Aktualisieren']);
    exit;
}

// Füge updated_at hinzu
$updates[] = "updated_at = NOW()";
// Füge user_id für WHERE-Klausel hinzu
$values[] = $userId;

$sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?";
logToFile("SQL Statement", $sql);
logToFile("Values", $values);

try {
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute($values);
    
    if ($result) {
        logToFile("SUCCESS: User aktualisiert", [
            'user_id' => $userId,
            'updated_fields' => array_keys($input)
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Profil erfolgreich aktualisiert'
        ]);
    } else {
        logToFile("ERROR: Update fehlgeschlagen", "Kein Fehler, aber auch kein Erfolg");
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Update fehlgeschlagen']);
    }
} catch (PDOException $e) {
    logToFile("ERROR: PDO Exception", $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Datenbankfehler: ' . $e->getMessage()]);
} catch (Exception $e) {
    logToFile("ERROR: Exception", $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Fehler: ' . $e->getMessage()]);
}

