<?php
/**
 * Reifenangebote API
 * API-Endpunkte für das Reifenangebote-Modul
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=UTF-8');
session_start();

// Fehler-Logging-Funktion
function logError($message, $context = []) {
    $logMessage = date('Y-m-d H:i:s') . " - TIRE_OFFERS_API: $message";
    if (!empty($context)) {
        $logMessage .= " | Context: " . json_encode($context);
    }
    error_log($logMessage);
}

// Prüfe Authentifizierung
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

require_once dirname(__DIR__) . '/config/database.php';
require_once dirname(__DIR__) . '/admin/includes/tire_offers_api.php';

$db = new Database();
$conn = $db->getConnection();

// Hole Autohaus-ID aus GET, POST oder Session
$autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? $_SESSION['current_autohaus_id'] ?? null;
if ($autohausId) {
    $_SESSION['current_autohaus_id'] = $autohausId;
}

// Erstelle API-Instanz mit Autohaus-ID
$api = new TireOffersAPI($autohausId);

$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'set_autohaus':
            // Setze Autohaus-ID in Session
            $autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? null;
            if ($autohausId) {
                $_SESSION['current_autohaus_id'] = $autohausId;
                echo json_encode(['success' => true, 'autohaus_id' => $autohausId]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Keine Autohaus-ID angegeben']);
            }
            break;
            
        case 'search_customers':
            // Suche Kunden nach Name
            $search = $_GET['search'] ?? '';
            $limit = intval($_GET['limit'] ?? 100);
            
            $customers = $api->searchCustomers($search, $limit);
            
            echo json_encode([
                'success' => true,
                'data' => $customers,
                'count' => count($customers)
            ]);
            break;
            
        case 'unified_search':
            // Universelle Suche: Name, Kennzeichen, Fahrgestellnummer, letzte 6 Ziffern VIN
            // OPTIMIERT für zuverlässigere Ergebnisse
            $search = trim($_GET['search'] ?? '');
            $limit = intval($_GET['limit'] ?? 100);
            
            if (empty($search) || strlen($search) < 2) {
                echo json_encode([
                    'success' => true,
                    'data' => [],
                    'count' => 0
                ]);
                break;
            }
            
            $allResults = [];
            
            // OPTIMIERT: Normalisiere Suchbegriff
            $searchNormalized = strtoupper(preg_replace('/\s+/', '', $search));
            $searchLower = strtolower($search);
            
            // 1. Suche nach Name (Nachname, Vorname, oder beides)
            try {
                $nameResults = $api->searchCustomers($search, $limit);
                if (is_array($nameResults) && !empty($nameResults)) {
                    $allResults = array_merge($allResults, $nameResults);
                }
            } catch (Exception $e) {
                logError("Error in name search", ['error' => $e->getMessage(), 'search' => $search]);
            }
            
            // 2. Suche nach Kennzeichen - OPTIMIERT: Schnellere Suche mit reduziertem Zeitraum
            try {
                // Versuche zuerst mit optimiertem Endpoint (sehr schnell)
                $licenseResult = null;
                try {
                    $encodedLicense = rawurlencode($search);
                    $result = $api->apiCall("/auftraege/by-license/$encodedLicense", 'GET', ['days' => 180]);
                    if (isset($result['success']) && $result['success'] && !empty($result['data'])) {
                        $licenseResult = $result['data'][0] ?? null; // Nimm ersten Treffer
                    }
                } catch (Exception $e) {
                    // Fallback auf normale Suche mit reduziertem Zeitraum
                    $licenseResult = $api->findCustomerByLicensePlate($search, 180);
                }
                
                if ($licenseResult && is_array($licenseResult)) {
                    $allResults[] = $licenseResult;
                }
            } catch (Exception $e) {
                logError("Error in license search", ['error' => $e->getMessage(), 'search' => $search]);
            }
            
            // 3. Suche nach Fahrgestellnummer (VIN) - OPTIMIERT: Reduzierter Zeitraum
            try {
                // OPTIMIERT: Nutze neue searchByVIN Funktion mit reduziertem Zeitraum
                $vinResults = $api->searchByVIN($search, 180);
                if (is_array($vinResults) && !empty($vinResults)) {
                    $allResults = array_merge($allResults, $vinResults);
                }
            } catch (Exception $e) {
                logError("Error in VIN search", ['error' => $e->getMessage(), 'search' => $search]);
            }
            
            // ENTFERNT: Teilstring-Suche in Aufträgen (zu langsam, wird durch andere Suchen abgedeckt)
            
            // Entferne Duplikate - OPTIMIERT: Bessere Duplikatserkennung
            $uniqueResults = [];
            $seen = [];
            foreach ($allResults as $result) {
                // Erstelle eindeutigen Key aus mehreren Feldern
                $key = null;
                if (isset($result['KundenAdrID']) && !empty($result['KundenAdrID'])) {
                    $key = 'kunde_' . $result['KundenAdrID'];
                } elseif (isset($result['AuftragNr']) && !empty($result['AuftragNr'])) {
                    $key = 'auftrag_' . $result['AuftragNr'];
                } else {
                    // Fallback: Kombination aus Name und Kennzeichen
                    $name = trim(($result['KundenName1'] ?? '') . ' ' . ($result['KundenVorname'] ?? ''));
                    $license = trim($result['ZulassungsKz'] ?? '');
                    $key = strtolower($name . '_' . $license);
                }
                
                if ($key && !isset($seen[$key])) {
                    $seen[$key] = true;
                    $uniqueResults[] = $result;
                }
            }
            
            // Limitiere auf $limit
            $uniqueResults = array_slice($uniqueResults, 0, $limit);
            
            echo json_encode([
                'success' => true,
                'data' => $uniqueResults,
                'count' => count($uniqueResults)
            ]);
            break;
            
        case 'search_by_license':
            // Suche nach Kennzeichen - OPTIMIERT für Performance
            $license = trim($_GET['license'] ?? '');
            
            if (empty($license)) {
                throw new Exception('Kennzeichen ist erforderlich');
            }
            
            // OPTIMIERT: Nutze verbesserte Suche mit reduziertem Zeitraum für schnellere Antwort
            $auftrag = $api->findCustomerByLicensePlate($license, 180);
            
            if ($auftrag) {
                echo json_encode([
                    'success' => true,
                    'data' => [$auftrag],
                    'count' => 1
                ]);
            } else {
                echo json_encode([
                    'success' => true,
                    'data' => [],
                    'count' => 0
                ]);
            }
            break;
            
        case 'get_termin_details':
            // Hole Termin-Details
            $auftragNr = $_GET['auftrag_nr'] ?? '';
            
            if (empty($auftragNr)) {
                throw new Exception('Auftragsnummer ist erforderlich');
            }
            
            $details = $api->getTerminDetails($auftragNr);
            
            if ($details) {
                echo json_encode([
                    'success' => true,
                    'data' => $details
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'error' => 'Termin nicht gefunden'
                ]);
            }
            break;
            
        case 'get_order_details':
            // Hole Auftragsdetails über Auftragsnummer - OPTIMIERT
            $orderNumber = $_GET['order_number'] ?? $_POST['order_number'] ?? '';
            $autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? null;
            
            if (empty($orderNumber)) {
                throw new Exception('Auftragsnummer ist erforderlich');
            }
            
            // Normalisiere Auftragsnummer (entferne Leerzeichen, konvertiere zu Integer für Vergleich)
            $orderNumber = trim($orderNumber);
            $orderNumberInt = (int)$orderNumber;
            
            // Erstelle neue API-Instanz mit Autohaus-ID
            $orderApi = new TireOffersAPI($autohausId);
            
            // OPTIMIERT: Health-Check zuerst - verhindert lange Wartezeiten wenn API nicht erreichbar ist
            $apiAvailable = false;
            $healthCheckStart = microtime(true);
            try {
                error_log("P2 API: [{$orderNumber}] Prüfe API-Verfügbarkeit...");
                $healthResult = $orderApi->apiCall("/system/health", 'GET', [], 3, 1); // 3 Sekunden Timeout, 1 Retry
                $apiAvailable = isset($healthResult['status']) && $healthResult['status'] === 'ok';
                $healthElapsed = round((microtime(true) - $healthCheckStart) * 1000);
                error_log("P2 API: [{$orderNumber}] Health-Check dauerte {$healthElapsed}ms - API verfügbar: " . ($apiAvailable ? 'JA' : 'NEIN'));
            } catch (Exception $e) {
                $healthElapsed = round((microtime(true) - $healthCheckStart) * 1000);
                error_log("P2 API: [{$orderNumber}] Health-Check fehlgeschlagen nach {$healthElapsed}ms: " . $e->getMessage());
                $apiAvailable = false;
            }
            
            // Wenn API nicht verfügbar, sofort abbrechen
            if (!$apiAvailable) {
                error_log("P2 API: [{$orderNumber}] API nicht verfügbar - breche ab");
                echo json_encode([
                    'success' => false,
                    'error' => 'Die P2 API ist derzeit nicht erreichbar. Bitte prüfen Sie die Verbindung oder versuchen Sie es später erneut.',
                    'api_status' => 'unavailable'
                ]);
                exit;
            }
            
            // OPTIMIERT: Versuche zuerst den neuen direkten Endpoint /auftraege/{orderNumber} (schnellster Weg!)
            $orderData = null;
            $terminDetails = null;
            $startTime = microtime(true);
            
            try {
                // NEU: Direkter Endpoint für einzelne Auftragsnummer - VIEL schneller!
                // Reduziertes Timeout auf 8 Sekunden (schnelleres Fehlschlagen wenn Endpoint nicht existiert)
                error_log("P2 API: [{$orderNumber}] Versuche direkten Endpoint /auftraege/{$orderNumber}");
                $result = $orderApi->apiCall("/auftraege/{$orderNumber}", 'GET', [], 8, 1);
                $elapsed = round((microtime(true) - $startTime) * 1000);
                error_log("P2 API: [{$orderNumber}] Direkter Endpoint dauerte {$elapsed}ms");
                
                if (isset($result['success']) && $result['success'] && !empty($result['data'])) {
                    $orderData = $result['data'];
                    error_log("P2 API: [{$orderNumber}] Auftrag über /auftraege/{$orderNumber} gefunden");
                } elseif (isset($result['success']) && !$result['success']) {
                    error_log("P2 API: [{$orderNumber}] Direkter Endpoint antwortete mit success=false: " . ($result['error'] ?? 'Unbekannter Fehler'));
                }
            } catch (Exception $e) {
                $elapsed = round((microtime(true) - $startTime) * 1000);
                error_log("P2 API: [{$orderNumber}] Error (Direkter Endpoint nach {$elapsed}ms): " . $e->getMessage());
                // Prüfe ob es ein 404 ist (Endpoint existiert nicht) - dann direkt zu Fallback
                if (strpos($e->getMessage(), '404') !== false || strpos($e->getMessage(), 'not found') !== false) {
                    error_log("P2 API: [{$orderNumber}] Endpoint /auftraege/{$orderNumber} existiert nicht - überspringe");
                }
            }
            
            // Fallback 1: Versuche /termine/{orderNumber} (enthält auch Positionen) - nur wenn direkter Endpoint fehlgeschlagen
            if (!$orderData) {
                $fallback1Start = microtime(true);
                try {
                    error_log("P2 API: [{$orderNumber}] Versuche Fallback 1 über /termine/{$orderNumber}");
                    // Reduziertes Timeout auf 10 Sekunden (schnelleres Fehlschlagen)
                    $terminDetails = $orderApi->getTerminDetails($orderNumber);
                    $elapsed = round((microtime(true) - $fallback1Start) * 1000);
                    error_log("P2 API: [{$orderNumber}] Fallback 1 (/termine) dauerte {$elapsed}ms");
                    
                    if ($terminDetails && isset($terminDetails['auftrag'])) {
                        // Extrahiere Auftragsdaten aus Termin-Details
                        $orderData = $terminDetails['auftrag'];
                        // Positionen sind bereits in $terminDetails['positionen']
                        error_log("P2 API: [{$orderNumber}] Auftrag über /termine gefunden");
                    } elseif ($terminDetails) {
                        error_log("P2 API: [{$orderNumber}] /termine antwortete, aber kein 'auftrag' Feld vorhanden");
                    }
                } catch (Exception $e) {
                    $elapsed = round((microtime(true) - $fallback1Start) * 1000);
                    error_log("P2 API: [{$orderNumber}] Error (Fallback 1 nach {$elapsed}ms): " . $e->getMessage());
                    // Wenn /termine auch timeoutet, ist die API wahrscheinlich überlastet
                    if (strpos($e->getMessage(), 'timed out') !== false || strpos($e->getMessage(), 'timeout') !== false) {
                        error_log("P2 API: [{$orderNumber}] API scheint überlastet zu sein - breche ab");
                        echo json_encode([
                            'success' => false,
                            'error' => 'Die P2 API antwortet zu langsam oder ist überlastet. Bitte versuchen Sie es später erneut.',
                            'api_status' => 'timeout'
                        ]);
                        exit;
                    }
                }
            }
            
            // Fallback 2: Versuche Basis-Auftragsdaten zu holen (nur wenn beide fehlgeschlagen)
            // WICHTIG: Dieser Fallback ist sehr langsam, daher nur als letzter Ausweg
            // ABER: Nur wenn die API grundsätzlich erreichbar ist (Health-Check war erfolgreich)
            if (!$orderData && !$terminDetails && $apiAvailable) {
                $fallback2Start = microtime(true);
                try {
                    // OPTIMIERT: Nutze kleineren Zeitraum (nur letzte 90 Tage für schnelleren Abruf)
                    // WICHTIG: Das Backend filtert NICHT nach order_number, daher müssen wir alle Daten laden
                    // und dann filtern. Das ist langsam, aber notwendig als letzter Fallback.
                    error_log("P2 API: [{$orderNumber}] Versuche letzten Fallback über /auftraege (90 Tage)");
                    $result = $orderApi->apiCall('/auftraege', 'GET', ['days' => 90], 8, 1);
                    $elapsed = round((microtime(true) - $fallback2Start) * 1000);
                    error_log("P2 API: [{$orderNumber}] Fallback 2 (/auftraege) dauerte {$elapsed}ms");
                    
                    if (isset($result['success']) && $result['success'] && !empty($result['data'])) {
                        $count = count($result['data']);
                        error_log("P2 API: [{$orderNumber}] Fallback 2: {$count} Aufträge geladen, suche nach Match...");
                        
                        // OPTIMIERT: Suche mit flexibler Vergleichslogik (String und Integer)
                        foreach ($result['data'] as $auftrag) {
                            if (isset($auftrag['AuftragNr'])) {
                                $auftragNr = $auftrag['AuftragNr'];
                                // Vergleiche sowohl als String als auch als Integer
                                if ((string)$auftragNr === (string)$orderNumber || 
                                    (int)$auftragNr === $orderNumberInt ||
                                    (string)$auftragNr === $orderNumber) {
                                    $orderData = $auftrag;
                                    error_log("P2 API: [{$orderNumber}] Auftrag über /auftraege gefunden (Match: {$auftragNr})");
                                    break;
                                }
                            }
                        }
                        if (!$orderData) {
                            error_log("P2 API: [{$orderNumber}] Kein Match in {$count} Aufträgen gefunden");
                        }
                    } else {
                        error_log("P2 API: [{$orderNumber}] Fallback 2: Keine Daten erhalten oder success=false");
                    }
                } catch (Exception $e2) {
                    $elapsed = round((microtime(true) - $fallback2Start) * 1000);
                    error_log("P2 API: [{$orderNumber}] Error (Fallback 2 nach {$elapsed}ms): " . $e2->getMessage());
                }
            }
            
            // NEU: Hole Positionen über dedizierten Endpoint /auftraege/positionen/{orderNumber}
            $positionen = [];
            if ($orderData || $terminDetails) {
                try {
                    error_log("P2 API: [{$orderNumber}] Hole Positionen über /auftraege/positionen/{$orderNumber}");
                    $posResult = $orderApi->apiCall("/auftraege/positionen/{$orderNumber}", 'GET', [], 10, 1);
                    if (isset($posResult['success']) && $posResult['success'] && !empty($posResult['data'])) {
                        $positionen = $posResult['data'];
                        $count = count($positionen);
                        error_log("P2 API: [{$orderNumber}] {$count} Positionen geladen");
                    } elseif (isset($terminDetails['positionen'])) {
                        // Fallback: Nutze Positionen aus terminDetails
                        $positionen = $terminDetails['positionen'];
                        error_log("P2 API: [{$orderNumber}] Nutze Positionen aus terminDetails");
                    }
                } catch (Exception $e) {
                    error_log("P2 API: [{$orderNumber}] Fehler beim Laden der Positionen: " . $e->getMessage());
                    // Fallback: Nutze Positionen aus terminDetails falls vorhanden
                    if (isset($terminDetails['positionen'])) {
                        $positionen = $terminDetails['positionen'];
                    }
                }
            }
            
            $totalElapsed = round((microtime(true) - $startTime) * 1000);
            error_log("P2 API: [{$orderNumber}] Gesamtzeit: {$totalElapsed}ms");
            
            if ($orderData || $terminDetails) {
                echo json_encode([
                    'success' => true,
                    'data' => [
                        'order' => $orderData,
                        'details' => $terminDetails,
                        'positionen' => $positionen
                    ]
                ]);
            } else {
                error_log("P2 API: Auftrag {$orderNumber} nicht gefunden (weder über /termine noch über /auftraege)");
                echo json_encode([
                    'success' => false,
                    'error' => "Auftrag {$orderNumber} nicht gefunden. Bitte prüfen Sie die Auftragsnummer."
                ]);
            }
            break;
            
        case 'update_position':
            // Aktualisiert den Text (Bezeichnung) einer Position - nur für Art CC, OP, TS
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Ungültige JSON-Daten: ' . json_last_error_msg());
            }
            
            $auftragNr = $data['auftrag_nr'] ?? $_GET['auftrag_nr'] ?? $_POST['auftrag_nr'] ?? '';
            $positionNr = $data['PositionNr'] ?? $data['position_nr'] ?? '';
            $bezeichnung = $data['Bezeichnung'] ?? '';
            
            if (empty($auftragNr) || empty($positionNr)) {
                throw new Exception('Auftragsnummer und Positionsnummer sind erforderlich');
            }
            
            if (empty($bezeichnung)) {
                throw new Exception('Bezeichnung darf nicht leer sein');
            }
            
            // Erstelle API-Instanz
            $orderApi = new TireOffersAPI($autohausId);
            
            // PUT-Request an Backend API
            error_log("P2 API: [{$auftragNr}] Aktualisiere Position {$positionNr}");
            $result = $orderApi->apiCall("/auftraege/positionen/{$auftragNr}", 'PUT', [
                'PositionNr' => (int)$positionNr,
                'Bezeichnung' => $bezeichnung
            ], 15, 1); // 15 Sekunden Timeout, 1 Retry
            
            if (isset($result['success']) && $result['success']) {
                error_log("P2 API: [{$auftragNr}] Position {$positionNr} erfolgreich aktualisiert");
                echo json_encode([
                    'success' => true,
                    'message' => $result['message'] ?? 'Position erfolgreich aktualisiert',
                    'affected_rows' => $result['affected_rows'] ?? 1
                ]);
            } else {
                $error = $result['error'] ?? 'Unbekannter Fehler';
                error_log("P2 API: [{$auftragNr}] Fehler beim Aktualisieren von Position {$positionNr}: {$error}");
                throw new Exception($error);
            }
            break;
            
        case 'create_offer':
            // Erstelle neues Angebot
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input) {
                throw new Exception('Ungültige Anfragedaten');
            }
            
            $current_user_id = $_SESSION['admin_user_id'] ?? null;
            
            // Generiere Token
            $token = bin2hex(random_bytes(32));
            $tokenExpires = date('Y-m-d H:i:s', strtotime('+24 hours'));
            
            // Bereite Angebotsdaten vor
            $offerData = [
                'new_tires' => $input['new_tires'] ?? [],
                'selected_services' => $input['selected_services'] ?? [],
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            // Berechne Gesamtbetrag
            $totalAmount = 0;
            foreach ($offerData['new_tires'] as $tire) {
                $totalAmount += ($tire['price'] + ($tire['mounting'] ?? 0)) * ($tire['quantity'] ?? 1);
            }
            
            // Addiere Service-Preise
            foreach ($offerData['selected_services'] as $serviceId) {
                $stmt = $conn->prepare("SELECT price FROM autohaus_services WHERE id = ? AND autohaus_id = ?");
                $stmt->execute([$serviceId, $input['autohaus_id']]);
                $service = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($service) {
                    $totalAmount += floatval($service['price']);
                }
            }
            
            $stmt = $conn->prepare("
                INSERT INTO tire_offers (
                    auftrag_nr, kunden_adr_id, customer_name, customer_firstname,
                    license_plate, vin, customer_street, customer_zip, customer_city,
                    customer_phone, customer_email, offer_data, letter_content,
                    employee_name, token, token_expires_at, status, total_amount,
                    autohaus_id, created_by
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $input['auftrag_nr'] ?? null,
                $input['kunden_adr_id'] ?? null,
                $input['customer_name'] ?? null,
                $input['customer_firstname'] ?? null,
                $input['license_plate'] ?? null,
                $input['vin'] ?? null,
                $input['customer_street'] ?? null,
                $input['customer_zip'] ?? null,
                $input['customer_city'] ?? null,
                $input['customer_phone'] ?? null,
                $input['customer_email'] ?? null,
                json_encode($offerData),
                $input['letter_content'] ?? null,
                $input['employee_name'] ?? null,
                $token,
                $tokenExpires,
                'draft',
                $totalAmount,
                $input['autohaus_id'] ?? null,
                $current_user_id
            ]);
            
            $offerId = $conn->lastInsertId();
            
            echo json_encode([
                'success' => true,
                'offer_id' => $offerId,
                'token' => $token,
                'message' => 'Angebot erfolgreich erstellt'
            ]);
            break;
            
        case 'get_offers':
            // Hole Angebote für einen Kunden
            $auftragNr = $_GET['auftrag_nr'] ?? null;
            $licensePlate = $_GET['license_plate'] ?? null;
            $autohausId = $_GET['autohaus_id'] ?? null;
            
            $sql = "SELECT * FROM tire_offers WHERE 1=1";
            $params = [];
            
            if ($auftragNr) {
                $sql .= " AND auftrag_nr = ?";
                $params[] = $auftragNr;
            }
            
            if ($licensePlate) {
                $sql .= " AND license_plate = ?";
                $params[] = $licensePlate;
            }
            
            if ($autohausId) {
                $sql .= " AND autohaus_id = ?";
                $params[] = $autohausId;
            }
            
            $sql .= " ORDER BY created_at DESC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Decode JSON fields
            foreach ($offers as &$offer) {
                if ($offer['offer_data']) {
                    $offer['offer_data'] = json_decode($offer['offer_data'], true);
                }
                if ($offer['customer_selection']) {
                    $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                }
            }
            
            echo json_encode([
                'success' => true,
                'offers' => $offers,
                'count' => count($offers)
            ]);
            break;
            
        case 'get_all':
            // Hole alle Angebote mit Filter (für Übersicht)
            $autohausId = $_GET['autohaus_id'] ?? null;
            $filter = $_GET['filter'] ?? 'all'; // 'all', 'pending', 'accepted', 'declined', 'new_request'
            $search = $_GET['search'] ?? '';
            
            if (!$autohausId) {
                throw new Exception('autohaus_id ist erforderlich');
            }
            
            $sql = "SELECT * FROM tire_offers WHERE autohaus_id = ?";
            $params = [$autohausId];
            
            // Status-Filter
            if ($filter === 'pending') {
                $sql .= " AND (status = 'sent' OR status = 'viewed')";
            } elseif ($filter === 'accepted') {
                $sql .= " AND status = 'accepted'";
            } elseif ($filter === 'declined') {
                $sql .= " AND status = 'declined'";
            } elseif ($filter === 'new_request') {
                $sql .= " AND status = 'new_request'";
            }
            // 'all' zeigt alle Status außer 'draft'
            
            // Suchfilter
            if (!empty($search)) {
                $sql .= " AND (customer_name LIKE ? OR customer_firstname LIKE ? OR license_plate LIKE ? OR auftrag_nr LIKE ?)";
                $searchParam = '%' . $search . '%';
                $params[] = $searchParam;
                $params[] = $searchParam;
                $params[] = $searchParam;
                $params[] = $searchParam;
            }
            
            $sql .= " ORDER BY created_at DESC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $offers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Decode JSON fields
            foreach ($offers as &$offer) {
                if ($offer['offer_data']) {
                    $offer['offer_data'] = json_decode($offer['offer_data'], true);
                }
                if ($offer['customer_selection']) {
                    $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
                }
            }
            
            echo json_encode([
                'success' => true,
                'offers' => $offers,
                'count' => count($offers)
            ]);
            break;
            
        case 'send_email':
            // POST - Email mit Angebot versenden
            logError("Send Email Action gestartet", ['request_method' => $_SERVER['REQUEST_METHOD']]);
            
            try {
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $error = 'JSON Parse Fehler: ' . json_last_error_msg();
                    logError($error, ['raw_input' => substr(file_get_contents('php://input'), 0, 500)]);
                    throw new Exception($error);
                }
                
                logError("JSON Daten empfangen", ['offer_id' => $data['offer_id'] ?? 'N/A', 'recipient_email' => $data['recipient_email'] ?? 'N/A']);
                
                $offerId = $data['offer_id'] ?? null;
                $recipientEmail = $data['recipient_email'] ?? '';
                
                if (!$offerId || empty($recipientEmail)) {
                    logError("Fehlende Parameter", ['offer_id' => $offerId, 'recipient_email' => $recipientEmail]);
                    throw new Exception('offer_id und recipient_email sind erforderlich');
                }
                
                // Lade Angebot
                logError("Lade Angebot", ['offer_id' => $offerId]);
                $stmt = $conn->prepare("SELECT * FROM tire_offers WHERE id = ?");
                $stmt->execute([$offerId]);
                $offer = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$offer) {
                    logError("Angebot nicht gefunden", ['offer_id' => $offerId]);
                    throw new Exception('Angebot nicht gefunden');
                }
                
                logError("Angebot geladen", ['offer_id' => $offerId, 'autohaus_id' => $offer['autohaus_id'] ?? 'N/A']);
                
                // Lade Autohaus-Daten für Email - verwende gleiche Logik wie wheelset_offers.php
                $stmt = $conn->prepare("
                    SELECT name, email_smtp_host, email_smtp_port, email_smtp_username, 
                           email_smtp_password, email_smtp_encryption, email_from_name
                    FROM autohaus WHERE id = ? AND email_smtp_host IS NOT NULL AND email_smtp_host != '' AND email_smtp_password IS NOT NULL AND email_smtp_password != ''
                ");
                $stmt->execute([$offer['autohaus_id']]);
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Falls kein Autohaus mit SMTP-Daten gefunden, versuche erstes aktives Autohaus
                if (!$autohaus || empty($autohaus['email_smtp_host']) || empty($autohaus['email_smtp_password'])) {
                    logError("Autohaus SMTP nicht gefunden, versuche Fallback", ['autohaus_id' => $offer['autohaus_id']]);
                    $stmt = $conn->prepare("
                        SELECT name, email_smtp_host, email_smtp_port, email_smtp_username, 
                               email_smtp_password, email_smtp_encryption, email_from_name
                        FROM autohaus 
                        WHERE is_active = 1 
                          AND email_smtp_host IS NOT NULL 
                          AND email_smtp_host != '' 
                          AND email_smtp_password IS NOT NULL 
                          AND email_smtp_password != ''
                        ORDER BY id LIMIT 1
                    ");
                    $stmt->execute();
                    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                }
                
                // Prüfe ob Autohaus mit SMTP-Daten gefunden wurde
                if (!$autohaus || empty($autohaus['email_smtp_host']) || empty($autohaus['email_smtp_password'])) {
                    logError("SMTP-Konfiguration fehlt", ['autohaus_id' => $offer['autohaus_id']]);
                    throw new Exception('SMTP-Konfiguration fehlt. Bitte konfigurieren Sie die SMTP-Einstellungen für das Autohaus in den Einstellungen.');
                }
                
                logError("SMTP-Konfiguration gefunden", [
                    'autohaus_name' => $autohaus['name'] ?? 'N/A',
                    'smtp_host' => $autohaus['email_smtp_host'] ?? 'N/A',
                    'smtp_port' => $autohaus['email_smtp_port'] ?? 'N/A',
                    'smtp_username' => substr($autohaus['email_smtp_username'] ?? 'N/A', 0, 20) . '...'
                ]);
                
                // Generiere PDF (vereinfacht - sollte in separater Datei sein)
                $pdfFilename = 'angebot_' . ($offer['auftrag_nr'] ?? $offer['id']) . '_' . date('Ymd_His') . '.pdf';
                
                // Erstelle Email-Inhalt
                logError("Lade send_tire_offer_email.php");
                $emailFile = __DIR__ . '/send_tire_offer_email.php';
                if (!file_exists($emailFile)) {
                    logError("send_tire_offer_email.php nicht gefunden", ['file_path' => $emailFile]);
                    throw new Exception('E-Mail-Template-Datei nicht gefunden: send_tire_offer_email.php');
                }
                require_once $emailFile;
                
                logError("Rufe sendTireOfferEmail auf", [
                    'offer_id' => $offerId,
                    'recipient_email' => $recipientEmail,
                    'token' => $offer['token'] ?? 'N/A'
                ]);
                
                $emailResult = sendTireOfferEmail($offer, $autohaus, $recipientEmail, $offer['token']);
                
                logError("sendTireOfferEmail Ergebnis", ['success' => $emailResult['success'] ?? false, 'error' => $emailResult['error'] ?? 'N/A']);
                
                if ($emailResult['success']) {
                    // Update Angebot Status - wenn es ein Entwurf war, ändere Status zu 'sent'
                    $newStatus = $offer['status'] === 'draft' ? 'sent' : $offer['status'];
                    
                    $stmt = $conn->prepare("
                        UPDATE tire_offers 
                        SET email_sent = 1, email_sent_at = NOW(), status = ?, pdf_filename = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$newStatus, $pdfFilename, $offerId]);
                    
                    logError("Angebot Status aktualisiert", ['offer_id' => $offerId, 'new_status' => $newStatus]);
                    
                    echo json_encode([
                        'success' => true,
                        'message' => 'Email erfolgreich versendet'
                    ]);
                } else {
                    $errorMsg = $emailResult['error'] ?? 'Fehler beim Versenden der Email';
                    logError("E-Mail Versendung fehlgeschlagen", ['error' => $errorMsg]);
                    throw new Exception($errorMsg);
                }
            } catch (Exception $e) {
                logError("Exception in send_email", [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => substr($e->getTraceAsString(), 0, 500)
                ]);
                throw $e;
            }
            break;
            
        case 'get_orders_by_license':
            // Hole alle Aufträge für ein Kennzeichen
            $licensePlate = $_GET['license_plate'] ?? '';
            $days = intval($_GET['days'] ?? 90);
            
            if (empty($licensePlate)) {
                throw new Exception('Kennzeichen ist erforderlich');
            }
            
            $orders = $api->getAllOrdersByLicensePlate($licensePlate, $days);
            
            echo json_encode([
                'success' => true,
                'orders' => $orders,
                'count' => count($orders)
            ]);
            break;
            
        case 'get_offer':
            // Hole ein einzelnes Angebot
            $offerId = $_GET['offer_id'] ?? null;
            
            if (!$offerId) {
                throw new Exception('offer_id ist erforderlich');
            }
            
            $stmt = $conn->prepare("SELECT * FROM tire_offers WHERE id = ?");
            $stmt->execute([$offerId]);
            $offer = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$offer) {
                throw new Exception('Angebot nicht gefunden');
            }
            
            // Decode JSON fields
            if ($offer['offer_data']) {
                $offer['offer_data'] = json_decode($offer['offer_data'], true);
            }
            if ($offer['customer_selection']) {
                $offer['customer_selection'] = json_decode($offer['customer_selection'], true);
            }
            
            echo json_encode([
                'success' => true,
                'offer' => $offer
            ]);
            break;
            
        case 'update_offer':
            // Aktualisiere Angebot (z.B. Auftragsnummer)
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['offer_id'])) {
                throw new Exception('offer_id ist erforderlich');
            }
            
            $offerId = $input['offer_id'];
            $updates = [];
            $params = [];
            
            if (isset($input['auftrag_nr'])) {
                $updates[] = "auftrag_nr = ?";
                $params[] = $input['auftrag_nr'];
            }
            
            if (empty($updates)) {
                throw new Exception('Keine Updates angegeben');
            }
            
            $params[] = $offerId;
            $sql = "UPDATE tire_offers SET " . implode(', ', $updates) . " WHERE id = ?";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            echo json_encode([
                'success' => true,
                'message' => 'Angebot aktualisiert'
            ]);
            break;
            
        case 'get_customer_email':
            // Hole E-Mail-Adresse aus API
            try {
                $kundenAdrId = $_GET['kunden_adr_id'] ?? '';
                $licensePlate = $_GET['license_plate'] ?? '';
                
                if (empty($kundenAdrId) && empty($licensePlate)) {
                    throw new Exception('Kunden-Adress-ID oder Kennzeichen ist erforderlich');
                }
                
                $email = null;
                
                // Versuche zuerst über Kunden-Adress-ID - suche in der Kundenliste
                // Die API hat keinen direkten Endpoint für einzelne Kunden, daher suchen wir in der Liste
                if (!empty($kundenAdrId)) {
                    try {
                        // Suche nach Kunden mit dieser ID (mit Suchbegriff falls möglich)
                        // Falls keine direkte Suche möglich, versuche über Aufträge
                        $result = $api->getTermine(90); // Hole Termine der letzten 90 Tage
                        if (is_array($result)) {
                            foreach ($result as $termin) {
                                if (isset($termin['KundenAdrID']) && $termin['KundenAdrID'] == $kundenAdrId) {
                                    // Prüfe verschiedene mögliche Email-Felder
                                    $email = $termin['Email'] ?? $termin['email'] ?? $termin['E-Mail'] ?? $termin['e_mail'] ?? null;
                                    if ($email) break;
                                }
                            }
                        }
                        
                        // Falls nicht gefunden, versuche auch in Kundenliste
                        if (empty($email)) {
                            $customerResult = $api->searchCustomers('', 1000);
                            if (is_array($customerResult)) {
                                foreach ($customerResult as $customer) {
                                    if (isset($customer['KundenAdrID']) && $customer['KundenAdrID'] == $kundenAdrId) {
                                        // Prüfe verschiedene mögliche Email-Felder
                                        $email = $customer['Email'] ?? $customer['email'] ?? $customer['E-Mail'] ?? $customer['e_mail'] ?? null;
                                        if ($email) break;
                                    }
                                }
                            }
                        }
                    } catch (Exception $e) {
                        error_log("Error fetching customer email by ID: " . $e->getMessage());
                    }
                }
                
                // Falls nicht gefunden, versuche über Kennzeichen
                if (empty($email) && !empty($licensePlate)) {
                    try {
                        $auftrag = $api->findCustomerByLicensePlate($licensePlate);
                        if ($auftrag) {
                            // Prüfe verschiedene mögliche Email-Felder
                            $email = $auftrag['Email'] ?? $auftrag['email'] ?? $auftrag['E-Mail'] ?? $auftrag['e_mail'] ?? null;
                        }
                    } catch (Exception $e) {
                        error_log("Error fetching customer email by license: " . $e->getMessage());
                    }
                }
                
                echo json_encode([
                    'success' => !empty($email),
                    'email' => $email,
                    'error' => empty($email) ? 'E-Mail-Adresse nicht gefunden. Bitte kontaktieren Sie den Kunden direkt.' : null
                ], JSON_UNESCAPED_UNICODE);
            } catch (Exception $e) {
                logError("Exception in get_customer_email", [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
                
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'email' => null,
                    'error' => 'Fehler beim Abrufen der E-Mail-Adresse: ' . $e->getMessage()
                ], JSON_UNESCAPED_UNICODE);
            }
            break;
            
        case 'autocomplete_customers':
            // Autocomplete für Kundennamen
            $search = $_GET['search'] ?? '';
            $limit = intval($_GET['limit'] ?? 10);
            
            if (empty($search) || strlen($search) < 2) {
                echo json_encode(['success' => true, 'data' => []]);
                break;
            }
            
            $customers = $api->searchCustomers($search, $limit);
            $suggestions = array_map(function($customer) {
                // Prüfe verschiedene mögliche Email-Felder
                $email = $customer['Email'] ?? $customer['email'] ?? $customer['E-Mail'] ?? $customer['e_mail'] ?? null;
                
                return [
                    'id' => $customer['KundenAdrID'] ?? null,
                    'name' => trim(($customer['KundenName1'] ?? '') . ' ' . ($customer['KundenVorname'] ?? '')),
                    'license_plate' => $customer['ZulassungsKz'] ?? null,
                    'email' => $email,
                    'full_data' => $customer // Vollständige Daten für die Suche
                ];
            }, $customers);
            
            echo json_encode([
                'success' => true,
                'data' => $suggestions
            ]);
            break;
            
        case 'autocomplete_license':
            // Autocomplete für Kennzeichen - OPTIMIERT für Performance
            $search = trim($_GET['search'] ?? '');
            $days = intval($_GET['days'] ?? 90); // Reduziert für schnellere Vorschläge
            
            if (empty($search) || strlen($search) < 2) {
                echo json_encode(['success' => true, 'data' => []]);
                break;
            }
            
            try {
                $result = $api->apiCall('/auftraege', 'GET', ['days' => $days]);
                if (isset($result['success']) && $result['success']) {
                    $auftraege = $result['data'] ?? [];
                    $licensePlates = [];
                    $seen = [];
                    
                    foreach ($auftraege as $auftrag) {
                        $license = $auftrag['ZulassungsKz'] ?? '';
                        if (!empty($license) && 
                            stripos($license, $search) !== false && 
                            !isset($seen[$license])) {
                            $seen[$license] = true;
                            // Prüfe verschiedene mögliche Email-Felder
                            $email = $auftrag['Email'] ?? $auftrag['email'] ?? $auftrag['E-Mail'] ?? $auftrag['e_mail'] ?? null;
                            
                            $licensePlates[] = [
                                'license_plate' => $license,
                                'customer_name' => trim(($auftrag['KundenName1'] ?? '') . ' ' . ($auftrag['KundenVorname'] ?? '')),
                                'auftrag_nr' => $auftrag['AuftragNr'] ?? null,
                                'email' => $email,
                                'full_data' => $auftrag // Vollständige Daten für die Suche
                            ];
                            
                            if (count($licensePlates) >= 10) break;
                        }
                    }
                    
                    echo json_encode([
                        'success' => true,
                        'data' => $licensePlates
                    ]);
                } else {
                    echo json_encode(['success' => true, 'data' => []]);
                }
            } catch (Exception $e) {
                error_log("Error in autocomplete_license: " . $e->getMessage());
                echo json_encode(['success' => true, 'data' => []]);
            }
            break;
            
        case 'validate_customer_link':
            // Validiere Verknüpfung zwischen Name und Kennzeichen
            $name = $_GET['name'] ?? '';
            $licensePlate = $_GET['license_plate'] ?? '';
            
            if (empty($name) || empty($licensePlate)) {
                echo json_encode([
                    'success' => false,
                    'warning' => 'Bitte geben Sie sowohl Name als auch Kennzeichen ein.'
                ]);
                break;
            }
            
            // Suche nach Kennzeichen
            $auftrag = $api->findCustomerByLicensePlate($licensePlate);
            
            if (!$auftrag) {
                echo json_encode([
                    'success' => false,
                    'warning' => 'Kein Auftrag mit diesem Kennzeichen gefunden. Bitte prüfen Sie das Kennzeichen.'
                ]);
                break;
            }
            
            // Prüfe ob Name übereinstimmt
            $auftragName = trim(($auftrag['KundenName1'] ?? '') . ' ' . ($auftrag['KundenVorname'] ?? ''));
            $searchName = trim($name);
            
            $nameMatch = stripos($auftragName, $searchName) !== false || 
                         stripos($searchName, $auftragName) !== false;
            
            echo json_encode([
                'success' => $nameMatch,
                'warning' => $nameMatch ? null : 'Warnung: Der Name stimmt nicht mit dem gefundenen Auftrag überein. Bitte prüfen Sie die Eingaben.',
                'auftrag_name' => $auftragName,
                'found_email' => $auftrag['Email'] ?? null
            ]);
            break;
            
        case 'check_health':
            // Prüfe API-Verbindung mit Details
            $healthDetails = $api->checkHealthWithDetails();
            echo json_encode([
                'success' => $healthDetails['success'],
                'api_available' => $healthDetails['success'],
                'url' => $healthDetails['url'] ?? '',
                'http_code' => $healthDetails['http_code'] ?? 0,
                'error' => $healthDetails['error'] ?? '',
                'details' => $healthDetails['details'] ?? [],
                'response_preview' => isset($healthDetails['response']) ? substr($healthDetails['response'], 0, 200) : ''
            ]);
            break;
            
        default:
            throw new Exception('Unbekannte Aktion: ' . $action);
    }
} catch (Exception $e) {
    logError("Unbehandelte Exception", [
        'action' => $action ?? 'N/A',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => substr($e->getTraceAsString(), 0, 1000)
    ]);
    
    http_response_code(500);
    
    // Erstelle detaillierte Fehlerantwort
    $errorResponse = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'action' => $action ?? 'N/A'
        ]
    ];
    
    // In Development: Mehr Details
    if (defined('DEBUG_MODE') && DEBUG_MODE) {
        $errorResponse['debug']['trace'] = explode("\n", substr($e->getTraceAsString(), 0, 2000));
    }
    
    echo json_encode($errorResponse, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}

