<?php
// Set headers based on request method
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // For GET requests (from Teams buttons), return HTML
    header('Content-Type: text/html; charset=UTF-8');
} else {
    // For POST requests, return JSON
    header('Content-Type: application/json');
}

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    // Handle both GET and POST requests
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // Handle GET request from Teams Action.OpenUrl
        $appointmentId = $_GET['appointment_id'] ?? null;
        $reactionType = $_GET['reaction_type'] ?? null;
        $user = $_GET['user'] ?? 'Teams User';
    } else {
        // Handle POST request (original functionality)
        $input = json_decode(file_get_contents('php://input'), true);
        $appointmentId = $input['appointment_id'] ?? null;
        $reactionType = $input['reaction_type'] ?? null;
        $user = $input['user'] ?? 'Unbekannt';
    }
    
    if (!$appointmentId || !$reactionType) {
        throw new Exception('Appointment ID und Reaction Type sind erforderlich');
    }
    
    // Validate reaction type
    $validReactions = ['tire_ordered', 'work_completed', 'customer_notified'];
    if (!in_array($reactionType, $validReactions)) {
        throw new Exception('Ungültiger Reaction Type');
    }
    
    // Get Teams message ID for this appointment
    $stmt = $conn->prepare("
        SELECT teams_message_id 
        FROM offer_tracking 
        WHERE appointment_id = ? 
        AND offer_type = 'sent' 
        AND teams_notification_sent = TRUE
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    
    $stmt->execute([$appointmentId]);
    $offerData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$offerData) {
        throw new Exception('Kein Angebot für diesen Termin gefunden');
    }
    
    $teamsMessageId = $offerData['teams_message_id'] ?: 'msg_' . $appointmentId . '_' . time();
    
    // Save reaction to database
    $stmt = $conn->prepare("
        INSERT INTO teams_reactions 
        (appointment_id, teams_message_id, reaction_type, reacted_by, reacted_at) 
        VALUES (?, ?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE 
        reacted_at = NOW()
    ");
    
    $stmt->execute([$appointmentId, $teamsMessageId, $reactionType, $user]);
    
    // Update offer tracking status based on reaction
    $statusUpdate = '';
    switch ($reactionType) {
        case 'tire_ordered':
            $statusUpdate = 'Neue Reifen bestellt';
            break;
        case 'work_completed':
            $statusUpdate = 'Arbeit abgeschlossen';
            break;
        case 'customer_notified':
            $statusUpdate = 'Kunde benachrichtigt';
            break;
    }
    
    // Update offer tracking with new status
    $stmt = $conn->prepare("
        UPDATE offer_tracking 
        SET status_update = ?, status_updated_at = NOW()
        WHERE appointment_id = ? 
        AND offer_type = 'sent'
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    
    $stmt->execute([$statusUpdate, $appointmentId]);
    
    // Send confirmation message back to Teams
    $confirmationMessage = [
        "@type" => "MessageCard",
        "@context" => "http://schema.org/extensions",
        "themeColor" => "28a745",
        "summary" => "Status aktualisiert: " . $statusUpdate,
        "sections" => [
            [
                "activityTitle" => "✅ Status aktualisiert!",
                "activitySubtitle" => "**" . $user . "** hat den Status geändert",
                "facts" => [
                    [
                        "name" => "📋 Neuer Status",
                        "value" => "**" . $statusUpdate . "**"
                    ],
                    [
                        "name" => "👤 Aktualisiert von",
                        "value" => $user
                    ],
                    [
                        "name" => "⏰ Zeitpunkt",
                        "value" => date('d.m.Y H:i:s')
                    ]
                ],
                "markdown" => true
            ]
        ]
    ];
    
    // Return success response
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // Return HTML page for GET requests (from Teams)
        ?>
        <!DOCTYPE html>
        <html lang="de">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Status aktualisiert</title>
            <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=Space+Grotesk:wght@300;400;500;600;700&display=swap" rel="stylesheet">
            <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
            <style>
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                
                body {
                    font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: 20px;
                }
                
                .status-container {
                    background: white;
                    border-radius: 16px;
                    box-shadow: 0 10px 40px rgba(0,0,0,0.1);
                    max-width: 500px;
                    width: 100%;
                    overflow: hidden;
                    position: relative;
                }
                
                .status-header {
                    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
                    padding: 2rem;
                    text-align: center;
                    color: white;
                    position: relative;
                }
                
                .status-header::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="10" cy="60" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="40" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
                    opacity: 0.3;
                }
                
                .success-icon {
                    font-size: 3rem;
                    margin-bottom: 1rem;
                    animation: pulse 2s infinite;
                    position: relative;
                    z-index: 1;
                }
                
                @keyframes pulse {
                    0% { transform: scale(1); }
                    50% { transform: scale(1.1); }
                    100% { transform: scale(1); }
                }
                
                .status-title {
                    font-size: 1.5rem;
                    font-weight: 600;
                    margin-bottom: 0.5rem;
                    position: relative;
                    z-index: 1;
                }
                
                .status-subtitle {
                    font-size: 0.9rem;
                    opacity: 0.9;
                    position: relative;
                    z-index: 1;
                }
                
                .status-content {
                    padding: 2rem;
                }
                
                .status-badge {
                    background: linear-gradient(135deg, #e8f5e8 0%, #d4edda 100%);
                    color: #2e7d32;
                    padding: 1rem 1.5rem;
                    border-radius: 12px;
                    margin-bottom: 1.5rem;
                    font-size: 1.1rem;
                    font-weight: 600;
                    text-align: center;
                    border-left: 4px solid #28a745;
                    box-shadow: 0 2px 8px rgba(40, 167, 69, 0.1);
                }
                
                .status-info {
                    color: #6c757d;
                    margin-bottom: 1.5rem;
                    line-height: 1.6;
                    text-align: center;
                }
                
                .status-details {
                    background: #f8f9fa;
                    border-radius: 12px;
                    padding: 1.5rem;
                    margin-bottom: 1.5rem;
                    border: 1px solid #e9ecef;
                }
                
                .detail-item {
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    padding: 0.5rem 0;
                    border-bottom: 1px solid #e9ecef;
                }
                
                .detail-item:last-child {
                    border-bottom: none;
                }
                
                .detail-label {
                    font-weight: 500;
                    color: #495057;
                    display: flex;
                    align-items: center;
                    gap: 0.5rem;
                }
                
                .detail-value {
                    color: #6c757d;
                    font-weight: 600;
                }
                
                .status-actions {
                    display: flex;
                    flex-direction: column;
                    gap: 1rem;
                }
                
                .btn-primary {
                    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
                    color: white;
                    border: none;
                    padding: 1rem 2rem;
                    border-radius: 12px;
                    font-size: 1rem;
                    font-weight: 600;
                    cursor: pointer;
                    transition: all 0.3s ease;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    gap: 0.5rem;
                    box-shadow: 0 4px 12px rgba(0, 123, 255, 0.3);
                }
                
                .btn-primary:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 6px 20px rgba(0, 123, 255, 0.4);
                }
                
                .auto-close-info {
                    text-align: center;
                    color: #6c757d;
                    font-size: 0.85rem;
                    padding: 1rem;
                    background: #f8f9fa;
                    border-radius: 8px;
                    border: 1px solid #e9ecef;
                }
                
                .countdown {
                    font-weight: 600;
                    color: #007bff;
                }
                
                @media (max-width: 480px) {
                    .status-container {
                        margin: 10px;
                    }
                    
                    .status-header {
                        padding: 1.5rem;
                    }
                    
                    .status-content {
                        padding: 1.5rem;
                    }
                    
                    .status-title {
                        font-size: 1.25rem;
                    }
                }
            </style>
            <script>
                // Auto-close after 5 seconds
                let countdown = 5;
                const countdownElement = document.getElementById('countdown');
                
                function updateCountdown() {
                    if (countdown > 0) {
                        countdownElement.textContent = countdown;
                        countdown--;
                        setTimeout(updateCountdown, 1000);
                    } else {
                        window.close();
                    }
                }
                
                // Start countdown when page loads
                window.onload = function() {
                    setTimeout(updateCountdown, 1000);
                };
            </script>
        </head>
        <body>
            <div class="status-container">
                <div class="status-header">
                    <div class="success-icon">✅</div>
                    <h1 class="status-title">Status erfolgreich aktualisiert!</h1>
                    <p class="status-subtitle">Der Status wurde erfolgreich in der Datenbank gespeichert</p>
                </div>
                
                <div class="status-content">
                    <div class="status-badge">
                        <?php echo htmlspecialchars($statusUpdate); ?>
                    </div>
                    
                    <div class="status-info">
                        Das Admin-Dashboard wurde automatisch aktualisiert und alle relevanten Systeme wurden benachrichtigt.
                    </div>
                    
                    <div class="status-details">
                        <div class="detail-item">
                            <span class="detail-label">
                                <i class="fas fa-info-circle"></i>
                                Status
                            </span>
                            <span class="detail-value"><?php echo htmlspecialchars($statusUpdate); ?></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label">
                                <i class="fas fa-user"></i>
                                Aktualisiert von
                            </span>
                            <span class="detail-value"><?php echo htmlspecialchars($user); ?></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label">
                                <i class="fas fa-clock"></i>
                                Zeitpunkt
                            </span>
                            <span class="detail-value"><?php echo date('d.m.Y H:i:s'); ?></span>
                        </div>
                    </div>
                    
                    <div class="status-actions">
                        <button class="btn-primary" onclick="window.close()">
                            <i class="fas fa-times"></i>
                            Fenster schließen
                        </button>
                        
                        <div class="auto-close-info">
                            <i class="fas fa-clock"></i>
                            Das Fenster schließt sich automatisch in <span class="countdown" id="countdown">5</span> Sekunden
                        </div>
                    </div>
                </div>
            </div>
        </body>
        </html>
        <?php
    } else {
        // Return JSON for POST requests
        echo json_encode([
            'success' => true,
            'message' => 'Reaktion erfolgreich gespeichert',
            'status_update' => $statusUpdate,
            'confirmation_message' => $confirmationMessage
        ]);
    }
    
} catch (Exception $e) {
    error_log("Error in teams_reaction.php: " . $e->getMessage());
    
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // Return HTML error page for GET requests
        ?>
        <!DOCTYPE html>
        <html lang="de">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Fehler</title>
            <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=Space+Grotesk:wght@300;400;500;600;700&display=swap" rel="stylesheet">
            <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
            <style>
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                
                body {
                    font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: 20px;
                }
                
                .status-container {
                    background: white;
                    border-radius: 16px;
                    box-shadow: 0 10px 40px rgba(0,0,0,0.1);
                    max-width: 500px;
                    width: 100%;
                    overflow: hidden;
                    position: relative;
                }
                
                .status-header {
                    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
                    padding: 2rem;
                    text-align: center;
                    color: white;
                    position: relative;
                }
                
                .status-header::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="10" cy="60" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="40" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
                    opacity: 0.3;
                }
                
                .error-icon {
                    font-size: 3rem;
                    margin-bottom: 1rem;
                    position: relative;
                    z-index: 1;
                }
                
                .status-title {
                    font-size: 1.5rem;
                    font-weight: 600;
                    margin-bottom: 0.5rem;
                    position: relative;
                    z-index: 1;
                }
                
                .status-subtitle {
                    font-size: 0.9rem;
                    opacity: 0.9;
                    position: relative;
                    z-index: 1;
                }
                
                .status-content {
                    padding: 2rem;
                }
                
                .error-badge {
                    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
                    color: #721c24;
                    padding: 1rem 1.5rem;
                    border-radius: 12px;
                    margin-bottom: 1.5rem;
                    font-size: 1rem;
                    text-align: center;
                    border-left: 4px solid #dc3545;
                    box-shadow: 0 2px 8px rgba(220, 53, 69, 0.1);
                }
                
                .status-actions {
                    display: flex;
                    flex-direction: column;
                    gap: 1rem;
                }
                
                .btn-primary {
                    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
                    color: white;
                    border: none;
                    padding: 1rem 2rem;
                    border-radius: 12px;
                    font-size: 1rem;
                    font-weight: 600;
                    cursor: pointer;
                    transition: all 0.3s ease;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    gap: 0.5rem;
                    box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
                }
                
                .btn-primary:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 6px 20px rgba(220, 53, 69, 0.4);
                }
                
                @media (max-width: 480px) {
                    .status-container {
                        margin: 10px;
                    }
                    
                    .status-header {
                        padding: 1.5rem;
                    }
                    
                    .status-content {
                        padding: 1.5rem;
                    }
                    
                    .status-title {
                        font-size: 1.25rem;
                    }
                }
            </style>
        </head>
        <body>
            <div class="status-container">
                <div class="status-header">
                    <div class="error-icon">❌</div>
                    <h1 class="status-title">Fehler aufgetreten</h1>
                    <p class="status-subtitle">Es ist ein unerwarteter Fehler aufgetreten</p>
                </div>
                
                <div class="status-content">
                    <div class="error-badge">
                        <?php echo htmlspecialchars($e->getMessage()); ?>
                    </div>
                    
                    <div class="status-actions">
                        <button class="btn-primary" onclick="window.close()">
                            <i class="fas fa-times"></i>
                            Fenster schließen
                        </button>
                    </div>
                </div>
            </div>
        </body>
        </html>
        <?php
    } else {
        // Return JSON error for POST requests
        echo json_encode([
            'success' => false,
            'error' => 'Fehler beim Speichern der Reaktion: ' . $e->getMessage()
        ]);
    }
}
?>
