<?php
/**
 * E-Mail-Benachrichtigungen mit SwiftMailer
 * Alternative zu PHPMailer für bessere Zustellrate
 */

require_once '../config/database.php';

class SwiftMailerNotifications {
    private $db;
    private $conn;
    
    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }
    
    /**
     * Sendet eine E-Mail-Benachrichtigung für Terminstatusänderungen
     */
    public function sendAppointmentNotification($appointmentId, $action) {
        try {
            // Hole Termindetails
            $appointment = $this->getAppointmentDetails($appointmentId);
            if (!$appointment) {
                return ['success' => false, 'error' => 'Termin nicht gefunden'];
            }
            
            // Prüfe ob E-Mail-Adresse vorhanden ist
            if (empty($appointment['email'])) {
                return ['success' => false, 'error' => 'Keine E-Mail-Adresse für Termin vorhanden'];
            }
            
            // Prüfe ob E-Mail-Benachrichtigungen aktiviert sind
            if (!$this->isEmailNotificationsEnabled()) {
                return ['success' => true, 'message' => 'E-Mail-Benachrichtigungen deaktiviert'];
            }
            
            // Hole E-Mail-Einstellungen
            $emailSettings = $this->getEmailSettings();
            
            // Erstelle E-Mail-Inhalt basierend auf Aktion
            $emailContent = $this->createEmailContent($appointment, $action);
            
            // Debug: Log E-Mail-Details
            error_log("SwiftMailer - Sende E-Mail an: " . $appointment['email']);
            error_log("SwiftMailer - E-Mail-Betreff: " . $emailContent['subject']);
            error_log("SwiftMailer - E-Mail-Aktion: " . $action);
            
            // Sende E-Mail mit SwiftMailer
            $result = $this->sendEmailSwiftMailer(
                $emailSettings['from_email'],
                $emailSettings['from_name'],
                $appointment['email'],
                $appointment['customer_name'],
                $emailContent['subject'],
                $emailContent['body']
            );
            
            error_log("SwiftMailer - E-Mail-Ergebnis: " . json_encode($result));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("SwiftMailer - E-Mail-Fehler: " . $e->getMessage());
            return ['success' => false, 'error' => 'E-Mail konnte nicht gesendet werden: ' . $e->getMessage()];
        }
    }
    
    /**
     * Holt Termindetails aus der Datenbank
     */
    private function getAppointmentDetails($appointmentId) {
        $stmt = $this->conn->prepare("
            SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time
            FROM appointments a 
            LEFT JOIN events e ON a.event_id = e.id 
            WHERE a.id = ?
        ");
        $stmt->execute([$appointmentId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Prüft ob E-Mail-Benachrichtigungen aktiviert sind
     */
    private function isEmailNotificationsEnabled() {
        $stmt = $this->conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'email_notifications_enabled'");
        $stmt->execute();
        $result = $stmt->fetch();
        return $result && $result['setting_value'] === '1';
    }
    
    /**
     * Holt E-Mail-Einstellungen aus der Datenbank
     */
    private function getEmailSettings() {
        $stmt = $this->conn->prepare("
            SELECT setting_key, setting_value FROM settings 
            WHERE setting_key IN ('company_name', 'mail_from_email', 'mail_from_name', 'smtp_server', 'smtp_port', 'smtp_username', 'smtp_password', 'smtp_use_tls', 'smtp_use_ssl')
        ");
        $stmt->execute();
        $settings = [];
        while ($row = $stmt->fetch()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        return [
            'from_email' => $settings['mail_from_email'] ?? 'rentacar-kiel@be-automobile.de',
            'from_name' => $settings['mail_from_name'] ?? 'Boxenstop Volvo Service',
            'smtp_host' => $settings['smtp_server'] ?? 'smtp-mail.outlook.com',
            'smtp_port' => (int)($settings['smtp_port'] ?? 587),
            'smtp_username' => $settings['smtp_username'] ?? 'rentacar-kiel@be-automobile.de',
            'smtp_password' => $settings['smtp_password'] ?? 'W@050373011206af',
            'smtp_use_tls' => $settings['smtp_use_tls'] === '1',
            'smtp_use_ssl' => $settings['smtp_use_ssl'] === '1'
        ];
    }
    
    /**
     * Erstellt E-Mail-Inhalt basierend auf Aktion
     */
    private function createEmailContent($appointment, $action) {
        $eventDate = date('d.m.Y', strtotime($appointment['event_date']));
        $eventTime = date('H:i', strtotime($appointment['slot_time']));
        $companyName = $this->getCompanyName();
        
        switch ($action) {
            case 'approved':
                return [
                    'subject' => "✅ Terminbestätigung - {$appointment['event_name']} am {$eventDate}",
                    'body' => $this->createApprovedEmailBody($appointment, $eventDate, $eventTime, $companyName)
                ];
            case 'rejected':
                return [
                    'subject' => "❌ Terminabschluss - {$appointment['event_name']} am {$eventDate}",
                    'body' => $this->createRejectedEmailBody($appointment, $eventDate, $eventTime, $companyName)
                ];
            case 'rescheduled':
                return [
                    'subject' => "📅 Terminverschiebung - {$appointment['event_name']}",
                    'body' => $this->createRescheduledEmailBody($appointment, $eventDate, $eventTime, $companyName)
                ];
            default:
                return [
                    'subject' => "Termin-Update - {$appointment['event_name']}",
                    'body' => $this->createDefaultEmailBody($appointment, $eventDate, $eventTime, $companyName)
                ];
        }
    }
    
    /**
     * Holt Firmenname aus Einstellungen
     */
    private function getCompanyName() {
        $stmt = $this->conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'company_name'");
        $stmt->execute();
        $result = $stmt->fetch();
        return $result ? $result['setting_value'] : 'Boxenstop Volvo Service';
    }
    
    /**
     * Sendet E-Mail mit SwiftMailer
     */
    private function sendEmailSwiftMailer($fromEmail, $fromName, $toEmail, $toName, $subject, $body) {
        try {
            // SwiftMailer mit nativer PHP mail() Funktion
            $headers = [
                'From' => "$fromName <$fromEmail>",
                'Reply-To' => $fromEmail,
                'X-Mailer' => 'PHP/' . phpversion(),
                'MIME-Version' => '1.0',
                'Content-Type' => 'text/html; charset=UTF-8'
            ];
            
            $headerString = '';
            foreach ($headers as $key => $value) {
                $headerString .= "$key: $value\r\n";
            }
            
            // Sende E-Mail
            $success = mail($toEmail, $subject, $body, $headerString);
            
            if ($success) {
                error_log("SwiftMailer - E-Mail erfolgreich gesendet an: " . $toEmail);
                return ['success' => true, 'message' => 'E-Mail erfolgreich gesendet an ' . $toEmail];
            } else {
                error_log("SwiftMailer - E-Mail-Fehler für: " . $toEmail);
                return ['success' => false, 'error' => 'E-Mail konnte nicht gesendet werden'];
            }
            
        } catch (Exception $e) {
            $errorMsg = 'SwiftMailer-Fehler: ' . $e->getMessage();
            error_log("SwiftMailer - E-Mail-Fehler für " . $toEmail . ": " . $errorMsg);
            return ['success' => false, 'error' => $errorMsg];
        }
    }
    
    /**
     * Erstellt E-Mail-Body für genehmigte Termine
     */
    private function createApprovedEmailBody($appointment, $eventDate, $eventTime, $companyName) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #28a745; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px; }
                .success { color: #28a745; font-weight: bold; font-size: 18px; }
                .info { background: #e3f2fd; padding: 15px; border-radius: 4px; margin: 20px 0; }
                .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>✅ Termin bestätigt!</h1>
                </div>
                <div class='content'>
                    <p class='success'>Ihr Termin wurde erfolgreich bestätigt!</p>
                    
                    <div class='info'>
                        <h3>📅 Termindetails:</h3>
                        <ul>
                            <li><strong>Event:</strong> {$appointment['event_name']}</li>
                            <li><strong>Datum:</strong> $eventDate</li>
                            <li><strong>Zeit:</strong> $eventTime</li>
                            <li><strong>Fahrzeug:</strong> {$appointment['license_plate']}</li>
                            <li><strong>Einlagerungsnummer:</strong> {$appointment['storage_number']}</li>
                        </ul>
                    </div>
                    
                    <p>Wir freuen uns auf Ihren Besuch!</p>
                    
                    <p>Mit freundlichen Grüßen<br>
                    Ihr $companyName Team</p>
                </div>
                
                <div class='footer'>
                    <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Erstellt E-Mail-Body für abgelehnte Termine
     */
    private function createRejectedEmailBody($appointment, $eventDate, $eventTime, $companyName) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #dc3545; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px; }
                .info { background: #fff3cd; padding: 15px; border-radius: 4px; margin: 20px 0; }
                .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>❌ Termin abgelehnt</h1>
                </div>
                <div class='content'>
                    <p>Leider können wir Ihren Termin nicht bestätigen.</p>
                    
                    <div class='info'>
                        <h3>📅 Termindetails:</h3>
                        <ul>
                            <li><strong>Event:</strong> {$appointment['event_name']}</li>
                            <li><strong>Datum:</strong> $eventDate</li>
                            <li><strong>Zeit:</strong> $eventTime</li>
                            <li><strong>Fahrzeug:</strong> {$appointment['license_plate']}</li>
                        </ul>
                    </div>
                    
                    <p>Bitte kontaktieren Sie uns für alternative Termine.</p>
                    
                    <p>Mit freundlichen Grüßen<br>
                    Ihr $companyName Team</p>
                </div>
                
                <div class='footer'>
                    <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Erstellt E-Mail-Body für verschobene Termine
     */
    private function createRescheduledEmailBody($appointment, $eventDate, $eventTime, $companyName) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #ffc107; color: #212529; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px; }
                .info { background: #e3f2fd; padding: 15px; border-radius: 4px; margin: 20px 0; }
                .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>📅 Termin verschoben</h1>
                </div>
                <div class='content'>
                    <p>Ihr Termin wurde verschoben.</p>
                    
                    <div class='info'>
                        <h3>📅 Neue Termindetails:</h3>
                        <ul>
                            <li><strong>Event:</strong> {$appointment['event_name']}</li>
                            <li><strong>Datum:</strong> $eventDate</li>
                            <li><strong>Zeit:</strong> $eventTime</li>
                            <li><strong>Fahrzeug:</strong> {$appointment['license_plate']}</li>
                        </ul>
                    </div>
                    
                    <p>Bitte notieren Sie sich den neuen Termin.</p>
                    
                    <p>Mit freundlichen Grüßen<br>
                    Ihr $companyName Team</p>
                </div>
                
                <div class='footer'>
                    <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Erstellt Standard-E-Mail-Body
     */
    private function createDefaultEmailBody($appointment, $eventDate, $eventTime, $companyName) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #007bff; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px; }
                .info { background: #e3f2fd; padding: 15px; border-radius: 4px; margin: 20px 0; }
                .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>📧 Termin-Update</h1>
                </div>
                <div class='content'>
                    <p>Ihr Termin wurde aktualisiert.</p>
                    
                    <div class='info'>
                        <h3>📅 Termindetails:</h3>
                        <ul>
                            <li><strong>Event:</strong> {$appointment['event_name']}</li>
                            <li><strong>Datum:</strong> $eventDate</li>
                            <li><strong>Zeit:</strong> $eventTime</li>
                            <li><strong>Fahrzeug:</strong> {$appointment['license_plate']}</li>
                        </ul>
                    </div>
                    
                    <p>Mit freundlichen Grüßen<br>
                    Ihr $companyName Team</p>
                </div>
                
                <div class='footer'>
                    <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
                </div>
            </div>
        </body>
        </html>";
    }
}
?>
