<?php
/**
 * Intelligente Reifensätze API - Komplett neu entwickelt
 * 
 * Features:
 * - Modulare Architektur
 * - Adaptive Lade-Strategien
 * - Intelligentes Parsing
 * - Smart Caching
 * - Robuste Fehlerbehandlung
 * - Performance-Optimierung
 */

require_once 'cache_manager.php';
require_once 'smart_parser.php';
require_once 'adaptive_loader.php';
require_once 'performance_monitor.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

session_start();
require_once dirname(__DIR__) . '/config/database.php';

/**
 * Hauptklasse für intelligente Reifensätze-Verwaltung
 */
class SmartWheelsetsAPI {
    private $autohausId;
    private $autohausConfig;
    private $performanceMonitor;
    private $adaptiveLoader;
    private $smartParser;
    private $cacheManager;
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $this->performanceMonitor = new PerformanceMonitor();
        $this->adaptiveLoader = new AdaptiveLoader($autohausId);
        // SmartParser wird nicht mehr benötigt, da AdaptiveLoader bereits parst
        // $this->smartParser = new SmartParser();
        
        // Initialisiere Cache nur wenn verfügbar
        try {
            if (isset($GLOBALS['conn']) && $GLOBALS['conn']) {
                $this->cacheManager = new CacheManager($GLOBALS['conn']);
            }
        } catch (Exception $e) {
            error_log("Cache nicht verfügbar: " . $e->getMessage());
        }
        
        $this->loadAutohausConfig();
    }
    
    /**
     * Lädt Autohaus-Konfiguration und bestimmt optimale Strategie
     */
    private function loadAutohausConfig() {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id, name, short_name, 
                    resy_username, resy_password, resy_dealer_id, resy_base_url,
                    COALESCE(last_wheelset_count, 0) as last_wheelset_count, 
                    last_load_time, 
                    COALESCE(performance_tier, 'small') as performance_tier
                FROM autohaus 
                WHERE id = ? AND is_active = 1
            ");
            $stmt->execute([$this->autohausId]);
            $this->autohausConfig = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$this->autohausConfig) {
                throw new Exception('Autohaus nicht gefunden');
            }
            
            // Bestimme Performance-Tier basierend auf historischen Daten
            $this->determinePerformanceTier();
            
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Autohaus-Konfiguration: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Bestimmt die optimale Performance-Strategie basierend auf Autohaus-Größe
     */
    private function determinePerformanceTier() {
        $lastCount = $this->autohausConfig['last_wheelset_count'] ?? 0;
        $lastLoadTime = $this->autohausConfig['last_load_time'] ?? 0;
        
        if ($lastCount > 1000) {
            $this->autohausConfig['performance_tier'] = 'large';
            $this->autohausConfig['max_pages'] = 50;
            $this->autohausConfig['timeout'] = 180;
        } elseif ($lastCount > 500) {
            $this->autohausConfig['performance_tier'] = 'medium';
            $this->autohausConfig['max_pages'] = 30;
            $this->autohausConfig['timeout'] = 120;
        } else {
            $this->autohausConfig['performance_tier'] = 'small';
            $this->autohausConfig['max_pages'] = 20;
            $this->autohausConfig['timeout'] = 60;
        }
    }
    
    /**
     * Hauptmethode zum Laden der Reifensätze
     */
    public function getWheelsets() {
        $this->performanceMonitor->start();
        
        try {
            // 1. Prüfe intelligenten Cache
            $cachedResult = $this->checkSmartCache();
            if ($cachedResult) {
                return $cachedResult;
            }
            
            // 2. Lade mit adaptiver Strategie (gibt bereits geparste Reifensätze zurück)
            $wheelsets = $this->adaptiveLoader->loadWheelsets($this->autohausConfig);
            
            // 3. Reifensätze sind bereits geparst, keine weitere Parsing nötig
            $parsedWheelsets = $wheelsets;
            
            // 4. Validiere und bereinige Daten
            $validatedWheelsets = $this->validateWheelsets($parsedWheelsets);
            
            // 5. Speichere im Cache
            $this->saveToCache($validatedWheelsets);
            
            // 6. Update Performance-Metriken
            $this->updatePerformanceMetrics(count($validatedWheelsets));
            
            $this->performanceMonitor->end();
            
            return [
                'success' => true,
                'wheelsets' => $validatedWheelsets,
                'count' => count($validatedWheelsets),
                'autohaus_id' => $this->autohausId,
                'performance_tier' => $this->autohausConfig['performance_tier'],
                'load_time' => $this->performanceMonitor->getDuration(),
                'cached' => false,
                'cache_time' => 'Neu geladen',
                'strategy' => $this->adaptiveLoader->getLastStrategy()
            ];
            
        } catch (Exception $e) {
            $this->performanceMonitor->end();
            error_log("Smart API Fehler: " . $e->getMessage());
            
            // Intelligenter Fallback
            return $this->handleError($e);
        }
    }
    
    /**
     * Intelligente Cache-Prüfung
     */
    private function checkSmartCache() {
        if (!$this->cacheManager) {
            return null;
        }
        
        // Prüfe verschiedene Cache-Ebenen
        $cacheLevels = [
            'processed' => $this->cacheManager->getCachedWheelsets($this->autohausId),
            'raw_html' => $this->cacheManager->getCachedHtml($this->autohausId),
            'partial' => $this->cacheManager->getPartialCache($this->autohausId)
        ];
        
        foreach ($cacheLevels as $level => $data) {
            if ($data !== null) {
                return [
                    'success' => true,
                    'wheelsets' => $data,
                    'count' => count($data),
                    'cached' => true,
                    'cache_level' => $level,
                    'cache_time' => 'Aus ' . $level . ' Cache'
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Validiert und bereinigt Reifensätze
     */
    private function validateWheelsets($wheelsets) {
        $validated = [];
        $duplicates = [];
        
        foreach ($wheelsets as $wheelset) {
            // Validiere erforderliche Felder
            if (empty($wheelset['satznummer']) || empty($wheelset['kennzeichen'])) {
                continue;
            }
            
            // Prüfe auf Duplikate
            $key = $wheelset['satznummer'] . '_' . $wheelset['kennzeichen'];
            if (isset($duplicates[$key])) {
                continue;
            }
            $duplicates[$key] = true;
            
            // Bereinige Daten
            $wheelset['satznummer'] = trim($wheelset['satznummer']);
            $wheelset['kennzeichen'] = trim($wheelset['kennzeichen']);
            $wheelset['halter'] = trim($wheelset['halter'] ?? 'Unbekannt');
            $wheelset['fahrzeug'] = trim($wheelset['fahrzeug'] ?? '');
            $wheelset['eingangsdatum'] = trim($wheelset['eingangsdatum'] ?? '');
            $wheelset['status'] = 'eingelagert';
            
            $validated[] = $wheelset;
        }
        
        return $validated;
    }
    
    /**
     * Speichert Daten im intelligenten Cache
     */
    private function saveToCache($wheelsets) {
        if (!$this->cacheManager) {
            return;
        }
        
        // Speichere in verschiedenen Cache-Ebenen
        $this->cacheManager->setCachedWheelsets($this->autohausId, 20, $wheelsets);
        
        // Speichere auch als Partial-Cache für große Autohäuser
        if (count($wheelsets) > 500) {
            $this->cacheManager->setPartialCache($this->autohausId, $wheelsets);
        }
    }
    
    /**
     * Aktualisiert Performance-Metriken
     */
    private function updatePerformanceMetrics($count) {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            // Prüfe ob die neuen Spalten existieren
            $stmt = $conn->prepare("SHOW COLUMNS FROM autohaus LIKE 'last_wheelset_count'");
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                // Neue Spalten existieren
                $stmt = $conn->prepare("
                    UPDATE autohaus 
                    SET last_wheelset_count = ?, last_load_time = NOW(), performance_tier = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $count,
                    $this->autohausConfig['performance_tier'],
                    $this->autohausId
                ]);
            } else {
                // Fallback für alte Datenbank-Struktur
                error_log("Performance-Spalten nicht verfügbar, überspringe Update");
            }
        } catch (Exception $e) {
            error_log("Fehler beim Update der Performance-Metriken: " . $e->getMessage());
        }
    }
    
    /**
     * Intelligente Fehlerbehandlung
     */
    private function handleError($exception) {
        $errorType = $this->classifyError($exception);
        
        switch ($errorType) {
            case 'timeout':
                return $this->handleTimeoutError();
            case 'network':
                return $this->handleNetworkError();
            case 'parsing':
                return $this->handleParsingError();
            default:
                return $this->handleGenericError($exception);
        }
    }
    
    /**
     * Klassifiziert Fehlertypen
     */
    private function classifyError($exception) {
        $message = $exception->getMessage();
        
        if (strpos($message, 'timeout') !== false || strpos($message, 'Maximum execution time') !== false) {
            return 'timeout';
        }
        if (strpos($message, 'curl') !== false || strpos($message, 'network') !== false) {
            return 'network';
        }
        if (strpos($message, 'parse') !== false || strpos($message, 'html') !== false) {
            return 'parsing';
        }
        
        return 'generic';
    }
    
    /**
     * Behandelt Timeout-Fehler
     */
    private function handleTimeoutError() {
        // Versuche mit reduzierter Seitenanzahl
        $reducedConfig = $this->autohausConfig;
        $reducedConfig['max_pages'] = min(10, $this->autohausConfig['max_pages']);
        
        try {
            $wheelsets = $this->adaptiveLoader->loadWheelsets($reducedConfig);
            $parsedWheelsets = $this->smartParser->parseWheelsets($wheelsets);
            $validatedWheelsets = $this->validateWheelsets($parsedWheelsets);
            
            return [
                'success' => true,
                'wheelsets' => $validatedWheelsets,
                'count' => count($validatedWheelsets),
                'warning' => 'Nur teilweise geladen (Timeout vermieden)',
                'cached' => false
            ];
        } catch (Exception $e) {
            return $this->getFallbackData();
        }
    }
    
    /**
     * Behandelt Netzwerk-Fehler
     */
    private function handleNetworkError() {
        // Versuche mit Retry-Logik
        for ($i = 0; $i < 3; $i++) {
            try {
                sleep($i * 2); // Exponential backoff
                $wheelsets = $this->adaptiveLoader->loadWheelsets($this->autohausConfig);
                $parsedWheelsets = $this->smartParser->parseWheelsets($wheelsets);
                $validatedWheelsets = $this->validateWheelsets($parsedWheelsets);
                
                return [
                    'success' => true,
                    'wheelsets' => $validatedWheelsets,
                    'count' => count($validatedWheelsets),
                    'info' => 'Nach ' . ($i + 1) . ' Versuchen erfolgreich',
                    'cached' => false
                ];
            } catch (Exception $e) {
                if ($i === 2) {
                    return $this->getFallbackData();
                }
            }
        }
    }
    
    /**
     * Behandelt Parsing-Fehler
     */
    private function handleParsingError() {
        // Versuche mit alternativen Parsing-Methoden
        try {
            $wheelsets = $this->adaptiveLoader->loadWheelsets($this->autohausConfig);
            $parsedWheelsets = $wheelsets; // Bereits geparst
            $validatedWheelsets = $this->validateWheelsets($parsedWheelsets);
            
            return [
                'success' => true,
                'wheelsets' => $validatedWheelsets,
                'count' => count($validatedWheelsets),
                'info' => 'Mit alternativer Parsing-Methode geladen',
                'cached' => false
            ];
        } catch (Exception $e) {
            return $this->getFallbackData();
        }
    }
    
    /**
     * Behandelt generische Fehler
     */
    private function handleGenericError($exception) {
        error_log("Generischer Fehler: " . $exception->getMessage());
        return $this->getFallbackData();
    }
    
    /**
     * Liefert Fallback-Daten
     */
    private function getFallbackData() {
        try {
            // Versuche lokale Datenbank
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id,
                    CONCAT(YEAR(created_at), '.', LPAD(MONTH(created_at), 2, '0'), '.', LPAD(DAY(created_at), 2, '0')) as satznummer,
                    kennzeichen,
                    fahrzeug_id as fahrzeug,
                    kunde as halter,
                    DATE_FORMAT(created_at, '%d.%m.%Y') as eingangsdatum,
                    'eingelagert' as status,
                    NULL as prufdoc_id
                FROM appointments 
                WHERE autohaus_id = ? 
                AND status IN ('confirmed', 'completed')
                ORDER BY created_at DESC
                LIMIT 50
            ");
            $stmt->execute([$this->autohausId]);
            $wheelsets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($wheelsets)) {
                return [
                    'success' => true,
                    'wheelsets' => $wheelsets,
                    'count' => count($wheelsets),
                    'info' => 'Fallback: Lokale Datenbank',
                    'cached' => false
                ];
            }
        } catch (Exception $e) {
            error_log("Fallback fehlgeschlagen: " . $e->getMessage());
        }
        
        // Letzter Fallback: Mock-Daten
        return [
            'success' => true,
            'wheelsets' => $this->getMockWheelsets(),
            'count' => 3,
            'warning' => 'Demo-Daten (System nicht erreichbar)',
            'cached' => false
        ];
    }
    
    /**
     * Generiert Mock-Daten
     */
    private function getMockWheelsets() {
        return [
            [
                'satznummer' => '2025.001.001',
                'kennzeichen' => 'KI-AB 123',
                'fahrzeug' => 'VIN123456789',
                'halter' => 'Max Mustermann GmbH',
                'eingangsdatum' => date('d.m.Y'),
                'status' => 'eingelagert',
                'prufdoc_id' => '12345'
            ],
            [
                'satznummer' => '2025.001.002',
                'kennzeichen' => 'KI-CD 456',
                'fahrzeug' => 'VIN987654321',
                'halter' => 'Anna Schmidt',
                'eingangsdatum' => date('d.m.Y', strtotime('-1 day')),
                'status' => 'eingelagert',
                'prufdoc_id' => '12346'
            ],
            [
                'satznummer' => '2025.001.003',
                'kennzeichen' => 'KI-EF 789',
                'fahrzeug' => 'VIN456789123',
                'halter' => 'Peter Müller & Co. KG',
                'eingangsdatum' => date('d.m.Y', strtotime('-2 days')),
                'status' => 'eingelagert',
                'prufdoc_id' => '12347'
            ]
        ];
    }
}

// API-Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['autohaus_id'])) {
        echo json_encode([
            'success' => false,
            'error' => 'Autohaus-ID ist erforderlich'
        ]);
        exit;
    }
    
    $autohausId = $input['autohaus_id'];
    
    try {
        $api = new SmartWheelsetsAPI($autohausId);
        $result = $api->getWheelsets();
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'API-Fehler: ' . $e->getMessage()
        ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'error' => 'Nur POST-Requests erlaubt'
    ]);
}
?>
