<?php
/**
 * Sendet WhatsApp-Bestätigung für einen Termin
 * Wird asynchron nach Buchung aufgerufen
 */

// Ermittle das Root-Verzeichnis des Projekts (unabhängig vom Ausführungsverzeichnis)
$rootDir = dirname(__DIR__);

// Lade benötigte Dateien mit absolutem Pfad
require_once $rootDir . '/config/database.php';
require_once $rootDir . '/api/autohaus_whatsapp_notifications.php';

header('Content-Type: application/json');

// Nur POST erlauben
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $appointmentId = $input['appointment_id'] ?? $_POST['appointment_id'] ?? null;
    
    if (!$appointmentId) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'appointment_id erforderlich']);
        exit;
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Hole Autohaus-ID für Termin
    $stmt = $conn->prepare("SELECT autohaus_id FROM appointments WHERE id = ?");
    $stmt->execute([$appointmentId]);
    $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$appointment) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Termin nicht gefunden']);
        exit;
    }
    
    $autohausId = $appointment['autohaus_id'];
    
    // Prüfe ob Bestätigung bereits gesendet wurde
    $stmt = $conn->prepare("SELECT whatsapp_confirmation_sent FROM appointments WHERE id = ?");
    $stmt->execute([$appointmentId]);
    $check = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($check && $check['whatsapp_confirmation_sent'] == 1) {
        echo json_encode(['success' => true, 'message' => 'Bestätigung wurde bereits gesendet']);
        exit;
    }
    
    // Sende WhatsApp-Bestätigung
    $whatsapp = new AutohausWhatsAppNotifications($autohausId);
    $result = $whatsapp->sendAppointmentNotification($appointmentId, 'confirmation');
    
    if ($result['success']) {
        // Markiere Bestätigung als gesendet
        $updateStmt = $conn->prepare("
            UPDATE appointments 
            SET whatsapp_confirmation_sent = 1, whatsapp_confirmation_sent_at = NOW() 
            WHERE id = ?
        ");
        $updateStmt->execute([$appointmentId]);
        
        echo json_encode([
            'success' => true,
            'message' => 'WhatsApp-Bestätigung erfolgreich gesendet',
            'appointment_id' => $appointmentId
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $result['error'] ?? 'Fehler beim Senden der Bestätigung'
        ]);
    }
    
} catch (Exception $e) {
    error_log("WhatsApp-Bestätigung Fehler: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Interner Fehler: ' . $e->getMessage()
    ]);
}

