<?php
/**
 * RESY Data Extraction API - URL Test Version
 * Testet verschiedene URL-Strukturen nach erfolgreichem Login
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true,
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies.txt'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    
    // Cookie-Verwaltung
    curl_setopt($ch, CURLOPT_COOKIEJAR, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIESESSION, false);
    curl_setopt($ch, CURLOPT_FRESH_CONNECT, false);
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// RESY Login
function performResyLogin($baseUrl, $username, $password) {
    debugLog("=== URL TEST LOGIN START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    // Cookie-Datei löschen
    $cookieFile = sys_get_temp_dir() . '/resy_cookies.txt';
    if (file_exists($cookieFile)) {
        unlink($cookieFile);
    }
    
    // Login-Seite laden
    $loginUrl = "$baseUrl/index.php?m=login&a=login";
    debugLog("Lade Login-Seite: $loginUrl");
    
    $response = httpRequest($loginUrl, [
        'cookie_file' => $cookieFile,
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Connection: keep-alive'
        ]
    ]);
    
    if ($response['http_code'] !== 200) {
        throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Login-Seite erfolgreich geladen");
    
    // Login-Daten senden
    $postData = [
        'FN' => 'login',
        'UserName' => $username,
        'Password' => $password,
        'button' => 'submit'
    ];
    
    $loginDataString = http_build_query($postData);
    debugLog("Login-Daten: " . str_replace($password, '***', $loginDataString));
    
    $loginResponse = httpRequest($loginUrl, [
        'method' => 'POST',
        'data' => $loginDataString,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Connection: keep-alive',
            'Referer: ' . $loginUrl
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Login-Response Code: " . $loginResponse['http_code']);
    
    // Login-Erfolg prüfen
    $responseBody = $loginResponse['body'];
    
    if (strpos($responseBody, 'Anmeldename') === false && 
        strpos($responseBody, 'Kennwort') === false &&
        strpos($responseBody, 'Anmeldung') === false) {
        debugLog("✅ Login erfolgreich!");
        debugLog("=== URL TEST LOGIN END ===");
        return $loginResponse;
    }
    
    throw new Exception("Login fehlgeschlagen");
}

// Teste verschiedene URL-Strukturen
function testUrlStructures($baseUrl, $lgsId) {
    debugLog("=== URL STRUCTURE TEST START ===");
    debugLog("LGS_ID: $lgsId");
    
    $urls = [
        // Standard URL-Struktur
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult",
        
        // Ohne FN Parameter
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId",
        
        // Mit verschiedenen FN Werten
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetEdit",
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetView",
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetDetails",
        
        // Mit verschiedenen a Werten
        "$baseUrl/index.php?m=wheelset&a=view&LGS_ID=$lgsId&FN=WheelsetResult",
        "$baseUrl/index.php?m=wheelset&a=details&LGS_ID=$lgsId&FN=WheelsetResult",
        
        // Mit verschiedenen m Werten
        "$baseUrl/index.php?m=wheel&a=edit&LGS_ID=$lgsId&FN=WheelsetResult",
        "$baseUrl/index.php?m=wheel_set&a=edit&LGS_ID=$lgsId&FN=WheelsetResult",
        
        // Mit verschiedenen Parameter-Reihenfolgen
        "$baseUrl/index.php?LGS_ID=$lgsId&m=wheelset&a=edit&FN=WheelsetResult",
        "$baseUrl/index.php?FN=WheelsetResult&LGS_ID=$lgsId&m=wheelset&a=edit",
        
        // Mit verschiedenen LGS_ID Formaten
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=" . str_replace('.', '', $lgsId) . "&FN=WheelsetResult",
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=" . str_replace('.', '', $lgsId) . "&FN=WheelsetResult",
        
        // Mit verschiedenen Pfaden
        "$baseUrl/wheelset/edit.php?LGS_ID=$lgsId&FN=WheelsetResult",
        "$baseUrl/wheelset/view.php?LGS_ID=$lgsId&FN=WheelsetResult",
        
        // Mit verschiedenen Endungen
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult&format=html",
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult&output=html"
    ];
    
    $results = [];
    
    foreach ($urls as $index => $url) {
        debugLog("Teste URL " . ($index + 1) . ": $url");
        
        try {
            $response = httpRequest($url, [
                'headers' => [
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                    'Referer: ' . $baseUrl . '/index.php?m=login&a=login'
                ],
                'cookie_file' => sys_get_temp_dir() . '/resy_cookies.txt'
            ]);
            
            $result = [
                'url' => $url,
                'http_code' => $response['http_code'],
                'length' => strlen($response['body']),
                'has_login_form' => strpos($response['body'], 'Anmeldename') !== false,
                'has_detail_data' => strpos($response['body'], 'WheelHALTERNAME') !== false,
                'has_wheel_data' => strpos($response['body'], 'Wheel') !== false,
                'has_error' => strpos($response['body'], 'Fehler') !== false || strpos($response['body'], 'Error') !== false,
                'preview' => substr($response['body'], 0, 200)
            ];
            
            $results[] = $result;
            
            debugLog("URL " . ($index + 1) . " - Code: " . $result['http_code'] . ", Length: " . $result['length'] . 
                    ", Login: " . ($result['has_login_form'] ? 'Ja' : 'Nein') . 
                    ", Detail: " . ($result['has_detail_data'] ? 'Ja' : 'Nein') . 
                    ", Wheel: " . ($result['has_wheel_data'] ? 'Ja' : 'Nein'));
            
            // Wenn Detail-Daten gefunden wurden, speichere die Antwort
            if ($result['has_detail_data']) {
                debugLog("✅ Detail-Daten gefunden in URL " . ($index + 1) . "!");
                file_put_contents(sys_get_temp_dir() . '/resy_detail_success.html', $response['body']);
                debugLog("=== URL STRUCTURE TEST END ===");
                return $response;
            }
            
        } catch (Exception $e) {
            debugLog("URL " . ($index + 1) . " Fehler: " . $e->getMessage());
            $results[] = [
                'url' => $url,
                'error' => $e->getMessage()
            ];
        }
    }
    
    debugLog("=== URL STRUCTURE TEST END ===");
    return $results;
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID',
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        return '';
    }
    
    $value = trim($element->textContent);
    
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
        }
    }
    
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
        }
    }
    
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== URL TEST RESY API Request Started ===");
    
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'url_test':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("URL Test für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            // Schritt 1: Login durchführen
            performResyLogin($autohaus['resy_base_url'], $autohaus['resy_username'], $autohaus['resy_password']);
            
            // Schritt 2: Verschiedene URL-Strukturen testen
            $urlTestResult = testUrlStructures($autohaus['resy_base_url'], $lgsId);
            
            if (is_array($urlTestResult)) {
                // Alle URLs getestet, aber keine funktioniert
                $result = [
                    'success' => false,
                    'error' => 'Keine funktionierende URL-Struktur gefunden',
                    'url_tests' => $urlTestResult,
                    'message' => 'Alle URL-Strukturen getestet, aber keine liefert Detail-Daten'
                ];
            } else {
                // Erfolgreiche URL gefunden
                $result = [
                    'success' => true,
                    'detailHtml' => $urlTestResult['body'],
                    'url' => $urlTestResult['url'],
                    'approach' => 'url_test',
                    'message' => 'Funktionierende URL-Struktur gefunden',
                    'lgs_id' => $lgsId
                ];
            }
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== URL TEST RESY API Request Ended ===");
?>
