<?php
/**
 * RESY Data Extraction API - Login Bypass Version
 * Versucht verschiedene Wege, um an die RESY-Daten zu kommen
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion mit erweiterten Optionen
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'cookies' => [],
        'follow_redirects' => true,
        'timeout' => 30,
        'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, $options['user_agent']);
    curl_setopt($ch, CURLOPT_COOKIEJAR, tempnam(sys_get_temp_dir(), 'resy_cookies'));
    curl_setopt($ch, CURLOPT_COOKIEFILE, tempnam(sys_get_temp_dir(), 'resy_cookies'));
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    if (!empty($options['cookies'])) {
        $cookieString = '';
        foreach ($options['cookies'] as $name => $value) {
            $cookieString .= $name . '=' . $value . '; ';
        }
        curl_setopt($ch, CURLOPT_COOKIE, rtrim($cookieString, '; '));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// Versuche verschiedene Wege, um an die RESY-Daten zu kommen
function tryMultipleApproaches($lgsId, $baseUrl) {
    debugLog("=== MULTIPLE APPROACHES START für LGS_ID: $lgsId ===");
    
    $approaches = [
        'direct_detail' => "Direkte Detailseite ohne Login",
        'search_page' => "Suchseite ohne Login",
        'public_api' => "Öffentliche API-Endpunkte",
        'alternative_urls' => "Alternative URL-Strukturen"
    ];
    
    foreach ($approaches as $approach => $description) {
        debugLog("Versuche Ansatz: $description");
        
        try {
            $result = tryApproach($approach, $lgsId, $baseUrl);
            if ($result['success']) {
                debugLog("Ansatz '$approach' erfolgreich!");
                return $result;
            }
        } catch (Exception $e) {
            debugLog("Ansatz '$approach' fehlgeschlagen: " . $e->getMessage());
        }
    }
    
    debugLog("Alle Ansätze fehlgeschlagen");
    return [
        'success' => false,
        'error' => 'Alle Ansätze fehlgeschlagen'
    ];
}

function tryApproach($approach, $lgsId, $baseUrl) {
    switch ($approach) {
        case 'direct_detail':
            return tryDirectDetail($lgsId, $baseUrl);
        case 'search_page':
            return trySearchPage($lgsId, $baseUrl);
        case 'public_api':
            return tryPublicApi($lgsId, $baseUrl);
        case 'alternative_urls':
            return tryAlternativeUrls($lgsId, $baseUrl);
        default:
            throw new Exception("Unbekannter Ansatz: $approach");
    }
}

function tryDirectDetail($lgsId, $baseUrl) {
    debugLog("Versuche direkte Detailseite...");
    
    $urls = [
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId&FN=WheelsetResult",
        "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$lgsId",
        "$baseUrl/wheelset/edit/$lgsId",
        "$baseUrl/api/wheelset/$lgsId"
    ];
    
    foreach ($urls as $url) {
        debugLog("Teste URL: $url");
        try {
            $response = httpRequest($url);
            debugLog("Response Code: " . $response['http_code']);
            
            if ($response['http_code'] === 200 && strpos($response['body'], 'WheelHALTERNAME') !== false) {
                debugLog("Detailseite erfolgreich geladen!");
                return [
                    'success' => true,
                    'detailHtml' => $response['body'],
                    'url' => $url,
                    'approach' => 'direct_detail'
                ];
            }
        } catch (Exception $e) {
            debugLog("URL fehlgeschlagen: " . $e->getMessage());
        }
    }
    
    throw new Exception("Direkte Detailseite nicht erreichbar");
}

function trySearchPage($lgsId, $baseUrl) {
    debugLog("Versuche Suchseite...");
    
    $searchUrl = "$baseUrl/index.php?m=wheelset&a=search";
    $response = httpRequest($searchUrl);
    
    if ($response['http_code'] === 200) {
        debugLog("Suchseite erfolgreich geladen");
        
        // Versuche die Suche zu simulieren
        $dom = new DOMDocument();
        @$dom->loadHTML($response['body']);
        $xpath = new DOMXPath($dom);
        
        // Suche nach dem Suchformular
        $form = $xpath->query('//form')->item(0);
        if ($form) {
            debugLog("Suchformular gefunden, simuliere Suche...");
            
            // Simuliere Suche mit LGS_ID
            $searchAction = $baseUrl . '/index.php?m=wheelset&a=search';
            $searchData = http_build_query([
                'LGS_ID' => $lgsId,
                'button' => 'submit'
            ]);
            
            $searchResponse = httpRequest($searchAction, [
                'method' => 'POST',
                'data' => $searchData,
                'headers' => [
                    'Content-Type: application/x-www-form-urlencoded',
                    'Referer: ' . $searchUrl
                ]
            ]);
            
            debugLog("Suche abgeschlossen, Response Code: " . $searchResponse['http_code']);
            
            if ($searchResponse['http_code'] === 200) {
                // Prüfe ob wir auf eine Detailseite weitergeleitet wurden
                if (strpos($searchResponse['body'], 'WheelHALTERNAME') !== false) {
                    debugLog("Detailseite nach Suche gefunden!");
                    return [
                        'success' => true,
                        'detailHtml' => $searchResponse['body'],
                        'url' => $searchAction,
                        'approach' => 'search_page'
                    ];
                }
                
                // Prüfe ob es Suchergebnisse gibt
                if (strpos($searchResponse['body'], 'Einlagerungsnummer') !== false) {
                    debugLog("Suchergebnisse gefunden, versuche Detail-Link zu finden...");
                    
                    // Suche nach Links zu Detailseiten
                    $links = $xpath->query('//a[contains(@href, "LGS_ID")]');
                    foreach ($links as $link) {
                        $href = $link->getAttribute('href');
                        if (strpos($href, $lgsId) !== false) {
                            debugLog("Detail-Link gefunden: $href");
                            
                            $detailUrl = $baseUrl . '/' . $href;
                            $detailResponse = httpRequest($detailUrl);
                            
                            if ($detailResponse['http_code'] === 200 && strpos($detailResponse['body'], 'WheelHALTERNAME') !== false) {
                                debugLog("Detailseite erfolgreich geladen!");
                                return [
                                    'success' => true,
                                    'detailHtml' => $detailResponse['body'],
                                    'url' => $detailUrl,
                                    'approach' => 'search_page'
                                ];
                            }
                        }
                    }
                }
            }
        }
        
        return [
            'success' => true,
            'searchHtml' => $response['body'],
            'approach' => 'search_page'
        ];
    }
    
    throw new Exception("Suchseite nicht erreichbar");
}

function tryPublicApi($lgsId, $baseUrl) {
    debugLog("Versuche öffentliche API...");
    
    $apiUrls = [
        "$baseUrl/api/wheelset/$lgsId",
        "$baseUrl/api/data/$lgsId",
        "$baseUrl/json/wheelset/$lgsId"
    ];
    
    foreach ($apiUrls as $url) {
        debugLog("Teste API URL: $url");
        try {
            $response = httpRequest($url, [
                'headers' => ['Accept: application/json']
            ]);
            
            if ($response['http_code'] === 200) {
                $data = json_decode($response['body'], true);
                if ($data) {
                    debugLog("API erfolgreich!");
                    return [
                        'success' => true,
                        'data' => $data,
                        'url' => $url,
                        'approach' => 'public_api'
                    ];
                }
            }
        } catch (Exception $e) {
            debugLog("API URL fehlgeschlagen: " . $e->getMessage());
        }
    }
    
    throw new Exception("Keine öffentliche API gefunden");
}

function tryAlternativeUrls($lgsId, $baseUrl) {
    debugLog("Versuche alternative URLs...");
    
    $alternativeUrls = [
        "$baseUrl/wheelset/$lgsId",
        "$baseUrl/detail/$lgsId",
        "$baseUrl/view/$lgsId",
        "$baseUrl/public/wheelset/$lgsId"
    ];
    
    foreach ($alternativeUrls as $url) {
        debugLog("Teste alternative URL: $url");
        try {
            $response = httpRequest($url);
            debugLog("Response Code: " . $response['http_code']);
            
            if ($response['http_code'] === 200 && strpos($response['body'], 'WheelHALTERNAME') !== false) {
                debugLog("Alternative URL erfolgreich!");
                return [
                    'success' => true,
                    'detailHtml' => $response['body'],
                    'url' => $url,
                    'approach' => 'alternative_urls'
                ];
            }
        } catch (Exception $e) {
            debugLog("Alternative URL fehlgeschlagen: " . $e->getMessage());
        }
    }
    
    throw new Exception("Keine alternative URLs gefunden");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    debugLog("=== HTML-ANALYSE FÜR FELD: $field ===");
    
    // Mapping der Feldnamen zu HTML-IDs
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID', // Autohaus ist die Ladestelle
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    debugLog("Suche nach Element mit ID: $htmlId");
    
    // Suche nach dem Element mit der entsprechenden ID
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        debugLog("Element mit ID '$htmlId' nicht gefunden für Feld '$field'");
        return '';
    }
    
    // Extrahiere den Textinhalt
    $value = trim($element->textContent);
    
    // Für Bemerkung-Felder: Suche nach textarea
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
            debugLog("Bemerkung aus textarea extrahiert: '$value'");
        }
    }
    
    debugLog("Feld '$field' (ID: $htmlId) extrahiert: '$value'");
    
    // Wenn Wert leer ist, suche nach <b> Tags innerhalb des Elements
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
            debugLog("Wert aus <b> Tag extrahiert: '$value'");
        }
    }
    
    debugLog("=== ENDE HTML-ANALYSE FÜR FELD: $field ===");
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== BYPASS RESY API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'navigate_bypass':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Bypass Navigation für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            $result = tryMultipleApproaches($lgsId, $autohaus['resy_base_url']);
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== BYPASS RESY API Request Ended ===");
?>
