<?php
// login_user.php - Benutzeranmeldung
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

require_once 'config.php';

// Helper Funktionen
function getDeviceInfo() {
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    if (strpos($user_agent, 'Mobile') !== false) {
        if (preg_match('/iPhone|iPad|iPod/i', $user_agent)) return 'iOS';
        if (preg_match('/Android/i', $user_agent)) return 'Android';
        return 'Mobile';
    }
    if (preg_match('/Mac|Macintosh/i', $user_agent)) return 'Mac';
    if (preg_match('/Windows/i', $user_agent)) return 'Windows';
    if (preg_match('/Linux/i', $user_agent)) return 'Linux';
    return 'Unknown';
}

function getClientIP() {
    $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP) !== false) {
                    return $ip;
                }
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
}

// Helper: Dekodiere Apple JWT Token
function decodeAppleToken($id_token) {
    if (empty($id_token)) {
        return null;
    }
    
    $parts = explode('.', $id_token);
    if (count($parts) !== 3) {
        return null;
    }
    
    $payload = $parts[1];
    $payload = str_replace(['-', '_'], ['+', '/'], $payload);
    
    $remainder = strlen($payload) % 4;
    if ($remainder) {
        $padlen = 4 - $remainder;
        $payload .= str_repeat('=', $padlen);
    }
    
    $decoded = base64_decode($payload);
    
    if ($decoded === false) {
        return null;
    }
    
    $data = json_decode($decoded, true);
    
    return $data;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Debug-Logging für Apple Sign In und Google Sign In
    error_log('Login User Input: ' . print_r($input, true));
    
    // Prüfen ob es sich um Apple Sign In oder Google Sign In handelt
    $isAppleSignIn = !empty($input['apple_user_id']);
    $isGoogleSignIn = !empty($input['google_user_id']);
    
    // WICHTIG: Bei Apple Sign In extrahiere die echte User ID aus dem Token
    $realAppleUserId = null;
    if ($isAppleSignIn && !empty($input['apple_identity_token'])) {
        $tokenData = decodeAppleToken($input['apple_identity_token']);
        if ($tokenData && isset($tokenData['sub'])) {
            $realAppleUserId = $tokenData['sub'];
            // Auch Email aus Token extrahieren falls vorhanden
            if (empty($input['email']) && isset($tokenData['email'])) {
                $input['email'] = $tokenData['email'];
                error_log('Extracted email from token: ' . $tokenData['email']);
            }
            error_log('Real Apple User ID (sub): ' . $realAppleUserId);
        }
    }
    
    if ($isAppleSignIn || $isGoogleSignIn) {
        // Für Apple Sign In und Google Sign In ist Email optional beim ersten Login
        // Apple liefert die Email nur beim ersten Mal
    } else {
        // Für normale Anmeldung ist E-Mail erforderlich
        if (empty($input['email'])) {
            throw new Exception('Email is required');
        }
    }
    
    // Apple Sign In Login
    if ($isAppleSignIn) {
        error_log('Apple Sign In - Starting Login Flow');
        
        // WICHTIG: Benutze die echte Apple User ID (sub) aus dem Token für Prüfung
        $searchAppleUserId = $realAppleUserId ?? $input['apple_user_id'];
        
        error_log('Searching for Apple User ID: ' . $searchAppleUserId);
        
        // Prüfe nach apple_user_id
        $stmt = $pdo->prepare("SELECT * FROM users WHERE apple_user_id = ? AND is_active = 1");
        $stmt->execute([$searchAppleUserId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        error_log('Apple Sign In - User by apple_user_id: ' . ($user ? 'Found' : 'Not Found'));
        
        if ($user) {
            error_log('Apple Sign In - Existing user found: ID ' . $user['id']);
            // Bestehender User - Login
        } else {
            // Neuer User - Registrierung
            error_log('Apple Sign In - Creating new user');
            
            // Wenn Email vorhanden, prüfe ob E-Mail bereits registriert ist
            if (!empty($input['email'])) {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND is_active = 1");
                $stmt->execute([$input['email']]);
                $existingUser = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($existingUser) {
                    error_log('Apple Sign In - Email exists: ' . $input['email']);
                    if (!$existingUser['apple_user_id']) {
                        error_log('Apple Sign In - Linking Apple ID to existing email account');
                        // E-Mail existiert bereits ohne Apple - füge Apple-ID hinzu
                        $stmt = $pdo->prepare("UPDATE users SET apple_user_id = ? WHERE id = ?");
                        $stmt->execute([$searchAppleUserId, $existingUser['id']]);
                        $user = $existingUser;
                        $user['apple_user_id'] = $searchAppleUserId;
                    } else {
                        error_log('Apple Sign In - Email already linked to different Apple ID');
                        throw new Exception('Diese E-Mail-Adresse ist bereits mit einem anderen Apple-Account verknüpft.');
                    }
                }
            }
            
            // Wenn immer noch kein User, erstelle neuen
            if (!$user) {
                error_log('Apple Sign In - Creating completely new user');
                error_log('Apple Sign In - Input data: ' . print_r($input, true));
                
                // WICHTIG: Bei Apple Sign In sollte Datenschutz bereits client-seitig geprüft werden
                // Hier zusätzliche Backend-Prüfung:
                if (!isset($input['datenschutz_accepted']) || $input['datenschutz_accepted'] !== true) {
                    error_log('Apple Sign In - Datenschutz nicht akzeptiert');
                    throw new Exception('Datenschutzerklärung muss akzeptiert werden');
                }
                
                // Neuer Apple-Benutzer - automatische Registrierung
                // WICHTIG: Apple liefert first_name/last_name nur beim ersten Login
                // Bei nachfolgenden Logins sind diese null
                $first_name = $input['first_name'] ?? null;
                $last_name = $input['last_name'] ?? null;
                
                error_log('Apple Sign In - Name data from input - first: ' . var_export($input['first_name'] ?? 'NULL', true) . ', last: ' . var_export($input['last_name'] ?? 'NULL', true));
                
                // Wenn keine Namen vorhanden, versuche sie aus der Email zu extrahieren
                if (empty($first_name) && !empty($email)) {
                    // Versuche Namen aus Email zu extrahieren (z.B. denizuludag439@gmail.com -> Deniz Uludag)
                    $emailParts = explode('@', $email);
                    $emailLocal = $emailParts[0];
                    
                    // Versuche Vorname.Nachname Format
                    if (strpos($emailLocal, '.') !== false) {
                        $parts = explode('.', $emailLocal);
                        if (count($parts) >= 2) {
                            $first_name = ucfirst($parts[0]);
                            $last_name = ucfirst($parts[1]);
                            error_log('⚠️ Extracted name from email: ' . $first_name . ' ' . $last_name);
                        }
                    } else {
                        // Fallback: Erste 6 Zeichen als Vorname
                        $first_name = ucfirst(substr($emailLocal, 0, 6));
                        $last_name = 'User';
                    }
                }
                
                // Wenn immer noch keine Namen, verwende Platzhalter
                if (empty($first_name)) {
                    $first_name = 'Vorname';
                    error_log('⚠️ No first_name from Apple - using placeholder');
                }
                if (empty($last_name)) {
                    $last_name = 'Nachname';
                    error_log('⚠️ No last_name from Apple - using placeholder');
                }
                
                // Generiere eindeutige Email für Apple User ohne Email
                $baseEmail = $input['email'];
                if (empty($baseEmail)) {
                    // Erstelle eindeutige Email basierend auf Apple User ID
                    $emailSuffix = substr(md5($input['apple_user_id']), 0, 8);
                    $email = 'apple_' . $emailSuffix . '@apple.local';
                    
                    // Prüfe ob Email bereits existiert (sollte nicht passieren mit hash)
                    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
                    $stmt->execute([$email]);
                    if ($stmt->fetch()) {
                        // Wenn doch existiert, füge Timestamp hinzu
                        $email = 'apple_' . $emailSuffix . '_' . time() . '@apple.local';
                    }
                } else {
                    $email = $baseEmail;
                }
                
                // Ignoriere Telefonnummer von Apple (wird nicht geliefert)
                // Wenn firstName/lastName null sind, behalte die Fallback-Werte
                if ($first_name !== 'Apple' && $last_name !== 'User') {
                    error_log('Apple Sign In - Creating user with real name: ' . $first_name . ' ' . $last_name . ' (' . $email . ')');
                } else {
                    error_log('Apple Sign In - Creating user with defaults: ' . $first_name . ' ' . $last_name . ' (' . $email . ')');
                }
                
                $auth_token = bin2hex(random_bytes(32));
                $token_expires = date('Y-m-d H:i:s', strtotime('+30 days'));
                
                // Registrierungsquelle und Device-Info
                $registration_source = 'apple_signin';
                $registration_device = $input['registration_device'] ?? getDeviceInfo();
                $registration_ip = $input['registration_ip'] ?? getClientIP();
                
                $sql = "INSERT INTO users (
                    first_name, last_name, email, phone_number, license_plate,
                    preferred_dealership_id, is_leasing_vehicle, is_contract_vehicle,
                    apple_user_id, auth_token, token_expires_at, email_verified, password_hash, 
                    verification_status, registration_source, registration_device, registration_ip
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $stmt = $pdo->prepare($sql);
                $result = $stmt->execute([
                    $first_name,
                    $last_name,
                    $email,
                    $input['phone_number'] ?? '',
                    $input['license_plate'] ?? '',
                    null,
                    false,
                    false,
                    $searchAppleUserId,
                    $auth_token,
                    $token_expires,
                    true,
                    null, // Kein Passwort für Apple-Benutzer
                    'pending', // Verifikation erforderlich
                    $registration_source,
                    $registration_device,
                    $registration_ip
                ]);
                
                if ($result) {
                    $user_id = $pdo->lastInsertId();
                    error_log('Apple Sign In - New user created with ID: ' . $user_id);
                    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                } else {
                    error_log('Apple Sign In - Failed to create user');
                }
            }
        }
        
        if ($user) {
            error_log('Apple Sign In - Final user: ID ' . $user['id'] . ', Email: ' . $user['email']);
            // Token aktualisieren und Login-Quelle speichern
            $auth_token = bin2hex(random_bytes(32));
            $token_expires = date('Y-m-d H:i:s', strtotime('+30 days'));
            
            $login_source = 'apple_signin';
            $login_device = getDeviceInfo();
            
            $stmt = $pdo->prepare("UPDATE users SET auth_token = ?, token_expires_at = ?, last_login_source = ?, last_login_device = ? WHERE id = ?");
            $stmt->execute([$auth_token, $token_expires, $login_source, $login_device, $user['id']]);
        }
        
    } elseif ($isGoogleSignIn) {
        // Google Sign In Login
        error_log('Google Sign In - Starting Login Flow');
        
        $googleUserId = $input['google_user_id'];
        error_log('Searching for Google User ID: ' . $googleUserId);
        
        // Prüfe nach google_user_id
        $stmt = $pdo->prepare("SELECT * FROM users WHERE google_user_id = ? AND is_active = 1");
        $stmt->execute([$googleUserId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        error_log('Google Sign In - User by google_user_id: ' . ($user ? 'Found' : 'Not Found'));
        
        if ($user) {
            error_log('Google Sign In - Existing user found: ID ' . $user['id']);
            // Bestehender User - Login
        } else {
            // Neuer User - Registrierung
            error_log('Google Sign In - Creating new user');
            
            // Wenn Email vorhanden, prüfe ob E-Mail bereits registriert ist
            if (!empty($input['email'])) {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND is_active = 1");
                $stmt->execute([$input['email']]);
                $existingUser = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($existingUser) {
                    error_log('Google Sign In - Email exists: ' . $input['email']);
                    if (!$existingUser['google_user_id']) {
                        error_log('Google Sign In - Linking Google ID to existing email account');
                        // E-Mail existiert bereits ohne Google - füge Google-ID hinzu
                        $stmt = $pdo->prepare("UPDATE users SET google_user_id = ? WHERE id = ?");
                        $stmt->execute([$googleUserId, $existingUser['id']]);
                        $user = $existingUser;
                        $user['google_user_id'] = $googleUserId;
                    } else {
                        error_log('Google Sign In - Email already linked to different Google ID');
                        throw new Exception('Diese E-Mail-Adresse ist bereits mit einem anderen Google-Account verknüpft.');
                    }
                }
            }
            
            // Wenn immer noch kein User, erstelle neuen
            if (!$user) {
                error_log('Google Sign In - Creating completely new user');
                error_log('Google Sign In - Input data: ' . print_r($input, true));
                
                // WICHTIG: Bei Google Sign In sollte Datenschutz bereits client-seitig geprüft werden
                // Hier zusätzliche Backend-Prüfung:
                if (!isset($input['datenschutz_accepted']) || $input['datenschutz_accepted'] !== true) {
                    error_log('Google Sign In - Datenschutz nicht akzeptiert');
                    throw new Exception('Datenschutzerklärung muss akzeptiert werden');
                }
                
                // Neuer Google-Benutzer - automatische Registrierung
                $first_name = $input['first_name'] ?? null;
                $last_name = $input['last_name'] ?? null;
                
                error_log('Google Sign In - Name data from input - first: ' . var_export($input['first_name'] ?? 'NULL', true) . ', last: ' . var_export($input['last_name'] ?? 'NULL', true));
                
                // Wenn keine Namen vorhanden, versuche sie aus der Email zu extrahieren
                $email = $input['email'] ?? null;
                if (empty($first_name) && !empty($email)) {
                    $emailParts = explode('@', $email);
                    $emailLocal = $emailParts[0];
                    
                    // Versuche Vorname.Nachname Format
                    if (strpos($emailLocal, '.') !== false) {
                        $parts = explode('.', $emailLocal);
                        if (count($parts) >= 2) {
                            $first_name = ucfirst($parts[0]);
                            $last_name = ucfirst($parts[1]);
                            error_log('⚠️ Extracted name from email: ' . $first_name . ' ' . $last_name);
                        }
                    } else {
                        // Fallback: Erste 6 Zeichen als Vorname
                        $first_name = ucfirst(substr($emailLocal, 0, 6));
                        $last_name = 'User';
                    }
                }
                
                // Wenn immer noch keine Namen, verwende Platzhalter
                if (empty($first_name)) {
                    $first_name = 'Vorname';
                    error_log('⚠️ No first_name from Google - using placeholder');
                }
                if (empty($last_name)) {
                    $last_name = 'Nachname';
                    error_log('⚠️ No last_name from Google - using placeholder');
                }
                
                // Generiere eindeutige Email für Google User ohne Email
                if (empty($email)) {
                    // Erstelle eindeutige Email basierend auf Google User ID
                    $emailSuffix = substr(md5($googleUserId), 0, 8);
                    $email = 'google_' . $emailSuffix . '@google.local';
                    
                    // Prüfe ob Email bereits existiert
                    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
                    $stmt->execute([$email]);
                    if ($stmt->fetch()) {
                        $email = 'google_' . $emailSuffix . '_' . time() . '@google.local';
                    }
                }
                
                $auth_token = bin2hex(random_bytes(32));
                $token_expires = date('Y-m-d H:i:s', strtotime('+30 days'));
                
                // Registrierungsquelle und Device-Info
                $registration_source = 'google_signin';
                $registration_device = $input['registration_device'] ?? getDeviceInfo();
                $registration_ip = $input['registration_ip'] ?? getClientIP();
                
                $sql = "INSERT INTO users (
                    first_name, last_name, email, phone_number, license_plate,
                    preferred_dealership_id, is_leasing_vehicle, is_contract_vehicle,
                    google_user_id, auth_token, token_expires_at, email_verified, password_hash, 
                    verification_status, registration_source, registration_device, registration_ip
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $stmt = $pdo->prepare($sql);
                $result = $stmt->execute([
                    $first_name,
                    $last_name,
                    $email,
                    $input['phone_number'] ?? '',
                    $input['license_plate'] ?? '',
                    null,
                    false,
                    false,
                    $googleUserId,
                    $auth_token,
                    $token_expires,
                    true,
                    null, // Kein Passwort für Google-Benutzer
                    'pending', // Verifikation erforderlich
                    $registration_source,
                    $registration_device,
                    $registration_ip
                ]);
                
                if ($result) {
                    $user_id = $pdo->lastInsertId();
                    error_log('Google Sign In - New user created with ID: ' . $user_id);
                    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                } else {
                    error_log('Google Sign In - Failed to create user');
                }
            }
        }
        
        if ($user) {
            error_log('Google Sign In - Final user: ID ' . $user['id'] . ', Email: ' . $user['email']);
            // Token aktualisieren und Login-Quelle speichern
            $auth_token = bin2hex(random_bytes(32));
            $token_expires = date('Y-m-d H:i:s', strtotime('+30 days'));
            
            $login_source = 'google_signin';
            $login_device = getDeviceInfo();
            
            $stmt = $pdo->prepare("UPDATE users SET auth_token = ?, token_expires_at = ?, last_login_source = ?, last_login_device = ? WHERE id = ?");
            $stmt->execute([$auth_token, $token_expires, $login_source, $login_device, $user['id']]);
        }
        
    } else {
        // Normale E-Mail/Passwort-Anmeldung
        if (empty($input['password'])) {
            throw new Exception('Password is required for email login');
        }
        
        $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND is_active = 1");
        $stmt->execute([$input['email']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            throw new Exception('Falsches Passwort oder E-Mail-Adresse');
        }
        
        if (!$user['password_hash'] || !password_verify($input['password'], $user['password_hash'])) {
            throw new Exception('Falsches Passwort oder E-Mail-Adresse');
        }
        
        // Prüfe Verifizierungsstatus
        // Rejected User können sich nicht anmelden
        if ($user['verification_status'] === 'rejected') {
            throw new Exception('Ihr Konto wurde abgelehnt. Bitte kontaktieren Sie den Support.');
        }
        
        // Pending User können sich anmelden und Dokumente hochladen
        // Die Blockierung erfolgt in booking.php (keine Terminbuchung möglich)
        
        // Token aktualisieren und Login-Quelle speichern
        $auth_token = bin2hex(random_bytes(32));
        $token_expires = date('Y-m-d H:i:s', strtotime('+30 days'));
        
        $login_source = 'web';
        $login_device = getDeviceInfo();
        
        $stmt = $pdo->prepare("UPDATE users SET auth_token = ?, token_expires_at = ?, last_login_source = ?, last_login_device = ? WHERE id = ?");
        $stmt->execute([$auth_token, $token_expires, $login_source, $login_device, $user['id']]);
    }
    
        if ($user && isset($auth_token)) {
            // Prüfe Verifizierungsstatus für alle Login-Methoden
            if ($user['verification_status'] === 'rejected') {
                // Rejected User können sich nicht anmelden
                throw new Exception('Ihr Konto wurde abgelehnt. Bitte kontaktieren Sie den Support.');
            }
            
            // Pending User können sich anmelden und Dokumente hochladen
            // Die Blockierung erfolgt in index.php und booking.php
            
            // Login-Aktivität in user_activity_log speichern
            try {
                // Stelle sicher, dass login_source definiert ist
                if (!isset($login_source)) {
                    $login_source = 'web'; // Fallback
                }
                
                $ip_address = getClientIP();
                $device_info = isset($login_device) ? $login_device : getDeviceInfo();
                $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
                
                // Prüfe ob Browser-Geolocation verfügbar ist (viel genauer als GeoIP)
                $browser_location = null;
                $location_data = null;
                $location_source = 'geoip'; // Standard: GeoIP
                
                if (isset($input['browser_location']) && is_array($input['browser_location']) && 
                    isset($input['browser_location']['latitude']) && isset($input['browser_location']['longitude'])) {
                    // Browser-Geolocation ist verfügbar (viel genauer!)
                    $browser_location = $input['browser_location'];
                    $location_source = 'browser_geolocation';
                    
                    // Verwende Browser-Geolocation als primäre Quelle
                    $location_data = [
                        'latitude' => floatval($browser_location['latitude']),
                        'longitude' => floatval($browser_location['longitude']),
                        'accuracy' => isset($browser_location['accuracy']) ? floatval($browser_location['accuracy']) : null,
                        'source' => 'browser_geolocation'
                    ];
                    
                    // Verwende Reverse Geocoding-Daten (aus Browser), falls verfügbar
                    if (isset($browser_location['city_info']) && is_array($browser_location['city_info'])) {
                        $city_info = $browser_location['city_info'];
                        $location_data['city'] = $city_info['city'] ?? null;
                        $location_data['region'] = $city_info['region'] ?? null;
                        $location_data['country'] = $city_info['country'] ?? null;
                        $location_data['postcode'] = $city_info['postcode'] ?? null;
                        
                        // Erstelle location-String aus Reverse Geocoding
                        $location_parts = array_filter([
                            $city_info['city'] ?? null,
                            $city_info['region'] ?? null,
                            $city_info['country'] ?? null
                        ]);
                        $location_data['location'] = !empty($location_parts) ? implode(', ', $location_parts) : null;
                    }
                    
                    // Fallback: Hole zusätzlich GeoIP-Daten für Stadt/Region-Informationen (nur wenn Reverse Geocoding fehlgeschlagen ist)
                    if (empty($location_data['location']) && $ip_address && $ip_address !== 'Unknown' && filter_var($ip_address, FILTER_VALIDATE_IP)) {
                        try {
                            $geoip_url = "http://ip-api.com/json/{$ip_address}?fields=status,country,regionName,city,lat,lon,isp";
                            $geoip_context = stream_context_create([
                                'http' => [
                                    'timeout' => 2,
                                    'method' => 'GET'
                                ]
                            ]);
                            $geoip_response = @file_get_contents($geoip_url, false, $geoip_context);
                            
                            if ($geoip_response) {
                                $geoip_data = json_decode($geoip_response, true);
                                if (isset($geoip_data['status']) && $geoip_data['status'] === 'success') {
                                    // Füge GeoIP-Stadt/Region-Info zu location_data hinzu (nur wenn nicht bereits vorhanden)
                                    if (empty($location_data['country'])) {
                                        $location_data['country'] = $geoip_data['country'] ?? null;
                                    }
                                    if (empty($location_data['region'])) {
                                        $location_data['region'] = $geoip_data['regionName'] ?? null;
                                    }
                                    if (empty($location_data['city'])) {
                                        $location_data['city'] = $geoip_data['city'] ?? null;
                                    }
                                    $location_data['isp'] = $geoip_data['isp'] ?? null;
                                    
                                    // Erstelle location-String
                                    $location_parts = array_filter([
                                        $location_data['city'],
                                        $location_data['region'],
                                        $location_data['country']
                                    ]);
                                    $location_data['location'] = !empty($location_parts) ? implode(', ', $location_parts) : null;
                                }
                            }
                        } catch (Exception $geoip_error) {
                            error_log("GeoIP Fehler: " . $geoip_error->getMessage());
                        }
                    }
                } else {
                    // Fallback: Hole Standort-Informationen via GeoIP-API (ip-api.com - kostenlos)
                    if ($ip_address && $ip_address !== 'Unknown' && filter_var($ip_address, FILTER_VALIDATE_IP)) {
                        try {
                            // ip-api.com - kostenlos, 45 Requests/Minute
                            $geoip_url = "http://ip-api.com/json/{$ip_address}?fields=status,country,regionName,city,lat,lon,isp";
                            $geoip_context = stream_context_create([
                                'http' => [
                                    'timeout' => 2, // 2 Sekunden Timeout
                                    'method' => 'GET'
                                ]
                            ]);
                            $geoip_response = @file_get_contents($geoip_url, false, $geoip_context);
                            
                            if ($geoip_response) {
                                $geoip_data = json_decode($geoip_response, true);
                                if (isset($geoip_data['status']) && $geoip_data['status'] === 'success') {
                                    $location_data = [
                                        'country' => $geoip_data['country'] ?? null,
                                        'region' => $geoip_data['regionName'] ?? null,
                                        'city' => $geoip_data['city'] ?? null,
                                        'latitude' => $geoip_data['lat'] ?? null,
                                        'longitude' => $geoip_data['lon'] ?? null,
                                        'isp' => $geoip_data['isp'] ?? null,
                                        'location' => trim(($geoip_data['city'] ?? '') . ', ' . ($geoip_data['regionName'] ?? '') . ', ' . ($geoip_data['country'] ?? ''), ', '),
                                        'source' => 'geoip'
                                    ];
                                }
                            }
                        } catch (Exception $geoip_error) {
                            // Fehler beim GeoIP sollte den Login nicht verhindern
                            error_log("GeoIP Fehler: " . $geoip_error->getMessage());
                        }
                    }
                }
                
                $metadata = json_encode([
                    'ip_address' => $ip_address,
                    'device' => $device_info,
                    'device_info' => $user_agent,
                    'source' => $login_source,
                    'login_source' => $login_source,
                    'location' => $location_data ? ($location_data['location'] ?? null) : null,
                    'location_data' => $location_data,
                    'location_source' => $location_source, // 'browser_geolocation' oder 'geoip'
                    'user_agent' => $user_agent
                ]);
                
                $description = "Login von " . $login_source . " - " . $device_info;
                
                $stmt = $pdo->prepare("
                    INSERT INTO user_activity_log (user_id, action_type, description, metadata)
                    VALUES (?, 'login', ?, ?)
                ");
                $stmt->execute([$user['id'], $description, $metadata]);
            } catch (Exception $log_error) {
                // Fehler beim Loggen sollte den Login nicht verhindern
                error_log("Fehler beim Speichern der Login-Aktivität: " . $log_error->getMessage());
            }
            
            // Passwort-Hash aus Antwort entfernen
            unset($user['password_hash']);
            
            echo json_encode([
                'success' => true,
                'user' => $user,
                'token' => $auth_token,
                'message' => $user['verification_status'] === 'pending' ? 'Login successful - Verifikation erforderlich' : 'Login successful'
            ]);
        } else {
            throw new Exception('Login failed');
        }
    
} catch (Exception $e) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
