<?php
/**
 * API zum Abrufen von Reifenempfehlungen basierend auf eingelagerten Sätzen
 * Liest CSV-Dateien aus volvo_tyre_engine und sucht nach passenden Reifen
 */

// Fehlerbehandlung: Zeige keine PHP-Fehler im JSON
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';

class TireRecommendationEngine {
    private $reifenCsvPath;
    private $komplettradCsvPath;
    public $reifenData = [];
    public $komplettradData = [];
    private static $csvCache = null;
    private static $cacheTimestamp = null;
    private static $cacheFile = null;
    
    public function __construct() {
        $basePath = dirname(__DIR__);
        $this->reifenCsvPath = $basePath . '/assets/ftp/volvo_tyre_engine/Reifen.csv';
        $this->komplettradCsvPath = $basePath . '/assets/ftp/volvo_tyre_engine/Komplettrad.csv';
        $this->cacheFile = $basePath . '/cache/tire_data_cache.php';
        
        error_log("TireRecommendationEngine: Initialisiere mit CSV-Pfad: " . $this->reifenCsvPath);
        error_log("TireRecommendationEngine: CSV-Datei existiert: " . (file_exists($this->reifenCsvPath) ? 'JA' : 'NEIN'));
        
        $this->loadCsvData();
        
        error_log("TireRecommendationEngine: Geladen " . count($this->reifenData) . " Reifen aus CSV");
    }
    
    /**
     * Lädt CSV-Daten in Arrays (mit Caching)
     */
    private function loadCsvData() {
        // Prüfe Cache (5 Minuten Gültigkeit)
        $cacheValid = false;
        if (self::$csvCache !== null && self::$cacheTimestamp !== null) {
            $cacheAge = time() - self::$cacheTimestamp;
            if ($cacheAge < 300) { // 5 Minuten
                $cacheValid = true;
                $this->reifenData = self::$csvCache['reifen'] ?? [];
                $this->komplettradData = self::$csvCache['komplettrad'] ?? [];
            }
        }
        
        // Prüfe Datei-Cache
        if (!$cacheValid && file_exists($this->cacheFile)) {
            $cacheData = include $this->cacheFile;
            if (is_array($cacheData) && isset($cacheData['timestamp']) && isset($cacheData['data'])) {
                $cacheAge = time() - $cacheData['timestamp'];
                if ($cacheAge < 300) { // 5 Minuten
                    $this->reifenData = $cacheData['data']['reifen'] ?? [];
                    $this->komplettradData = $cacheData['data']['komplettrad'] ?? [];
                    self::$csvCache = $cacheData['data'];
                    self::$cacheTimestamp = $cacheData['timestamp'];
                    return;
                }
            }
        }
        
        // Lade aus CSV-Dateien
        if (file_exists($this->reifenCsvPath)) {
            $this->reifenData = $this->parseCsvFile($this->reifenCsvPath);
        }
        
        if (file_exists($this->komplettradCsvPath)) {
            $this->komplettradData = $this->parseCsvFile($this->komplettradCsvPath);
        }
        
        // Speichere im Cache
        self::$csvCache = [
            'reifen' => $this->reifenData,
            'komplettrad' => $this->komplettradData
        ];
        self::$cacheTimestamp = time();
        
        // Speichere in Datei-Cache
        $cacheDir = dirname($this->cacheFile);
        if (!is_dir($cacheDir)) {
            @mkdir($cacheDir, 0755, true);
        }
        if (is_writable($cacheDir)) {
            $cacheContent = "<?php\nreturn " . var_export([
                'timestamp' => time(),
                'data' => self::$csvCache
            ], true) . ";\n";
            @file_put_contents($this->cacheFile, $cacheContent);
        }
    }
    
    /**
     * Parst CSV-Datei und gibt Array zurück
     */
    private function parseCsvFile($filePath) {
        $data = [];
        $handle = fopen($filePath, 'r');
        
        if ($handle === false) {
            error_log("parseCsvFile: Konnte Datei nicht öffnen: " . $filePath);
            return $data;
        }
        
        // Lese Header-Zeile
        $headers = fgetcsv($handle, 0, ';');
        if ($headers === false) {
            fclose($handle);
            error_log("parseCsvFile: Konnte Header nicht lesen");
            return $data;
        }
        
        // Logge Header für Debugging
        error_log("parseCsvFile: Gefundene Spalten: " . implode(', ', $headers));
        
        // Normalisiere Header-Namen (entferne Leerzeichen, konvertiere zu lowercase)
        $normalizedHeaders = [];
        $headerMap = []; // Mapping für verschiedene mögliche Spaltennamen
        foreach ($headers as $index => $header) {
            $normalized = strtolower(trim($header));
            $normalizedHeaders[] = $normalized;
            
            // Erstelle Mapping für verschiedene mögliche Spaltennamen
            if (stripos($normalized, 'dimension') !== false || stripos($normalized, 'größe') !== false || stripos($normalized, 'groesse') !== false) {
                $headerMap['dimension'] = $normalized;
            }
            // WICHTIG: Suche nach "hersteller" (Name), aber NICHT nach "hersteller-nr" (Artikelnummer)
            // Prüfe zuerst, ob es "hersteller-nr" ist - dann überspringe es für das Hersteller-Mapping
            $isHerstellerNr = (stripos($normalized, 'hersteller-nr') !== false || 
                               stripos($normalized, 'hersteller-nummer') !== false || 
                               stripos($normalized, 'artikel') !== false);
            
            // Jetzt prüfe auf "hersteller", "marke" oder "brand" (ohne "-nr")
            if (!$isHerstellerNr && 
                (stripos($normalized, 'hersteller') !== false || 
                 stripos($normalized, 'marke') !== false || 
                 stripos($normalized, 'brand') !== false)) {
                // Nur wenn noch nicht gesetzt oder wenn dieser Header besser passt
                if (empty($headerMap['hersteller']) || 
                    (stripos($normalized, 'hersteller') !== false && stripos($headerMap['hersteller'], 'hersteller') === false)) {
                    $headerMap['hersteller'] = $normalized;
                }
            }
            if (stripos($normalized, 'profil') !== false || stripos($normalized, 'modell') !== false || stripos($normalized, 'model') !== false) {
                $headerMap['profil'] = $normalized;
            }
        }
        
        // Fallback: Wenn keine Header gefunden wurden, verwende Standard-Indizes (falls CSV feste Reihenfolge hat)
        // Typische Reihenfolge: nr, ean, dimension, hersteller, profil, ...
        if (empty($headerMap['dimension']) && isset($normalizedHeaders[2])) {
            $headerMap['dimension'] = $normalizedHeaders[2];
            error_log("parseCsvFile: Verwende Index 2 für Dimension (Fallback)");
        }
        if (empty($headerMap['hersteller'])) {
            // WICHTIG: Suche explizit nach "hersteller" (ohne "-nr" oder "-nummer")
            // Prüfe alle Header und finde den, der "hersteller" enthält, aber NICHT "hersteller-nr"
            $foundHerstellerIndex = null;
            foreach ($normalizedHeaders as $idx => $header) {
                if (stripos($header, 'hersteller') !== false && 
                    stripos($header, 'hersteller-nr') === false && 
                    stripos($header, 'hersteller-nummer') === false &&
                    stripos($header, 'artikel') === false) {
                    $foundHerstellerIndex = $idx;
                    $headerMap['hersteller'] = $header;
                    error_log("parseCsvFile: Gefunden 'hersteller' bei Index $idx: '$header'");
                    break;
                }
            }
            // Wenn immer noch nicht gefunden, versuche Index 3 (aber nur wenn es NICHT "hersteller-nr" ist)
            if (empty($headerMap['hersteller']) && isset($normalizedHeaders[3])) {
                $headerAt3 = $normalizedHeaders[3];
                if (stripos($headerAt3, 'hersteller-nr') === false && 
                    stripos($headerAt3, 'hersteller-nummer') === false &&
                    stripos($headerAt3, 'artikel') === false) {
                    $headerMap['hersteller'] = $headerAt3;
                    error_log("parseCsvFile: Verwende Index 3 für Hersteller (Fallback): '$headerAt3'");
                } else {
                    error_log("parseCsvFile: WARNUNG - Index 3 enthält 'hersteller-nr', überspringe Fallback");
                }
            }
        }
        if (empty($headerMap['profil']) && isset($normalizedHeaders[4])) {
            $headerMap['profil'] = $normalizedHeaders[4];
            error_log("parseCsvFile: Verwende Index 4 für Profil (Fallback)");
        }
        
        error_log("parseCsvFile: Header-Mapping: " . json_encode($headerMap));
        error_log("parseCsvFile: Alle Header (erste 10): " . json_encode(array_slice($normalizedHeaders, 0, 10)));
        
        $rowCount = 0;
        $validRowCount = 0;
        
        // Lese Datenzeilen
        while (($row = fgetcsv($handle, 0, ';')) !== false) {
            $rowCount++;
            
            if (count($row) !== count($normalizedHeaders)) {
                // Überspringe ungültige Zeilen, aber logge sie
                if ($rowCount <= 5) {
                    error_log("parseCsvFile: Zeile $rowCount übersprungen (Spaltenanzahl stimmt nicht: " . count($row) . " vs " . count($normalizedHeaders) . ")");
                }
                continue;
            }
            
            $item = [];
            foreach ($normalizedHeaders as $index => $header) {
                $item[$header] = $row[$index] ?? '';
            }
            
            // Verwende Header-Mapping für Dimension, Hersteller, Profil
            $dimension = '';
            $hersteller = '';
            $profil = '';
            
            // Suche Dimension
            if (isset($headerMap['dimension'])) {
                $dimension = trim($item[$headerMap['dimension']] ?? '');
            } else {
                // Fallback: Suche nach verschiedenen möglichen Spaltennamen
                foreach (['dimension', 'größe', 'groesse', 'reifengröße', 'reifengroesse'] as $key) {
                    if (isset($item[$key])) {
                        $dimension = trim($item[$key]);
                        break;
                    }
                }
            }
            
            // Suche Hersteller
            if (isset($headerMap['hersteller'])) {
                $hersteller = trim($item[$headerMap['hersteller']] ?? '');
            } else {
                foreach (['hersteller', 'marke', 'brand', 'reifenhersteller'] as $key) {
                    if (isset($item[$key])) {
                        $hersteller = trim($item[$key]);
                        break;
                    }
                }
            }
            
            // Suche Profil
            if (isset($headerMap['profil'])) {
                $profil = trim($item[$headerMap['profil']] ?? '');
            } else {
                foreach (['profil', 'modell', 'model', 'reifenmodell'] as $key) {
                    if (isset($item[$key])) {
                        $profil = trim($item[$key]);
                        break;
                    }
                }
            }
            
            // Normalisiere die Spaltennamen für konsistente Verwendung
            $item['dimension'] = $dimension;
            $item['hersteller'] = $hersteller;
            $item['profil'] = $profil;
            
            // Nur Zeilen mit gültigen Daten hinzufügen
            if (!empty($dimension) || !empty($hersteller)) {
                $data[] = $item;
                $validRowCount++;
                
                // Logge erste paar Zeilen für Debugging
                if ($validRowCount <= 3) {
                    error_log("parseCsvFile: Beispiel-Zeile $validRowCount: Dimension='$dimension', Hersteller='$hersteller', Profil='$profil'");
                }
            }
        }
        
        fclose($handle);
        error_log("parseCsvFile: $validRowCount gültige Zeilen von $rowCount insgesamt gelesen");
        return $data;
    }
    
    /**
     * Normalisiert Dimension für Vergleich (z.B. "195/65 R 15" -> "195/65R15")
     * Behandelt auch Fälle wie "150/R15" oder "150/ R 15"
     */
    private function normalizeDimension($dimension) {
        if (empty($dimension)) {
            return '';
        }
        
        // Entferne Leerzeichen und konvertiere zu Großbuchstaben
        $normalized = strtoupper(trim($dimension));
        
        // Entferne zusätzliche Informationen am Ende (wie "91 T", "91Y", "95H XL", etc.)
        // Pattern: Breite/Verhältnis R Durchmesser [Zusatzinfos]
        // Beispiele: "195/65 R 15 91 T" -> "195/65R15", "235/40R18 91Y" -> "235/40R18"
        if (preg_match('/^(\d+\/\d*)\s*R\s*(\d+)/i', $normalized, $matches)) {
            // Extrahiere nur Breite/Verhältnis und Durchmesser
            $normalized = $matches[1] . 'R' . $matches[2];
        } else {
            // Fallback: Entferne alle Leerzeichen
            $normalized = preg_replace('/\s+/', '', $normalized);
            // Normalisiere "R" Format - entferne Leerzeichen um R
            $normalized = preg_replace('/\s*R\s*/', 'R', $normalized);
        }
        
        return $normalized;
    }
    
    /**
     * Normalisiert Hersteller-Namen für Vergleich
     * Entfernt Leerzeichen, konvertiert zu Kleinbuchstaben, entfernt Sonderzeichen
     */
    private function normalizeManufacturer($manufacturer) {
        if (empty($manufacturer)) {
            return '';
        }
        
        $normalized = strtolower(trim($manufacturer));
        // Entferne alle Leerzeichen
        $normalized = preg_replace('/\s+/', '', $normalized);
        // Entferne Sonderzeichen (behalte nur Buchstaben und Zahlen)
        $normalized = preg_replace('/[^a-z0-9]/', '', $normalized);
        
        return $normalized;
    }
    
    /**
     * Berechnet Preis-Leistungs-Score für einen Reifen
     * Ziel: Mittlerer Preis, gute Marge (aber nicht die höchste), gute Qualität
     */
    private function calculateValueScore($tire) {
        $price = floatval($tire['price'] ?? 0);
        $margin = floatval($tire['margin'] ?? 0);
        $marginPercent = floatval($tire['margin_percent'] ?? 0);
        
        // Qualitätsscore basierend auf EU-Label
        $qualityScore = 0;
        $euLabel = strtoupper($tire['eu_label'] ?? '');
        
        // Rollwiderstand: A = 3, B = 2, C = 1, D = 0
        if (preg_match('/\b([A-D])\b/', $euLabel, $matches)) {
            $rollingResistance = $matches[1];
            $qualityScore += ($rollingResistance === 'A' ? 3 : ($rollingResistance === 'B' ? 2 : ($rollingResistance === 'C' ? 1 : 0)));
        }
        
        // Nasshaftung: A = 3, B = 2, C = 1, D = 0
        if (preg_match('/\b([A-D])\b/', substr($euLabel, 2), $matches)) {
            $wetGrip = $matches[1];
            $qualityScore += ($wetGrip === 'A' ? 3 : ($wetGrip === 'B' ? 2 : ($wetGrip === 'C' ? 1 : 0)));
        }
        
        // Normalisiere Qualitätsscore (0-6)
        $qualityScore = min(6, $qualityScore);
        
        // Preis-Leistungs-Score:
        // - Gute Marge (15-30% ist optimal, nicht zu hoch, nicht zu niedrig)
        // - Mittlerer Preis (nicht zu billig, nicht zu teuer)
        // - Gute Qualität
        
        $marginScore = 0;
        if ($marginPercent >= 15 && $marginPercent <= 30) {
            // Optimale Marge: 15-30%
            $marginScore = 10;
        } else if ($marginPercent >= 10 && $marginPercent < 15) {
            // Gute Marge: 10-15%
            $marginScore = 7;
        } else if ($marginPercent > 30 && $marginPercent <= 40) {
            // Sehr hohe Marge (aber nicht optimal)
            $marginScore = 8;
        } else if ($marginPercent > 40) {
            // Zu hohe Marge (nicht optimal für Preis-Leistung)
            $marginScore = 5;
        } else {
            // Niedrige Marge
            $marginScore = 3;
        }
        
        // Preis-Score: Mittlerer Preis ist besser (nicht zu billig, nicht zu teuer)
        // Wir nehmen an, dass die Preise bereits sortiert sind
        // Für jetzt: Niedriger Preis = besser (aber mit Qualitäts-Bonus)
        $priceScore = $price > 0 ? (100 / ($price / 10)) : 0; // Normalisiere Preis
        
        // Gesamt-Score: Qualität (40%) + Marge (40%) + Preis (20%)
        $totalScore = ($qualityScore * 4) + ($marginScore * 4) + ($priceScore * 0.2);
        
        return $totalScore;
    }
    
    /**
     * Normalisiert Einsatzgebiet für Vergleich
     */
    private function normalizeApplicationArea($area) {
        if (empty($area)) {
            return '';
        }
        
        $normalized = strtolower(trim($area));
        
        // Mappings
        $mappings = [
            'summer' => 'sommer',
            'sommer' => 'sommer',
            'winter' => 'winter',
            'allseason' => 'ganzjahres',
            'ganzjahres' => 'ganzjahres',
            'ganzjahresreifen' => 'ganzjahres',
            'all year' => 'ganzjahres'
        ];
        
        foreach ($mappings as $key => $value) {
            if (stripos($normalized, $key) !== false) {
                return $value;
            }
        }
        
        return $normalized;
    }
    
    /**
     * Normalisiert Modell-Namen für Vergleich
     * Entfernt Leerzeichen, konvertiert zu Kleinbuchstaben, normalisiert Sonderzeichen
     */
    private function normalizeModel($model) {
        if (empty($model)) {
            return '';
        }
        
        $normalized = strtolower(trim($model));
        // Normalisiere Leerzeichen (mehrere zu einem)
        $normalized = preg_replace('/\s+/', ' ', $normalized);
        // Entferne führende/abschließende Leerzeichen
        $normalized = trim($normalized);
        
        return $normalized;
    }
    
    /**
     * Prüft ob zwei Modell-Namen übereinstimmen (flexibel)
     * Berücksichtigt Teilstrings, Leerzeichen-Variationen, etc.
     */
    private function modelsMatch($model1, $model2) {
        if (empty($model1) || empty($model2)) {
            return false;
        }
        
        $norm1 = $this->normalizeModel($model1);
        $norm2 = $this->normalizeModel($model2);
        
        // Exakte Übereinstimmung
        if ($norm1 === $norm2) {
            return true;
        }
        
        // Einer enthält den anderen (Teilstring-Match)
        if (strpos($norm1, $norm2) !== false || strpos($norm2, $norm1) !== false) {
            return true;
        }
        
        // Entferne Leerzeichen und prüfe erneut
        $norm1NoSpace = preg_replace('/\s+/', '', $norm1);
        $norm2NoSpace = preg_replace('/\s+/', '', $norm2);
        
        if ($norm1NoSpace === $norm2NoSpace) {
            return true;
        }
        
        // Einer enthält den anderen ohne Leerzeichen
        if (strpos($norm1NoSpace, $norm2NoSpace) !== false || strpos($norm2NoSpace, $norm1NoSpace) !== false) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Extrahiert Breite und Durchmesser aus einer Dimension
     * Gibt Array zurück: ['width' => '205', 'ratio' => '65', 'diameter' => '16', 'normalized' => '205/65R16']
     */
    private function parseDimension($dimension) {
        $normalized = $this->normalizeDimension($dimension);
        
        // Pattern: Breite/VerhältnisRDurchmesser oder Breite/RDurchmesser
        // Beispiel: "205/65R16" oder "205/R16"
        if (preg_match('/^(\d+)\/(\d*)R(\d+)$/', $normalized, $matches)) {
            return [
                'width' => $matches[1],
                'ratio' => $matches[2] ?? '',
                'diameter' => $matches[3],
                'normalized' => $normalized
            ];
        }
        
        return null;
    }
    
    /**
     * Prüft ob zwei Dimensionen kompatibel sind (gleiche Breite und Durchmesser)
     */
    private function dimensionsMatch($dim1, $dim2) {
        $parsed1 = $this->parseDimension($dim1);
        $parsed2 = $this->parseDimension($dim2);
        
        if (!$parsed1 || !$parsed2) {
            // Fallback: Exakte Übereinstimmung
            return $this->normalizeDimension($dim1) === $this->normalizeDimension($dim2);
        }
        
        // Dimensionen passen, wenn Breite und Durchmesser übereinstimmen
        return $parsed1['width'] === $parsed2['width'] && 
               $parsed1['diameter'] === $parsed2['diameter'];
    }
    
    /**
     * Sucht nach exakten Matches (1:1 Empfehlungen)
     * Wichtig: Dimension ist am wichtigsten (Breite + Durchmesser), dann Marke, Modell kann abweichen
     */
    public function findExactMatches($dimension, $manufacturer = '', $model = '') {
        $matches = [];
        
        if (empty($dimension)) {
            error_log("findExactMatches: Leere Dimension übergeben");
            return $matches;
        }
        
        // Normalisiere Suchkriterien
        $searchManufacturerNorm = $this->normalizeManufacturer($manufacturer);
        $searchModelNorm = $this->normalizeModel($model);
        
        // Optimierung: Parse Dimension einmal vor der Schleife
        $searchParsed = $this->parseDimension($dimension);
        if (!$searchParsed) {
            error_log("findExactMatches: Konnte Dimension nicht parsen: " . $dimension);
            return $matches;
        }
        
        error_log("findExactMatches: Suche nach Dimension=" . $dimension . " (parsed: " . json_encode($searchParsed) . "), Manufacturer=" . $manufacturer . " (normalized: " . $searchManufacturerNorm . "), Model=" . $model . " (normalized: " . $searchModelNorm . ")");
        error_log("findExactMatches: Durchsuche " . count($this->reifenData) . " Reifen in der Datenbank");
        
        // Debug: Zeige erste paar Reifen aus CSV für Vergleich
        $sampleCount = min(5, count($this->reifenData));
        for ($i = 0; $i < $sampleCount; $i++) {
            $sampleTire = $this->reifenData[$i];
            $sampleDim = $sampleTire['dimension'] ?? '';
            $sampleMan = $sampleTire['hersteller'] ?? '';
            $sampleMod = $sampleTire['profil'] ?? '';
            error_log("  Beispiel CSV-Reifen " . ($i + 1) . ": Dimension='" . $sampleDim . "', Hersteller='" . $sampleMan . "', Profil='" . $sampleMod . "'");
        }
        
        $matchCount = 0;
        $checkedCount = 0;
        foreach ($this->reifenData as $tire) {
            $checkedCount++;
            $tireDimension = $tire['dimension'] ?? '';
            
            if (empty($tireDimension)) {
                continue;
            }
            
            // PRIORITÄT 1: Dimension muss übereinstimmen (Breite + Durchmesser, Verhältnis kann abweichen)
            // Optimierung: Schneller Vergleich mit bereits geparster Dimension
            $tireParsed = $this->parseDimension($tireDimension);
            if (!$tireParsed) {
                // Fallback: Normalisierter Vergleich
                $normalizedSearch = $this->normalizeDimension($dimension);
                $normalizedTire = $this->normalizeDimension($tireDimension);
                if ($normalizedSearch !== $normalizedTire) {
                    // Logge nur erste paar nicht-matches für Debugging
                    if ($checkedCount <= 5) {
                        error_log("  × Dimension stimmt nicht: Such='$dimension' (norm: $normalizedSearch) vs CSV='$tireDimension' (norm: $normalizedTire)");
                    }
                    continue;
                }
            } else {
                // Schneller Vergleich: Breite und Durchmesser
                if ($searchParsed['width'] !== $tireParsed['width'] || 
                    $searchParsed['diameter'] !== $tireParsed['diameter']) {
                    // Logge nur erste paar nicht-matches für Debugging
                    if ($checkedCount <= 5) {
                        error_log("  × Dimension stimmt nicht: Such=" . $searchParsed['width'] . "/" . $searchParsed['diameter'] . " vs CSV=" . $tireParsed['width'] . "/" . $tireParsed['diameter']);
                    }
                    continue;
                }
            }
            
            // Dimension stimmt überein - weiter mit Hersteller und Modell
            // WICHTIG: hersteller = Herstellername (z.B. "Continental", "Dunlop")
            //          hersteller-nr = Artikelnummer vom Hersteller (z.B. "528970")
            $tireManufacturer = trim($tire['hersteller'] ?? '');
            $tireManufacturerNorm = $this->normalizeManufacturer($tireManufacturer);
            $tireModel = trim($tire['profil'] ?? '');
            
            // PRIORITÄT 2: Wenn Hersteller angegeben, bevorzuge diese, aber akzeptiere auch andere
            $matchPriority = 3; // Standard: Nur Dimension übereinstimmend
            $matchType = 'recommended';
            
            if (!empty($searchManufacturerNorm)) {
                if ($tireManufacturerNorm === $searchManufacturerNorm) {
                    // Marke stimmt überein - höhere Priorität
                    $matchPriority = 1;
                    $matchType = 'exact';
                    
                    // Wenn auch Modell angegeben, prüfe Modell
                    if (!empty($searchModelNorm)) {
                        if ($this->modelsMatch($tireModel, $model)) {
                            // Exaktes Modell-Match - höchste Priorität
                            $matchPriority = 0;
                            error_log("  ✓ Exaktes Match gefunden: " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                        } else {
                            // Marke stimmt, aber Modell weicht ab - immer noch gute Empfehlung
                            $matchPriority = 1;
                            error_log("  ✓ Marke-Match gefunden (Modell abweichend): " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                        }
                    } else {
                        error_log("  ✓ Marke-Match gefunden (kein Modell-Vergleich): " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                    }
                } else {
                    // Marke weicht ab, aber Dimension stimmt - niedrigere Priorität, aber immer noch Empfehlung
                    $matchPriority = 2;
                    $matchType = 'alternative';
                    error_log("  → Alternative gefunden (andere Marke): " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                }
            } else {
                // Kein Hersteller angegeben - nur Dimension-Match
                // Prüfe Modell, wenn angegeben
                if (!empty($searchModelNorm) && !empty($tireModel)) {
                    if ($this->modelsMatch($tireModel, $model)) {
                        // Modell stimmt überein - höhere Priorität
                        $matchPriority = 1;
                        $matchType = 'exact';
                        error_log("  ✓ Modell-Match gefunden (nur Dimension + Modell): " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                    } else {
                        $matchPriority = 2;
                        $matchType = 'alternative';
                        error_log("  → Empfehlung gefunden (Dimension stimmt, Modell abweichend): " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                    }
                } else {
                    error_log("  → Empfehlung gefunden (nur Dimension): " . $tireDimension . " - " . $tireManufacturer . " " . $tireModel);
                }
            }
            
            $formattedTire = $this->formatTireData($tire, $matchType);
            $formattedTire['match_priority'] = $matchPriority;
            $matches[] = $formattedTire;
            $matchCount++;
        }
        
        error_log("findExactMatches: Insgesamt " . $matchCount . " Matches gefunden");
        
        // Sortiere nach Priorität (niedrigere Zahl = höhere Priorität)
        usort($matches, function($a, $b) {
            $priorityA = $a['match_priority'] ?? 999;
            $priorityB = $b['match_priority'] ?? 999;
            if ($priorityA !== $priorityB) {
                return $priorityA <=> $priorityB;
            }
            // Dann nach Preis
            $priceA = floatval($a['price'] ?? 0);
            $priceB = floatval($b['price'] ?? 0);
            return $priceA <=> $priceB;
        });
        
        return $matches;
    }
    
    /**
     * Sucht nach alternativen Reifen (gleiche Dimension, andere Hersteller)
     * Wichtig: Dimension ist am wichtigsten! (wird jetzt in findExactMatches behandelt)
     * Diese Funktion wird nicht mehr verwendet, da alle Empfehlungen in findExactMatches sind
     */
    public function findAlternatives($dimension, $excludeManufacturer = '', $excludeModel = '') {
        // Diese Funktion wird nicht mehr benötigt, da findExactMatches jetzt alle Empfehlungen behandelt
        // (auch solche mit abweichender Marke)
        return [];
    }
    
    /**
     * Formatiert Reifendaten für die Ausgabe
     */
    public function formatTireData($tire, $matchType = 'alternative') {
        $price = floatval(str_replace(',', '.', $tire['vk-brutto'] ?? $tire['vk-brutto'] ?? '0'));
        $priceNet = floatval(str_replace(',', '.', $tire['vk-netto'] ?? $tire['vk-netto'] ?? '0'));
        $ek = floatval(str_replace(',', '.', $tire['ot-ek'] ?? $tire['ot-ek'] ?? '0'));
        $oemEk = floatval(str_replace(',', '.', $tire['oem-ek'] ?? $tire['oem-ek'] ?? '0'));
        $rabatt = floatval(str_replace(',', '.', $tire['rabatt'] ?? $tire['rabatt'] ?? '0'));
        $kb = floatval(str_replace(',', '.', $tire['kb'] ?? $tire['kb'] ?? '0'));
        
        // EU-Label zusammenstellen
        $euLabel = '';
        $rollwiderstand = trim($tire['rollwiderstand'] ?? '');
        $nasshaftung = trim($tire['nasshaftung'] ?? '');
        $geraeuschklasse = trim($tire['geräuschklasse'] ?? '');
        $geraeuschemission = trim($tire['geräuschemission'] ?? '');
        
        if (!empty($rollwiderstand) || !empty($nasshaftung) || !empty($geraeuschklasse)) {
            $euLabel = trim(($rollwiderstand ?: '-') . ' ' . ($nasshaftung ?: '-') . ' ' . ($geraeuschklasse ?: '-'));
            if (!empty($geraeuschemission)) {
                $euLabel .= ' ' . $geraeuschemission;
            }
        }
        
        // Berechne Marge (Verkaufspreis - Einkaufspreis)
        // Verwende den niedrigeren EK (ot-ek oder oem-ek)
        $usedEk = $ek > 0 ? $ek : ($oemEk > 0 ? $oemEk : 0);
        $margin = $price - $usedEk;
        $marginPercent = $price > 0 ? ($margin / $price) * 100 : 0;
        
        return [
            'ean' => trim($tire['ean'] ?? ''),
            'dimension' => trim($tire['dimension'] ?? ''),
            'manufacturer' => trim($tire['hersteller'] ?? ''),
            'model' => trim($tire['profil'] ?? ''),
            'brand_model' => trim(($tire['hersteller'] ?? '') . ' - ' . ($tire['profil'] ?? '')),
            'oem_part_number' => trim($tire['oem-ersatzteil-nr'] ?? ''),
            'part_number' => trim($tire['ersatzteil-nr'] ?? ''),
            'manufacturer_number' => trim($tire['hersteller-nr'] ?? ''),
            'execution' => trim($tire['ausfuehrung'] ?? ''),
            'marking' => trim($tire['markierung'] ?? ''),
            'runflat' => trim($tire['runflat'] ?? ''),
            'vehicle_type' => trim($tire['fahrzeugart'] ?? ''),
            'application_area' => trim($tire['einsatzgebiet'] ?? ''),
            'eu_label' => $euLabel,
            'noise_emission' => $geraeuschemission,
            'noise_class' => $geraeuschklasse,
            'wet_grip' => $nasshaftung,
            'rolling_resistance' => $rollwiderstand,
            'price' => $price,
            'price_net' => $priceNet,
            'ek' => $ek,
            'oem_ek' => $oemEk,
            'ek_used' => $usedEk,
            'margin' => $margin,
            'margin_percent' => $marginPercent,
            'rabatt' => $rabatt,
            'kb' => $kb,
            'kb_date' => trim($tire['kb-datum'] ?? ''),
            'stock' => trim($tire['eigenes lager'] ?? ''),
            'warehouse' => trim($tire['lager'] ?? ''),
            'notes' => trim($tire['bemerkungen'] ?? ''),
            'match_type' => $matchType
        ];
    }
    
    /**
     * Hauptfunktion: Sucht Empfehlungen basierend auf aktuellen Reifendaten
     */
    public function getRecommendations($currentTires) {
        $recommendations = [
            'exact_matches' => [],
            'alternatives' => []
        ];
        
        error_log("getRecommendations: Starte Suche mit " . count($currentTires) . " aktuellen Reifen");
        error_log("getRecommendations: CSV-Datenbank enthält " . count($this->reifenData) . " Reifen");
        
        if (empty($currentTires) || !is_array($currentTires)) {
            error_log("getRecommendations: Keine aktuellen Reifen übergeben");
            return $recommendations;
        }
        
        if (empty($this->reifenData)) {
            error_log("getRecommendations: WARNUNG - CSV-Datenbank ist leer!");
            return $recommendations;
        }
        
        // Verarbeite jeden aktuellen Reifen
        foreach ($currentTires as $index => $tire) {
            error_log("getRecommendations: Verarbeite Reifen " . ($index + 1) . ": " . json_encode($tire));
            
            $dimension = $tire['dimension'] ?? $tire['size'] ?? $tire['reifengroesse'] ?? '';
            $manufacturer = $tire['manufacturer'] ?? $tire['hersteller'] ?? $tire['brand'] ?? '';
            $model = $tire['model'] ?? $tire['profil'] ?? $tire['reifenmodell'] ?? '';
            
            error_log("getRecommendations: Extrahiert - Dimension='$dimension', Manufacturer='$manufacturer', Model='$model'");
            
            if (empty($dimension)) {
                error_log("getRecommendations: Überspringe Reifen " . ($index + 1) . " - keine Dimension");
                continue;
            }
            
            // Suche alle Empfehlungen (findExactMatches behandelt jetzt alle Fälle)
            $allMatches = $this->findExactMatches($dimension, $manufacturer, $model);
            error_log("getRecommendations: Gefunden " . count($allMatches) . " Matches für Reifen " . ($index + 1));
            
            // Extrahiere Einsatzgebiet aus aktuellem Reifen
            $currentApplicationArea = $tire['type'] ?? $tire['season'] ?? '';
            // Normalisiere Einsatzgebiet (SUMMER -> Sommer, WINTER -> Winter, ALLSEASON -> Ganzjahres)
            $normalizedApplicationArea = $this->normalizeApplicationArea($currentApplicationArea);
            
            if (!empty($allMatches)) {
                // WICHTIG: Dimension und Einsatzgebiet müssen IMMER zu 100% übereinstimmen!
                foreach ($allMatches as $match) {
                    $priority = $match['match_priority'] ?? 999;
                    
                    // Prüfe Einsatzgebiet - MUSS zu 100% übereinstimmen!
                    $matchApplicationArea = $this->normalizeApplicationArea($match['application_area'] ?? '');
                    $applicationAreaMatches = false;
                    
                    if (!empty($normalizedApplicationArea) && !empty($matchApplicationArea)) {
                        // Beide sind vorhanden - müssen exakt übereinstimmen
                        $applicationAreaMatches = ($matchApplicationArea === $normalizedApplicationArea);
                    } else if (empty($normalizedApplicationArea) && empty($matchApplicationArea)) {
                        // Beide sind leer - das zählt als Übereinstimmung
                        $applicationAreaMatches = true;
                    } else {
                        // Einer ist vorhanden, der andere nicht - KEINE Übereinstimmung
                        $applicationAreaMatches = false;
                    }
                    
                    // Prüfe Dimension - MUSS zu 100% übereinstimmen!
                    $matchDimension = $this->normalizeDimension($match['dimension'] ?? '');
                    $searchDimension = $this->normalizeDimension($dimension);
                    $dimensionMatches = ($matchDimension === $searchDimension);
                    
                    // Nur wenn BEIDE (Dimension UND Einsatzgebiet) zu 100% übereinstimmen, wird der Reifen verwendet
                    if (!$dimensionMatches || !$applicationAreaMatches) {
                        error_log("getRecommendations: Reifen übersprungen - Dimension: " . ($dimensionMatches ? 'OK' : 'FEHLT') . ", Einsatzgebiet: " . ($applicationAreaMatches ? 'OK' : 'FEHLT'));
                        continue;
                    }
                    
                    if ($priority === 0) {
                        // NUR exakte 1:1 Matches (Dimension + Marke + Modell + Einsatzgebiet)
                        $recommendations['exact_matches'][] = $match;
                    } else if ($priority >= 2) {
                        // Alternativen: andere Marke, aber gleiche Dimension + Einsatzgebiet (beide bereits geprüft)
                        $recommendations['alternatives'][] = $match;
                    }
                    // Priorität 1 (Marke stimmt, aber Modell nicht) wird nicht mehr angezeigt
                }
            } else {
                error_log("getRecommendations: Keine Matches gefunden für Reifen " . ($index + 1) . " - versuche nur mit Dimension");
                // Versuche nochmal mit nur Dimension (ohne Marke/Modell-Filter)
                $dimensionOnlyMatches = $this->findExactMatches($dimension, '', '');
                if (!empty($dimensionOnlyMatches)) {
                    error_log("getRecommendations: Gefunden " . count($dimensionOnlyMatches) . " Matches nur mit Dimension");
                    // Filtere nach Einsatzgebiet - MUSS zu 100% übereinstimmen!
                    foreach ($dimensionOnlyMatches as $match) {
                        // Prüfe Dimension - MUSS zu 100% übereinstimmen!
                        $matchDimension = $this->normalizeDimension($match['dimension'] ?? '');
                        $searchDimension = $this->normalizeDimension($dimension);
                        if ($matchDimension !== $searchDimension) {
                            continue;
                        }
                        
                        // Prüfe Einsatzgebiet - MUSS zu 100% übereinstimmen!
                        $matchApplicationArea = $this->normalizeApplicationArea($match['application_area'] ?? '');
                        $applicationAreaMatches = false;
                        
                        if (!empty($normalizedApplicationArea) && !empty($matchApplicationArea)) {
                            $applicationAreaMatches = ($matchApplicationArea === $normalizedApplicationArea);
                        } else if (empty($normalizedApplicationArea) && empty($matchApplicationArea)) {
                            $applicationAreaMatches = true;
                        }
                        
                        if ($applicationAreaMatches) {
                            $recommendations['alternatives'][] = $match;
                        }
                    }
                } else {
                    error_log("getRecommendations: Auch mit nur Dimension keine Matches gefunden!");
                }
            }
        }
        
        // Entferne Duplikate basierend auf EAN
        $recommendations['exact_matches'] = $this->removeDuplicates($recommendations['exact_matches']);
        $recommendations['alternatives'] = $this->removeDuplicates($recommendations['alternatives']);
        
        // Filtere Reifen ohne Preis (0,00€) heraus
        $recommendations['exact_matches'] = array_filter($recommendations['exact_matches'], function($tire) {
            $price = floatval($tire['price'] ?? 0);
            return $price > 0;
        });
        $recommendations['alternatives'] = array_filter($recommendations['alternatives'], function($tire) {
            $price = floatval($tire['price'] ?? 0);
            return $price > 0;
        });
        
        // Setze Arrays zurück (array_filter behält Indizes)
        $recommendations['exact_matches'] = array_values($recommendations['exact_matches']);
        $recommendations['alternatives'] = array_values($recommendations['alternatives']);
        
        // WICHTIG: Nur EIN 1:1 Match anzeigen - wähle den mit dem besten Preis-Leistungs-Verhältnis
        $bestValueTire = null;
        if (count($recommendations['exact_matches']) > 0) {
            // Sortiere nach Preis-Leistung: Berechne Score basierend auf Preis, Marge und Qualität
            usort($recommendations['exact_matches'], function($a, $b) {
                // Berechne Preis-Leistungs-Score für jeden Reifen
                $scoreA = $this->calculateValueScore($a);
                $scoreB = $this->calculateValueScore($b);
                
                // Höherer Score = besseres Preis-Leistungs-Verhältnis
                return $scoreB <=> $scoreA;
            });
            
            // Nimm nur den besten (ersten) Reifen für 1:1 Match
            $bestValueTire = array_shift($recommendations['exact_matches']);
            $recommendations['exact_matches'] = [$bestValueTire];
        }
        
        // Finde Reifen mit höchster Marge (kann derselbe sein wie bestValueTire)
        $highestMarginTire = null;
        $allTiresForMargin = [];
        
        // Sammle alle Reifen, die als exact_match in Frage kommen (auch die, die nicht als 1:1 Match gewählt wurden)
        if (!empty($currentTires)) {
            $firstTire = $currentTires[0];
            $dimension = $firstTire['dimension'] ?? '';
            $manufacturer = $firstTire['manufacturer'] ?? '';
            $model = $firstTire['model'] ?? '';
            
            // Suche alle Matches erneut (inkl. Priorität 0)
            $allPossibleMatches = $this->findExactMatches($dimension, $manufacturer, $model);
            
            // Filtere nach Preis > 0 UND Dimension + Einsatzgebiet müssen zu 100% übereinstimmen
            $firstTire = $currentTires[0];
            $currentApplicationArea = $firstTire['type'] ?? $firstTire['season'] ?? '';
            $normalizedCurrentApplicationArea = $this->normalizeApplicationArea($currentApplicationArea);
            $normalizedSearchDimension = $this->normalizeDimension($dimension);
            
            $allPossibleMatches = array_filter($allPossibleMatches, function($tire) use ($normalizedSearchDimension, $normalizedCurrentApplicationArea) {
                // Preis muss > 0 sein
                if (floatval($tire['price'] ?? 0) <= 0) {
                    return false;
                }
                
                // Dimension muss zu 100% übereinstimmen
                $matchDimension = $this->normalizeDimension($tire['dimension'] ?? '');
                if ($matchDimension !== $normalizedSearchDimension) {
                    return false;
                }
                
                // Einsatzgebiet muss zu 100% übereinstimmen
                $matchApplicationArea = $this->normalizeApplicationArea($tire['application_area'] ?? '');
                if (!empty($normalizedCurrentApplicationArea) && !empty($matchApplicationArea)) {
                    if ($matchApplicationArea !== $normalizedCurrentApplicationArea) {
                        return false;
                    }
                } else if (empty($normalizedCurrentApplicationArea) && empty($matchApplicationArea)) {
                    // Beide leer = OK
                } else {
                    // Einer vorhanden, der andere nicht = NICHT OK
                    return false;
                }
                
                return true;
            });
            $allPossibleMatches = array_values($allPossibleMatches);
            
            // Finde den mit höchster Marge
            if (!empty($allPossibleMatches)) {
                usort($allPossibleMatches, function($a, $b) {
                    $marginA = floatval($a['margin'] ?? 0);
                    $marginB = floatval($b['margin'] ?? 0);
                    return $marginB <=> $marginA;
                });
                $highestMarginTire = $allPossibleMatches[0];
            }
        }
        
        // Füge höchste Marge zu exact_matches hinzu, wenn es ein anderer Reifen ist
        if ($highestMarginTire && $bestValueTire) {
            if ($highestMarginTire['ean'] !== $bestValueTire['ean']) {
                $recommendations['exact_matches'][] = $highestMarginTire;
            }
        } else if ($highestMarginTire && !$bestValueTire) {
            // Wenn kein 1:1 Match, aber höchste Marge vorhanden
            $recommendations['exact_matches'][] = $highestMarginTire;
        }
        
        // Filtere Alternativen: Nur 2-3 beste (nach Preis sortiert, niedrigster zuerst)
        usort($recommendations['alternatives'], function($a, $b) {
            $priceA = floatval($a['price'] ?? 0);
            $priceB = floatval($b['price'] ?? 0);
            return $priceA <=> $priceB;
        });
        $recommendations['alternatives'] = array_slice($recommendations['alternatives'], 0, 3);
        
        // Prüfe warum kein 1:1 Match gefunden wurde (wenn keines vorhanden)
        $noMatchReason = null;
        if (empty($recommendations['exact_matches']) && !empty($currentTires)) {
            $firstTire = $currentTires[0];
            $dimension = $firstTire['dimension'] ?? '';
            $manufacturer = $firstTire['manufacturer'] ?? '';
            $model = $firstTire['model'] ?? '';
            
            // Prüfe ob Dimension-Matches vorhanden sind
            $dimensionMatches = $this->findExactMatches($dimension, '', '');
            $dimensionMatches = array_filter($dimensionMatches, function($tire) {
                return floatval($tire['price'] ?? 0) > 0;
            });
            
            if (empty($dimensionMatches)) {
                $noMatchReason = "Dimension nicht in Datenbank vorhanden";
            } else {
                // Prüfe ob Marke vorhanden ist
                $manufacturerMatches = array_filter($dimensionMatches, function($tire) use ($manufacturer) {
                    $tireManufacturer = strtolower(trim($tire['manufacturer'] ?? ''));
                    $searchManufacturer = strtolower(trim($manufacturer));
                    return $tireManufacturer === $searchManufacturer;
                });
                
                if (empty($manufacturerMatches)) {
                    $noMatchReason = "Marke nicht in Datenbank vorhanden";
                } else {
                    // Prüfe ob Modell vorhanden ist
                    $modelMatches = array_filter($manufacturerMatches, function($tire) use ($model) {
                        return $this->modelsMatch($tire['model'] ?? '', $model);
                    });
                    
                    if (empty($modelMatches)) {
                        $noMatchReason = "Modell nicht vorhanden";
                    } else {
                        $noMatchReason = "Kein Reifen mit gültigem Preis gefunden";
                    }
                }
            }
        }
        
        if ($noMatchReason) {
            $recommendations['no_match_reason'] = $noMatchReason;
        }
        
        error_log("getRecommendations: Ergebnis - " . count($recommendations['exact_matches']) . " exakte Matches, " . count($recommendations['alternatives']) . " Alternativen");
        if ($noMatchReason) {
            error_log("getRecommendations: Kein 1:1 Match - Grund: " . $noMatchReason);
        }
        
        return $recommendations;
    }
    
    /**
     * Entfernt Duplikate basierend auf EAN
     */
    private function removeDuplicates($tires) {
        $seen = [];
        $unique = [];
        
        foreach ($tires as $tire) {
            $ean = $tire['ean'] ?? '';
            if (empty($ean) || !isset($seen[$ean])) {
                $seen[$ean] = true;
                $unique[] = $tire;
            }
        }
        
        return $unique;
    }
}

// API-Endpunkt
try {
    $action = $_GET['action'] ?? $_POST['action'] ?? 'get_recommendations';
    
    if ($action === 'get_all_tires') {
        try {
            $engine = new TireRecommendationEngine();
            
            // Lade alle Reifen aus der CSV
            $allTires = [];
            if (!empty($engine->reifenData)) {
                foreach ($engine->reifenData as $tire) {
                    try {
                        $allTires[] = $engine->formatTireData($tire, 'all');
                    } catch (Exception $e) {
                        // Überspringe fehlerhafte Zeilen
                        error_log("Fehler beim Formatieren eines Reifens: " . $e->getMessage());
                        continue;
                    }
                }
            }
            
            echo json_encode([
                'success' => true,
                'tires' => $allTires,
                'count' => count($allTires)
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], JSON_UNESCAPED_UNICODE);
        }
        
    } else if ($action === 'get_recommendations') {
        $engine = new TireRecommendationEngine();
        
        // Lade aktuelle Reifendaten aus POST oder GET
        $currentTires = [];
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            $currentTires = $input['current_tires'] ?? [];
        } else {
            // Für GET: Lade aus wheelset_number
            $wheelsetNumber = $_GET['wheelset_number'] ?? '';
            
            if (!empty($wheelsetNumber)) {
                // Lade wheelset-Daten aus der Datenbank
                $db = new Database();
                $conn = $db->getConnection();
                
                $stmt = $conn->prepare("
                    SELECT reifengroesse, reifenmodell, reifenart, saison 
                    FROM stored_wheelsets_v2 
                    WHERE satznummer = ? AND archiviert = 0
                    LIMIT 1
                ");
                $stmt->execute([$wheelsetNumber]);
                $wheelset = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($wheelset) {
                    // Parse Reifenmodell (Format: "Hersteller Modell")
                    $reifenmodell = $wheelset['reifenmodell'] ?? '';
                    $parts = explode(' ', $reifenmodell, 2);
                    $manufacturer = $parts[0] ?? '';
                    $model = $parts[1] ?? '';
                    
                    $currentTires[] = [
                        'dimension' => $wheelset['reifengroesse'] ?? '',
                        'manufacturer' => $manufacturer,
                        'model' => $model,
                        'type' => $wheelset['reifenart'] ?? '',
                        'season' => $wheelset['saison'] ?? ''
                    ];
                }
            }
        }
        
        if (!isset($engine)) {
            $engine = new TireRecommendationEngine();
        }
        
        // Debug-Informationen sammeln
        $basePath = dirname(__DIR__);
        $csvPath = $basePath . '/assets/ftp/volvo_tyre_engine/Reifen.csv';
        
        $debugInfo = [
            'csv_file_exists' => file_exists($csvPath),
            'csv_file_path' => $csvPath,
            'csv_data_count' => count($engine->reifenData),
            'current_tires_count' => count($currentTires),
            'current_tires' => $currentTires
        ];
        
        // Prüfe erste paar CSV-Zeilen für Debugging
        if (!empty($engine->reifenData)) {
            $debugInfo['sample_csv_rows'] = array_slice($engine->reifenData, 0, 3);
            // Zeige Spaltennamen der ersten Zeile
            if (!empty($engine->reifenData[0])) {
                $debugInfo['csv_columns'] = array_keys($engine->reifenData[0]);
                // Zeige Beispiel-Daten für Dimension, Hersteller, Profil
                $firstRow = $engine->reifenData[0];
                $debugInfo['sample_data'] = [
                    'dimension' => $firstRow['dimension'] ?? 'NICHT GEFUNDEN',
                    'hersteller' => $firstRow['hersteller'] ?? 'NICHT GEFUNDEN',
                    'profil' => $firstRow['profil'] ?? 'NICHT GEFUNDEN'
                ];
            }
        } else {
            $debugInfo['error'] = 'CSV-Datenbank ist leer!';
        }
        
        $recommendations = $engine->getRecommendations($currentTires);
        
        // Füge Debug-Info zu den Empfehlungen hinzu
        $debugInfo['recommendations_count'] = [
            'exact_matches' => count($recommendations['exact_matches']),
            'alternatives' => count($recommendations['alternatives'])
        ];
        
        echo json_encode([
            'success' => true,
            'recommendations' => $recommendations,
            'debug' => $debugInfo
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        
    } else {
        throw new Exception('Unbekannte Aktion');
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

