<?php
/**
 * API-Endpoint für P2 API-Datenabfrage nach Kennzeichen
 * OPTIMIERT: Probiert mehrere Kennzeichen-Varianten und verwendet längeren Zeitraum
 */

// Unterdrücke alle Ausgaben außer JSON (verhindert HTML-Fehler)
ob_start();
error_reporting(E_ALL);
ini_set('display_errors', 0); // Keine HTML-Fehler ausgeben
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ob_end_clean();
    exit(0);
}

// Error Handler für saubere JSON-Antworten
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    // Ignoriere Warnungen, logge aber Fehler
    if ($errno === E_WARNING || $errno === E_NOTICE) {
        error_log("P2 API Warning: $errstr in $errfile:$errline");
        return true; // Weiter ausführen
    }
    // Kritische Fehler als JSON zurückgeben
    ob_clean();
    echo json_encode([
        'success' => false,
        'error' => 'PHP Fehler: ' . $errstr
    ]);
    exit;
});

try {
    require_once dirname(__DIR__) . '/config/database.php';
    require_once dirname(__DIR__) . '/admin/includes/tire_offers_api.php';
} catch (Exception $e) {
    ob_clean();
    echo json_encode([
        'success' => false,
        'error' => 'Fehler beim Laden der Abhängigkeiten: ' . $e->getMessage()
    ]);
    exit;
}

/**
 * Generiert verschiedene Varianten eines Kennzeichens für die Suche
 * z.B. "KI-SD 221" -> ["KI-SD 221", "KI-SD221", "KISD221", "KI SD 221"]
 */
function generateLicensePlateVariants($licensePlate) {
    $variants = [];
    
    // Original
    $variants[] = trim($licensePlate);
    
    // Entferne alle Leerzeichen
    $noSpaces = preg_replace('/\s+/', '', trim($licensePlate));
    if ($noSpaces !== trim($licensePlate)) {
        $variants[] = $noSpaces;
    }
    
    // Entferne alle Bindestriche
    $noDashes = str_replace('-', '', trim($licensePlate));
    if ($noDashes !== trim($licensePlate)) {
        $variants[] = $noDashes;
    }
    
    // Entferne alle Leerzeichen und Bindestriche
    $noSpacesNoDashes = preg_replace('/[\s\-]+/', '', trim($licensePlate));
    if ($noSpacesNoDashes !== trim($licensePlate) && $noSpacesNoDashes !== $noSpaces && $noSpacesNoDashes !== $noDashes) {
        $variants[] = $noSpacesNoDashes;
    }
    
    // Ersetze Bindestriche durch Leerzeichen
    $dashToSpace = str_replace('-', ' ', trim($licensePlate));
    if ($dashToSpace !== trim($licensePlate)) {
        $variants[] = $dashToSpace;
    }
    
    // Ersetze Leerzeichen durch Bindestriche
    $spaceToDash = preg_replace('/\s+/', '-', trim($licensePlate));
    if ($spaceToDash !== trim($licensePlate)) {
        $variants[] = $spaceToDash;
    }
    
    // Normalisiere: Großbuchstaben, keine Leerzeichen/Bindestriche
    $normalized = strtoupper(preg_replace('/[\s\-]+/', '', trim($licensePlate)));
    if ($normalized !== trim($licensePlate)) {
        $variants[] = $normalized;
    }
    
    // Entferne Duplikate und leere Werte
    $variants = array_unique(array_filter($variants));
    
    return array_values($variants);
}

try {
    // Get license plate from request
    $input = json_decode(file_get_contents('php://input'), true);
    $licensePlate = $input['license_plate'] ?? $_GET['license_plate'] ?? '';
    
    if (empty($licensePlate)) {
        echo json_encode([
            'success' => false,
            'error' => 'Kennzeichen fehlt'
        ]);
        exit;
    }
    
    // Get autohaus_id from request (optional, falls vorhanden)
    $autohausId = $input['autohaus_id'] ?? $_GET['autohaus_id'] ?? null;
    
    // Initialize P2 API
    $api = new TireOffersAPI($autohausId);
    
    // OPTIMIERT: Schneller Health-Check vor der Suche (verhindert lange Wartezeiten)
    $apiAvailable = false;
    $healthCheckError = null;
    try {
        $healthResult = $api->apiCall("/system/health", 'GET', [], 2, 1); // 2 Sekunden Timeout, 1 Retry
        $apiAvailable = isset($healthResult['status']) && $healthResult['status'] === 'ok';
    } catch (Exception $e) {
        $healthCheckError = $e->getMessage();
        error_log("P2 API Health-Check fehlgeschlagen: " . $healthCheckError);
        $apiAvailable = false;
    }
    
    if (!$apiAvailable) {
        ob_clean();
        $errorMsg = 'P2 API ist nicht erreichbar.';
        if ($healthCheckError) {
            if (strpos($healthCheckError, 'timed out') !== false) {
                $errorMsg .= ' Die Verbindung zum Server (100.82.207.72:5001) hat ein Timeout.';
            } elseif (strpos($healthCheckError, 'couldn\'t connect') !== false || strpos($healthCheckError, 'CURLE_COULDNT_CONNECT') !== false) {
                $errorMsg .= ' Die Verbindung zum Server (100.82.207.72:5001) konnte nicht hergestellt werden.';
            } else {
                $errorMsg .= ' Fehler: ' . $healthCheckError;
            }
        } else {
            $errorMsg .= ' Bitte prüfen Sie die Verbindung zum Server (100.82.207.72:5001).';
        }
        
        echo json_encode([
            'success' => false,
            'error' => $errorMsg,
            'data' => null,
            'api_unavailable' => true,
            'error_details' => $healthCheckError
        ]);
        exit;
    }
    
    // OPTIMIERT: Probiere nur das Original-Kennzeichen mit sehr kurzem Zeitraum (maximal schnell)
    // Nur 90 Tage (3 Monate) für extrem schnelle Suche
    $customerData = null;
    $lastError = null;
    $errorDetails = [];
    
    try {
        // Direkt den optimierten Endpoint verwenden - nur Original-Kennzeichen, nur 90 Tage
        // Mit Retry-Mechanismus (2 Versuche)
        $encodedLicense = rawurlencode(trim($licensePlate));
        $result = $api->apiCall("/auftraege/by-license/$encodedLicense", 'GET', ['days' => 90], 5, 2);
        
        if (isset($result['success']) && $result['success'] && !empty($result['data'])) {
            // Nimm ersten Treffer (neuesten)
            $customerData = $result['data'][0] ?? null;
            if ($customerData) {
                error_log("P2 API: Gefunden mit Kennzeichen '$licensePlate' (Zeitraum: 90 Tage)");
            }
        } elseif (isset($result['success']) && $result['success'] && empty($result['data'])) {
            // API antwortet, aber keine Daten gefunden
            $lastError = "Keine Aufträge für Kennzeichen '$licensePlate' in den letzten 90 Tagen gefunden";
        }
    } catch (Exception $e) {
        $lastError = $e->getMessage();
        $errorDetails = [
            'type' => 'search_error',
            'message' => $lastError,
            'license_plate' => $licensePlate,
            'endpoint' => "/auftraege/by-license/$encodedLicense"
        ];
        error_log("P2 API: Fehler bei Suche: " . $lastError);
    }
    
    // Bereinige Output Buffer vor JSON-Ausgabe
    ob_clean();
    
    if ($customerData) {
        echo json_encode([
            'success' => true,
            'data' => $customerData
        ]);
    } else {
        // Detaillierte Fehlermeldung mit genauen Informationen
        $errorMsg = 'Keine Daten in P2 API gefunden';
        
        if ($lastError) {
            // Analysiere Fehlertyp für präzise Meldung
            if (strpos($lastError, 'timed out') !== false || strpos($lastError, 'CURLE_OPERATION_TIMEDOUT') !== false) {
                $errorMsg = "Timeout: Die P2 API hat nicht rechtzeitig geantwortet (>5 Sekunden). Die Datenbankabfrage dauert möglicherweise zu lange oder der Server ist überlastet.";
            } elseif (strpos($lastError, 'couldn\'t connect') !== false || strpos($lastError, 'CURLE_COULDNT_CONNECT') !== false) {
                $errorMsg = "Verbindungsfehler: Die Verbindung zum P2 API Server (100.82.207.72:5001) konnte nicht hergestellt werden. Bitte prüfen Sie die Netzwerkverbindung.";
            } elseif (strpos($lastError, 'HTTP') !== false) {
                $errorMsg = "API-Fehler: " . $lastError;
            } elseif (strpos($lastError, 'Keine Aufträge') !== false) {
                $errorMsg = $lastError;
            } else {
                $errorMsg = "Fehler bei der Suche: " . $lastError;
            }
        } else {
            $errorMsg .= " für Kennzeichen '$licensePlate' in den letzten 90 Tagen.";
        }
        
        error_log("P2 API: $errorMsg");
        
        echo json_encode([
            'success' => false,
            'error' => $errorMsg,
            'data' => null,
            'error_details' => $errorDetails ?? null
        ]);
    }
    
} catch (Exception $e) {
    ob_clean();
    error_log("P2 API Error: " . $e->getMessage());
    error_log("P2 API Stack Trace: " . $e->getTraceAsString());
    echo json_encode([
        'success' => false,
        'error' => 'Fehler bei der API-Anfrage: ' . $e->getMessage()
    ]);
} catch (Throwable $e) {
    ob_clean();
    error_log("P2 API Fatal Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Kritischer Fehler: ' . $e->getMessage()
    ]);
}

