<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Nur POST-Requests erlauben
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

require_once '../config/database.php';

class ResyCSVImport {
    private $conn;
    
    public function __construct() {
        $db = new Database();
        $this->conn = $db->getConnection();
    }
    
    public function importCSV($csvContent) {
        try {
            // CSV mit Semikolon als Trennzeichen parsen
            $lines = explode("\n", $csvContent);
            $header = str_getcsv($lines[0], ';');
            $importedCount = 0;
            $errors = [];
            $processedSets = [];
            
            // CSV-Header validieren (RESY-Format)
            $requiredColumns = ['Satznummer', 'Kennzeichen', 'Fahrgestellnr.', 'Haltername', 'Eingangsdatum', 'Autohaus'];
            foreach ($requiredColumns as $col) {
                if (!in_array($col, $header)) {
                    throw new Exception("Erforderliche Spalte '{$col}' fehlt in der CSV-Datei");
                }
            }
            
            // Daten gruppieren nach Satznummer (jede Position ist eine separate Zeile)
            $groupedData = [];
            
            for ($i = 1; $i < count($lines); $i++) {
                if (empty(trim($lines[$i]))) continue;
                
                $row = str_getcsv($lines[$i], ';');
                
                // Debug: Log Spalten-Anzahl
                error_log("CSV Import - Row $i: " . count($row) . " columns, Header: " . count($header));
                
                // Spalten-Anzahl anpassen falls nötig
                if (count($row) > count($header)) {
                    $row = array_slice($row, 0, count($header));
                } elseif (count($row) < count($header)) {
                    $row = array_pad($row, count($header), '');
                }
                
                $data = array_combine($header, $row);
                $satznummer = $this->normalizeStorageNumber($data['Satznummer']);
                
                // Debug: Log was gefunden wird
                error_log("CSV Import - Processing row $i: Satznummer=$satznummer, Position=" . ($data['Position'] ?? 'N/A'));
                
                if (!isset($groupedData[$satznummer])) {
                    $groupedData[$satznummer] = [
                        'base_data' => $data,
                        'tires' => []
                    ];
                }
                
                // Reifendaten für diese Position hinzufügen
                $position = $data['Position'] ?? 'UNKNOWN';
                $groupedData[$satznummer]['tires'][$position] = $data;
            }
            
            error_log("CSV Import - Grouped " . count($groupedData) . " tire sets");
            
            // Gruppierte Daten verarbeiten
            foreach ($groupedData as $satznummer => $setData) {
                try {
                    $this->importTireSetData($setData);
                    $importedCount++;
                    $processedSets[] = $satznummer;
                } catch (Exception $e) {
                    $errors[] = "Satz {$satznummer}: " . $e->getMessage();
                }
            }
            
            return [
                'success' => true,
                'imported_count' => $importedCount,
                'total_rows' => count($lines) - 1,
                'processed_sets' => $processedSets,
                'errors' => $errors,
                'message' => "{$importedCount} Reifensätze erfolgreich importiert"
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    private function importTireSetData($setData) {
        $baseData = $setData['base_data'];
        $tires = $setData['tires'];
        
        // Einlagerungsnummer normalisieren
        $storageNumber = $this->normalizeStorageNumber($baseData['Satznummer']);
        
        // Prüfen ob bereits vorhanden
        $stmt = $this->conn->prepare("SELECT id FROM resy_tire_data WHERE einlagerungsnummer = ?");
        $stmt->execute([$storageNumber]);
        if ($stmt->fetch()) {
            throw new Exception("Einlagerungsnummer {$storageNumber} bereits vorhanden");
        }
        
        // Autohaus-ID ermitteln
        $autohausId = $this->getAutohausId($baseData['Autohaus']);
        
        // Datum konvertieren
        $eingangsdatum = $this->convertDate($baseData['Eingangsdatum']);
        
        // Reifendaten aus gruppierten Daten extrahieren
        $tireData = $this->extractTireDataFromRESYFormat($tires);
        
        // In Datenbank speichern
        $stmt = $this->conn->prepare("
            INSERT INTO resy_tire_data (
                einlagerungsnummer, satznummer, eingangsdatum, serviceart, autohaus_id,
                kennzeichen, fin, haltername, fahrzeugtyp, baujahr, km_stand,
                reifen_vl, reifen_vr, reifen_hl, reifen_hr, images, bemerkung,
                created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        
        $stmt->execute([
            $storageNumber,
            $storageNumber,
            $eingangsdatum,
            $baseData['Serviceart'] ?? 'Lagern, Reinigen, Prüfen',
            $autohausId,
            $baseData['Kennzeichen'],
            $baseData['Fahrgestellnr.'],
            $baseData['Haltername'],
            $baseData['Fahrzeugbezeichnung'] ?? '',
            $baseData['Baujahr'] ?? '',
            $baseData['Km-Stand'] ?? '',
            json_encode($tireData['reifen_vl'] ?? []),
            json_encode($tireData['reifen_vr'] ?? []),
            json_encode($tireData['reifen_hl'] ?? []),
            json_encode($tireData['reifen_hr'] ?? []),
            json_encode($tireData['images'] ?? []),
            $baseData['Kunden-Bemerkung'] ?? ''
        ]);
    }
    
    private function importTireData($data) {
        // Einlagerungsnummer normalisieren
        $storageNumber = $this->normalizeStorageNumber($data['Satznummer']);
        
        // Prüfen ob bereits vorhanden
        $stmt = $this->conn->prepare("SELECT id FROM resy_tire_data WHERE einlagerungsnummer = ?");
        $stmt->execute([$storageNumber]);
        if ($stmt->fetch()) {
            throw new Exception("Einlagerungsnummer {$storageNumber} bereits vorhanden");
        }
        
        // Autohaus-ID ermitteln
        $autohausId = $this->getAutohausId($data['Autohaus']);
        
        // Datum konvertieren
        $eingangsdatum = $this->convertDate($data['Eingangsdatum']);
        
        // Reifendaten extrahieren (falls vorhanden)
        $tireData = $this->extractTireDataFromCSV($data);
        
        // In Datenbank speichern
        $stmt = $this->conn->prepare("
            INSERT INTO resy_tire_data (
                einlagerungsnummer, satznummer, eingangsdatum, serviceart, autohaus_id,
                kennzeichen, fin, haltername, fahrzeugtyp, baujahr, km_stand,
                reifen_vl, reifen_vr, reifen_hl, reifen_hr, images, bemerkung,
                created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        
        $stmt->execute([
            $storageNumber,
            $storageNumber,
            $eingangsdatum,
            $data['Serviceart'] ?? 'Lagern, Reinigen, Prüfen',
            $autohausId,
            $data['Kennzeichen'],
            $data['FIN'],
            $data['Haltername'],
            $data['Fzg. Bezeichnung'] ?? '',
            $data['Baujahr'] ?? '',
            $data['KM-Stand'] ?? '',
            json_encode($tireData['reifen_vl'] ?? []),
            json_encode($tireData['reifen_vr'] ?? []),
            json_encode($tireData['reifen_hl'] ?? []),
            json_encode($tireData['reifen_hr'] ?? []),
            json_encode($tireData['images'] ?? []),
            $data['Bemerkung'] ?? ''
        ]);
    }
    
    private function normalizeStorageNumber($number) {
        // Format: 9.716.669 -> 9.716.669
        return preg_replace('/[^0-9.]/', '', $number);
    }
    
    private function getAutohausId($autohausName) {
        $stmt = $this->conn->prepare("SELECT id FROM autohaus WHERE name LIKE ? OR short_name LIKE ?");
        $stmt->execute(["%{$autohausName}%", "%{$autohausName}%"]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['id'] : 1; // Default zu Kiel
    }
    
    private function convertDate($dateString) {
        // Verschiedene Datumsformate unterstützen
        $formats = ['d.m.Y', 'Y-m-d', 'd/m/Y', 'm/d/Y'];
        
        foreach ($formats as $format) {
            $date = DateTime::createFromFormat($format, $dateString);
            if ($date !== false) {
                return $date->format('Y-m-d');
            }
        }
        
        // Fallback: aktuelles Datum
        return date('Y-m-d');
    }
    
    private function extractTireDataFromRESYFormat($tires) {
        $tireData = [
            'reifen_vl' => [],
            'reifen_vr' => [],
            'reifen_hl' => [],
            'reifen_hr' => [],
            'images' => []
        ];
        
        // Reifendaten aus gruppierten RESY-Daten extrahieren
        $positions = ['VL', 'VR', 'HL', 'HR'];
        
        foreach ($positions as $pos) {
            $key = 'reifen_' . strtolower($pos);
            
            if (isset($tires[$pos])) {
                $tire = $tires[$pos];
                $tireData[$key] = [
                    'position' => $pos,
                    'marke' => $tire['Reifenhersteller'] ?? 'Unbekannt',
                    'modell' => $tire['Reifenmodell'] ?? 'Unbekannt',
                    'groesse' => $tire['Reifengröße-Scannerdaten'] ?? $tire['Reifengröße-Satz'] ?? 'Unbekannt',
                    'volltext' => ($tire['Reifenhersteller'] ?? 'Unbekannt') . ' ' . ($tire['Reifenmodell'] ?? 'Unbekannt'),
                    'art' => $tire['Reifenart'] ?? 'Unbekannt',
                    'dot' => $tire['DOT'] ?? '',
                    'profiltiefe' => [
                        'i' => $this->extractTreadDepth($tire, 'I'),
                        'm' => $this->extractTreadDepth($tire, 'M'),
                        'a' => $this->extractTreadDepth($tire, 'A')
                    ],
                    'luftdruck' => $tire['Luftdruck'] ? $tire['Luftdruck'] . ' bar' : '2.5 bar',
                    'empfehlung' => $tire['Empfehlung'] ?? 'keine Fehler',
                    'zustand' => $this->getTireCondition($tire['Empfehlung'] ?? 'keine Fehler'),
                    'reifenvorschaden' => $tire['Reifen-Vorschaden'] ?? 'N',
                    'felgenvorschaden' => $tire['Felgen-Vorschaden'] ?? 'N',
                    'notlauf' => $tire['Notlaufeigenschaften'] ?? 'N'
                ];
            } else {
                // Fallback falls Position nicht vorhanden
                $tireData[$key] = [
                    'position' => $pos,
                    'marke' => 'Unbekannt',
                    'modell' => 'Unbekannt',
                    'groesse' => 'Unbekannt',
                    'volltext' => 'Unbekannt',
                    'art' => 'Unbekannt',
                    'dot' => '',
                    'profiltiefe' => ['i' => '0', 'm' => '0', 'a' => '0'],
                    'luftdruck' => '2.5 bar',
                    'empfehlung' => 'keine Fehler',
                    'zustand' => 'Unbekannt'
                ];
            }
        }
        
        return $tireData;
    }
    
    private function extractTreadDepth($tire, $position) {
        // Profiltiefe aus verschiedenen Spalten extrahieren
        $profiltiefe = $tire['Profiltiefe'] ?? '';
        
        // Falls spezifische Position-Profiltiefen vorhanden sind
        $posKey = 'Profiltiefe ' . $position;
        if (isset($tire[$posKey])) {
            return str_replace(',', '.', str_replace(' mm', '', $tire[$posKey]));
        }
        
        // Fallback auf allgemeine Profiltiefe
        if ($profiltiefe) {
            return str_replace(',', '.', str_replace(' mm', '', $profiltiefe));
        }
        
        return '0';
    }
    
    private function extractTireDataFromCSV($data) {
        $tireData = [
            'reifen_vl' => [],
            'reifen_vr' => [],
            'reifen_hl' => [],
            'reifen_hr' => [],
            'images' => []
        ];
        
        // Reifendaten aus CSV-Spalten extrahieren (falls vorhanden)
        $positions = ['VL', 'VR', 'HL', 'HR'];
        
        foreach ($positions as $pos) {
            $key = 'reifen_' . strtolower($pos);
            $tireData[$key] = [
                'position' => $pos,
                'marke' => $data["Reifenhersteller {$pos}"] ?? 'Unbekannt',
                'modell' => $data["Reifenmodell {$pos}"] ?? 'Unbekannt',
                'groesse' => $data["Reifengröße {$pos}"] ?? 'Unbekannt',
                'volltext' => ($data["Reifenhersteller {$pos}"] ?? 'Unbekannt') . ' ' . ($data["Reifenmodell {$pos}"] ?? 'Unbekannt'),
                'art' => $data['Reifenart'] ?? 'Unbekannt',
                'dot' => $data["DOT {$pos}"] ?? '',
                'profiltiefe' => [
                    'i' => $data["Profil I {$pos}"] ?? '0',
                    'm' => $data["Profil M {$pos}"] ?? '0',
                    'a' => $data["Profil A {$pos}"] ?? '0'
                ],
                'luftdruck' => '2.5 bar',
                'empfehlung' => $data["Empfehlung {$pos}"] ?? 'keine Fehler',
                'zustand' => 'Gut'
            ];
        }
        
        return $tireData;
    }
    
    public function createSampleCSV() {
        // RESY-Format CSV mit Semikolon als Trennzeichen
        $csv = '"Satznummer";"Position";"Saison";"Fahrzeug";"Km-Stand";"Lagerplatz";"Ladestelle";"Anzahl Raeder";"Anzahl Radkappen";"Info 1";"Info 2";"Felgen-Vorschaden";"Reifen-Vorschaden";"Notlaufeigenschaften";"Reifengröße-Scannerdaten";"Reifengröße-Satz";"Reifenmodell";"Reifenart";"Reifenhersteller";"Serviceart";"Profil";"Luftdruck";"Run flat";"Herstellerkennung";"Radkappen";"Felgenbreite";"Einpresstiefe";"Teilenummer";"Felgenart";"Felgenhersteller";"DOT";"Prüfdatum";"Vorgangsnummer";"Platzordnung durch";"Platzordnung am";"Pruefung durch";"Pruefung am";"Reinigung durch";"Reinigung am";"Auswuchtung durch";"Auswuchtung  am";"Reifenfehler";"Empfehlung";"Empfehlung Felgen";"Eingangsdatum";"Ausgangsdatum";"letzte & Anmeldung durch ";"Wiedervorlage-Termin";"Kunden-Bemerkung";"Profiltiefe";"Fahrzeugbezeichnung";"Anrede";"Haltername";"Haltervorname";"Straße";"PLZ";"Ort";"E-Mail";"Fahrgestellnr.";"Kennzeichen";"Aktion";"Datum der Bestellung";"Bestell-Termin";"Bestellung erledigt am";"Autohaus";"Empfänger-Autohaus";"Handy-Nr";"RF-Hersteller Satz";"RF-Modell Satz";"Reifengroeße VL";"Reifenhersteller VL";"Reifenmodell VL";"Profiltiefe VL";"Reifengroeße HL";"Reifenhersteller HL";"Reifenmodell HL";"Profiltiefe HL";"Reifengroeße VR";"Reifenhersteller VR";"Reifenmodell VR";"Profiltiefe VR";"Reifengroeße HR";"Reifenhersteller HR";"Reifenmodell HR";"Profiltiefe HR";"&nbsp"' . "\n";
        
        // Beispiel-Daten für einen kompletten Reifensatz (4 Positionen)
        $csv .= '"9.716.991";"VL";"Sommer 2025";"503198";"";"05.0.015.007.3";"";"4";"";"";"";"N";"N";"N";"275/35 R 22V 104";"";"Scorpion Winter";"Winterreifen";"Pirelli";"Lagern, Reinigen, Prüfen";"6,9";"";"N";"";"N";"";"";"";"";"";"2722";"2025-03-27 10:19";"";"NICO WOLTER";"2025-08-26 08:02";"SYSTEM";"2025-03-27 10:19";"SASCHA MOTEJAT";"2025-08-05 08:24";"";"";"";"keine Fehler";"keine Fehler";"2025-03-27 10:04";"";"";"";"";"6,9";"";"";"Arte Agentur f. soz. Dienste UG";"";"";"";"";"";"YV1LFH5V2P1964399";"ECK-TE28E";"";"2025-03-27 14:28";"";"";"113 B & E / Kiel";"";"";"";"";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,9";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,8";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,5";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,7";""' . "\n";
        $csv .= '"9.716.991";"VR";"Sommer 2025";"503198";"";"05.0.015.007.3";"";"4";"";"";"";"N";"N";"N";"275/35 R 22V 104";"";"Scorpion Winter";"Winterreifen";"Pirelli";"Lagern, Reinigen, Prüfen";"6,5";"";"N";"";"N";"";"";"";"";"";"2722";"2025-03-27 10:19";"";"NICO WOLTER";"2025-08-26 08:02";"SYSTEM";"2025-03-27 10:19";"SASCHA MOTEJAT";"2025-08-05 08:24";"";"";"Schnittverletzung";"leichte Fehler";"keine Fehler";"2025-03-27 10:04";"";"";"";"Beschädigung am Reifen VR vor der Montage prüfen; OJ 11.08.2025";"6,5";"";"";"Arte Agentur f. soz. Dienste UG";"";"";"";"";"";"YV1LFH5V2P1964399";"ECK-TE28E";"";"2025-03-27 14:28";"";"";"113 B & E / Kiel";"";"";"";"";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,9";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,8";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,5";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,7";""' . "\n";
        $csv .= '"9.716.991";"HL";"Sommer 2025";"503198";"";"05.0.015.007.3";"";"4";"";"";"";"N";"N";"N";"275/35 R 22V 104";"";"Scorpion Winter";"Winterreifen";"Pirelli";"Lagern, Reinigen, Prüfen";"6,8";"";"N";"";"N";"";"";"";"";"";"2722";"2025-03-27 10:19";"";"NICO WOLTER";"2025-08-26 08:02";"SYSTEM";"2025-03-27 10:19";"SASCHA MOTEJAT";"2025-08-05 08:24";"";"";"";"keine Fehler";"keine Fehler";"2025-03-27 10:04";"";"";"";"";"6,8";"";"";"Arte Agentur f. soz. Dienste UG";"";"";"";"";"";"YV1LFH5V2P1964399";"ECK-TE28E";"";"2025-03-27 14:28";"";"";"113 B & E / Kiel";"";"";"";"";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,9";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,8";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,5";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,7";""' . "\n";
        $csv .= '"9.716.991";"HR";"Sommer 2025";"503198";"";"05.0.015.007.3";"";"4";"";"";"";"N";"N";"N";"275/35 R 22V 104";"";"Scorpion Winter";"Winterreifen";"Pirelli";"Lagern, Reinigen, Prüfen";"6,7";"";"N";"";"N";"";"";"";"";"";"2722";"2025-03-27 10:19";"";"NICO WOLTER";"2025-08-26 08:02";"SYSTEM";"2025-03-27 10:19";"SASCHA MOTEJAT";"2025-08-05 08:24";"";"";"";"keine Fehler";"keine Fehler";"2025-03-27 10:04";"";"";"";"";"6,7";"";"";"Arte Agentur f. soz. Dienste UG";"";"";"";"";"";"YV1LFH5V2P1964399";"ECK-TE28E";"";"2025-03-27 14:28";"";"";"113 B & E / Kiel";"";"";"";"";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,9";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,8";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,5";"275/35 R 22";"Pirelli";"Scorpion Winter";"6,7";""' . "\n";
        
        return $csv;
    }
}

// API-Handler
try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        $input = $_POST;
    }
    
    $action = $input['action'] ?? '';
    
    $importer = new ResyCSVImport();
    
    if ($action === 'import') {
        $csvContent = $input['csv_content'] ?? '';
        if (empty($csvContent)) {
            throw new Exception('CSV-Inhalt ist erforderlich');
        }
        
        $result = $importer->importCSV($csvContent);
        echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        
    } elseif ($action === 'sample') {
        $csv = $importer->createSampleCSV();
        echo json_encode([
            'success' => true,
            'csv_content' => $csv,
            'message' => 'Beispiel-CSV erstellt'
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        
    } else {
        throw new Exception('Ungültige Aktion');
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
}
?>
