<?php
/**
 * Helper-Funktion: Erstellt eine Einlagerungsnummer über RESY API
 * 
 * Diese Funktion erstellt eine Pickup Order in RESY und gibt die vergebene
 * Einlagerungsnummer (wheelSetNumber) zurück.
 * 
 * @param int $autohausId Autohaus-ID
 * @param array $data Kunden- und Fahrzeugdaten
 * @return array|null Array mit wheelSetNumber und Status, oder null bei Fehler
 */

require_once dirname(__DIR__) . '/config/database.php';
require_once __DIR__ . '/resy_rest_api_complete.php';

function createStorageNumberViaResy($autohausId, $data) {
    try {
        $api = new ResyRestAPIComplete($autohausId);
        
        // Validiere erforderliche Felder
        $required = ['licenceNo', 'VIN', 'appointment', 'location', 'tireCount', 'tireCapCount', 'tireType'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                throw new Exception("Erforderliches Feld fehlt: {$field}");
            }
        }
        
        // Erstelle Pickup Order
        $order = [
            'appointment' => $data['appointment'], // ISO 8601 Format
            'location' => $data['location'],
            'licenceNo' => $data['licenceNo'],
            'VIN' => $data['VIN'],
            'tireCount' => intval($data['tireCount']),
            'tireCapCount' => intval($data['tireCapCount']),
            'tireType' => strtoupper($data['tireType']), // SUMMER, WINTER, ALLSEASON
        ];
        
        // Optionale Felder
        if (isset($data['servicePackageIdent'])) {
            $order['servicePackageIdent'] = $data['servicePackageIdent'];
        }
        if (isset($data['offset'])) {
            $order['offset'] = intval($data['offset']);
        }
        if (isset($data['wheelSetNumber'])) {
            $order['wheelSetNumber'] = intval($data['wheelSetNumber']);
        }
        
        // Erstelle Order
        $result = $api->createPickupOrder($api->customerNo, [$order]);
        
        // Prüfe Antwort
        if (isset($result['pickupOrders']) && is_array($result['pickupOrders']) && count($result['pickupOrders']) > 0) {
            $createdOrder = $result['pickupOrders'][0];
            
            if (isset($createdOrder['wheelSetNumber'])) {
                return [
                    'success' => true,
                    'wheelSetNumber' => $createdOrder['wheelSetNumber'],
                    'status' => $createdOrder['status'] ?? null,
                    'statusCode' => $createdOrder['statusCode'] ?? null,
                    'appointment' => $createdOrder['appointment'] ?? null,
                    'order' => $createdOrder
                ];
            }
        }
        
        throw new Exception("Keine wheelSetNumber in der API-Antwort gefunden");
        
    } catch (Exception $e) {
        error_log("RESY createStorageNumber Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

// Beispiel-Verwendung:
if (php_sapi_name() === 'cli' && basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    echo "=== RESY Einlagerungsnummer erstellen - Beispiel ===\n\n";
    
    $autohausId = 1; // Beispiel-Autohaus
    
    $data = [
        'licenceNo' => 'HH-AB123',
        'VIN' => 'WBA12345678901234',
        'appointment' => date('c', strtotime('+7 days')), // ISO 8601
        'location' => 'Einfahrt',
        'tireCount' => 4,
        'tireCapCount' => 4,
        'tireType' => 'WINTER', // SUMMER, WINTER, ALLSEASON
        'servicePackageIdent' => 'L', // Optional
    ];
    
    $result = createStorageNumberViaResy($autohausId, $data);
    
    if ($result['success']) {
        echo "✅ Einlagerungsnummer erstellt: {$result['wheelSetNumber']}\n";
        echo "Status: {$result['status']}\n";
    } else {
        echo "❌ Fehler: {$result['error']}\n";
    }
}







