<?php
/**
 * RESY Tire Status API - Immer zuerst RESY prüfen, dann in DB speichern
 * 
 * Priorität:
 * 1. RESY-Login und Live-Daten abrufen
 * 2. Daten in Datenbank speichern
 * 3. Daten an Frontend zurückgeben
 */

// Debug-Logging aktivieren
function debugLog($message) {
    error_log("RESY_API_DEBUG: $message");
}

// Setze Content-Type
header('Content-Type: application/json; charset=utf-8');

// Debug-Logging starten
debugLog("API gestartet");

// ---------- Input-Verarbeitung ----------
$input = file_get_contents('php://input');
$requestData = json_decode($input, true) ?: [];

debugLog("JSON Input: " . ($input ?: 'null'));

// Parameter extrahieren
$storageNumber = $requestData['storage_number'] ?? $_POST['storage_number'] ?? $_GET['storage_number'] ?? '';
$appointmentDate = $requestData['appointment_date'] ?? $_POST['appointment_date'] ?? $_GET['appointment_date'] ?? '';
$eventName = $requestData['event_name'] ?? $_POST['event_name'] ?? $_GET['event_name'] ?? '';
$autohausId = $requestData['autohaus_id'] ?? $_POST['autohaus_id'] ?? $_GET['autohaus_id'] ?? 1;

debugLog("Storage Number: $storageNumber");
debugLog("Appointment Date: $appointmentDate");
debugLog("Event Name: $eventName");

if (!$storageNumber) {
    debugLog("ERROR: storage_number fehlt");
    echo json_encode(['success' => false, 'error' => 'storage_number fehlt']);
    exit;
}

// ---------- Datenbank-Verbindung ----------
try {
    require_once __DIR__ . '/../config/database.php';
    $db = new Database();
    $pdo = $db->getConnection();
    
    debugLog("Versuche DB-Verbindung für Autohaus-ID: $autohausId");
    
    // Autohaus-Daten laden
    $stmt = $pdo->prepare("SELECT resy_username, resy_password, resy_base_url FROM autohaus WHERE id = ?");
    $stmt->execute([$autohausId]);
    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$autohaus) {
        throw new RuntimeException("Autohaus mit ID $autohausId nicht gefunden");
    }
    
    debugLog("DB-Verbindung erfolgreich");
    debugLog("Autohaus-Daten: " . json_encode($autohaus));
    
    $RESY_USER = $autohaus['resy_username'];
    $RESY_PASS = $autohaus['resy_password'];
    $BASE_URL = $autohaus['resy_base_url'];
    
    debugLog("RESY_USER: $RESY_USER");
    debugLog("BASE_URL: $BASE_URL");
    
} catch (Exception $e) {
    debugLog("DB-Fehler: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Datenbankfehler: ' . $e->getMessage()]);
    exit;
}

// ---------- HTTP-Helper-Funktionen ----------
function http(array $opts = []): array {
    $UA = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
    $cookieFile = tempnam(sys_get_temp_dir(), 'res') . '.tmp';
    
    $base = [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT      => $UA,
        CURLOPT_CONNECTTIMEOUT => 30,
        CURLOPT_TIMEOUT        => 60,
        CURLOPT_COOKIEJAR      => $cookieFile,
        CURLOPT_COOKIEFILE     => $cookieFile,
        CURLOPT_HEADER         => false,
        CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
        CURLOPT_ENCODING       => 'gzip,deflate',
        CURLOPT_REFERER        => 'https://resy.dtm-reifen.de/',
    ];
    
    $ch = curl_init();
    curl_setopt_array($ch, $base + $opts);
    $body = curl_exec($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $info = curl_getinfo($ch);
    debugLog("cURL Info: HTTP $status, URL: " . $info['url']);
    curl_close($ch);
    return [$status, $body];
}

function absUrl(string $base, string $path): string {
    if (preg_match('~^https?://~i', $path)) return $path;
    
    if (strpos($path, './') === 0) {
        $path = substr($path, 2);
    }
    
    return rtrim($base, '/') . '/' . ltrim($path, '/');
}

function xp(string $html): DOMXPath {
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($html, LIBXML_NOWARNING|LIBXML_NOERROR|LIBXML_NONET|LIBXML_COMPACT);
    return new DOMXPath($doc);
}

function q1(DOMXPath $x, string $query, $context = null) {
    $nodes = $x->query($query, $context);
    return $nodes->length > 0 ? $nodes->item(0) : null;
}

function nodeText($node): string {
    return $node ? trim($node->textContent) : '';
}

function deFloat(string $str): float {
    $str = str_replace(',', '.', $str);
    $str = preg_replace('/[^0-9.-]/', '', $str);
    return floatval($str);
}

// ---------- Hauptfunktion ----------
$resySuccess = false;
$scrapedData = null;

// Versuche zuerst RESY-Login
try {
    debugLog("Versuche RESY-Login für Einlagerungsnummer: $storageNumber");

    // 1) Login
    debugLog("Versuche Login-Seite zu laden: " . absUrl($BASE_URL, '/index.php?m=login&a=login'));
    [$st, $loginHtml] = http([CURLOPT_URL => absUrl($BASE_URL, '/index.php?m=login&a=login'), CURLOPT_HTTPGET => true]);
    debugLog("Login-Seite Response: HTTP $st, Länge: " . strlen($loginHtml));
    
    if ($st >= 400) {
        debugLog("Login-Seite Fehler: HTTP $st");
        throw new RuntimeException("Login-Seite HTTP $st - Möglicherweise ist RESY nicht erreichbar");
    }

    $x = xp($loginHtml);
    $form = q1($x, "//form[.//input[@type='password']]");
    if (!$form) {
        debugLog("Login-Formular nicht gefunden");
        throw new RuntimeException("Login-Formular nicht gefunden");
    }

    $action = $form->getAttribute('action') ?: '/index.php?m=login';
    debugLog("Login-Formular gefunden: action=$action");
    
    $fields = [];
    
    // Alle Input-Felder sammeln
    foreach ($x->query(".//input", $form) as $in) {
        $type = strtolower($in->getAttribute('type'));
        $name = $in->getAttribute('name');
        if ($name === '' || in_array($type, ['submit','button','image'])) continue;
        $fields[$name] = $in->getAttribute('value');
    }
    
    debugLog("Gefundene Felder: " . json_encode($fields));
    
    // Username/Password-Feldnamen finden - RESY-spezifisch
    $userField = 'UserName';  // RESY verwendet 'UserName'
    $passField = 'Password';  // RESY verwendet 'Password'
    
    debugLog("RESY-Feldnamen: UserField=$userField, PassField=$passField");
    
    if (!$userField || !$passField) {
        debugLog("Login-Feldnamen nicht erkannt");
        throw new RuntimeException("Login-Feldnamen nicht erkannt");
    }

    $fields[$userField] = $RESY_USER;
    $fields[$passField] = $RESY_PASS;
    
    debugLog("Finale Login-Daten: " . json_encode($fields));

    [$st, $afterLogin] = http([
        CURLOPT_URL => absUrl($BASE_URL, $action),
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($fields),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
    ]);
    
    debugLog("Login-POST Response: HTTP $st, Länge: " . strlen($afterLogin));
    
    if ($st >= 400) {
        debugLog("Login-POST Fehler: HTTP $st");
        throw new RuntimeException("Login-POST HTTP $st");
    }
    
    // Prüfe ob Login erfolgreich war
    if (strpos($afterLogin, 'Anmeldename') !== false) {
        debugLog("Login fehlgeschlagen - Login-Formular noch vorhanden");
        throw new RuntimeException("Login fehlgeschlagen - Anmeldedaten möglicherweise falsch");
    }
    
    debugLog("Login erfolgreich - Weiterleitung zur Suche");

    // 2) Zur Suche navigieren
    debugLog("Versuche Suche-Seite zu laden: " . absUrl($BASE_URL, '/index.php?m=wheelset&a=search'));
    [$st, $searchHtml] = http([CURLOPT_URL => absUrl($BASE_URL, '/index.php?m=wheelset&a=search'), CURLOPT_HTTPGET => true]);
    debugLog("Suche-Seite Response: HTTP $st, Länge: " . strlen($searchHtml));
    
    if ($st >= 400) {
        debugLog("Suche-Seite Fehler: HTTP $st");
        debugLog("Response Content: " . substr($searchHtml, 0, 500));
        throw new RuntimeException("Suche-Seite HTTP $st - Login möglicherweise fehlgeschlagen");
    }

    // 3) Suche nach Einlagerungsnummer durchführen
    $x = xp($searchHtml);
    $searchForm = q1($x, "//form[.//input[@name='LGS_ID']]");
    
    if (!$searchForm) {
        debugLog("Suchformular nicht gefunden");
        throw new RuntimeException("Suchformular nicht gefunden");
    }
    
    $searchAction = $searchForm->getAttribute('action') ?: '/index.php?m=wheelset&a=search';
    debugLog("Suchformular gefunden: action=$searchAction");
    
    $searchFields = [];
    foreach ($x->query(".//input", $searchForm) as $in) {
        $name = $in->getAttribute('name');
        if ($name) $searchFields[$name] = $in->getAttribute('value');
    }
    
    $searchFields['LGS_ID'] = $storageNumber;
    $searchFields['KD_ID'] = '453'; // 113 B & E / Kiel (basierend auf der HTML)
    
    debugLog("Suchfelder: " . json_encode($searchFields));
    
    [$st, $searchResult] = http([
        CURLOPT_URL => absUrl($BASE_URL, $searchAction),
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($searchFields),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
    ]);
    
    debugLog("Suche-Response: HTTP $st, Länge: " . strlen($searchResult));
    
    if ($st >= 400) {
        debugLog("Suche-Fehler: HTTP $st");
        throw new RuntimeException("Suche HTTP $st");
    }
    
    // 4) Detail-Link finden
    $x = xp($searchResult);
    $detailLink = null;
    
    // Suche nach Link mit LGS_ID
    $links = $x->query("//a[contains(@href, 'LGS_ID')]");
    foreach ($links as $link) {
        $href = $link->getAttribute('href');
        if (strpos($href, $storageNumber) !== false) {
            $detailLink = $href;
            break;
        }
    }
    
    if (!$detailLink) {
        debugLog("Detail-Link nicht gefunden für Einlagerungsnummer: $storageNumber");
        throw new RuntimeException("Einlagerungsnummer $storageNumber nicht im RESY-System gefunden");
    }
    
    debugLog("Detail-Link gefunden: $detailLink");
    
    // 5) Detail-Seite laden
    [$st, $detailHtml] = http([CURLOPT_URL => absUrl($BASE_URL, $detailLink), CURLOPT_HTTPGET => true]);
    debugLog("Detail-Seite Response: HTTP $st, Länge: " . strlen($detailHtml));
    
    if ($st >= 400) {
        debugLog("Detail-Seite Fehler: HTTP $st");
        throw new RuntimeException("Detail-Seite HTTP $st");
    }
    
    // 6) Daten scrapen
    $x = xp($detailHtml);
    
    // Einlagerungsinformationen
    $einlagerung = [
        'satznummer'    => nodeText(q1($x, "//p[@id='WheelLGS_ID']/b")),
        'eingangsdatum' => nodeText(q1($x, "//p[@id='WheelEINGANG']/b")),
        'serviceart'    => nodeText(q1($x, "//p[@id='WheelSERV_ID']/b")),
        'ausgangsdatum' => nodeText(q1($x, "//p[@id='WheelAUSGANG']/b")),
        'autohaus'      => nodeText(q1($x, "//p[@id='WheelKD_ID']/b")),
    ];
    
    // Fahrzeugdaten
    $fahrzeug = [
        'kennzeichen' => nodeText(q1($x, "//p[@id='WheelKENNZEICHEN']/b")),
        'fin'         => nodeText(q1($x, "//p[@id='WheelFAHRGESTNR']/b")),
    ];
    
    // Reifendaten aus Tabelle
    $reifen = [];
    $rows = $x->query("//table[@id='WheelsetViewWheels']//tbody//tr");
    
    foreach ($rows as $row) {
        $cells = $x->query(".//td", $row);
        if ($cells->length < 14) continue;
        
        $pos = nodeText($cells->item(0));
        $size = nodeText($cells->item(1));
        $brand = nodeText($cells->item(5));
        $model = nodeText($cells->item(6));
        $dot = nodeText($cells->item(8));
        $type = nodeText($cells->item(11));
        $empfehlung = nodeText($cells->item(13));
        
        // Profiltiefen aus den Boxen
        $box = q1($x, "//div[@id='$pos']");
        if ($box) {
            $innen = deFloat(nodeText(q1($x, ".//div[@id='Profilinnen']//p/b", $box)));
            $mitte = deFloat(nodeText(q1($x, ".//div[@id='Profilmitte']//p/b", $box)));
            $aussen = deFloat(nodeText(q1($x, ".//div[@id='Profilaussen']//p/b", $box)));
            
            $avg = round(($innen + $mitte + $aussen) / 3, 1);
            
            // Zustand bestimmen
            $state = 'unbekannt';
            if ($avg >= 6.0) {
                $state = 'sehr_gut';
            } elseif ($avg >= 4.0) {
                $state = 'gut';
            } elseif ($avg >= 2.0) {
                $state = 'befriedigend';
            } else {
                $state = 'kritisch';
            }
            
            // Status aus Empfehlung
            $status = 'keine_fehler';
            if (strpos(strtolower($empfehlung), 'austausch') !== false) {
                $status = 'austausch_notwendig';
            } elseif (strpos(strtolower($empfehlung), 'prüf') !== false) {
                $status = 'pruefen';
            }
            
            $reifen[] = [
                'pos' => $pos,
                'size' => $size,
                'brand' => $brand,
                'model' => $model,
                'dot' => $dot,
                'type' => $type,
                'profile' => [
                    'innen' => $innen,
                    'mitte' => $mitte,
                    'aussen' => $aussen,
                ],
                'avg' => $avg,
                'state' => $state,
                'status' => $status,
                'empfehlung' => $empfehlung,
            ];
        }
    }
    
    // 7) Daten in Datenbank speichern
    debugLog("Speichere Daten in Datenbank...");
    
    // Prüfe ob Datensatz bereits existiert
    $stmt = $pdo->prepare("SELECT id FROM resy_tire_data WHERE einlagerungsnummer = ? OR satznummer = ?");
    $stmt->execute([$storageNumber, $storageNumber]);
    $existingId = $stmt->fetchColumn();
    
    // Bereite Reifendaten für JSON-Speicherung vor
    $reifenData = [];
    foreach ($reifen as $tire) {
        $pos = strtolower($tire['pos']);
        $reifenData["reifen_$pos"] = json_encode([
            'position' => $tire['pos'],
            'marke' => $tire['brand'],
            'modell' => $tire['model'],
            'groesse' => $tire['size'],
            'art' => $tire['type'],
            'dot' => $tire['dot'],
            'profiltiefe' => [
                'i' => $tire['profile']['innen'],
                'm' => $tire['profile']['mitte'],
                'a' => $tire['profile']['aussen'],
            ],
            'empfehlung' => $tire['empfehlung'],
            'zustand' => $tire['status'],
        ]);
    }
    
    if ($existingId) {
        // Update vorhandenen Datensatz
        $stmt = $pdo->prepare("
            UPDATE resy_tire_data SET 
                satznummer = ?, eingangsdatum = ?, serviceart = ?, 
                autohaus_id = ?, kennzeichen = ?, fin = ?, 
                reifen_vl = ?, reifen_vr = ?, reifen_hl = ?, reifen_hr = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $einlagerung['satznummer'],
            $einlagerung['eingangsdatum'],
            $einlagerung['serviceart'],
            $autohausId,
            $fahrzeug['kennzeichen'],
            $fahrzeug['fin'],
            $reifenData['reifen_vl'] ?? '',
            $reifenData['reifen_vr'] ?? '',
            $reifenData['reifen_hl'] ?? '',
            $reifenData['reifen_hr'] ?? '',
            $existingId
        ]);
        debugLog("Datensatz aktualisiert: ID $existingId");
    } else {
        // Neuen Datensatz erstellen
        $stmt = $pdo->prepare("
            INSERT INTO resy_tire_data 
            (einlagerungsnummer, satznummer, eingangsdatum, serviceart, autohaus_id, 
             kennzeichen, fin, reifen_vl, reifen_vr, reifen_hl, reifen_hr, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        $stmt->execute([
            $storageNumber,
            $einlagerung['satznummer'],
            $einlagerung['eingangsdatum'],
            $einlagerung['serviceart'],
            $autohausId,
            $fahrzeug['kennzeichen'],
            $fahrzeug['fin'],
            $reifenData['reifen_vl'] ?? '',
            $reifenData['reifen_vr'] ?? '',
            $reifenData['reifen_hl'] ?? '',
            $reifenData['reifen_hr'] ?? '',
        ]);
        debugLog("Neuer Datensatz erstellt");
    }
    
    // 8) Response erstellen
    $response = [
        'success' => true,
        'data' => [
            'einlagerung' => $einlagerung,
            'fahrzeug' => $fahrzeug,
            'reifen' => $reifen,
        ],
        'metadata' => [
            'timestamp' => date('Y-m-d H:i:s'),
            'storage_number' => $storageNumber,
            'source' => 'RESY_LIVE_DATA',
            'cached' => true,
        ]
    ];
    
    debugLog("RESY-Response erstellt: " . json_encode($response));
    echo json_encode($response);
    
} catch (Exception $e) {
    debugLog("Fehler: " . $e->getMessage());
    
    $response = [
        'success' => false,
        'error' => $e->getMessage(),
        'metadata' => [
            'timestamp' => date('Y-m-d H:i:s'),
            'storage_number' => $storageNumber,
            'source' => 'RESY_ERROR'
        ]
    ];
    
    echo json_encode($response);
}
?>
