<?php
/**
 * Backup System - Vollständiges Backup-System für Boxenstop
 * Erstellt Backups von Datenbank, Dateien und Konfigurationen
 */

class BackupSystem {
    private $backupDir;
    private $conn;
    
    public function __construct($database) {
        $this->conn = $database;
        $this->backupDir = __DIR__ . '/../backups/';
        
        // Erstelle Backup-Verzeichnis falls nicht vorhanden
        if (!is_dir($this->backupDir)) {
            mkdir($this->backupDir, 0755, true);
        }
    }
    
    /**
     * Erstellt ein vollständiges Backup
     */
    public function createFullBackup($description = '') {
        $timestamp = date('Y-m-d_H-i-s');
        $backupName = 'backup_' . $timestamp;
        $backupPath = $this->backupDir . $backupName;
        
        if (!is_dir($backupPath)) {
            mkdir($backupPath, 0755, true);
        }
        
        $results = [
            'database' => $this->backupDatabase($backupPath, $backupName),
            'files' => $this->backupFiles($backupPath),
            'config' => $this->backupConfig($backupPath)
        ];
        
        // Erstelle Backup-Manifest
        $manifest = [
            'timestamp' => date('Y-m-d H:i:s'),
            'description' => $description,
            'backup_name' => $backupName,
            'results' => $results,
            'php_version' => phpversion(),
            'server_info' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'
        ];
        
        file_put_contents($backupPath . '/manifest.json', json_encode($manifest, JSON_PRETTY_PRINT));
        
        return [
            'success' => true,
            'backup_path' => $backupPath,
            'backup_name' => $backupName,
            'manifest' => $manifest
        ];
    }
    
    /**
     * Erstellt ein Datenbank-Backup
     */
    private function backupDatabase($backupPath, $backupName) {
        try {
            $dbBackupPath = $backupPath . '/database';
            if (!is_dir($dbBackupPath)) {
                mkdir($dbBackupPath, 0755, true);
            }
            
            // Hole alle Tabellen
            $tables = [];
            $stmt = $this->conn->query("SHOW TABLES");
            while ($row = $stmt->fetch(PDO::FETCH_NUM)) {
                $tables[] = $row[0];
            }
            
            $sqlDump = "-- Database Backup: " . date('Y-m-d H:i:s') . "\n";
            $sqlDump .= "-- Backup Name: $backupName\n\n";
            
            foreach ($tables as $table) {
                // Tabellen-Struktur
                $createTable = $this->conn->query("SHOW CREATE TABLE `$table`");
                $createTableRow = $createTable->fetch(PDO::FETCH_NUM);
                $sqlDump .= "\n-- Table structure for `$table`\n";
                $sqlDump .= "DROP TABLE IF EXISTS `$table`;\n";
                $sqlDump .= $createTableRow[1] . ";\n\n";
                
                // Tabellen-Daten
                $sqlDump .= "-- Dumping data for table `$table`\n";
                $data = $this->conn->query("SELECT * FROM `$table`");
                $rows = $data->fetchAll(PDO::FETCH_ASSOC);
                
                if (count($rows) > 0) {
                    $columns = array_keys($rows[0]);
                    $sqlDump .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES\n";
                    
                    $values = [];
                    foreach ($rows as $row) {
                        $rowValues = [];
                        foreach ($row as $value) {
                            if ($value === null) {
                                $rowValues[] = 'NULL';
                            } else {
                                $rowValues[] = $this->conn->quote($value);
                            }
                        }
                        $values[] = "(" . implode(", ", $rowValues) . ")";
                    }
                    $sqlDump .= implode(",\n", $values) . ";\n\n";
                }
            }
            
            file_put_contents($dbBackupPath . '/database_backup.sql', $sqlDump);
            
            return [
                'success' => true,
                'tables_backed_up' => count($tables),
                'file' => $dbBackupPath . '/database_backup.sql',
                'size' => filesize($dbBackupPath . '/database_backup.sql')
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Erstellt ein Backup wichtiger Dateien
     */
    private function backupFiles($backupPath) {
        try {
            $filesBackupPath = $backupPath . '/files';
            if (!is_dir($filesBackupPath)) {
                mkdir($filesBackupPath, 0755, true);
            }
            
            $importantFiles = [
                '../config/database.php',
                '../config/database_optimized.php',
                '../.htaccess',
                '../composer.json',
                '../composer.lock'
            ];
            
            $backedUpFiles = [];
            foreach ($importantFiles as $file) {
                $fullPath = __DIR__ . '/' . $file;
                if (file_exists($fullPath)) {
                    $fileName = basename($file);
                    copy($fullPath, $filesBackupPath . '/' . $fileName);
                    $backedUpFiles[] = $fileName;
                }
            }
            
            return [
                'success' => true,
                'files_backed_up' => count($backedUpFiles),
                'files' => $backedUpFiles
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Erstellt ein Backup der Konfiguration
     */
    private function backupConfig($backupPath) {
        try {
            $config = [
                'php_version' => phpversion(),
                'php_extensions' => get_loaded_extensions(),
                'server_info' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'date' => date('Y-m-d H:i:s')
            ];
            
            file_put_contents($backupPath . '/config.json', json_encode($config, JSON_PRETTY_PRINT));
            
            return [
                'success' => true,
                'file' => $backupPath . '/config.json'
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Listet alle Backups auf
     */
    public function listBackups() {
        $backups = [];
        
        if (is_dir($this->backupDir)) {
            $dirs = scandir($this->backupDir);
            foreach ($dirs as $dir) {
                if ($dir !== '.' && $dir !== '..' && is_dir($this->backupDir . $dir)) {
                    $manifestPath = $this->backupDir . $dir . '/manifest.json';
                    if (file_exists($manifestPath)) {
                        $manifest = json_decode(file_get_contents($manifestPath), true);
                        $backups[] = [
                            'name' => $dir,
                            'timestamp' => $manifest['timestamp'] ?? 'Unknown',
                            'description' => $manifest['description'] ?? '',
                            'path' => $this->backupDir . $dir,
                            'manifest' => $manifest
                        ];
                    }
                }
            }
        }
        
        // Sortiere nach Timestamp (neueste zuerst)
        usort($backups, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });
        
        return $backups;
    }
    
    /**
     * Löscht ein Backup
     */
    public function deleteBackup($backupName) {
        $backupPath = $this->backupDir . $backupName;
        
        if (!is_dir($backupPath)) {
            return ['success' => false, 'error' => 'Backup nicht gefunden'];
        }
        
        // Lösche rekursiv
        $this->deleteDirectory($backupPath);
        
        return ['success' => true];
    }
    
    /**
     * Löscht ein Verzeichnis rekursiv
     */
    private function deleteDirectory($dir) {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
        }
        
        return rmdir($dir);
    }
    
    /**
     * Bereinigt alte Backups (behält nur die letzten N Backups)
     */
    public function cleanupOldBackups($keepCount = 10) {
        $backups = $this->listBackups();
        
        if (count($backups) <= $keepCount) {
            return ['success' => true, 'deleted' => 0];
        }
        
        $toDelete = array_slice($backups, $keepCount);
        $deleted = 0;
        
        foreach ($toDelete as $backup) {
            if ($this->deleteBackup($backup['name'])['success']) {
                $deleted++;
            }
        }
        
        return ['success' => true, 'deleted' => $deleted];
    }
    
    /**
     * Prüft ob Backup-Verzeichnis verfügbar ist
     */
    public function isAvailable() {
        return is_dir($this->backupDir) && is_writable($this->backupDir);
    }
}
?>
















