<?php
/**
 * admin_approve_user.php - VOLLSTÄNDIGE IMPLEMENTIERUNG
 * 
 * Diese Datei zeigt die komplette Implementierung für:
 * - Account-Freischaltung (approved)
 * - Account-Ablehnung (rejected)
 * - Account-Sperrung (suspended)
 * - Zukünftige Erweiterungen
 * 
 * MIT AUTOMATISCHEN PUSH-BENACHRICHTIGUNGEN
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

require_once 'config.php';
require_once 'send_push_notification.php'; // Push-Funktion einbinden

// ============================================================================
// HELPER FUNKTIONEN
// ============================================================================

/**
 * Prüft ob User Admin-Berechtigung hat
 */
function isAdminUser($adminUserId) {
    global $pdo;
    if (!$pdo || !$adminUserId) return false;
    
    $stmt = $pdo->prepare("
        SELECT id FROM admin_users 
        WHERE id = ? AND is_active = 1 AND role IN ('admin', 'super_admin')
        LIMIT 1
    ");
    $stmt->execute([$adminUserId]);
    return $stmt->fetch() !== false;
}

/**
 * Lädt User-Informationen
 */
function getUserInfo($userId) {
    global $pdo;
    if (!$pdo || !$userId) return null;
    
    $stmt = $pdo->prepare("
        SELECT id, email, first_name, last_name, verification_status, status, phone_number
        FROM users 
        WHERE id = ? AND is_active = 1
        LIMIT 1
    ");
    $stmt->execute([$userId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Formatiert Benutzername für Nachrichten
 */
function formatUserName($user) {
    $firstName = trim($user['first_name'] ?? '');
    $lastName = trim($user['last_name'] ?? '');
    $name = trim($firstName . ' ' . $lastName);
    
    if (empty($name)) {
        $name = $user['email'] ?? 'Benutzer';
    }
    
    return $name;
}

/**
 * Sendet Push-Benachrichtigung mit Typ-spezifischer Nachricht
 */
function sendUserStatusPushNotification($userId, $action, $reason = null, $adminUserId = null) {
    global $pdo;
    
    $user = getUserInfo($userId);
    if (!$user) {
        error_log("sendUserStatusPushNotification: User ID $userId nicht gefunden");
        return false;
    }
    
    $userName = formatUserName($user);
    
    // Nachrichten-Templates basierend auf Aktion
    $notifications = [
        'approved' => [
            'title' => 'Ihr Konto wurde freigeschaltet!',
            'body' => "Guten Tag $userName, Ihr Konto wurde erfolgreich verifiziert. Sie können nun Termine buchen.",
            'customData' => [
                'type' => 'account_approved',
                'user_id' => $userId,
                'verification_status' => 'approved',
                'action' => 'approved'
            ]
        ],
        'rejected' => [
            'title' => 'Ihr Konto wurde abgelehnt',
            'body' => "Guten Tag $userName, leider konnte Ihr Konto nicht freigeschaltet werden." . 
                     ($reason ? " Grund: $reason" : " Bitte kontaktieren Sie uns für weitere Informationen."),
            'customData' => [
                'type' => 'account_rejected',
                'user_id' => $userId,
                'verification_status' => 'rejected',
                'action' => 'rejected',
                'reason' => $reason
            ]
        ],
        'suspended' => [
            'title' => 'Ihr Konto wurde gesperrt',
            'body' => "Guten Tag $userName, Ihr Konto wurde vorübergehend gesperrt." . 
                     ($reason ? " Grund: $reason" : " Bitte kontaktieren Sie den Support."),
            'customData' => [
                'type' => 'account_suspended',
                'user_id' => $userId,
                'status' => 'suspended',
                'action' => 'suspended',
                'reason' => $reason
            ]
        ],
        'reactivated' => [
            'title' => 'Ihr Konto wurde reaktiviert',
            'body' => "Guten Tag $userName, Ihr Konto wurde wieder aktiviert. Sie können es nun wieder nutzen.",
            'customData' => [
                'type' => 'account_reactivated',
                'user_id' => $userId,
                'status' => 'approved',
                'action' => 'reactivated'
            ]
        ],
        'document_uploaded' => [
            'title' => 'Verifikationsdokument erhalten',
            'body' => "Guten Tag $userName, wir haben Ihr Verifikationsdokument erhalten und prüfen es nun.",
            'customData' => [
                'type' => 'document_uploaded',
                'user_id' => $userId,
                'action' => 'document_uploaded'
            ]
        ],
        'verification_requested' => [
            'title' => 'Verifikation erforderlich',
            'body' => "Guten Tag $userName, bitte laden Sie ein Verifikationsdokument hoch, um Ihr Konto zu aktivieren.",
            'customData' => [
                'type' => 'verification_requested',
                'user_id' => $userId,
                'action' => 'verification_requested'
            ]
        ]
    ];
    
    // Prüfe ob Aktion unterstützt wird
    if (!isset($notifications[$action])) {
        error_log("sendUserStatusPushNotification: Unbekannte Aktion '$action' für User ID $userId");
        return false;
    }
    
    $notification = $notifications[$action];
    
    // Grund zur Nachricht hinzufügen, falls vorhanden
    if ($reason && in_array($action, ['rejected', 'suspended'])) {
        $notification['body'] = str_replace(
            ' Bitte kontaktieren',
            "\n\nGrund: $reason\n\nBitte kontaktieren",
            $notification['body']
        );
    }
    
    // Push-Benachrichtigung senden
    $result = @sendPushNotification(
        $userId,
        $notification['title'],
        $notification['body'],
        $notification['customData']
    );
    
    if ($result) {
        error_log("Push-Benachrichtigung gesendet: User ID $userId, Aktion: $action");
    } else {
        error_log("Push-Benachrichtigung fehlgeschlagen: User ID $userId, Aktion: $action");
    }
    
    return $result;
}

// ============================================================================
// HAUPT-LOGIK
// ============================================================================

try {
    // Input lesen
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Ungültige Eingabe');
    }
    
    // Aktion bestimmen (approved, rejected, suspended, etc.)
    $action = $input['action'] ?? 'approved';
    
    // User ID prüfen
    $userId = (int)($input['user_id'] ?? 0);
    if (!$userId) {
        throw new Exception('user_id ist erforderlich');
    }
    
    // Admin-Berechtigung prüfen
    // Option 1: Session (wenn Admin-Dashboard verwendet)
    session_start();
    $adminUserId = $_SESSION['admin_user_id'] ?? null;
    
    // Option 2: Bearer Token (wenn API verwendet)
    if (!$adminUserId) {
        $headers = getallheaders();
        if ($headers) {
            foreach ($headers as $key => $value) {
                if (strtolower($key) === 'authorization' && stripos($value, 'bearer ') === 0) {
                    $token = trim(substr($value, 7));
                    // Prüfe ob Token Admin-Token ist
                    $stmt = $pdo->prepare("SELECT id FROM admin_users WHERE auth_token = ? AND is_active = 1 LIMIT 1");
                    $stmt->execute([$token]);
                    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
                    if ($admin) {
                        $adminUserId = (int)$admin['id'];
                    }
                    break;
                }
            }
        }
    }
    
    if (!$adminUserId || !isAdminUser($adminUserId)) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Keine Admin-Berechtigung']);
        exit;
    }
    
    // User-Informationen laden
    $user = getUserInfo($userId);
    if (!$user) {
        throw new Exception('Benutzer nicht gefunden');
    }
    
    // Grund für Aktion (optional)
    $reason = $input['reason'] ?? null;
    
    // Status-Änderung durchführen
    $success = false;
    $message = '';
    $pushSent = false;
    
    switch ($action) {
        case 'approved':
            // User freischalten
            $stmt = $pdo->prepare("
                UPDATE users 
                SET verification_status = 'approved',
                    status = 'approved',
                    approved_by = ?,
                    approved_at = NOW(),
                    rejected_by = NULL,
                    rejected_at = NULL,
                    rejection_reason = NULL,
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$adminUserId, $userId]);
            $success = true;
            $message = 'Benutzer erfolgreich freigeschaltet';
            $pushSent = sendUserStatusPushNotification($userId, 'approved', null, $adminUserId);
            break;
            
        case 'rejected':
            // User ablehnen
            if (empty($reason)) {
                $reason = 'Verifikationsdokument entspricht nicht den Anforderungen';
            }
            
            $stmt = $pdo->prepare("
                UPDATE users 
                SET verification_status = 'rejected',
                    status = 'rejected',
                    rejected_by = ?,
                    rejected_at = NOW(),
                    rejection_reason = ?,
                    approved_by = NULL,
                    approved_at = NULL,
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$adminUserId, $reason, $userId]);
            $success = true;
            $message = 'Benutzer wurde abgelehnt';
            $pushSent = sendUserStatusPushNotification($userId, 'rejected', $reason, $adminUserId);
            break;
            
        case 'suspended':
            // User sperren
            if (empty($reason)) {
                $reason = 'Verstoß gegen Nutzungsbedingungen';
            }
            
            $stmt = $pdo->prepare("
                UPDATE users 
                SET status = 'suspended',
                    is_active = 0,
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            $success = true;
            $message = 'Benutzer wurde gesperrt';
            $pushSent = sendUserStatusPushNotification($userId, 'suspended', $reason, $adminUserId);
            break;
            
        case 'reactivated':
            // User reaktivieren (nach Sperre)
            $stmt = $pdo->prepare("
                UPDATE users 
                SET status = 'approved',
                    is_active = 1,
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            $success = true;
            $message = 'Benutzer wurde reaktiviert';
            $pushSent = sendUserStatusPushNotification($userId, 'reactivated', null, $adminUserId);
            break;
            
        case 'request_verification':
            // Verifikation anfordern (z.B. wenn Dokument fehlt)
            $stmt = $pdo->prepare("
                UPDATE users 
                SET verification_status = 'pending',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            $success = true;
            $message = 'Verifikationsanfrage gesendet';
            $pushSent = sendUserStatusPushNotification($userId, 'verification_requested', $reason, $adminUserId);
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
    // Erfolgsmeldung mit Push-Status
    $responseMessage = $message;
    if ($pushSent) {
        $responseMessage .= ' und Benachrichtigung gesendet';
    } elseif ($action === 'approved' || $action === 'rejected') {
        // Bei wichtigen Aktionen warnen, wenn Push fehlgeschlagen
        error_log("WARNING: Push-Benachrichtigung konnte nicht gesendet werden für User ID $userId, Aktion: $action");
    }
    
    // Activity Log (optional, falls vorhanden)
    if (function_exists('logAdminActivity')) {
        logAdminActivity($adminUserId, $action, [
            'target_user_id' => $userId,
            'reason' => $reason
        ]);
    }
    
    echo json_encode([
        'success' => $success,
        'message' => $responseMessage,
        'action' => $action,
        'push_sent' => $pushSent,
        'user_id' => $userId
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    error_log("admin_approve_user.php ERROR: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
