<?php
/**
 * add_vehicle.php
 * Fügt ein Fahrzeug für einen Benutzer hinzu
 * Unterstützt sowohl Bearer Token als auch PHP Session
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Api-Key, X-Timestamp, X-Nonce, X-Signature');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Session starten
session_start();

// Log-Funktion (optional, kann später entfernt werden)
function logToFile($message, $data = null) {
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    $logFile = $logDir . '/add_vehicle_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message";
    if ($data !== null) {
        $logMessage .= "\n" . (is_string($data) ? $data : json_encode($data, JSON_PRETTY_PRINT));
    }
    $logMessage .= "\n" . str_repeat('-', 80) . "\n";
    @file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// Database connection helper - verwendet config.php wie upload_verification_document.php
function getDbConnection() {
    static $pdo = null;
    if ($pdo !== null) return $pdo;
    
    logToFile("getDbConnection: Versuche DB-Verbindung herzustellen");
    
    try {
        // Methode 1: Versuche config.php
        $configPath = __DIR__ . '/../config.php';
        logToFile("getDbConnection: Prüfe config.php", $configPath);
        
        if (file_exists($configPath)) {
            logToFile("getDbConnection: config.php gefunden, lade...");
            require_once $configPath;
            
            if (isset($pdo) && $pdo instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$pdo)");
                return $pdo;
            }
            
            if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$GLOBALS['pdo'])");
                $pdo = $GLOBALS['pdo'];
                return $pdo;
            }
        }
        
        // Methode 2: Versuche config/database.php
        $dbConfigPath = __DIR__ . '/../config/database.php';
        if (file_exists($dbConfigPath)) {
            require_once $dbConfigPath;
            if (class_exists('Database')) {
                try {
                    $db = new Database();
                    $pdo = $db->getConnection();
                    if ($pdo instanceof PDO) {
                        logToFile("getDbConnection SUCCESS", "Verbindung via Database-Klasse hergestellt");
                        return $pdo;
                    }
                } catch (Exception $e) {
                    logToFile("getDbConnection WARNING", "Database-Klasse Fehler: " . $e->getMessage());
                }
            }
        }
        
        // Methode 3: Fallback mit Constants
        if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER')) {
            $host = DB_HOST;
            $dbname = DB_NAME;
            $user = DB_USER;
            $pass = defined('DB_PASS') ? DB_PASS : '';
            try {
                $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $user, $pass);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                logToFile("getDbConnection SUCCESS", "Verbindung via Constants hergestellt");
                return $pdo;
            } catch (PDOException $e) {
                logToFile("getDbConnection ERROR", "Constants-Verbindung fehlgeschlagen: " . $e->getMessage());
            }
        }
        
        logToFile("getDbConnection ERROR", "Alle Verbindungsversuche fehlgeschlagen");
        return null;
        
    } catch (Exception $e) {
        logToFile("getDbConnection ERROR", "Exception: " . $e->getMessage());
        return null;
    }
}

// Bearer Token extrahieren
function getBearerToken(): ?string {
    $hdrs = getallheaders();
    if (!$hdrs) return null;
    foreach ($hdrs as $k => $v) {
        if (strtolower($k) === 'authorization' && stripos($v, 'bearer ') === 0) {
            return trim(substr($v, 7));
        }
    }
    return null;
}

// User-ID aus Token ermitteln
function getUserIdFromToken(string $token) {
    $pdo = getDbConnection();
    if (!$pdo) return null;
    try {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE auth_token = ? AND token_expires_at > NOW() AND is_active = 1 LIMIT 1");
        $stmt->execute([$token]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['id'] : null;
    } catch (Exception $e) {
        error_log("getUserIdFromToken error: " . $e->getMessage());
        return null;
    }
}

// User-ID ermitteln (aus Session oder Token)
$userId = null;

// 1. Prüfe Session (für Backend-Kompatibilität)
if (isset($_SESSION['user_id']) || (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'])) {
    $userId = (int)($_SESSION['user_id'] ?? 0);
    if ($userId > 0) {
        logToFile("User-ID aus Session", $userId);
    }
}

// 2. Prüfe Bearer Token (falls keine Session)
if (!$userId || $userId === 0) {
    $token = getBearerToken();
    if ($token) {
        logToFile("Bearer Token gefunden", substr($token, 0, 20) . '...');
        $userId = getUserIdFromToken($token);
        if ($userId) {
            // Setze Session für zukünftige Requests
            $_SESSION['user_id'] = $userId;
            $_SESSION['user_logged_in'] = true;
            logToFile("User-ID aus Token", $userId);
        } else {
            logToFile("Token ungültig", "Token konnte nicht zu User-ID aufgelöst werden");
        }
    }
}

// Kein User gefunden
if (!$userId || $userId === 0) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Nicht angemeldet']);
    exit;
}

// Input lesen
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Ungültige Eingabe']);
    exit;
}

logToFile("Add Vehicle Request", [
    'user_id' => $userId,
    'input' => $input
]);

// Erforderliche Felder prüfen
$requiredFields = ['license_plate'];
foreach ($requiredFields as $field) {
    if (!isset($input[$field]) || trim($input[$field]) === '') {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Feld '$field' ist erforderlich"]);
        exit;
    }
}

// Datenbankverbindung
$pdo = getDbConnection();
if (!$pdo) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Datenbankverbindung fehlgeschlagen']);
    exit;
}

try {
    // Prüfe, ob bereits ein Fahrzeug mit diesem Kennzeichen existiert
    $stmt = $pdo->prepare("SELECT id FROM user_vehicles WHERE user_id = ? AND license_plate = ? AND is_active = 1 LIMIT 1");
    $stmt->execute([$userId, trim($input['license_plate'])]);
    $existing = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existing) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Ein Fahrzeug mit diesem Kennzeichen existiert bereits']);
        exit;
    }
    
    // Prüfe, ob bereits ein Standard-Fahrzeug existiert
    $isDefault = isset($input['is_default']) && ($input['is_default'] === true || $input['is_default'] === '1' || $input['is_default'] === 1);
    
    // Wenn dieses Fahrzeug als Standard markiert werden soll, entferne Standard-Markierung von anderen
    if ($isDefault) {
        $stmt = $pdo->prepare("UPDATE user_vehicles SET is_default = 0 WHERE user_id = ? AND is_active = 1");
        $stmt->execute([$userId]);
    } else {
        // Prüfe, ob der User bereits ein Standard-Fahrzeug hat
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM user_vehicles WHERE user_id = ? AND is_default = 1 AND is_active = 1");
        $stmt->execute([$userId]);
        $hasDefault = $stmt->fetchColumn() > 0;
        
        // Wenn kein Standard-Fahrzeug existiert, mache dieses zum Standard
        if (!$hasDefault) {
            $isDefault = true;
        }
    }
    
    // Fahrzeug einfügen
    $notes = isset($input['vehicle_notes']) ? trim($input['vehicle_notes']) : (isset($input['notes']) ? trim($input['notes']) : '');
    $stmt = $pdo->prepare("
        INSERT INTO user_vehicles (user_id, license_plate, vehicle_brand, vehicle_model, vehicle_year, is_default, notes, is_active, created_at, updated_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, 1, NOW(), NOW())
    ");
    
    $result = $stmt->execute([
        $userId,
        trim($input['license_plate']),
        isset($input['vehicle_brand']) ? trim($input['vehicle_brand']) : null,
        isset($input['vehicle_model']) ? trim($input['vehicle_model']) : null,
        isset($input['vehicle_year']) && $input['vehicle_year'] !== '' ? (int)$input['vehicle_year'] : null,
        $isDefault ? 1 : 0,
        $notes
    ]);
    
    if ($result) {
        $vehicleId = (int)$pdo->lastInsertId();
        
        logToFile("SUCCESS: Fahrzeug hinzugefügt", [
            'user_id' => $userId,
            'vehicle_id' => $vehicleId,
            'license_plate' => trim($input['license_plate'])
        ]);
        
        // Fahrzeug-Daten zurückgeben
        $stmt = $pdo->prepare("SELECT * FROM user_vehicles WHERE id = ? AND user_id = ? AND is_active = 1 LIMIT 1");
        $stmt->execute([$vehicleId, $userId]);
        $vehicle = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Fahrzeug erfolgreich hinzugefügt',
            'vehicle_id' => $vehicleId,
            'vehicle' => [
                'id' => (int)$vehicle['id'],
                'user_id' => (int)$vehicle['user_id'],
                'license_plate' => $vehicle['license_plate'],
                'vehicle_brand' => $vehicle['vehicle_brand'],
                'vehicle_model' => $vehicle['vehicle_model'],
                'vehicle_year' => (int)$vehicle['vehicle_year'],
                'is_default' => (bool)$vehicle['is_default'],
                'notes' => $vehicle['notes'] ?? ''
            ]
        ]);
    } else {
        logToFile("ERROR: INSERT fehlgeschlagen", "Kein Fehler, aber auch kein Erfolg");
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Fahrzeug konnte nicht hinzugefügt werden']);
    }
    
} catch (PDOException $e) {
    logToFile("ERROR: PDO Exception", $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Datenbankfehler: ' . $e->getMessage()]);
} catch (Exception $e) {
    logToFile("ERROR: Exception", $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Fehler: ' . $e->getMessage()]);
}

