<?php
/**
 * Widget: Registrierungsquellen-Statistik
 * Zeigt Verteilung der Registrierungsquellen (Web, iOS, Android, Apple Sign-In)
 */

require_once __DIR__ . '/../../config/database.php';

$db = new Database();
$conn = $db->getConnection();

// Statistiken abrufen
$stmt = $conn->prepare("
    SELECT 
        registration_source,
        COUNT(*) as count,
        COUNT(CASE WHEN verification_status = 'approved' THEN 1 END) as approved_count,
        COUNT(CASE WHEN verification_status = 'pending' THEN 1 END) as pending_count,
        COUNT(CASE WHEN verification_status = 'rejected' THEN 1 END) as rejected_count
    FROM users
    GROUP BY registration_source
    ORDER BY count DESC
");
$stmt->execute();
$stats = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Icon-Mapping
$source_icons = [
    'web' => 'fas fa-globe',
    'ios' => 'fab fa-apple',
    'android' => 'fab fa-android',
    'apple_signin' => 'fab fa-apple',
    'unknown' => 'fas fa-question-circle'
];

// Name-Mapping
$source_names = [
    'web' => 'Web-Browser',
    'ios' => 'iOS App',
    'android' => 'Android App',
    'apple_signin' => 'Apple Sign-In',
    'unknown' => 'Unbekannt'
];

// Farbe-Mapping
$source_colors = [
    'web' => '#007bff',
    'ios' => '#000000',
    'android' => '#3DDC84',
    'apple_signin' => '#A8B1B9',
    'unknown' => '#6c757d'
];
?>

<div class="widget-card">
    <div class="widget-header">
        <h3>
            <i class="fas fa-chart-pie"></i>
            Registrierungsquellen
        </h3>
    </div>
    
    <div class="widget-content">
        <?php if (empty($stats)): ?>
            <div class="empty-state">
                <i class="fas fa-chart-line"></i>
                <p>Noch keine Daten verfügbar</p>
            </div>
        <?php else: ?>
            <!-- Registrierungsquellen Liste -->
            <div class="registration-sources">
                <?php foreach ($stats as $stat): ?>
                    <?php
                    $source = $stat['registration_source'];
                    $total = (int)$stat['count'];
                    $approved = (int)$stat['approved_count'];
                    $pending = (int)$stat['pending_count'];
                    $rejected = (int)$stat['rejected_count'];
                    
                    $approved_percent = $total > 0 ? round(($approved / $total) * 100) : 0;
                    $pending_percent = $total > 0 ? round(($pending / $total) * 100) : 0;
                    $rejected_percent = $total > 0 ? round(($rejected / $total) * 100) : 0;
                    ?>
                    <div class="source-item">
                        <div class="source-header">
                            <div class="source-info">
                                <i class="<?php echo $source_icons[$source] ?? $source_icons['unknown']; ?>" 
                                   style="color: <?php echo $source_colors[$source] ?? $source_colors['unknown']; ?>"></i>
                                <span class="source-name"><?php echo htmlspecialchars($source_names[$source] ?? $source); ?></span>
                            </div>
                            <span class="source-count"><?php echo $total; ?></span>
                        </div>
                        
                        <div class="source-status">
                            <div class="status-bar">
                                <?php if ($approved_percent > 0): ?>
                                    <div class="status-segment approved" style="width: <?php echo $approved_percent; ?>%;" 
                                         title="Freigeschaltet: <?php echo $approved; ?>"></div>
                                <?php endif; ?>
                                <?php if ($pending_percent > 0): ?>
                                    <div class="status-segment pending" style="width: <?php echo $pending_percent; ?>%;" 
                                         title="Wartend: <?php echo $pending; ?>"></div>
                                <?php endif; ?>
                                <?php if ($rejected_percent > 0): ?>
                                    <div class="status-segment rejected" style="width: <?php echo $rejected_percent; ?>%;" 
                                         title="Abgelehnt: <?php echo $rejected; ?>"></div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="status-legend">
                                <?php if ($approved > 0): ?>
                                    <span class="legend-item approved">
                                        <span class="legend-dot"></span>
                                        <?php echo $approved; ?> freigeschaltet
                                    </span>
                                <?php endif; ?>
                                <?php if ($pending > 0): ?>
                                    <span class="legend-item pending">
                                        <span class="legend-dot"></span>
                                        <?php echo $pending; ?> wartend
                                    </span>
                                <?php endif; ?>
                                <?php if ($rejected > 0): ?>
                                    <span class="legend-item rejected">
                                        <span class="legend-dot"></span>
                                        <?php echo $rejected; ?> abgelehnt
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Gesamt-Statistik -->
            <div class="total-stats">
                <?php
                $total_users = array_sum(array_column($stats, 'count'));
                $total_approved = array_sum(array_column($stats, 'approved_count'));
                $total_pending = array_sum(array_column($stats, 'pending_count'));
                $total_rejected = array_sum(array_column($stats, 'rejected_count'));
                ?>
                <div class="stat-row">
                    <span>Gesamt:</span>
                    <strong><?php echo $total_users; ?> User</strong>
                </div>
                <div class="stat-row">
                    <span>Freigeschaltet:</span>
                    <strong class="text-success"><?php echo $total_approved; ?></strong>
                </div>
                <div class="stat-row">
                    <span>Wartend:</span>
                    <strong class="text-warning"><?php echo $total_pending; ?></strong>
                </div>
                <div class="stat-row">
                    <span>Abgelehnt:</span>
                    <strong class="text-danger"><?php echo $total_rejected; ?></strong>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.registration-sources {
    margin-bottom: 1.5rem;
}

.source-item {
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 8px;
}

.source-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.75rem;
}

.source-info {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.source-name {
    font-weight: 600;
    color: #495057;
}

.source-count {
    background: #007bff;
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-weight: 600;
    font-size: 0.9rem;
}

.source-status {
    margin-top: 0.5rem;
}

.status-bar {
    display: flex;
    height: 20px;
    border-radius: 4px;
    overflow: hidden;
    margin-bottom: 0.5rem;
}

.status-segment {
    height: 100%;
    transition: all 0.3s;
}

.status-segment.approved {
    background: #28a745;
}

.status-segment.pending {
    background: #ffc107;
}

.status-segment.rejected {
    background: #dc3545;
}

.status-legend {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
    font-size: 0.85rem;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.legend-dot {
    width: 12px;
    height: 12px;
    border-radius: 50%;
}

.legend-item.approved .legend-dot {
    background: #28a745;
}

.legend-item.pending .legend-dot {
    background: #ffc107;
}

.legend-item.rejected .legend-dot {
    background: #dc3545;
}

.total-stats {
    background: #e9ecef;
    padding: 1rem;
    border-radius: 8px;
    margin-top: 1.5rem;
}

.stat-row {
    display: flex;
    justify-content: space-between;
    padding: 0.5rem 0;
    border-bottom: 1px solid #dee2e6;
}

.stat-row:last-child {
    border-bottom: none;
}

.stat-row span {
    color: #6c757d;
}

.stat-row strong {
    color: #495057;
}

.text-success {
    color: #28a745 !important;
}

.text-warning {
    color: #ffc107 !important;
}

.text-danger {
    color: #dc3545 !important;
}
</style>

