<?php
/**
 * WhatsApp Logs - Erweiterte Übersicht mit Analytics und Features
 */

session_start();

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once 'includes/helpers.php';

$db = new Database();
$conn = $db->getConnection();

// Get current user data with autohaus information
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;
$can_switch_autohaus = false;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.can_switch_autohaus, a.name as autohaus_name, u.username
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_role = $user_data['role'] ?? 'admin';
    $current_user_autohaus_id = $user_data['autohaus_id'];
    $current_user_autohaus_name = $user_data['autohaus_name'];
    $current_user_name = $user_data['username'] ?? 'Administrator';
    $can_switch_autohaus = $user_data['can_switch_autohaus'] == 1;
}

// Zeitraum-Filter
$timeRange = $_GET['time_range'] ?? '7d';
$filterStatus = $_GET['status'] ?? 'all';
$filterAutohaus = $_GET['autohaus_id'] ?? null;
$filterActionType = $_GET['action_type'] ?? 'all';
$filterSearch = $_GET['search'] ?? '';
$filterDateFrom = $_GET['date_from'] ?? null;
$filterDateTo = $_GET['date_to'] ?? null;
$page = intval($_GET['page'] ?? 1);
$perPage = intval($_GET['per_page'] ?? 50);
$offset = ($page - 1) * $perPage;

// Zeitraum-Berechnung
$dateFrom = null;
$dateTo = date('Y-m-d');
switch ($timeRange) {
    case 'today':
        $dateFrom = date('Y-m-d');
        break;
    case '7d':
        $dateFrom = date('Y-m-d', strtotime('-7 days'));
        break;
    case '30d':
        $dateFrom = date('Y-m-d', strtotime('-30 days'));
        break;
    case '90d':
        $dateFrom = date('Y-m-d', strtotime('-90 days'));
        break;
    case 'custom':
        if ($filterDateFrom) $dateFrom = $filterDateFrom;
        if ($filterDateTo) $dateTo = $filterDateTo;
        break;
}

// Lade Autohäuser für Filter
$autohausStmt = $conn->query("SELECT id, name FROM autohaus ORDER BY name");
$autohauser = $autohausStmt->fetchAll(PDO::FETCH_ASSOC);

// Baue Query
$where = ['1=1'];
$params = [];

if ($dateFrom) {
    $where[] = 'DATE(wl.created_at) >= ?';
    $params[] = $dateFrom;
}

if ($dateTo) {
    $where[] = 'DATE(wl.created_at) <= ?';
    $params[] = $dateTo;
}

if ($filterStatus !== 'all') {
    $where[] = 'wl.status = ?';
    $params[] = $filterStatus;
}

if ($filterAutohaus) {
    $where[] = 'wl.autohaus_id = ?';
    $params[] = $filterAutohaus;
}

if ($filterActionType !== 'all') {
    $where[] = 'wl.action_type = ?';
    $params[] = $filterActionType;
}

if ($filterSearch) {
    $where[] = '(wl.phone_number LIKE ? OR wl.customer_name LIKE ? OR wl.message LIKE ?)';
    $searchTerm = '%' . $filterSearch . '%';
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$whereClause = implode(' AND ', $where);

// Count
$countStmt = $conn->prepare("
    SELECT COUNT(*) 
    FROM whatsapp_logs wl
    WHERE $whereClause
");
$countStmt->execute($params);
$totalLogs = $countStmt->fetchColumn();

// Lade Logs
$logsStmt = $conn->prepare("
    SELECT 
        wl.*,
        a.name as autohaus_name,
        ap.customer_name as appointment_customer_name,
        ap.slot_date,
        ap.slot_time
    FROM whatsapp_logs wl
    LEFT JOIN autohaus a ON wl.autohaus_id = a.id
    LEFT JOIN appointments ap ON wl.appointment_id = ap.id
    WHERE $whereClause
    ORDER BY wl.created_at DESC
    LIMIT $perPage OFFSET $offset
");
$logsStmt->execute($params);
$logs = $logsStmt->fetchAll(PDO::FETCH_ASSOC);

// Erweiterte Statistiken
$statsQuery = "
    SELECT 
        status,
        COUNT(*) as count
    FROM whatsapp_logs wl
    WHERE " . ($dateFrom ? "DATE(wl.created_at) >= ?" : "1=1") . "
    " . ($dateTo ? "AND DATE(wl.created_at) <= ?" : "") . "
    GROUP BY status
";
$statsParams = [];
if ($dateFrom) $statsParams[] = $dateFrom;
if ($dateTo) $statsParams[] = $dateTo;
$statsStmt = $conn->prepare($statsQuery);
$statsStmt->execute($statsParams);
$stats = $statsStmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Erfolgsrate berechnen
$totalWithStatus = array_sum($stats);
$successRate = $totalWithStatus > 0 ? round(($stats['success'] ?? 0) / $totalWithStatus * 100, 1) : 0;

// Tägliche Statistiken für Chart (letzte 30 Tage)
$dailyStatsQuery = "
    SELECT 
        DATE(created_at) as date,
        status,
        COUNT(*) as count
    FROM whatsapp_logs
    WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
    GROUP BY DATE(created_at), status
    ORDER BY date ASC
";
$dailyStatsStmt = $conn->query($dailyStatsQuery);
$dailyStats = $dailyStatsStmt->fetchAll(PDO::FETCH_ASSOC);

// Top Fehler
$topErrorsQuery = "
    SELECT 
        error_message,
        COUNT(*) as count
    FROM whatsapp_logs
    WHERE status = 'error' AND error_message IS NOT NULL AND error_message != ''
    " . ($dateFrom ? "AND DATE(created_at) >= ?" : "") . "
    " . ($dateTo ? "AND DATE(created_at) <= ?" : "") . "
    GROUP BY error_message
    ORDER BY count DESC
    LIMIT 10
";
$topErrorsParams = [];
if ($dateFrom) $topErrorsParams[] = $dateFrom;
if ($dateTo) $topErrorsParams[] = $dateTo;
$topErrorsStmt = $conn->prepare($topErrorsQuery);
$topErrorsStmt->execute($topErrorsParams);
$topErrors = $topErrorsStmt->fetchAll(PDO::FETCH_ASSOC);

// Autohaus-Vergleich
$autohausStatsQuery = "
    SELECT 
        a.name as autohaus_name,
        COUNT(*) as total,
        SUM(CASE WHEN wl.status = 'success' THEN 1 ELSE 0 END) as success,
        SUM(CASE WHEN wl.status = 'error' THEN 1 ELSE 0 END) as error,
        SUM(CASE WHEN wl.status = 'pending' THEN 1 ELSE 0 END) as pending
    FROM whatsapp_logs wl
    LEFT JOIN autohaus a ON wl.autohaus_id = a.id
    WHERE " . ($dateFrom ? "DATE(wl.created_at) >= ?" : "1=1") . "
    " . ($dateTo ? "AND DATE(wl.created_at) <= ?" : "") . "
    GROUP BY a.id, a.name
    ORDER BY total DESC
    LIMIT 10
";
$autohausStatsParams = [];
if ($dateFrom) $autohausStatsParams[] = $dateFrom;
if ($dateTo) $autohausStatsParams[] = $dateTo;
$autohausStatsStmt = $conn->prepare($autohausStatsQuery);
$autohausStatsStmt->execute($autohausStatsParams);
$autohausStats = $autohausStatsStmt->fetchAll(PDO::FETCH_ASSOC);

// Aktionstypen für Filter
$actionTypesStmt = $conn->query("SELECT DISTINCT action_type FROM whatsapp_logs WHERE action_type IS NOT NULL ORDER BY action_type");
$actionTypes = $actionTypesStmt->fetchAll(PDO::FETCH_COLUMN);
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WhatsApp Analytics & Logs - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
    :root {
        --whatsapp-green: #25D366;
        --whatsapp-dark: #128C7E;
        --primary: #0056b3;
        --success: #10b981;
        --error: #ef4444;
        --warning: #f59e0b;
        --text-muted: #6c757d;
    }

    .logs-container {
        padding: 2rem;
        max-width: 1800px;
        margin: 0 auto;
    }

    .page-header {
        background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
        color: white;
        padding: 2rem;
        border-radius: 16px;
        margin-bottom: 2rem;
        box-shadow: 0 4px 20px rgba(37, 211, 102, 0.3);
    }

    .page-header h1 {
        margin: 0 0 0.5rem 0;
        font-size: 2.5rem;
        font-weight: 700;
        display: flex;
        align-items: center;
        gap: 1rem;
    }

    .page-header p {
        margin: 0;
        opacity: 0.95;
        font-size: 1.1rem;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        transition: transform 0.2s, box-shadow 0.2s;
        border-left: 4px solid var(--primary);
    }

    .stat-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }

    .stat-card.success {
        border-left-color: var(--success);
    }

    .stat-card.error {
        border-left-color: var(--error);
    }

    .stat-card.warning {
        border-left-color: var(--warning);
    }

    .stat-card.info {
        border-left-color: var(--primary);
    }

    .stat-card-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 0.5rem;
    }

    .stat-card h3 {
        margin: 0;
        font-size: 0.9rem;
        color: var(--text-muted);
        font-weight: 500;
    }

    .stat-card-icon {
        font-size: 1.5rem;
        opacity: 0.7;
    }

    .stat-card .value {
        font-size: 2.5rem;
        font-weight: 700;
        color: var(--primary);
        margin: 0.5rem 0;
    }

    .stat-card.success .value {
        color: var(--success);
    }

    .stat-card.error .value {
        color: var(--error);
    }

    .stat-card.warning .value {
        color: var(--warning);
    }

    .stat-card-footer {
        font-size: 0.85rem;
        color: var(--text-muted);
        margin-top: 0.5rem;
    }

    .charts-section {
        display: grid;
        grid-template-columns: 2fr 1fr;
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .chart-card {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }

    .chart-card h3 {
        margin: 0 0 1rem 0;
        font-size: 1.1rem;
        font-weight: 600;
        color: #333;
    }

    .filters-card {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        margin-bottom: 1.5rem;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }

    .filters-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 1rem;
    }

    .filters-header h3 {
        margin: 0;
        font-size: 1.1rem;
        font-weight: 600;
    }

    .quick-filters {
        display: flex;
        gap: 0.5rem;
        flex-wrap: wrap;
        margin-bottom: 1rem;
    }

    .quick-filter-btn {
        padding: 0.5rem 1rem;
        border: 2px solid #dee2e6;
        background: white;
        border-radius: 8px;
        cursor: pointer;
        font-size: 0.9rem;
        transition: all 0.2s;
        text-decoration: none;
        color: #333;
        display: inline-block;
    }

    .quick-filter-btn:hover {
        border-color: var(--primary);
        background: #f8f9fa;
    }

    .quick-filter-btn.active {
        background: var(--primary);
        color: white;
        border-color: var(--primary);
    }

    .filters-form {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
        gap: 1rem;
        align-items: end;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
    }

    .filter-group label {
        font-size: 0.85rem;
        font-weight: 500;
        margin-bottom: 0.5rem;
        color: #555;
    }

    .form-control {
        padding: 0.6rem;
        border: 1px solid #dee2e6;
        border-radius: 8px;
        font-size: 0.9rem;
    }

    .form-control:focus {
        outline: none;
        border-color: var(--primary);
        box-shadow: 0 0 0 3px rgba(0, 86, 179, 0.1);
    }

    .btn {
        padding: 0.6rem 1.2rem;
        border: none;
        border-radius: 8px;
        font-size: 0.9rem;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s;
        text-decoration: none;
        display: inline-block;
    }

    .btn-primary {
        background: var(--primary);
        color: white;
    }

    .btn-primary:hover {
        background: #004494;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-secondary:hover {
        background: #5a6268;
    }

    .btn-success {
        background: var(--success);
        color: white;
    }

    .btn-danger {
        background: var(--error);
        color: white;
    }

    .action-buttons {
        display: flex;
        gap: 0.5rem;
        margin-top: 1rem;
    }

    .logs-table-card {
        background: white;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 1.5rem;
    }

    .table-header {
        padding: 1.5rem;
        border-bottom: 1px solid #dee2e6;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .table-header h3 {
        margin: 0;
        font-size: 1.1rem;
        font-weight: 600;
    }

    .table-actions {
        display: flex;
        gap: 0.5rem;
    }

    .logs-table {
        overflow-x: auto;
    }

    .logs-table table {
        width: 100%;
        border-collapse: collapse;
    }

    .logs-table th {
        background: #f8f9fa;
        padding: 1rem;
        text-align: left;
        font-weight: 600;
        font-size: 0.85rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        color: #555;
        border-bottom: 2px solid #dee2e6;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    .logs-table th.sortable {
        cursor: pointer;
        user-select: none;
    }

    .logs-table th.sortable:hover {
        background: #e9ecef;
    }

    .logs-table td {
        padding: 1rem;
        border-bottom: 1px solid #f0f0f0;
        font-size: 0.9rem;
    }

    .logs-table tr:hover {
        background: #f8f9fa;
    }

    .logs-table tr.clickable {
        cursor: pointer;
    }

    .status-badge {
        display: inline-block;
        padding: 0.4rem 0.9rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-success {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        color: white;
    }

    .status-error {
        background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        color: white;
    }

    .status-pending {
        background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        color: white;
    }
    
    .error-row {
        background-color: rgba(239, 68, 68, 0.05);
        border-left: 3px solid #ef4444;
    }
    
    .error-row:hover {
        background-color: rgba(239, 68, 68, 0.1);
    }
    
    .pending-row {
        background-color: rgba(245, 158, 11, 0.05);
        border-left: 3px solid #f59e0b;
    }
    
    .pending-row:hover {
        background-color: rgba(245, 158, 11, 0.1);
    }
    
    .error-text {
        color: #ef4444;
        font-weight: 500;
    }

    .action-badge {
        display: inline-block;
        padding: 0.3rem 0.7rem;
        border-radius: 6px;
        font-size: 0.75rem;
        font-weight: 500;
        background: #e9ecef;
        color: #495057;
    }

    .text-truncate {
        max-width: 200px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }

    .pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 0.5rem;
        margin-top: 1.5rem;
        padding: 1rem;
    }

    .pagination a, .pagination span {
        padding: 0.5rem 1rem;
        border: 1px solid #dee2e6;
        border-radius: 8px;
        text-decoration: none;
        color: var(--primary);
        transition: all 0.2s;
    }

    .pagination a:hover {
        background: #f8f9fa;
        border-color: var(--primary);
    }

    .pagination .active {
        background: var(--primary);
        color: white;
        border-color: var(--primary);
    }

    .modal {
        display: none;
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        backdrop-filter: blur(4px);
    }

    .modal.active {
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .modal-content {
        background: white;
        border-radius: 16px;
        padding: 2rem;
        max-width: 800px;
        width: 90%;
        max-height: 90vh;
        overflow-y: auto;
        box-shadow: 0 10px 40px rgba(0,0,0,0.2);
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 1.5rem;
        padding-bottom: 1rem;
        border-bottom: 2px solid #dee2e6;
    }

    .modal-header h2 {
        margin: 0;
        font-size: 1.5rem;
    }

    .modal-close {
        background: none;
        border: none;
        font-size: 1.5rem;
        cursor: pointer;
        color: #6c757d;
        padding: 0.5rem;
    }

    .modal-close:hover {
        color: #333;
    }

    .detail-row {
        display: grid;
        grid-template-columns: 150px 1fr;
        gap: 1rem;
        padding: 0.75rem 0;
        border-bottom: 1px solid #f0f0f0;
    }

    .detail-label {
        font-weight: 600;
        color: #555;
    }

    .detail-value {
        color: #333;
    }

    .error-list {
        list-style: none;
        padding: 0;
        margin: 0;
    }

    .error-list li {
        padding: 0.5rem;
        margin-bottom: 0.5rem;
        background: #fee;
        border-left: 3px solid var(--error);
        border-radius: 4px;
    }

    .top-errors-card {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 1.5rem;
    }

    .top-errors-card h3 {
        margin: 0 0 1rem 0;
        font-size: 1.1rem;
        font-weight: 600;
    }

    .error-item {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 0.75rem;
        margin-bottom: 0.5rem;
        background: #f8f9fa;
        border-radius: 8px;
        border-left: 3px solid var(--error);
    }

    .error-text {
        flex: 1;
        font-size: 0.85rem;
        color: #555;
    }

    .error-count {
        font-weight: 600;
        color: var(--error);
        margin-left: 1rem;
    }

    @media (max-width: 1200px) {
        .charts-section {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .logs-container {
            padding: 1rem;
        }

        .stats-grid {
            grid-template-columns: 1fr;
        }

        .filters-form {
            grid-template-columns: 1fr;
        }
    }
    </style>
</head>
<body class="admin-body" data-user-id="<?php echo htmlspecialchars($current_user_id); ?>">
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        <main class="admin-main">
            <div class="logs-container">
                <!-- Page Header -->
                <div class="page-header">
                    <h1><i class="fab fa-whatsapp"></i> WhatsApp Analytics & Logs</h1>
                    <p>Umfassende Übersicht und Analyse aller WhatsApp-Versand-Logs mit erweiterten Features</p>
                </div>

                <!-- Statistiken -->
                <div class="stats-grid">
                    <div class="stat-card success">
                        <div class="stat-card-header">
                            <h3>Erfolgreich</h3>
                            <i class="fas fa-check-circle stat-card-icon"></i>
                        </div>
                        <div class="value"><?php echo number_format($stats['success'] ?? 0); ?></div>
                        <div class="stat-card-footer">Erfolgreich versendet</div>
                    </div>
                    <div class="stat-card error">
                        <div class="stat-card-header">
                            <h3>Fehler</h3>
                            <i class="fas fa-exclamation-circle stat-card-icon"></i>
                        </div>
                        <div class="value"><?php echo number_format($stats['error'] ?? 0); ?></div>
                        <div class="stat-card-footer">Fehlgeschlagene Versuche</div>
                    </div>
                    <div class="stat-card warning">
                        <div class="stat-card-header">
                            <h3>Ausstehend</h3>
                            <i class="fas fa-clock stat-card-icon"></i>
                        </div>
                        <div class="value"><?php echo number_format($stats['pending'] ?? 0); ?></div>
                        <div class="stat-card-footer">In Bearbeitung</div>
                    </div>
                    <div class="stat-card info">
                        <div class="stat-card-header">
                            <h3>Erfolgsrate</h3>
                            <i class="fas fa-chart-line stat-card-icon"></i>
                        </div>
                        <div class="value"><?php echo $successRate; ?>%</div>
                        <div class="stat-card-footer">Von <?php echo number_format($totalWithStatus); ?> Nachrichten</div>
                    </div>
                    <div class="stat-card info">
                        <div class="stat-card-header">
                            <h3>Gesamt (Filter)</h3>
                            <i class="fas fa-list stat-card-icon"></i>
                        </div>
                        <div class="value"><?php echo number_format($totalLogs); ?></div>
                        <div class="stat-card-footer">Nachrichten im Filter</div>
                    </div>
                </div>

                <!-- Charts Section -->
                <div class="charts-section">
                    <div class="chart-card">
                        <h3><i class="fas fa-chart-area"></i> Versand-Statistik (30 Tage)</h3>
                        <canvas id="dailyChart" height="80"></canvas>
                    </div>
                    <div class="chart-card">
                        <h3><i class="fas fa-chart-pie"></i> Status-Verteilung</h3>
                        <canvas id="statusChart" height="80"></canvas>
                    </div>
                </div>

                <!-- Top Fehler -->
                <?php if (!empty($topErrors)): ?>
                <div class="top-errors-card">
                    <h3><i class="fas fa-exclamation-triangle"></i> Top 10 Fehler</h3>
                    <?php foreach ($topErrors as $error): ?>
                    <div class="error-item">
                        <div class="error-text"><?php echo htmlspecialchars(substr($error['error_message'], 0, 150)); ?></div>
                        <div class="error-count"><?php echo $error['count']; ?>x</div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>

                <!-- Filter -->
                <div class="filters-card">
                    <div class="filters-header">
                        <h3><i class="fas fa-filter"></i> Filter & Suche</h3>
                        <div class="action-buttons">
                            <a href="whatsapp_logs.php" class="btn btn-secondary"><i class="fas fa-redo"></i> Zurücksetzen</a>
                            <button onclick="exportData('csv')" class="btn btn-success"><i class="fas fa-download"></i> CSV Export</button>
                        </div>
                    </div>
                    
                    <div class="quick-filters">
                        <a href="?time_range=today" class="quick-filter-btn <?php echo $timeRange === 'today' ? 'active' : ''; ?>">Heute</a>
                        <a href="?time_range=7d" class="quick-filter-btn <?php echo $timeRange === '7d' ? 'active' : ''; ?>">7 Tage</a>
                        <a href="?time_range=30d" class="quick-filter-btn <?php echo $timeRange === '30d' ? 'active' : ''; ?>">30 Tage</a>
                        <a href="?time_range=90d" class="quick-filter-btn <?php echo $timeRange === '90d' ? 'active' : ''; ?>">90 Tage</a>
                        <a href="?time_range=custom" class="quick-filter-btn <?php echo $timeRange === 'custom' ? 'active' : ''; ?>">Benutzerdefiniert</a>
                    </div>

                    <form method="GET" action="" class="filters-form">
                        <input type="hidden" name="time_range" value="<?php echo htmlspecialchars($timeRange); ?>">
                        
                        <div class="filter-group">
                            <label>Zeitraum von:</label>
                            <input type="date" name="date_from" value="<?php echo htmlspecialchars($filterDateFrom ?? ''); ?>" class="form-control">
                        </div>
                        
                        <div class="filter-group">
                            <label>Zeitraum bis:</label>
                            <input type="date" name="date_to" value="<?php echo htmlspecialchars($filterDateTo ?? ''); ?>" class="form-control">
                        </div>
                        
                        <div class="filter-group">
                            <label>Status:</label>
                            <select name="status" class="form-control">
                                <option value="all" <?php echo $filterStatus === 'all' ? 'selected' : ''; ?>>Alle</option>
                                <option value="success" <?php echo $filterStatus === 'success' ? 'selected' : ''; ?>>Erfolgreich</option>
                                <option value="error" <?php echo $filterStatus === 'error' ? 'selected' : ''; ?>>Fehler</option>
                                <option value="pending" <?php echo $filterStatus === 'pending' ? 'selected' : ''; ?>>Ausstehend</option>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>Autohaus:</label>
                            <select name="autohaus_id" class="form-control">
                                <option value="">Alle</option>
                                <?php foreach ($autohauser as $ah): ?>
                                    <option value="<?php echo $ah['id']; ?>" <?php echo $filterAutohaus == $ah['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($ah['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>Aktionstyp:</label>
                            <select name="action_type" class="form-control">
                                <option value="all">Alle</option>
                                <?php foreach ($actionTypes as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type); ?>" <?php echo $filterActionType === $type ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($type); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>Suche (Telefon, Name, Nachricht):</label>
                            <input type="text" name="search" value="<?php echo htmlspecialchars($filterSearch); ?>" class="form-control" placeholder="Suchen...">
                        </div>
                        
                        <div class="filter-group">
                            <label>Einträge pro Seite:</label>
                            <select name="per_page" class="form-control">
                                <option value="25" <?php echo $perPage == 25 ? 'selected' : ''; ?>>25</option>
                                <option value="50" <?php echo $perPage == 50 ? 'selected' : ''; ?>>50</option>
                                <option value="100" <?php echo $perPage == 100 ? 'selected' : ''; ?>>100</option>
                                <option value="200" <?php echo $perPage == 200 ? 'selected' : ''; ?>>200</option>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label>&nbsp;</label>
                            <button type="submit" class="btn btn-primary"><i class="fas fa-search"></i> Filter anwenden</button>
                        </div>
                    </form>
                </div>

                <!-- Logs Tabelle -->
                <div class="logs-table-card">
                    <div class="table-header">
                        <h3><i class="fas fa-table"></i> Logs (<?php echo number_format($totalLogs); ?> Einträge)</h3>
                        <div class="table-actions">
                            <?php if ($filterStatus === 'error' || $filterStatus === 'all'): ?>
                                <button onclick="retryFailed()" class="btn btn-success"><i class="fas fa-redo"></i> Fehlgeschlagene erneut versuchen</button>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="logs-table">
                        <table>
                            <thead>
                                <tr>
                                    <th class="sortable" onclick="sortTable('created_at')">Datum/Zeit <i class="fas fa-sort"></i></th>
                                    <th>Kunde</th>
                                    <th>Telefon</th>
                                    <th>Autohaus</th>
                                    <th>Aktion</th>
                                    <th class="sortable" onclick="sortTable('status')">Status <i class="fas fa-sort"></i></th>
                                    <th>Nachricht</th>
                                    <th>Fehler</th>
                                    <th>Aktionen</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($logs)): ?>
                                    <tr>
                                        <td colspan="9" style="text-align: center; padding: 3rem; color: var(--text-muted);">
                                            <i class="fas fa-inbox" style="font-size: 3rem; margin-bottom: 1rem; display: block; opacity: 0.3;"></i>
                                            Keine Logs gefunden
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($logs as $log): ?>
                                        <tr class="clickable <?php echo $log['status'] === 'error' ? 'error-row' : ($log['status'] === 'pending' ? 'pending-row' : ''); ?>" onclick="showLogDetail(<?php echo htmlspecialchars(json_encode($log)); ?>)">
                                            <td><?php echo date('d.m.Y H:i', strtotime($log['created_at'])); ?></td>
                                            <td><?php echo htmlspecialchars($log['customer_name'] ?? $log['appointment_customer_name'] ?? '-'); ?></td>
                                            <td><?php echo htmlspecialchars($log['phone_number']); ?></td>
                                            <td><?php echo htmlspecialchars($log['autohaus_name'] ?? '-'); ?></td>
                                            <td><span class="action-badge"><?php echo htmlspecialchars($log['action_type'] ?? '-'); ?></span></td>
                                            <td>
                                                <span class="status-badge status-<?php echo $log['status']; ?>">
                                                    <?php 
                                                    $statusLabels = [
                                                        'success' => 'Erfolgreich',
                                                        'error' => 'Fehler',
                                                        'pending' => 'Ausstehend'
                                                    ];
                                                    echo $statusLabels[$log['status']] ?? ucfirst($log['status']);
                                                    ?>
                                                </span>
                                            </td>
                                            <td class="text-truncate" title="<?php echo htmlspecialchars($log['message'] ?? '-'); ?>">
                                                <?php echo htmlspecialchars(substr($log['message'] ?? '-', 0, 80)); ?>
                                            </td>
                                            <td class="text-truncate <?php echo $log['error_message'] ? 'error-text' : ''; ?>" title="<?php echo htmlspecialchars($log['error_message'] ?? ''); ?>">
                                                <?php echo $log['error_message'] ? htmlspecialchars(substr($log['error_message'], 0, 80)) : '-'; ?>
                                            </td>
                                            <td onclick="event.stopPropagation();">
                                                <button onclick="showLogDetail(<?php echo htmlspecialchars(json_encode($log)); ?>)" class="btn btn-primary" style="padding: 0.3rem 0.6rem; font-size: 0.8rem;">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php if ($log['status'] === 'error'): ?>
                                                    <button onclick="retrySingle(<?php echo $log['id']; ?>)" class="btn btn-success" style="padding: 0.3rem 0.6rem; font-size: 0.8rem;">
                                                        <i class="fas fa-redo"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Pagination -->
                <?php if ($totalLogs > $perPage): ?>
                    <div class="pagination">
                        <?php
                        $totalPages = ceil($totalLogs / $perPage);
                        $queryParams = $_GET;
                        
                        if ($page > 1):
                            $queryParams['page'] = $page - 1;
                            echo '<a href="?' . http_build_query($queryParams) . '">&laquo; Zurück</a>';
                        endif;
                        
                        for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++):
                            $queryParams['page'] = $i;
                            if ($i == $page):
                                echo '<span class="active">' . $i . '</span>';
                            else:
                                echo '<a href="?' . http_build_query($queryParams) . '">' . $i . '</a>';
                            endif;
                        endfor;
                        
                        if ($page < $totalPages):
                            $queryParams['page'] = $page + 1;
                            echo '<a href="?' . http_build_query($queryParams) . '">Weiter &raquo;</a>';
                        endif;
                        ?>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <!-- Detail Modal -->
    <div id="logModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fab fa-whatsapp"></i> Log-Details</h2>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            <div id="modalBody"></div>
        </div>
    </div>

    <script>
    // Chart.js Konfiguration
    const dailyData = <?php echo json_encode($dailyStats); ?>;
    const statusData = <?php echo json_encode($stats); ?>;

    // Tägliche Statistik Chart
    const dailyCtx = document.getElementById('dailyChart');
    if (dailyCtx) {
        const dailyLabels = [];
        const successData = [];
        const errorData = [];
        const pendingData = [];
        
        // Daten für Chart vorbereiten
        const dailyMap = {};
        dailyData.forEach(item => {
            if (!dailyMap[item.date]) {
                dailyMap[item.date] = { success: 0, error: 0, pending: 0 };
            }
            dailyMap[item.date][item.status] = item.count;
        });
        
        Object.keys(dailyMap).sort().forEach(date => {
            dailyLabels.push(new Date(date).toLocaleDateString('de-DE', { day: '2-digit', month: '2-digit' }));
            successData.push(dailyMap[date].success || 0);
            errorData.push(dailyMap[date].error || 0);
            pendingData.push(dailyMap[date].pending || 0);
        });

        new Chart(dailyCtx, {
            type: 'line',
            data: {
                labels: dailyLabels,
                datasets: [
                    {
                        label: 'Erfolgreich',
                        data: successData,
                        borderColor: '#10b981',
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        tension: 0.4
                    },
                    {
                        label: 'Fehler',
                        data: errorData,
                        borderColor: '#ef4444',
                        backgroundColor: 'rgba(239, 68, 68, 0.1)',
                        tension: 0.4
                    },
                    {
                        label: 'Ausstehend',
                        data: pendingData,
                        borderColor: '#f59e0b',
                        backgroundColor: 'rgba(245, 158, 11, 0.1)',
                        tension: 0.4
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    legend: {
                        position: 'top',
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }

    // Status-Verteilung Chart
    const statusCtx = document.getElementById('statusChart');
    if (statusCtx) {
        new Chart(statusCtx, {
            type: 'doughnut',
            data: {
                labels: ['Erfolgreich', 'Fehler', 'Ausstehend'],
                datasets: [{
                    data: [
                        statusData.success || 0,
                        statusData.error || 0,
                        statusData.pending || 0
                    ],
                    backgroundColor: [
                        '#10b981',
                        '#ef4444',
                        '#f59e0b'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    legend: {
                        position: 'bottom',
                    }
                }
            }
        });
    }

    // Modal Funktionen
    function showLogDetail(log) {
        const modal = document.getElementById('logModal');
        const modalBody = document.getElementById('modalBody');
        
        const html = `
            <div class="detail-row">
                <div class="detail-label">ID:</div>
                <div class="detail-value">${log.id}</div>
            </div>
            <div class="detail-row">
                <div class="detail-label">Datum/Zeit:</div>
                <div class="detail-value">${new Date(log.created_at).toLocaleString('de-DE')}</div>
            </div>
            <div class="detail-row">
                <div class="detail-label">Kunde:</div>
                <div class="detail-value">${log.customer_name || log.appointment_customer_name || '-'}</div>
            </div>
            <div class="detail-row">
                <div class="detail-label">Telefon:</div>
                <div class="detail-value">${log.phone_number}</div>
            </div>
            <div class="detail-row">
                <div class="detail-label">Autohaus:</div>
                <div class="detail-value">${log.autohaus_name || '-'}</div>
            </div>
            <div class="detail-row">
                <div class="detail-label">Aktionstyp:</div>
                <div class="detail-value">${log.action_type || '-'}</div>
            </div>
            <div class="detail-row">
                <div class="detail-label">Status:</div>
                <div class="detail-value">
                    <span class="status-badge status-${log.status}">${log.status}</span>
                </div>
            </div>
            ${log.appointment_id ? `
            <div class="detail-row">
                <div class="detail-label">Termin:</div>
                <div class="detail-value">
                    ${log.slot_date ? new Date(log.slot_date).toLocaleDateString('de-DE') : '-'} 
                    ${log.slot_time || ''}
                </div>
            </div>
            ` : ''}
            <div class="detail-row">
                <div class="detail-label">Nachricht:</div>
                <div class="detail-value" style="white-space: pre-wrap;">${log.message || '-'}</div>
            </div>
            ${log.error_message ? `
            <div class="detail-row">
                <div class="detail-label">Fehler:</div>
                <div class="detail-value" style="color: #ef4444;">${log.error_message}</div>
            </div>
            ` : ''}
            ${log.response_data ? `
            <div class="detail-row">
                <div class="detail-label">Response:</div>
                <div class="detail-value" style="white-space: pre-wrap; font-family: monospace; font-size: 0.85rem;">${log.response_data}</div>
            </div>
            ` : ''}
        `;
        
        modalBody.innerHTML = html;
        modal.classList.add('active');
    }

    function closeModal() {
        document.getElementById('logModal').classList.remove('active');
    }

    // Modal schließen bei Klick außerhalb
    document.getElementById('logModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });

    // Export Funktion
    function exportData(format) {
        const params = new URLSearchParams(window.location.search);
        params.set('export', format);
        window.location.href = 'whatsapp_logs_export.php?' + params.toString();
    }

    // Retry Funktionen
    function retrySingle(logId) {
        if (!confirm('Möchten Sie diese Nachricht erneut versenden?')) return;
        
        fetch('api/retry_whatsapp.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ log_id: logId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Nachricht wird erneut versendet!');
                location.reload();
            } else {
                alert('Fehler: ' + (data.message || 'Unbekannter Fehler'));
            }
        })
        .catch(error => {
            alert('Fehler beim Senden der Anfrage');
        });
    }

    function retryFailed() {
        if (!confirm('Möchten Sie alle fehlgeschlagenen Nachrichten erneut versenden?')) return;
        
        const params = new URLSearchParams(window.location.search);
        fetch('api/retry_whatsapp_bulk.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                filters: {
                    status: 'error',
                    date_from: params.get('date_from'),
                    date_to: params.get('date_to'),
                    autohaus_id: params.get('autohaus_id')
                }
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert(`${data.count} Nachrichten werden erneut versendet!`);
                location.reload();
            } else {
                alert('Fehler: ' + (data.message || 'Unbekannter Fehler'));
            }
        })
        .catch(error => {
            alert('Fehler beim Senden der Anfrage');
        });
    }

    // Sortierung
    let sortColumn = '';
    let sortDirection = 'DESC';

    function sortTable(column) {
        if (sortColumn === column) {
            sortDirection = sortDirection === 'ASC' ? 'DESC' : 'ASC';
        } else {
            sortColumn = column;
            sortDirection = 'ASC';
        }
        
        const params = new URLSearchParams(window.location.search);
        params.set('sort', column);
        params.set('order', sortDirection);
        window.location.href = '?' + params.toString();
    }

    // Auto-Refresh alle 30 Sekunden (optional)
    // setInterval(() => {
    //     if (document.visibilityState === 'visible') {
    //         location.reload();
    //     }
    // }, 30000);
    </script>
</body>
</html>
