<?php
/**
 * 🔄 QUICK ACTIONS SCHEMA UPDATE
 * Erweitert die quick_actions Tabelle um neue Felder für Sharing und erweiterte Features
 */

require_once '../config/database_optimized.php';

try {
    $db = DatabaseOptimized::getInstance();
    $conn = $db->getConnection();
    
    echo "🔄 Aktualisiere quick_actions Tabelle...\n";
    
    // Check if table exists
    $stmt = $conn->query("SHOW TABLES LIKE 'quick_actions'");
    if ($stmt->rowCount() == 0) {
        echo "❌ Tabelle quick_actions existiert nicht. Führe zuerst setup_quick_actions.php aus.\n";
        exit;
    }
    
    // Check current structure
    $stmt = $conn->query("DESCRIBE quick_actions");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $updates = [];
    
    // Add missing columns
    if (!in_array('description', $columns)) {
        $updates[] = "ADD COLUMN description VARCHAR(255) DEFAULT '' AFTER title";
    }
    
    if (!in_array('text_color', $columns)) {
        $updates[] = "ADD COLUMN text_color VARCHAR(7) DEFAULT '#ffffff' AFTER background_color";
    }
    
    if (!in_array('target', $columns)) {
        $updates[] = "ADD COLUMN target VARCHAR(10) DEFAULT '_self' AFTER url";
    }
    
    if (!in_array('share_token', $columns)) {
        $updates[] = "ADD COLUMN share_token VARCHAR(32) DEFAULT NULL AFTER autohaus_id";
    }
    
    if (!in_array('is_public', $columns)) {
        $updates[] = "ADD COLUMN is_public TINYINT DEFAULT 0 AFTER share_token";
    }
    
    if (!in_array('created_by', $columns)) {
        $updates[] = "ADD COLUMN created_by VARCHAR(100) DEFAULT NULL AFTER is_public";
    }
    
    // Execute updates
    if (!empty($updates)) {
        $sql = "ALTER TABLE quick_actions " . implode(', ', $updates);
        $conn->exec($sql);
        echo "✅ Tabelle erfolgreich aktualisiert mit " . count($updates) . " neuen Spalten.\n";
    } else {
        echo "ℹ️ Tabelle ist bereits aktuell.\n";
    }
    
    // Add indexes for better performance
    $indexes = [
        "CREATE INDEX IF NOT EXISTS idx_share_token ON quick_actions(share_token)",
        "CREATE INDEX IF NOT EXISTS idx_is_public ON quick_actions(is_public)",
        "CREATE INDEX IF NOT EXISTS idx_created_by ON quick_actions(created_by)"
    ];
    
    foreach ($indexes as $index_sql) {
        try {
            $conn->exec($index_sql);
        } catch (PDOException $e) {
            // Index might already exist, ignore error
        }
    }
    
    echo "✅ Indizes aktualisiert.\n";
    
    // Update existing records with default values
    $update_sql = "
        UPDATE quick_actions 
        SET 
            description = COALESCE(description, ''),
            text_color = COALESCE(text_color, '#ffffff'),
            target = COALESCE(target, '_self'),
            is_public = COALESCE(is_public, 0)
        WHERE description IS NULL OR text_color IS NULL OR target IS NULL OR is_public IS NULL
    ";
    
    $stmt = $conn->prepare($update_sql);
    $stmt->execute();
    $affected = $stmt->rowCount();
    
    if ($affected > 0) {
        echo "✅ {$affected} bestehende Einträge aktualisiert.\n";
    }
    
    echo "🎉 Quick Actions Schema Update erfolgreich abgeschlossen!\n";
    
} catch (Exception $e) {
    echo "❌ Fehler beim Update: " . $e->getMessage() . "\n";
}
?>
