<?php
/**
 * Test-Script für alle WhatsApp-Aktionen
 * Testet confirmation, reminder_24h, reminder_48h, approved, rejected, cancellation, rescheduled
 */

session_start();
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../api/autohaus_whatsapp_notifications.php';

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    die('Bitte als Admin einloggen');
}

header('Content-Type: application/json');

$db = new Database();
$conn = $db->getConnection();

// Test-Parameter
$autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? 1;
$testPhone = $_GET['phone'] ?? $_POST['phone'] ?? '01782465658';
$action = $_GET['action'] ?? $_POST['action'] ?? 'all';

// Test-Daten
$testData = [
    'customer_name' => 'Test Kunde',
    'phone' => $testPhone,
    'email' => 'test@example.com',
    'license_plate' => 'TEST-123',
    'autohaus_id' => intval($autohausId),
];

$results = [];

try {
    // Prüfe und erstelle whatsapp_logs Tabelle falls nötig
    $whatsappNotifications = new AutohausWhatsAppNotifications($testData['autohaus_id']);
    $whatsappNotifications->ensureWhatsAppLogsTableAndColumnsExist();

    // Finde ein aktives Event für das Autohaus
    $stmt = $conn->prepare("
        SELECT id, name, event_date, start_time, end_time, slot_duration, max_monteure
        FROM events 
        WHERE autohaus_id = ? AND is_active = 1 AND event_date >= CURDATE()
        ORDER BY event_date ASC, start_time ASC
        LIMIT 1
    ");
    $stmt->execute([$testData['autohaus_id']]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$event) {
        echo json_encode(['success' => false, 'error' => 'Kein aktives Event für dieses Autohaus gefunden']);
        exit;
    }

    // Finde einen verfügbaren Slot
    $eventDate = $event['event_date'];
    $startTime = strtotime($event['start_time']);
    $endTime = strtotime($event['end_time']);
    $slotDuration = $event['slot_duration'] * 60;
    $maxMonteure = $event['max_monteure'];

    // Hole bereits gebuchte Slots
    $bookedSlotsStmt = $conn->prepare("
        SELECT slot_time, COUNT(*) as booked_count
        FROM appointments 
        WHERE event_id = ? AND slot_date = ? AND status IN ('booked', 'approved', 'vorbereitet')
        GROUP BY slot_time
    ");
    $bookedSlotsStmt->execute([$event['id'], $eventDate]);
    $bookedSlots = $bookedSlotsStmt->fetchAll(PDO::FETCH_KEY_PAIR);

    $availableSlot = null;
    $currentTime = $startTime;

    while ($currentTime < $endTime) {
        $slotTime = date('H:i:s', $currentTime);
        $bookedCount = $bookedSlots[$slotTime] ?? 0;

        if ($bookedCount < $maxMonteure) {
            $availableSlot = $slotTime;
            break;
        }
        $currentTime += $slotDuration;
    }
    
    if (!$availableSlot) {
        echo json_encode([
            'success' => false,
            'error' => 'Kein verfügbarer Slot für dieses Event gefunden'
        ]);
        exit;
    }

    // Definiere Test-Aktionen
    $testActions = [];
    
    if ($action === 'all' || $action === 'confirmation') {
        $testActions['confirmation'] = [
            'label' => 'Termin-Bestätigung',
            'description' => 'Wird sofort nach Buchung gesendet',
            'slot_date' => $eventDate,
            'slot_time' => $availableSlot
        ];
    }
    
    if ($action === 'all' || $action === 'reminder_24h') {
        // Termin für morgen
        $testActions['reminder_24h'] = [
            'label' => 'Erinnerung 24h',
            'description' => '24 Stunden vor Termin',
            'slot_date' => date('Y-m-d', strtotime('+1 day')),
            'slot_time' => $availableSlot
        ];
    }
    
    if ($action === 'all' || $action === 'reminder_48h') {
        // Termin für übermorgen
        $testActions['reminder_48h'] = [
            'label' => 'Erinnerung 48h',
            'description' => '48 Stunden vor Termin',
            'slot_date' => date('Y-m-d', strtotime('+2 days')),
            'slot_time' => $availableSlot
        ];
    }
    
    if ($action === 'all' || $action === 'approved') {
        $testActions['approved'] = [
            'label' => 'Termin genehmigt',
            'description' => 'Bei Genehmigung',
            'slot_date' => $eventDate,
            'slot_time' => $availableSlot,
            'status' => 'approved'
        ];
    }
    
    if ($action === 'all' || $action === 'rejected') {
        $testActions['rejected'] = [
            'label' => 'Termin abgelehnt',
            'description' => 'Bei Ablehnung',
            'slot_date' => $eventDate,
            'slot_time' => $availableSlot,
            'status' => 'rejected'
        ];
    }
    
    if ($action === 'all' || $action === 'cancellation') {
        $testActions['cancellation'] = [
            'label' => 'Termin storniert',
            'description' => 'Bei Stornierung',
            'slot_date' => $eventDate,
            'slot_time' => $availableSlot,
            'status' => 'cancelled'
        ];
    }
    
    if ($action === 'all' || $action === 'rescheduled') {
        $testActions['rescheduled'] = [
            'label' => 'Termin verschoben',
            'description' => 'Bei Verschiebung',
            'slot_date' => date('Y-m-d', strtotime('+3 days')),
            'slot_time' => $availableSlot,
            'status' => 'rescheduled'
        ];
    }
    
    if ($action === 'all' || $action === 'review_request') {
        // Termin für gestern (24 Stunden zurück) - simuliert abgeschlossenen Termin
        $testActions['review_request'] = [
            'label' => 'Bewertungsanfrage',
            'description' => '24 Stunden nach Termin',
            'slot_date' => date('Y-m-d', strtotime('-1 day')),
            'slot_time' => $availableSlot,
            'status' => 'completed'
        ];
    }

    // Führe Tests durch
    foreach ($testActions as $actionType => $actionConfig) {
        try {
            // Erstelle Test-Termin
            $stmt = $conn->prepare("
                INSERT INTO appointments (
                    event_id, autohaus_id, customer_name, license_plate, phone, email, 
                    slot_time, slot_date, status, source
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'admin_test')
            ");
            
            $appointmentStatus = $actionConfig['status'] ?? 'booked';
            
            $stmt->execute([
                $event['id'],
                $testData['autohaus_id'],
                $testData['customer_name'] . ' (' . $actionConfig['label'] . ')',
                $testData['license_plate'],
                $testData['phone'],
                $testData['email'],
                $actionConfig['slot_time'],
                $actionConfig['slot_date'],
                $appointmentStatus
            ]);
            
            $appointmentId = $conn->lastInsertId();
            
            // Sende WhatsApp-Nachricht
            $whatsapp = new AutohausWhatsAppNotifications($testData['autohaus_id']);
            $result = $whatsapp->sendAppointmentNotification($appointmentId, $actionType);
            
            // Speichere Ergebnis
            $results[$actionType] = [
                'action' => $actionType,
                'label' => $actionConfig['label'],
                'appointment_id' => $appointmentId,
                'success' => $result['success'],
                'message' => $result['message'] ?? null,
                'error' => $result['error'] ?? null,
                'slot_date' => $actionConfig['slot_date'],
                'slot_time' => $actionConfig['slot_time']
            ];
            
            // Lösche Test-Termin nach kurzer Verzögerung (damit Logs gespeichert werden)
            if ($actionType !== 'cancellation') {
                // Für cancellation: Termin wird gelöscht, daher nicht löschen
                usleep(500000); // 0.5 Sekunden warten
                $deleteStmt = $conn->prepare("DELETE FROM appointments WHERE id = ? AND source = 'admin_test'");
                $deleteStmt->execute([$appointmentId]);
            }
            
        } catch (Exception $e) {
            $results[$actionType] = [
                'action' => $actionType,
                'label' => $actionConfig['label'] ?? $actionType,
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    // Hole Logs für die Tests
    $logStmt = $conn->prepare("
        SELECT * FROM whatsapp_logs 
        WHERE autohaus_id = ? AND phone_number = ? 
        AND created_at >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)
        ORDER BY created_at DESC
    ");
    $logStmt->execute([$testData['autohaus_id'], $testData['phone']]);
    $logs = $logStmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'test_phone' => $testPhone,
        'autohaus_id' => $testData['autohaus_id'],
        'results' => $results,
        'logs' => $logs,
        'summary' => [
            'total_tested' => count($results),
            'successful' => count(array_filter($results, fn($r) => $r['success'] ?? false)),
            'failed' => count(array_filter($results, fn($r) => !($r['success'] ?? false)))
        ]
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ], JSON_PRETTY_PRINT);
}
?>

