<?php
/**
 * Test-Script für HelloMateo API-Verbindung
 * Direkter Test ohne Datenbank
 */

header('Content-Type: application/json');
require_once '../config/database.php';

session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht angemeldet']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$apiKey = $input['api_key'] ?? '';
$autohausId = $input['autohaus_id'] ?? null;

if (empty($apiKey)) {
    echo json_encode([
        'success' => false,
        'error' => 'Kein API-Key übergeben'
    ]);
    exit;
}

// API-Key bereinigen - entferne alle Whitespace-Zeichen
$apiKey = trim($apiKey);
$apiKey = preg_replace('/\s+/', '', $apiKey); // Entferne alle Leerzeichen, Tabs, Newlines

// Teste verschiedene Endpunkte
$testEndpoints = [
    [
        'name' => 'Templates Liste',
        'url' => 'https://integration.getmateo.com/api/v1/templates',
        'method' => 'GET'
    ],
    [
        'name' => 'Health/Status',
        'url' => 'https://integration.getmateo.com/api/v1/health',
        'method' => 'GET'
    ],
    [
        'name' => 'Root Endpoint',
        'url' => 'https://integration.getmateo.com/api/v1/',
        'method' => 'GET'
    ]
];

$results = [];

foreach ($testEndpoints as $endpoint) {
    $ch = curl_init();
    
    // WICHTIG: Authorization Header exakt formatieren - KEINE zusätzlichen Leerzeichen!
    $authHeader = 'Bearer ' . $apiKey;
    
    curl_setopt($ch, CURLOPT_URL, $endpoint['url']);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: ' . $authHeader,
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    
    error_log("test_hellomateo_api.php: Teste {$endpoint['name']} - URL: {$endpoint['url']}, API-Key Länge: " . strlen($apiKey));
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    $curlInfo = curl_getinfo($ch);
    curl_close($ch);
    
    $result = [
        'endpoint' => $endpoint['name'],
        'url' => $endpoint['url'],
        'http_code' => $httpCode,
        'curl_error' => $curlError ?: null,
        'response_length' => strlen($response),
        'response_preview' => substr($response, 0, 500)
    ];
    
    if ($response) {
        $decoded = json_decode($response, true);
        if ($decoded) {
            $result['response_json'] = $decoded;
            if (isset($decoded['message'])) {
                $result['api_message'] = $decoded['message'];
            }
            if (isset($decoded['hint'])) {
                $result['api_hint'] = $decoded['hint'];
            }
            if (isset($decoded['code'])) {
                $result['api_code'] = $decoded['code'];
            }
        } else {
            $result['response_raw'] = $response;
        }
    }
    
    $results[] = $result;
    
    // Bei Erfolg: Stoppe weitere Tests
    if ($httpCode === 200) {
        break;
    }
}

$success = count(array_filter($results, function($r) { return $r['http_code'] === 200; })) > 0;
$httpCode = $results[0]['http_code'] ?? 0;

// Speichere Test-Status in Datenbank (wenn autohaus_id übergeben wurde und Test erfolgreich)
if ($success && $autohausId && !empty($autohausId)) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Prüfe ob Spalte existiert, sonst erstelle sie
        $stmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_api_key_test_successful'");
        if ($stmt->rowCount() === 0) {
            $conn->exec("ALTER TABLE autohaus ADD COLUMN whatsapp_api_key_test_successful BOOLEAN DEFAULT FALSE");
        }
        
        // Speichere erfolgreichen Test
        $stmt = $conn->prepare("UPDATE autohaus SET whatsapp_api_key_test_successful = 1 WHERE id = ?");
        $stmt->execute([$autohausId]);
        error_log("test_hellomateo_api.php: Test-Status erfolgreich gespeichert für Autohaus ID: " . $autohausId);
    } catch (Exception $e) {
        error_log("test_hellomateo_api.php: Fehler beim Speichern des Test-Status: " . $e->getMessage());
    }
} elseif (!$success && $autohausId && !empty($autohausId)) {
    // Bei Fehler: Setze Test-Status auf false
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $stmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_api_key_test_successful'");
        if ($stmt->rowCount() > 0) {
            $stmt = $conn->prepare("UPDATE autohaus SET whatsapp_api_key_test_successful = 0 WHERE id = ?");
            $stmt->execute([$autohausId]);
        }
    } catch (Exception $e) {
        // Ignoriere Fehler
    }
}

// API-Key Analyse (ohne den kompletten Key zu zeigen)
$apiKeyInfo = [
    'length' => strlen($apiKey),
    'starts_with' => substr($apiKey, 0, 20),
    'ends_with' => substr($apiKey, -20),
    'has_spaces' => strpos($apiKey, ' ') !== false,
    'has_newlines' => strpos($apiKey, "\n") !== false || strpos($apiKey, "\r") !== false,
    'is_jwt' => preg_match('/^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+$/', $apiKey)
];

echo json_encode([
    'success' => $success,
    'api_key_info' => $apiKeyInfo,
    'test_results' => $results,
    'autohaus_id' => $autohausId,
    'recommendation' => $httpCode === 401 || $httpCode === 403 
        ? 'Der API-Key ist ungültig. Bitte überprüfen Sie den API-Key in Ihrem HelloMateo-Dashboard.'
        : ($httpCode === 200 
            ? 'API-Verbindung erfolgreich!' 
            : 'Unbekannter Fehler. Bitte überprüfen Sie die API-Dokumentation.')
], JSON_PRETTY_PRINT);

?>

