<?php
/**
 * Test-Script: Terminbuchung mit WhatsApp-Bestätigung
 * Testet ob ein Termin korrekt gebucht wird und eine WhatsApp-Nachricht gesendet wird
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../api/autohaus_whatsapp_notifications.php';

header('Content-Type: application/json');

// Test-Daten
$testData = [
    'customer_name' => 'Uludag',
    'phone' => '01782465658',
    'email' => 'test@example.com',
    'license_plate' => 'TEST-123',
    'autohaus_id' => 1, // Standard Autohaus
];

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Erstelle Log-Tabelle falls nicht vorhanden oder füge fehlende Spalten hinzu
    try {
        // Prüfe ob Tabelle existiert
        $tableCheck = $conn->query("SHOW TABLES LIKE 'whatsapp_logs'");
        if ($tableCheck->rowCount() === 0) {
            // Tabelle existiert nicht - erstelle sie
            $conn->exec("
                CREATE TABLE whatsapp_logs (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    appointment_id INT,
                    autohaus_id INT,
                    phone_number VARCHAR(20),
                    customer_name VARCHAR(255),
                    action_type VARCHAR(50),
                    status VARCHAR(20),
                    message TEXT,
                    error_message TEXT,
                    created_at DATETIME,
                    updated_at DATETIME,
                    INDEX idx_appointment (appointment_id),
                    INDEX idx_autohaus (autohaus_id),
                    INDEX idx_status (status),
                    INDEX idx_created (created_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
            ");
        } else {
            // Tabelle existiert - füge fehlende Spalten hinzu
            $columns = $conn->query("SHOW COLUMNS FROM whatsapp_logs")->fetchAll(PDO::FETCH_COLUMN);
            
            if (!in_array('customer_name', $columns)) {
                $conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN customer_name VARCHAR(255) AFTER phone_number");
            }
            if (!in_array('action_type', $columns)) {
                $conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN action_type VARCHAR(50) AFTER customer_name");
            }
            if (!in_array('status', $columns)) {
                $conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN status VARCHAR(20) AFTER action_type");
            }
            if (!in_array('message', $columns)) {
                $conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN message TEXT AFTER status");
            }
            if (!in_array('error_message', $columns)) {
                $conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN error_message TEXT AFTER message");
            }
            if (!in_array('updated_at', $columns)) {
                $conn->exec("ALTER TABLE whatsapp_logs ADD COLUMN updated_at DATETIME AFTER created_at");
            }
        }
    } catch (Exception $e) {
        // Fehler beim Erstellen/Anpassen der Tabelle
        error_log("Fehler beim Erstellen/Anpassen der whatsapp_logs Tabelle: " . $e->getMessage());
    }
    
    // Hole das erste verfügbare Event
    $stmt = $conn->prepare("
        SELECT id, name, event_date, start_time, slot_duration, max_monteure
        FROM events 
        WHERE is_active = 1 
        AND event_date >= CURDATE()
        AND autohaus_id = ?
        ORDER BY event_date ASC, start_time ASC
        LIMIT 1
    ");
    $stmt->execute([$testData['autohaus_id']]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        echo json_encode([
            'success' => false,
            'error' => 'Kein verfügbares Event gefunden'
        ]);
        exit;
    }
    
    // Berechne verfügbare Slots für das Event
    $eventDate = $event['event_date'];
    $startTime = strtotime($event['start_time']);
    $endTime = strtotime($event['end_time'] ?? '18:00:00');
    $slotDuration = $event['slot_duration'] ?? 30;
    
    // Hole bereits gebuchte Slots
    $stmt = $conn->prepare("
        SELECT slot_time, COUNT(*) as booked_count
        FROM appointments 
        WHERE event_id = ? 
        AND slot_date = ? 
        AND status IN ('pending', 'approved', 'booked', 'vorbereitet', 'completed')
        GROUP BY slot_time
    ");
    $stmt->execute([$event['id'], $eventDate]);
    $bookedSlots = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Finde ersten verfügbaren Slot
    $availableSlot = null;
    $currentTime = $startTime;
    $maxMonteure = $event['max_monteure'] ?? 5;
    
    while ($currentTime < $endTime) {
        $slotTime = date('H:i:s', $currentTime);
        $bookedCount = $bookedSlots[$slotTime] ?? 0;
        
        if ($bookedCount < $maxMonteure) {
            $availableSlot = $slotTime;
            break;
        }
        
        $currentTime += ($slotDuration * 60);
    }
    
    if (!$availableSlot) {
        echo json_encode([
            'success' => false,
            'error' => 'Kein verfügbarer Slot für dieses Event gefunden'
        ]);
        exit;
    }
    
    // Erstelle Termin
    $stmt = $conn->prepare("
        INSERT INTO appointments (
            event_id, autohaus_id, customer_name, license_plate, phone, email, 
            slot_time, slot_date, status, source
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'booked', 'admin_test')
    ");
    
    $stmt->execute([
        $event['id'],
        $testData['autohaus_id'],
        $testData['customer_name'],
        $testData['license_plate'],
        $testData['phone'],
        $testData['email'],
        $availableSlot,
        $eventDate
    ]);
    
    $appointmentId = $conn->lastInsertId();
    
    // Log WhatsApp-Versand
    $logStmt = $conn->prepare("
        INSERT INTO whatsapp_logs (
            appointment_id, autohaus_id, phone_number, customer_name, 
            action_type, status, message, error_message, created_at
        ) VALUES (?, ?, ?, ?, 'test_booking', 'pending', ?, NULL, NOW())
    ");
    $logStmt->execute([
        $appointmentId,
        $testData['autohaus_id'],
        $testData['phone'],
        $testData['customer_name'],
        'Termin gebucht - Test'
    ]);
    $logId = $conn->lastInsertId();
    
    // Sende WhatsApp-Bestätigung
    $whatsapp = new AutohausWhatsAppNotifications($testData['autohaus_id']);
    $result = $whatsapp->sendAppointmentNotification($appointmentId, 'confirmation');
    
    // Update Log
    $updateStmt = $conn->prepare("
        UPDATE whatsapp_logs 
        SET status = ?, 
            message = ?, 
            error_message = ?,
            updated_at = NOW()
        WHERE id = ?
    ");
    
    $status = $result['success'] ? 'success' : 'error';
    $message = $result['message'] ?? $result['error'] ?? 'Unbekannt';
    $errorMessage = $result['success'] ? null : ($result['error'] ?? 'Unbekannter Fehler');
    
    $updateStmt->execute([
        $status,
        $message,
        $errorMessage,
        $logId
    ]);
    
    // Update Appointment
    if ($result['success']) {
        $updateAppointment = $conn->prepare("
            UPDATE appointments 
            SET whatsapp_confirmation_sent = 1, 
                whatsapp_confirmation_sent_at = NOW()
            WHERE id = ?
        ");
        $updateAppointment->execute([$appointmentId]);
    }
    
    echo json_encode([
        'success' => true,
        'appointment_id' => $appointmentId,
        'event' => [
            'id' => $event['id'],
            'name' => $event['name'],
            'date' => $eventDate,
            'time' => $availableSlot
        ],
        'whatsapp_result' => $result,
        'log_id' => $logId
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
}

