<?php
/**
 * 🎯 Tasks & Reminders Manager
 * Professionelle Verwaltungsseite für Aufgaben und Erinnerungen
 */

session_start();
require_once '../config/database_optimized.php';
require_once 'includes/helpers.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';
$current_theme = 'light';

if ($current_user_id) {
    $user_data = $db->getUserData($current_user_id);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
        $current_user_role = $user_data['role'] ?? 'admin';
        $current_theme = $user_data['theme'] ?? 'light';
    }
}

// Handle form submissions
$action = $_GET['action'] ?? 'list';
$task_id = $_GET['id'] ?? null;
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $post_action = $_POST['action'] ?? 'create';
    
    try {
        if ($post_action === 'create' || $post_action === 'update') {
            $title = $_POST['title'] ?? '';
            $description = $_POST['description'] ?? '';
            $priority = $_POST['priority'] ?? 'medium';
            $status = $_POST['status'] ?? 'pending';
            $due_date = $_POST['due_date'] ?? null;
            $due_time = $_POST['due_time'] ?? null;
            $reminder_date = $_POST['reminder_date'] ?? null;
            $reminder_time = $_POST['reminder_time'] ?? null;
            $assigned_to = $_POST['assigned_to'] ?? null;
            $category = $_POST['category'] ?? 'general';
            
            if (empty($title)) {
                throw new Exception('Titel ist erforderlich');
            }
            
            if ($post_action === 'create') {
                $stmt = $conn->prepare("
                    INSERT INTO dashboard_tasks 
                    (user_id, title, description, priority, status, due_date, due_time, 
                     reminder_date, reminder_time, assigned_to, created_by, category, is_active)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
                ");
                
                $stmt->execute([
                    null,
                    $title,
                    $description,
                    $priority,
                    $status,
                    $due_date ?: null,
                    $due_time ?: null,
                    $reminder_date ?: null,
                    $reminder_time ?: null,
                    $assigned_to ?: null,
                    $current_user_id,
                    $category
                ]);
                
                $message = 'Aufgabe erfolgreich erstellt!';
            } else {
                $task_id = $_POST['id'] ?? null;
                if (!$task_id) {
                    throw new Exception('Aufgaben-ID erforderlich');
                }
                
                $stmt = $conn->prepare("
                    UPDATE dashboard_tasks 
                    SET title = ?, description = ?, priority = ?, status = ?, 
                        due_date = ?, due_time = ?, reminder_date = ?, reminder_time = ?, 
                        assigned_to = ?, category = ?
                    WHERE id = ?
                ");
                
                $stmt->execute([
                    $title,
                    $description,
                    $priority,
                    $status,
                    $due_date ?: null,
                    $due_time ?: null,
                    $reminder_date ?: null,
                    $reminder_time ?: null,
                    $assigned_to ?: null,
                    $category,
                    $task_id
                ]);
                
                $message = 'Aufgabe erfolgreich aktualisiert!';
            }
        } elseif ($post_action === 'delete' && $task_id) {
            $stmt = $conn->prepare("UPDATE dashboard_tasks SET is_active = 0 WHERE id = ?");
            $stmt->execute([$task_id]);
            $message = 'Aufgabe erfolgreich gelöscht!';
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Get filters
$filter_status = $_GET['status'] ?? 'all';
$filter_priority = $_GET['priority'] ?? 'all';
$filter_category = $_GET['category'] ?? 'all';
$search = $_GET['search'] ?? '';

// Build query
$sql = "
    SELECT t.*, 
           u1.username as assigned_to_name,
           u1.full_name as assigned_to_full_name,
           u2.username as created_by_name,
           u2.full_name as created_by_full_name
    FROM dashboard_tasks t
    LEFT JOIN admin_users u1 ON t.assigned_to = u1.id
    LEFT JOIN admin_users u2 ON t.created_by = u2.id
    WHERE t.is_active = 1
";

$params = [];

if ($filter_status !== 'all') {
    $sql .= " AND t.status = ?";
    $params[] = $filter_status;
}

if ($filter_priority !== 'all') {
    $sql .= " AND t.priority = ?";
    $params[] = $filter_priority;
}

if ($filter_category !== 'all') {
    $sql .= " AND t.category = ?";
    $params[] = $filter_category;
}

if ($search) {
    $sql .= " AND (t.title LIKE ? OR t.description LIKE ?)";
    $searchTerm = "%{$search}%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

// User filter
$sql .= " AND (t.user_id IS NULL OR t.user_id = ? OR t.assigned_to = ?)";
$params[] = $current_user_id;
$params[] = $current_user_id;

$sql .= " ORDER BY 
    CASE t.priority 
        WHEN 'urgent' THEN 1
        WHEN 'high' THEN 2
        WHEN 'medium' THEN 3
        WHEN 'low' THEN 4
    END,
    t.due_date ASC,
    t.due_time ASC";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get task for editing
$task = null;
if ($action === 'edit' && $task_id) {
    $stmt = $conn->prepare("SELECT * FROM dashboard_tasks WHERE id = ? AND is_active = 1");
    $stmt->execute([$task_id]);
    $task = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Get all users for assignment
$stmt = $conn->prepare("SELECT id, username, full_name FROM admin_users WHERE is_active = 1 ORDER BY username ASC");
$stmt->execute();
$all_users = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$stats = [
    'total' => count($tasks),
    'pending' => count(array_filter($tasks, fn($t) => $t['status'] === 'pending')),
    'in_progress' => count(array_filter($tasks, fn($t) => $t['status'] === 'in_progress')),
    'completed' => count(array_filter($tasks, fn($t) => $t['status'] === 'completed')),
    'urgent' => count(array_filter($tasks, fn($t) => $t['priority'] === 'urgent' && in_array($t['status'], ['pending', 'in_progress'])))
];
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🎯 Aufgaben & Erinnerungen • Boxenstop</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link rel="stylesheet" href="assets/css/admin-zoom.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: #1e293b;
        }

        [data-theme="dark"] body {
            background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);
            color: #f1f5f9;
        }

        .admin-container {
            display: flex;
            min-height: 100vh;
        }

        .admin-main {
            flex: 1;
            padding: var(--space-8, 2rem);
            margin-left: 300px;
        }

        .page-header {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 1.5rem;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
        }

        [data-theme="dark"] .page-header {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .page-header h1 {
            font-size: 2rem;
            font-weight: 800;
            margin-bottom: 1rem;
        }

        .stats-bar {
            display: flex;
            gap: 1rem;
            margin-top: 1rem;
            flex-wrap: wrap;
        }

        .stat-badge {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: rgba(99, 102, 241, 0.1);
            border-radius: 0.75rem;
            font-size: 0.875rem;
            font-weight: 600;
        }

        .filters-bar {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 1rem;
            padding: 1.5rem;
            margin-bottom: 2rem;
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
            align-items: center;
        }

        [data-theme="dark"] .filters-bar {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .filter-group label {
            font-size: 0.75rem;
            font-weight: 600;
            color: #64748b;
        }

        [data-theme="dark"] .filter-group label {
            color: #94a3b8;
        }

        .filter-group select,
        .filter-group input {
            padding: 0.5rem;
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 0.5rem;
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
        }

        [data-theme="dark"] .filter-group select,
        [data-theme="dark"] .filter-group input {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
            color: #f1f5f9;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 0.75rem;
            font-weight: 600;
            cursor: pointer;
            border: none;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            text-decoration: none;
        }

        .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
        }

        .tasks-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 1.5rem;
        }

        .task-card {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }

        [data-theme="dark"] .task-card {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .task-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        }

        .task-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 1rem;
        }

        .task-title {
            font-size: 1.125rem;
            font-weight: 700;
            color: #1e293b;
        }

        [data-theme="dark"] .task-title {
            color: #f1f5f9;
        }

        .priority-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 0.5rem;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .priority-urgent { background: #FEE2E2; color: #991B1B; }
        .priority-high { background: #FEF3C7; color: #92400E; }
        .priority-medium { background: #DBEAFE; color: #1E40AF; }
        .priority-low { background: #F3F4F6; color: #374151; }

        [data-theme="dark"] .priority-urgent { background: rgba(239, 68, 68, 0.2); color: #FCA5A5; }
        [data-theme="dark"] .priority-high { background: rgba(245, 158, 11, 0.2); color: #FCD34D; }
        [data-theme="dark"] .priority-medium { background: rgba(59, 130, 246, 0.2); color: #93C5FD; }
        [data-theme="dark"] .priority-low { background: rgba(107, 114, 128, 0.2); color: #D1D5DB; }

        .task-description {
            color: #64748b;
            font-size: 0.875rem;
            margin-bottom: 1rem;
            line-height: 1.6;
        }

        [data-theme="dark"] .task-description {
            color: #94a3b8;
        }

        .task-meta {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
            font-size: 0.875rem;
            color: #64748b;
        }

        [data-theme="dark"] .task-meta {
            color: #94a3b8;
        }

        .task-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
        }

        .btn-success {
            background: #10b981;
            color: white;
        }

        .btn-danger {
            background: #ef4444;
            color: white;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(10px);
            z-index: 10000;
            align-items: center;
            justify-content: center;
        }

        .modal.active {
            display: flex;
        }

        .modal-content {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 1.5rem;
            padding: 2rem;
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }

        [data-theme="dark"] .modal-content {
            background: rgba(0, 0, 0, 0.8);
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            font-size: 0.875rem;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 0.5rem;
            background: rgba(255, 255, 255, 0.6);
        }

        [data-theme="dark"] .form-group input,
        [data-theme="dark"] .form-group select,
        [data-theme="dark"] .form-group textarea {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
            color: #f1f5f9;
        }

        .alert {
            padding: 1rem;
            border-radius: 0.75rem;
            margin-bottom: 1rem;
        }

        .alert-success {
            background: #D1FAE5;
            color: #065F46;
        }

        .alert-error {
            background: #FEE2E2;
            color: #991B1B;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="page-header">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <h1>🎯 Aufgaben & Erinnerungen</h1>
                        <p style="color: #64748b; margin-top: 0.5rem;">Verwalten Sie Ihre Aufgaben und Erinnerungen professionell</p>
                    </div>
                    <button class="btn btn-primary" onclick="openCreateModal()">
                        <i class="fas fa-plus"></i>
                        Neue Aufgabe
                    </button>
                </div>
                
                <div class="stats-bar">
                    <div class="stat-badge">
                        <i class="fas fa-tasks"></i>
                        <span>Gesamt: <?php echo $stats['total']; ?></span>
                    </div>
                    <div class="stat-badge">
                        <i class="fas fa-clock"></i>
                        <span>Ausstehend: <?php echo $stats['pending']; ?></span>
                    </div>
                    <div class="stat-badge">
                        <i class="fas fa-spinner"></i>
                        <span>In Bearbeitung: <?php echo $stats['in_progress']; ?></span>
                    </div>
                    <div class="stat-badge">
                        <i class="fas fa-check-circle"></i>
                        <span>Erledigt: <?php echo $stats['completed']; ?></span>
                    </div>
                    <?php if ($stats['urgent'] > 0): ?>
                        <div class="stat-badge" style="background: rgba(239, 68, 68, 0.1); color: #991B1B;">
                            <i class="fas fa-exclamation-triangle"></i>
                            <span>Dringend: <?php echo $stats['urgent']; ?></span>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <div class="filters-bar">
                <form method="GET" style="display: flex; gap: 1rem; flex-wrap: wrap; align-items: end; flex: 1;">
                    <div class="filter-group">
                        <label>Status</label>
                        <select name="status" onchange="this.form.submit()">
                            <option value="all" <?php echo $filter_status === 'all' ? 'selected' : ''; ?>>Alle</option>
                            <option value="pending" <?php echo $filter_status === 'pending' ? 'selected' : ''; ?>>Ausstehend</option>
                            <option value="in_progress" <?php echo $filter_status === 'in_progress' ? 'selected' : ''; ?>>In Bearbeitung</option>
                            <option value="completed" <?php echo $filter_status === 'completed' ? 'selected' : ''; ?>>Erledigt</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label>Priorität</label>
                        <select name="priority" onchange="this.form.submit()">
                            <option value="all" <?php echo $filter_priority === 'all' ? 'selected' : ''; ?>>Alle</option>
                            <option value="urgent" <?php echo $filter_priority === 'urgent' ? 'selected' : ''; ?>>Dringend</option>
                            <option value="high" <?php echo $filter_priority === 'high' ? 'selected' : ''; ?>>Hoch</option>
                            <option value="medium" <?php echo $filter_priority === 'medium' ? 'selected' : ''; ?>>Mittel</option>
                            <option value="low" <?php echo $filter_priority === 'low' ? 'selected' : ''; ?>>Niedrig</option>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label>Kategorie</label>
                        <select name="category" onchange="this.form.submit()">
                            <option value="all" <?php echo $filter_category === 'all' ? 'selected' : ''; ?>>Alle</option>
                            <option value="general" <?php echo $filter_category === 'general' ? 'selected' : ''; ?>>Allgemein</option>
                            <option value="appointment" <?php echo $filter_category === 'appointment' ? 'selected' : ''; ?>>Termin</option>
                            <option value="event" <?php echo $filter_category === 'event' ? 'selected' : ''; ?>>Event</option>
                            <option value="storage" <?php echo $filter_category === 'storage' ? 'selected' : ''; ?>>Lager</option>
                        </select>
                    </div>
                    <div class="filter-group" style="flex: 1; min-width: 200px;">
                        <label>Suche</label>
                        <input type="text" name="search" placeholder="Titel oder Beschreibung..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i>
                        Suchen
                    </button>
                    <?php if ($search || $filter_status !== 'all' || $filter_priority !== 'all' || $filter_category !== 'all'): ?>
                        <a href="tasks_manager.php" class="btn" style="background: #6b7280; color: white;">
                            <i class="fas fa-times"></i>
                            Zurücksetzen
                        </a>
                    <?php endif; ?>
                </form>
            </div>

            <?php if (empty($tasks)): ?>
                <div style="text-align: center; padding: 4rem; background: rgba(255, 255, 255, 0.8); border-radius: 1rem;">
                    <i class="fas fa-tasks" style="font-size: 4rem; color: #9ca3af; margin-bottom: 1rem;"></i>
                    <h3 style="margin-bottom: 0.5rem;">Keine Aufgaben gefunden</h3>
                    <p style="color: #64748b;">Erstellen Sie Ihre erste Aufgabe oder passen Sie die Filter an.</p>
                    <button class="btn btn-primary" onclick="openCreateModal()" style="margin-top: 1.5rem;">
                        <i class="fas fa-plus"></i>
                        Neue Aufgabe erstellen
                    </button>
                </div>
            <?php else: ?>
                <div class="tasks-grid">
                    <?php foreach ($tasks as $task_item): ?>
                        <div class="task-card">
                            <div class="task-header">
                                <h3 class="task-title"><?php echo htmlspecialchars($task_item['title']); ?></h3>
                                <span class="priority-badge priority-<?php echo $task_item['priority']; ?>">
                                    <?php echo ucfirst($task_item['priority']); ?>
                                </span>
                            </div>
                            
                            <?php if ($task_item['description']): ?>
                                <p class="task-description"><?php echo nl2br(htmlspecialchars($task_item['description'])); ?></p>
                            <?php endif; ?>
                            
                            <div class="task-meta">
                                <div><i class="fas fa-info-circle"></i> Status: <strong><?php echo ucfirst(str_replace('_', ' ', $task_item['status'])); ?></strong></div>
                                <?php if ($task_item['due_date']): ?>
                                    <div><i class="fas fa-calendar"></i> Fällig: <strong><?php echo date('d.m.Y', strtotime($task_item['due_date'])); ?></strong>
                                        <?php if ($task_item['due_time']): ?>
                                            um <?php echo date('H:i', strtotime($task_item['due_time'])); ?>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ($task_item['assigned_to_name']): ?>
                                    <div><i class="fas fa-user"></i> Zugewiesen an: <strong><?php echo htmlspecialchars($task_item['assigned_to_full_name'] ?? $task_item['assigned_to_name']); ?></strong></div>
                                <?php endif; ?>
                                <?php if ($task_item['category']): ?>
                                    <div><i class="fas fa-tag"></i> Kategorie: <strong><?php echo ucfirst($task_item['category']); ?></strong></div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="task-actions">
                                <a href="?action=edit&id=<?php echo $task_item['id']; ?>" class="btn btn-sm" style="background: #3b82f6; color: white;">
                                    <i class="fas fa-edit"></i>
                                    Bearbeiten
                                </a>
                                <?php if ($task_item['status'] !== 'completed'): ?>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Aufgabe als erledigt markieren?');">
                                        <input type="hidden" name="action" value="update">
                                        <input type="hidden" name="id" value="<?php echo $task_item['id']; ?>">
                                        <input type="hidden" name="title" value="<?php echo htmlspecialchars($task_item['title']); ?>">
                                        <input type="hidden" name="description" value="<?php echo htmlspecialchars($task_item['description'] ?? ''); ?>">
                                        <input type="hidden" name="priority" value="<?php echo $task_item['priority']; ?>">
                                        <input type="hidden" name="status" value="completed">
                                        <input type="hidden" name="due_date" value="<?php echo $task_item['due_date'] ?? ''; ?>">
                                        <input type="hidden" name="due_time" value="<?php echo $task_item['due_time'] ?? ''; ?>">
                                        <input type="hidden" name="assigned_to" value="<?php echo $task_item['assigned_to'] ?? ''; ?>">
                                        <input type="hidden" name="category" value="<?php echo $task_item['category'] ?? 'general'; ?>">
                                        <button type="submit" class="btn btn-sm btn-success">
                                            <i class="fas fa-check"></i>
                                            Erledigt
                                        </button>
                                    </form>
                                <?php endif; ?>
                                <form method="POST" style="display: inline;" onsubmit="return confirm('Aufgabe wirklich löschen?');">
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="id" value="<?php echo $task_item['id']; ?>">
                                    <button type="submit" class="btn btn-sm btn-danger">
                                        <i class="fas fa-trash"></i>
                                        Löschen
                                    </button>
                                </form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </main>
    </div>

    <!-- Create/Edit Modal -->
    <div id="taskModal" class="modal <?php echo ($action === 'create' || $action === 'edit') ? 'active' : ''; ?>">
        <div class="modal-content">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                <h2><?php echo $action === 'edit' ? 'Aufgabe bearbeiten' : 'Neue Aufgabe erstellen'; ?></h2>
                <button onclick="closeModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="<?php echo $action === 'edit' ? 'update' : 'create'; ?>">
                <?php if ($action === 'edit' && $task): ?>
                    <input type="hidden" name="id" value="<?php echo $task['id']; ?>">
                <?php endif; ?>
                
                <div class="form-group">
                    <label>Titel *</label>
                    <input type="text" name="title" required value="<?php echo htmlspecialchars($task['title'] ?? ''); ?>" placeholder="Aufgabentitel">
                </div>
                
                <div class="form-group">
                    <label>Beschreibung</label>
                    <textarea name="description" rows="4" placeholder="Detaillierte Beschreibung..."><?php echo htmlspecialchars($task['description'] ?? ''); ?></textarea>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label>Priorität</label>
                        <select name="priority">
                            <option value="low" <?php echo ($task['priority'] ?? 'medium') === 'low' ? 'selected' : ''; ?>>Niedrig</option>
                            <option value="medium" <?php echo ($task['priority'] ?? 'medium') === 'medium' ? 'selected' : ''; ?>>Mittel</option>
                            <option value="high" <?php echo ($task['priority'] ?? 'medium') === 'high' ? 'selected' : ''; ?>>Hoch</option>
                            <option value="urgent" <?php echo ($task['priority'] ?? 'medium') === 'urgent' ? 'selected' : ''; ?>>Dringend</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Status</label>
                        <select name="status">
                            <option value="pending" <?php echo ($task['status'] ?? 'pending') === 'pending' ? 'selected' : ''; ?>>Ausstehend</option>
                            <option value="in_progress" <?php echo ($task['status'] ?? 'pending') === 'in_progress' ? 'selected' : ''; ?>>In Bearbeitung</option>
                            <option value="completed" <?php echo ($task['status'] ?? 'pending') === 'completed' ? 'selected' : ''; ?>>Erledigt</option>
                        </select>
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label>Fälligkeitsdatum</label>
                        <input type="date" name="due_date" value="<?php echo $task['due_date'] ?? ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Fälligkeitszeit</label>
                        <input type="time" name="due_time" value="<?php echo $task['due_time'] ?? ''; ?>">
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label>Erinnerungsdatum</label>
                        <input type="date" name="reminder_date" value="<?php echo $task['reminder_date'] ?? ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Erinnerungszeit</label>
                        <input type="time" name="reminder_time" value="<?php echo $task['reminder_time'] ?? ''; ?>">
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label>Zugewiesen an</label>
                        <select name="assigned_to">
                            <option value="">Niemandem zuweisen</option>
                            <?php foreach ($all_users as $user): ?>
                                <option value="<?php echo $user['id']; ?>" <?php echo ($task['assigned_to'] ?? '') == $user['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($user['full_name'] ?? $user['username']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Kategorie</label>
                        <select name="category">
                            <option value="general" <?php echo ($task['category'] ?? 'general') === 'general' ? 'selected' : ''; ?>>Allgemein</option>
                            <option value="appointment" <?php echo ($task['category'] ?? 'general') === 'appointment' ? 'selected' : ''; ?>>Termin</option>
                            <option value="event" <?php echo ($task['category'] ?? 'general') === 'event' ? 'selected' : ''; ?>>Event</option>
                            <option value="storage" <?php echo ($task['category'] ?? 'general') === 'storage' ? 'selected' : ''; ?>>Lager</option>
                        </select>
                    </div>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary" style="flex: 1;">
                        <i class="fas fa-save"></i>
                        Speichern
                    </button>
                    <button type="button" class="btn" onclick="closeModal()" style="background: #6b7280; color: white;">
                        Abbrechen
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function openCreateModal() {
            window.location.href = '?action=create';
        }

        function closeModal() {
            window.location.href = 'tasks_manager.php';
        }

        // Close modal on outside click
        document.getElementById('taskModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    </script>
</body>
</html>

