<?php
/**
 * 🚀 QUICK ACTIONS MANAGER
 * Volvo Apple Design - Benutzerdefinierte Schnellaktionen verwalten
 * 
 * Features:
 * - CRUD für Schnellaktionen
 * - Drag & Drop Sortierung
 * - Icon Picker
 * - Color Picker
 * - Vorschau-Funktion
 * - Dark/Light Mode Support
 */

session_start();
require_once '../config/database_optimized.php';
require_once '../includes/performance_monitor.php';

// 🔐 Security Check
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

// 👤 User Data & Theme
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';
$current_theme = $_SESSION['current_theme'] ?? 'light';

if ($current_user_id) {
    $user_data = $db->getUserData($current_user_id);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Administrator';
        $current_user_role = $user_data['role'] ?? 'admin';
        $current_theme = $user_data['theme'] ?? $_SESSION['current_theme'] ?? 'light';
    }
}

// 🎨 Theme Toggle Handler
if (isset($_POST['toggle_theme'])) {
    $new_theme = ($current_theme === 'light') ? 'dark' : 'light';
    
    if ($current_user_id) {
        try {
            $stmt = $conn->prepare("UPDATE admin_users SET theme = ? WHERE id = ?");
            $stmt->execute([$new_theme, $current_user_id]);
            $current_theme = $new_theme;
            $_SESSION['current_theme'] = $current_theme;
        } catch (Exception $e) {
            $current_theme = $new_theme;
            $_SESSION['current_theme'] = $current_theme;
        }
    } else {
        $current_theme = $new_theme;
        $_SESSION['current_theme'] = $current_theme;
    }
}

// 🚀 CRUD Operations
$message = '';
$message_type = '';

// Create Quick Action
if (isset($_POST['create_action'])) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO quick_actions (title, description, icon, url, target, background_color, text_color, position_order, user_id) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $position = $conn->query("SELECT COALESCE(MAX(position_order), 0) + 1 FROM quick_actions")->fetchColumn();
        
        $stmt->execute([
            $_POST['title'],
            $_POST['description'],
            $_POST['icon'],
            $_POST['url'],
            $_POST['target'],
            $_POST['background_color'],
            $_POST['text_color'],
            $position,
            $current_user_id
        ]);
        
        $message = 'Schnellaktion erfolgreich erstellt!';
        $message_type = 'success';
    } catch (Exception $e) {
        $message = 'Fehler beim Erstellen: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Update Quick Action
if (isset($_POST['update_action'])) {
    try {
        $stmt = $conn->prepare("
            UPDATE quick_actions 
            SET title = ?, description = ?, icon = ?, url = ?, target = ?, 
                background_color = ?, text_color = ?, updated_at = CURRENT_TIMESTAMP
            WHERE id = ?
        ");
        
        $stmt->execute([
            $_POST['title'],
            $_POST['description'],
            $_POST['icon'],
            $_POST['url'],
            $_POST['target'],
            $_POST['background_color'],
            $_POST['text_color'],
            $_POST['action_id']
        ]);
        
        $message = 'Schnellaktion erfolgreich aktualisiert!';
        $message_type = 'success';
    } catch (Exception $e) {
        $message = 'Fehler beim Aktualisieren: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Delete Quick Action
if (isset($_POST['delete_action'])) {
    try {
        $stmt = $conn->prepare("DELETE FROM quick_actions WHERE id = ?");
        $stmt->execute([$_POST['action_id']]);
        
        $message = 'Schnellaktion erfolgreich gelöscht!';
        $message_type = 'success';
    } catch (Exception $e) {
        $message = 'Fehler beim Löschen: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Toggle Active Status
if (isset($_POST['toggle_active'])) {
    try {
        $stmt = $conn->prepare("UPDATE quick_actions SET is_active = NOT is_active WHERE id = ?");
        $stmt->execute([$_POST['action_id']]);
        
        $message = 'Status erfolgreich geändert!';
        $message_type = 'success';
    } catch (Exception $e) {
        $message = 'Fehler beim Ändern des Status: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Update Order
if (isset($_POST['update_order'])) {
    try {
        $orders = json_decode($_POST['order_data'], true);
        
        $conn->beginTransaction();
        $stmt = $conn->prepare("UPDATE quick_actions SET position_order = ? WHERE id = ?");
        
        foreach ($orders as $order) {
            $stmt->execute([$order['position'], $order['id']]);
        }
        
        $conn->commit();
        
        $message = 'Reihenfolge erfolgreich aktualisiert!';
        $message_type = 'success';
    } catch (Exception $e) {
        $conn->rollback();
        $message = 'Fehler beim Aktualisieren der Reihenfolge: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Get all quick actions
$quick_actions = $conn->query("
    SELECT *, 
           CASE WHEN user_id IS NULL THEN 'System' ELSE 'Benutzer' END as source
    FROM quick_actions 
    ORDER BY position_order ASC, id ASC
")->fetchAll(PDO::FETCH_ASSOC);

// Popular icons for picker
$popular_icons = [
    'fas fa-plus', 'fas fa-calendar-check', 'fas fa-warehouse', 'fas fa-external-link-alt',
    'fas fa-users', 'fas fa-cog', 'fas fa-chart-bar', 'fas fa-envelope',
    'fas fa-phone', 'fas fa-home', 'fas fa-car', 'fas fa-tools',
    'fas fa-clipboard-list', 'fas fa-search', 'fas fa-download', 'fas fa-upload',
    'fas fa-print', 'fas fa-share', 'fas fa-edit', 'fas fa-trash',
    'fas fa-star', 'fas fa-heart', 'fas fa-bookmark', 'fas fa-flag'
];
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="googlebot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="bingbot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <title>Schnellaktionen Manager - Boxenstop</title>
    <!-- Optimized CSS Loading -->
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/admin-zoom.css">
    <link rel="stylesheet" href="assets/css/modern-sidebar.css">
    <link rel="stylesheet" href="assets/css/admin-inline-styles.css">
    <!-- Optimized Font Loading -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <!-- Font Awesome - Optimized -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" integrity="sha512-9usAa10IRO0HhonpyAIVpjrylPvoDwiPUiKdWk5t3PyolY1cOd4DSE0Ga+ri4AuTroPR5aQvXU9xC6qOPnzFeg==" crossorigin="anonymous" referrerpolicy="no-referrer">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.css">
    
    <!-- Custom Quick Actions Styles -->
    <style>
        /* Quick Actions Manager Specific Styles */
        :root {
            /* Volvo Colors */
            --volvo-navy: #1a1a1a;
            --volvo-blue: #007aff;
            --volvo-light: #f5f5f7;
            
            /* Apple System Colors */
            --system-blue: #007aff;
            --system-green: #34c759;
            --system-orange: #ff9500;
            --system-red: #ff3b30;
            --system-purple: #af52de;
            --system-gray: #8e8e93;
            
            /* Background Colors */
            --bg-primary: #ffffff;
            --bg-secondary: #f5f5f7;
            --bg-tertiary: #ffffff;
            --bg-elevated: #ffffff;
            
            /* Text Colors */
            --label-primary: #000000;
            --label-secondary: #3c3c43;
            --label-tertiary: #3c3c43;
            --label-quaternary: #2c2c2e;
            
            /* Separator Colors */
            --separator-opaque: #c6c6c8;
            --separator: rgba(60, 60, 67, 0.29);
            
            /* Fill Colors */
            --fill-primary: rgba(120, 120, 128, 0.2);
            --fill-secondary: rgba(120, 120, 128, 0.16);
            --fill-tertiary: rgba(118, 118, 128, 0.12);
            
            /* System Spacing */
            --spacing-xs: 4px;
            --spacing-sm: 8px;
            --spacing-md: 16px;
            --spacing-lg: 24px;
            --spacing-xl: 32px;
            --spacing-2xl: 48px;
            
            /* Border Radius */
            --radius-xs: 4px;
            --radius-sm: 8px;
            --radius-md: 12px;
            --radius-lg: 16px;
            --radius-xl: 20px;
            
            /* Typography */
            --font-system: -apple-system, BlinkMacSystemFont, 'SF Pro Display', sans-serif;
            --font-mono: 'SF Mono', Monaco, Consolas, monospace;
            
            /* Shadows */
            --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.1);
            --shadow-md: 0 4px 12px rgba(0, 0, 0, 0.15);
            --shadow-lg: 0 10px 24px rgba(0, 0, 0, 0.15);
            
            /* Transitions */
            --ease-out: cubic-bezier(0.25, 0.46, 0.45, 0.94);
            --ease-in-out: cubic-bezier(0.645, 0.045, 0.355, 1);
            --spring: cubic-bezier(0.175, 0.885, 0.32, 1.275);
        }
        
        /* Reset & Base Styles */
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        html {
            font-size: 16px;
        }
        
        body {
            font-family: var(--font-system);
            font-size: 17px;
            line-height: 1.47059;
            color: var(--label-primary);
            background: var(--bg-secondary);
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }
        
        /* App Container */
        .app {
            display: flex;
            min-height: 100vh;
            background: var(--bg-secondary);
        }
        
        /* Sidebar */
        .sidebar {
            width: 280px;
            background: var(--bg-elevated);
            border-right: 1px solid var(--separator-opaque);
            display: flex;
            flex-direction: column;
            position: fixed;
            left: 0;
            top: 0;
            height: 100vh;
            z-index: 100;
        }
        
        .sidebar-header {
            padding: var(--spacing-lg);
            border-bottom: 1px solid var(--separator-opaque);
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }
        
        .logo-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, var(--volvo-navy), var(--system-blue));
            border-radius: var(--radius-md);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 18px;
            box-shadow: var(--shadow-sm);
        }
        
        .logo-text {
            display: flex;
            flex-direction: column;
        }
        
        .logo-title {
            font-size: 19px;
            font-weight: 600;
            color: var(--label-primary);
            line-height: 1.2;
        }
        
        .logo-subtitle {
            font-size: 13px;
            color: var(--label-secondary);
            font-weight: 400;
        }
        
        /* Navigation */
        .nav {
            flex: 1;
            padding: var(--spacing-md) 0;
            overflow-y: auto;
        }
        
        .nav-section {
            margin-bottom: var(--spacing-lg);
        }
        
        .nav-title {
            font-size: 13px;
            font-weight: 600;
            color: var(--label-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            padding: 0 var(--spacing-lg) var(--spacing-sm);
        }
        
        .nav-item {
            display: flex;
            align-items: center;
            padding: var(--spacing-sm) var(--spacing-lg);
            color: var(--label-primary);
            text-decoration: none;
            transition: background-color 0.2s var(--ease-out);
            font-size: 17px;
            font-weight: 400;
        }
        
        .nav-item:hover {
            background: var(--fill-tertiary);
        }
        
        .nav-item.active {
            background: var(--system-blue);
            color: white;
            font-weight: 500;
        }
        
        .nav-item-icon {
            width: 24px;
            height: 24px;
            margin-right: var(--spacing-md);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 16px;
        }
        
        /* Main Content */
        .main {
            flex: 1;
            margin-left: 280px;
            background: var(--bg-secondary);
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        
        /* Header */
        .header {
            background: var(--bg-primary);
            border-bottom: 1px solid var(--separator-opaque);
            padding: var(--spacing-xl);
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
        }
        
        .header-title {
            font-size: 34px;
            font-weight: 700;
            color: var(--label-primary);
            letter-spacing: -0.5px;
            line-height: 1.2;
        }
        
        .header-subtitle {
            font-size: 17px;
            color: var(--label-secondary);
            margin-top: var(--spacing-xs);
        }
        
        .header-actions {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }
        
        /* Buttons */
        .btn {
            font-family: var(--font-system);
            font-size: 17px;
            font-weight: 400;
            padding: var(--spacing-sm) var(--spacing-lg);
            border-radius: var(--radius-lg);
            border: none;
            cursor: pointer;
            transition: all 0.2s var(--ease-out);
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-sm);
            text-decoration: none;
            white-space: nowrap;
        }
        
        .btn-primary {
            background: var(--system-blue);
            color: white;
            box-shadow: var(--shadow-sm);
        }
        
        .btn-primary:hover {
            background: #0056d3;
            box-shadow: var(--shadow-md);
            transform: translateY(-1px);
        }
        
        .btn-secondary {
            background: var(--fill-secondary);
            color: var(--label-primary);
        }
        
        .btn-secondary:hover {
            background: var(--fill-primary);
        }
        
        .theme-toggle {
            width: 40px;
            height: 40px;
            border-radius: var(--radius-md);
            background: var(--fill-secondary);
            border: none;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s var(--ease-out);
            color: var(--label-primary);
        }
        
        .theme-toggle:hover {
            background: var(--fill-primary);
            transform: scale(1.05);
        }
        
        /* Content Area */
        .content {
            flex: 1;
            padding: var(--spacing-xl);
            max-width: 1200px;
        }
        
        /* Content Header */
        .content-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--spacing-2xl);
        }
        
        .content-title {
            font-size: 28px;
            font-weight: 700;
            color: var(--label-primary);
        }
        
        /* Cards Grid */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(340px, 1fr));
            gap: var(--spacing-lg);
        }
        
        .card {
            background: var(--bg-primary);
            border-radius: var(--radius-xl);
            border: 1px solid var(--separator-opaque);
            padding: var(--spacing-lg);
            transition: all 0.3s var(--ease-out);
            cursor: grab;
            position: relative;
            overflow: hidden;
        }
        
        .card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
            border-color: var(--system-blue);
        }
        
        .card:active {
            cursor: grabbing;
        }
        
        .card-header {
            display: flex;
            align-items: flex-start;
            gap: var(--spacing-md);
            margin-bottom: var(--spacing-lg);
        }
        
        .card-icon {
            width: 56px;
            height: 56px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
            flex-shrink: 0;
            box-shadow: var(--shadow-sm);
        }
        
        .card-content {
            flex: 1;
            min-width: 0;
        }
        
        .card-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--label-primary);
            margin-bottom: var(--spacing-xs);
            line-height: 1.3;
        }
        
        .card-description {
            font-size: 15px;
            color: var(--label-secondary);
            line-height: 1.4;
            margin-bottom: var(--spacing-md);
        }
        
        .card-url {
            font-family: var(--font-mono);
            font-size: 13px;
            color: var(--system-blue);
            background: var(--fill-tertiary);
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--radius-xs);
            word-break: break-all;
        }
        
        .card-actions {
            display: flex;
            gap: var(--spacing-sm);
            margin-top: var(--spacing-lg);
            padding-top: var(--spacing-lg);
            border-top: 1px solid var(--separator-opaque);
        }
        
        .card-btn {
            flex: 1;
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--radius-sm);
            border: none;
            font-size: 15px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s var(--ease-out);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: var(--spacing-xs);
        }
        
        .card-btn-edit {
            background: var(--system-blue);
            color: white;
        }
        
        .card-btn-edit:hover {
            background: #0056d3;
        }
        
        .card-btn-delete {
            background: var(--system-red);
            color: white;
        }
        
        .card-btn-delete:hover {
            background: #d70015;
        }
        
        .drag-handle {
            position: absolute;
            top: var(--spacing-md);
            right: var(--spacing-md);
            color: var(--label-tertiary);
            cursor: grab;
            padding: var(--spacing-xs);
            border-radius: var(--radius-xs);
            transition: all 0.2s var(--ease-out);
        }
        
        .drag-handle:hover {
            color: var(--system-blue);
            background: var(--fill-tertiary);
        }
        
        /* Alerts */
        .alert {
            padding: var(--spacing-md);
            border-radius: var(--radius-lg);
            margin-bottom: var(--spacing-lg);
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-weight: 500;
        }
        
        .alert-success {
            background: rgba(52, 199, 89, 0.1);
            color: var(--system-green);
            border: 1px solid rgba(52, 199, 89, 0.2);
        }
        
        .alert-error {
            background: rgba(255, 59, 48, 0.1);
            color: var(--system-red);
            border: 1px solid rgba(255, 59, 48, 0.2);
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s var(--ease-out);
            }
            
            .main {
                margin-left: 0;
            }
            
            .header {
                padding: var(--spacing-lg);
            }
            
            .header-content {
                flex-direction: column;
                align-items: flex-start;
                gap: var(--spacing-md);
            }
            
            .content {
                padding: var(--spacing-lg);
            }
            
            .cards-grid {
                grid-template-columns: 1fr;
            }
        }
            --shadow-medium: rgba(0, 0, 0, 0.15);
            --shadow-heavy: rgba(0, 0, 0, 0.25);
            
            /* 🧱 Spacing System */
            --space-xs: 0.25rem;
            --space-sm: 0.5rem;
            --space-md: 1rem;
            --space-lg: 1.5rem;
            --space-xl: 2rem;
            --space-2xl: 3rem;
            --space-3xl: 4rem;
            
            /* 📐 Border Radius */
            --radius-sm: 6px;
            --radius-md: 12px;
            --radius-lg: 16px;
            --radius-xl: 20px;
            --radius-full: 9999px;
            
            /* 🎭 Shadows */
            --shadow-sm: 0 1px 3px var(--shadow-light);
            --shadow-md: 0 4px 12px var(--shadow-medium);
            --shadow-lg: 0 8px 24px var(--shadow-medium);
            --shadow-xl: 0 16px 48px var(--shadow-heavy);
            
            /* 🎬 Animations */
            --transition-fast: 0.15s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-normal: 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-slow: 0.5s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        /* 🌙 Dark Theme */
        [data-theme="dark"] {
            --bg-primary: #ffffff;
            --bg-secondary: #f8f9fa;
            --bg-tertiary: #f1f3f4;
            --text-primary: #1a1a1a;
            --text-secondary: #6e6e73;
            --text-tertiary: #8e8e93;
            --border-primary: #d1d5db;
            --border-secondary: #e5e7eb;
            --shadow-light: rgba(0, 0, 0, 0.1);
            --shadow-medium: rgba(0, 0, 0, 0.15);
            --shadow-heavy: rgba(0, 0, 0, 0.25);
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, var(--bg-secondary) 0%, var(--bg-tertiary) 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
            margin: 0;
            padding: 0;
            transition: all var(--transition-normal);
        }
        
        /* 🎯 App Layout */
        .app-layout {
            position: relative;
            min-height: 100vh;
        }
        
        /* 🧭 Smart Sidebar (like performance.php) */
        .smart-sidebar {
            width: 300px;
            background: linear-gradient(180deg, #ffffff 0%, #f8f9fa 100%);
            border-right: 1px solid #e1e5e9;
            box-shadow: 4px 0 20px rgba(0, 0, 0, 0.08);
            display: flex;
            flex-direction: column;
            z-index: 1000;
            transition: all var(--transition-normal);
            position: fixed;
            left: 0;
            top: 0;
            height: 100vh;
        }
        
        /* 🏢 Logo Section */
        .logo-section {
            padding: var(--space-xl);
            border-bottom: 1px solid #e1e5e9;
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
        }
        
        .logo-container {
            display: flex;
            align-items: center;
            gap: var(--space-md);
        }
        
        .logo-icon {
            width: 48px;
            height: 48px;
            background: linear-gradient(135deg, var(--volvo-blue), var(--volvo-accent));
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.5rem;
            box-shadow: 0 4px 15px rgba(0, 120, 212, 0.3);
        }
        
        .logo-text .logo-title {
            font-family: 'SF Pro Display', -apple-system, BlinkMacSystemFont, sans-serif;
            font-size: 1.25rem;
            font-weight: 800;
            color: var(--text-primary);
            line-height: 1.2;
        }
        
        .logo-text .logo-subtitle {
            font-size: 0.85rem;
            color: var(--text-secondary);
            font-weight: 500;
        }
        
        /* 🧭 Navigation */
        .smart-nav {
            flex: 1;
            padding: var(--space-lg);
            overflow-y: auto;
        }
        
        .nav-section {
            margin-bottom: var(--space-xl);
        }
        
        .nav-section-title {
            padding: 0 var(--space-md) var(--space-sm);
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.1em;
            color: #8e8e93;
            margin-bottom: var(--space-sm);
        }
        
        .nav-items {
            display: flex;
            flex-direction: column;
            gap: var(--space-xs);
        }
        
        .nav-item {
            display: flex;
            align-items: center;
            gap: var(--space-md);
            padding: var(--space-md);
            text-decoration: none;
            color: #6e6e73;
            border-radius: var(--radius-lg);
            transition: all 0.2s ease;
            position: relative;
            font-weight: 500;
            font-size: 0.95rem;
        }
        
        .nav-item:hover {
            background: rgba(0, 120, 212, 0.08);
            color: var(--volvo-blue);
            transform: translateX(4px);
        }
        
        .nav-item.active {
            background: linear-gradient(135deg, var(--volvo-blue), var(--volvo-accent));
            color: white;
            box-shadow: 0 4px 15px rgba(0, 120, 212, 0.3);
        }
        
        .nav-item.active:hover {
            transform: translateX(0);
            box-shadow: 0 6px 20px rgba(0, 120, 212, 0.4);
        }
        
        .nav-icon {
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
        }
        
        /* 📱 Main Content */
        .main-content {
            flex: 1;
            padding: var(--space-2xl);
            overflow-y: auto;
            margin-left: 300px;
            min-height: 100vh;
            background: var(--bg-secondary);
        }
            background: var(--bg-secondary);
        }
        
        /* 🎯 Header (Enhanced Volvo-Apple Style) */
        .header {
            background: var(--bg-primary);
            border-radius: var(--radius-xl);
            padding: var(--space-2xl);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-lg);
            border: 1px solid var(--border-primary);
            position: relative;
            overflow: hidden;
        }
        
        .header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 6px;
            background: linear-gradient(90deg, var(--volvo-blue), var(--volvo-accent), var(--volvo-success));
        }
        
        .header::after {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(0, 120, 212, 0.03) 0%, transparent 70%);
            pointer-events: none;
        }
        
        .header-content {
            display: flex;
            align-items: center;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: var(--space-lg);
            position: relative;
            z-index: 1;
        }
        
        .header-left h1 {
            font-family: 'SF Pro Display', -apple-system, BlinkMacSystemFont, sans-serif;
            font-size: 2.75rem;
            font-weight: 900;
            background: linear-gradient(135deg, var(--volvo-blue), var(--volvo-accent));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: var(--space-sm);
            letter-spacing: -0.03em;
            line-height: 1.1;
        }
        
        .header-left p {
            font-size: 1.25rem;
            color: var(--text-secondary);
            font-weight: 400;
            line-height: 1.4;
        }
        
        .header-right {
            display: flex;
            align-items: center;
            gap: var(--space-lg);
        }
        
        /* 🎛️ Theme Toggle */
        .theme-toggle {
            background: var(--bg-secondary);
            border: 1px solid var(--border-primary);
            border-radius: var(--radius-full);
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--text-secondary);
            cursor: pointer;
            transition: all var(--transition-fast);
        }
        
        .theme-toggle:hover {
            background: var(--volvo-accent);
            color: white;
            transform: scale(1.05);
            box-shadow: var(--shadow-md);
        }
        
        /* 📨 Message */
        .message {
            padding: var(--space-lg);
            border-radius: var(--radius-lg);
            margin-bottom: var(--space-xl);
            display: flex;
            align-items: center;
            gap: var(--space-md);
            font-weight: 500;
        }
        
        .message.success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--volvo-success);
            border: 1px solid rgba(16, 185, 129, 0.2);
        }
        
        .message.error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--volvo-error);
            border: 1px solid rgba(239, 68, 68, 0.2);
        }
        
        /* 🎮 Actions (Enhanced Volvo-Apple Style) */
        .actions-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--space-xl);
            flex-wrap: wrap;
            gap: var(--space-md);
        }
        
        .actions-title {
            font-family: 'SF Pro Display', -apple-system, BlinkMacSystemFont, sans-serif;
            font-size: 2rem;
            font-weight: 800;
            color: var(--text-primary);
            letter-spacing: -0.02em;
        }
        
        .add-action-btn {
            background: linear-gradient(135deg, var(--volvo-blue), var(--volvo-accent));
            color: white;
            border: none;
            border-radius: var(--radius-xl);
            padding: var(--space-md) var(--space-xl);
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: var(--space-sm);
            cursor: pointer;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            font-family: 'SF Pro Display', -apple-system, BlinkMacSystemFont, sans-serif;
            font-size: 1rem;
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            box-shadow: 0 8px 25px rgba(0, 120, 212, 0.4);
        }
        
        .add-action-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: inherit;
            z-index: -1;
            transition: transform 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .add-action-btn::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            background: rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            transform: translate(-50%, -50%);
            transition: width 0.6s ease, height 0.6s ease;
            z-index: -1;
        }
        
        .add-action-btn:hover {
            transform: translateY(-3px) scale(1.03);
            box-shadow: 0 15px 35px rgba(0, 120, 212, 0.5);
        }
        
        .add-action-btn:hover::after {
            width: 300px;
            height: 300px;
        }
        
        .add-action-btn:active {
            transform: translateY(-1px) scale(1.01);
            transition: all 0.15s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .add-action-btn i {
            font-size: 1.1em;
            transition: transform 0.3s ease;
        }
        
        .add-action-btn:hover i {
            transform: scale(1.2) rotate(90deg);
        }
            gap: var(--space-sm);
            cursor: pointer;
            transition: all var(--transition-fast);
            box-shadow: var(--shadow-md);
        }
        
        .add-action-btn:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        /* 📋 Actions Grid (Enhanced Volvo-Apple Style) */
        .actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(340px, 1fr));
            gap: var(--space-xl);
            margin-bottom: var(--space-xl);
        }
        
        .action-item {
            background: var(--bg-primary);
            border-radius: var(--radius-2xl);
            padding: var(--space-xl);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.08);
            border: 1px solid var(--border-primary);
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            cursor: grab;
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
        }
        
        .action-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--volvo-blue), var(--volvo-accent));
            transform: scaleX(0);
            transition: transform 0.3s ease;
        }
        
        .action-item:hover::before {
            transform: scaleX(1);
        }
        
        .action-item:hover {
            transform: translateY(-8px) scale(1.02);
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
            border-color: var(--volvo-blue);
        }
        
        .action-item:active {
            cursor: grabbing;
            transform: translateY(-4px) scale(1.01);
            transition: all 0.15s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .action-item.sortable-ghost {
            opacity: 0.3;
            transform: scale(0.95);
        }
        
        .action-preview {
            display: flex;
            align-items: center;
            gap: var(--space-md);
            margin-bottom: var(--space-lg);
            padding: var(--space-md);
            border-radius: var(--radius-lg);
            border: 2px dashed var(--border-secondary);
            background: rgba(0, 120, 212, 0.05);
            transition: all 0.3s ease;
        }
        
        .action-preview:hover {
            border-color: var(--volvo-blue);
            background: rgba(0, 120, 212, 0.1);
        }
        
        .action-icon {
            width: 64px;
            height: 64px;
            border-radius: var(--radius-xl);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.75rem;
            flex-shrink: 0;
            color: white;
            position: relative;
            overflow: hidden;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
        }
        
        .action-icon::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(45deg, transparent, rgba(255, 255, 255, 0.2), transparent);
            transform: translateX(-100%);
            transition: transform 0.6s ease;
        }
        
        .action-item:hover .action-icon::before {
            transform: translateX(100%);
        }
        
        .action-text {
            flex: 1;
            min-width: 0;
        }
        
        .action-title {
            font-family: 'SF Pro Display', -apple-system, BlinkMacSystemFont, sans-serif;
            font-weight: 700;
            font-size: 1.25rem;
            color: var(--text-primary);
            margin-bottom: var(--space-xs);
            letter-spacing: -0.01em;
            line-height: 1.3;
        }
        
        .action-description {
            font-size: 0.95rem;
            color: var(--text-secondary);
            line-height: 1.5;
            margin-bottom: var(--space-md);
        }
        
        .action-url {
            font-family: 'SF Mono', Monaco, 'Cascadia Code', monospace;
            font-size: 0.85rem;
            color: var(--volvo-blue);
            background: rgba(0, 120, 212, 0.1);
            padding: var(--space-xs) var(--space-sm);
            border-radius: var(--radius-md);
            word-break: break-all;
            border: 1px solid rgba(0, 120, 212, 0.2);
            display: block;
            margin-bottom: var(--space-lg);
        }
        
        .action-controls {
            display: flex;
            gap: var(--space-sm);
            flex-wrap: wrap;
            padding-top: var(--space-lg);
            border-top: 1px solid var(--border-secondary);
        }
        
        .control-btn {
            border: none;
            border-radius: var(--radius-lg);
            padding: var(--space-sm) var(--space-md);
            cursor: pointer;
            font-size: 0.9rem;
            font-weight: 600;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            gap: var(--space-xs);
            flex: 1;
            justify-content: center;
        }
        
        .control-btn.edit {
            background: linear-gradient(135deg, var(--volvo-blue), var(--volvo-accent));
            color: white;
            box-shadow: 0 4px 15px rgba(0, 120, 212, 0.3);
        }
        
        .control-btn.edit:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 120, 212, 0.4);
        }
        
        .control-btn.delete {
            background: linear-gradient(135deg, var(--volvo-danger), #ff453a);
            color: white;
            box-shadow: 0 4px 15px rgba(255, 69, 58, 0.3);
        }
        
        .control-btn.delete:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(255, 69, 58, 0.4);
        }
        
        .control-btn.toggle {
            background: linear-gradient(135deg, var(--volvo-warning), #ffb000);
            color: white;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
        }
        
        .control-btn.toggle:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(245, 158, 11, 0.4);
        }
        
        .control-btn:hover {
            transform: scale(1.05);
        }
        
        /* 🛠️ Modal */
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            opacity: 0;
            visibility: hidden;
            transition: all var(--transition-normal);
        }
        
        .modal.active {
            opacity: 1;
            visibility: visible;
        }
        
        .modal-content {
            background: var(--bg-primary);
            border-radius: var(--radius-xl);
            padding: var(--space-2xl);
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-xl);
            transform: scale(0.9);
            transition: all var(--transition-normal);
        }
        
        .modal.active .modal-content {
            transform: scale(1);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--space-xl);
        }
        
        .modal-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text-primary);
        }
        
        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            padding: var(--space-sm);
            border-radius: var(--radius-sm);
            transition: all var(--transition-fast);
        }
        
        .modal-close:hover {
            background: var(--bg-secondary);
            color: var(--text-primary);
        }
        
        /* 📝 Form */
        .form-group {
            margin-bottom: var(--space-lg);
        }
        
        .form-label {
            display: block;
            margin-bottom: var(--space-sm);
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .form-input {
            width: 100%;
            padding: var(--space-md);
            border: 1px solid var(--border-primary);
            border-radius: var(--radius-md);
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-size: 0.875rem;
            transition: all var(--transition-fast);
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--volvo-accent);
            box-shadow: 0 0 0 3px rgba(0, 120, 212, 0.1);
        }
        
        .form-select {
            width: 100%;
            padding: var(--space-md);
            border: 1px solid var(--border-primary);
            border-radius: var(--radius-md);
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-size: 0.875rem;
            cursor: pointer;
        }
        
        .color-input {
            width: 60px;
            height: 40px;
            border: 1px solid var(--border-primary);
            border-radius: var(--radius-sm);
            cursor: pointer;
        }
        
        /* 🎨 Icon Picker */
        .icon-picker {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(40px, 1fr));
            gap: var(--space-sm);
            max-height: 200px;
            overflow-y: auto;
            border: 1px solid var(--border-primary);
            border-radius: var(--radius-md);
            padding: var(--space-md);
            background: var(--bg-secondary);
        }
        
        .icon-option {
            width: 40px;
            height: 40px;
            border: 1px solid var(--border-primary);
            border-radius: var(--radius-sm);
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all var(--transition-fast);
            background: var(--bg-primary);
        }
        
        .icon-option:hover,
        .icon-option.selected {
            border-color: var(--volvo-accent);
            background: var(--volvo-accent);
            color: white;
        }
        
        .form-actions {
            display: flex;
            gap: var(--space-md);
            justify-content: flex-end;
            margin-top: var(--space-xl);
        }
        
        .btn {
            padding: var(--space-md) var(--space-xl);
            border: none;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            transition: all var(--transition-fast);
            display: flex;
            align-items: center;
            gap: var(--space-sm);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--volvo-blue), var(--volvo-accent));
            color: white;
        }
        
        .btn-secondary {
            background: var(--bg-secondary);
            color: var(--text-primary);
            border: 1px solid var(--border-primary);
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }
        
        /* 📱 Responsive */
        @media (max-width: 768px) {
            .smart-sidebar {
                transform: translateX(-100%);
                position: fixed;
                z-index: 1001;
            }
            
            .main-content {
                margin-left: 0;
                padding: var(--space-md);
            }
            
            .container {
                padding: var(--space-lg);
            }
            
            .header-content {
                flex-direction: column;
                text-align: center;
            }
            
            .actions-header {
                flex-direction: column;
                gap: var(--space-lg);
            }
            
            .actions-grid {
                grid-template-columns: 1fr;
            }
            
            .form-actions {
                flex-direction: column;
            }
        }
        
        /* 🎭 Custom Scrollbar */
        ::-webkit-scrollbar {
            width: 8px;
        }
        
        ::-webkit-scrollbar-track {
            background: var(--bg-secondary);
        }
        
        ::-webkit-scrollbar-thumb {
            background: var(--border-secondary);
            border-radius: var(--radius-full);
        }
        
        ::-webkit-scrollbar-thumb:hover {
            background: var(--volvo-accent);
        }
        
        .inactive {
            opacity: 0.6;
        }
        
        .drag-handle {
            cursor: grab;
            color: var(--text-tertiary);
            position: absolute;
            top: var(--space-md);
            right: var(--space-md);
        }
        
        .drag-handle:active {
            cursor: grabbing;
        }
        
        .action-source {
            position: absolute;
            top: var(--space-md);
            left: var(--space-md);
            padding: var(--space-xs) var(--space-sm);
            background: var(--bg-secondary);
            border-radius: var(--radius-sm);
            font-size: 0.75rem;
            color: var(--text-tertiary);
        }
    </style>
</head>
<body>
    <div class="app">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="logo">
                    <div class="logo-icon">
                        <i class="fas fa-rocket"></i>
                    </div>
                    <div class="logo-text">
                        <div class="logo-title">Boxxenstopp</div>
                        <div class="logo-subtitle">Quick Actions</div>
                    </div>
                </div>
            </div>
            
            <nav class="nav">
                <div class="nav-section">
                    <div class="nav-title">Management</div>
                    <a href="index.php" class="nav-item">
                        <div class="nav-item-icon">
                            <i class="fas fa-home"></i>
                        </div>
                        Dashboard
                    </a>
                    <a href="quick_actions_manager.php" class="nav-item active">
                        <div class="nav-item-icon">
                            <i class="fas fa-rocket"></i>
                        </div>
                        Schnellaktionen
                    </a>
                    <a href="settings.php" class="nav-item">
                        <div class="nav-item-icon">
                            <i class="fas fa-cog"></i>
                        </div>
                        Einstellungen
                    </a>
                </div>
                
                <div class="nav-section">
                    <div class="nav-title">System</div>
                    <a href="performance.php" class="nav-item">
                        <div class="nav-item-icon">
                            <i class="fas fa-chart-line"></i>
                        </div>
                        Performance
                    </a>
                    <a href="professional_admin_test.php" class="nav-item">
                        <div class="nav-item-icon">
                            <i class="fas fa-flask"></i>
                        </div>
                        Tests
                    </a>
                </div>
            </nav>
        </aside>
        
        <!-- Main Content -->
        <main class="main">
            <header class="header">
                <div class="header-content">
                    <div>
                        <h1 class="header-title">Schnellaktionen Manager</h1>
                        <p class="header-subtitle">Konfiguriere deine persönlichen Dashboard-Schnellaktionen</p>
                    </div>
                    <div class="header-actions">
                        <form method="POST" style="display: inline;">
                            <button type="submit" name="toggle_theme" class="theme-toggle" title="Theme wechseln">
                                <i class="fas fa-<?php echo $current_theme === 'light' ? 'moon' : 'sun'; ?>"></i>
                            </button>
                        </form>
                    </div>
                </div>
            </header>
            
            <div class="content">
                <!-- Messages -->
                <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <?php endif; ?>
                
                <!-- Content Header -->
                <div class="content-header">
                    <h2 class="content-title">Deine Schnellaktionen</h2>
                    <button class="btn btn-primary" onclick="openModal()">
                        <i class="fas fa-plus"></i>
                        Neue Aktion hinzufügen
                    </button>
                </div>
                
                <!-- Actions Grid -->
                <div class="cards-grid" id="sortable-actions">
                    <?php foreach ($quick_actions as $action): ?>
                    <div class="card<?php echo $action['is_active'] ? '' : ' inactive'; ?>" data-id="<?php echo $action['id']; ?>">
                        <div class="drag-handle">
                            <i class="fas fa-grip-vertical"></i>
                        </div>
                        
                        <div class="card-header">
                            <div class="card-icon" style="background-color: <?php echo htmlspecialchars($action['background_color']); ?>;">
                                <i class="<?php echo htmlspecialchars($action['icon']); ?>"></i>
                            </div>
                            <div class="card-content">
                                <h3 class="card-title"><?php echo htmlspecialchars($action['title']); ?></h3>
                                <p class="card-description"><?php echo htmlspecialchars($action['description']); ?></p>
                                <div class="card-url"><?php echo htmlspecialchars($action['url']); ?></div>
                            </div>
                        </div>
                        
                        <div class="card-actions">
                            <button class="card-btn card-btn-edit" onclick="editAction(<?php echo htmlspecialchars(json_encode($action)); ?>)">
                                <i class="fas fa-edit"></i>
                                Bearbeiten
                            </button>
                            
                            <form method="POST" style="display: inline; flex: 1;">
                                <input type="hidden" name="action_id" value="<?php echo $action['id']; ?>">
                                <button type="submit" name="delete_action" class="card-btn card-btn-delete" 
                                        onclick="return confirm('Diese Aktion wirklich löschen?')">
                                    <i class="fas fa-trash"></i>
                                    Löschen
                                </button>
                            </form>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </main>
    </div>

    <!-- Modal -->
    <div class="modal" id="actionModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title" id="modalTitle">Neue Schnellaktion</h3>
                <button type="button" class="modal-close" onclick="closeModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form method="POST" id="actionForm">
                <input type="hidden" name="action_id" id="actionId">
                
                <div class="form-group">
                    <label class="form-label">Titel *</label>
                    <input type="text" name="title" id="actionTitle" class="form-input" required placeholder="z.B. Neues Event">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Beschreibung</label>
                    <input type="text" name="description" id="actionDescription" class="form-input" placeholder="z.B. Erstelle ein neues Event">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Icon *</label>
                    <input type="text" name="icon" id="actionIcon" class="form-input" required placeholder="fas fa-plus" readonly>
                    <div class="icon-picker" id="iconPicker">
                        <?php foreach ($popular_icons as $icon): ?>
                        <div class="icon-option" data-icon="<?php echo $icon; ?>" onclick="selectIcon('<?php echo $icon; ?>')">
                            <i class="<?php echo $icon; ?>"></i>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">URL *</label>
                    <input type="text" name="url" id="actionUrl" class="form-input" required placeholder="z.B. events.php?action=create">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Ziel</label>
                    <select name="target" id="actionTarget" class="form-select">
                        <option value="_self">Gleiches Fenster</option>
                        <option value="_blank">Neues Fenster</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: var(--space-lg);">
                    <div class="form-group" style="flex: 1;">
                        <label class="form-label">Hintergrundfarbe</label>
                        <input type="color" name="background_color" id="actionBgColor" class="color-input" value="#007bff">
                    </div>
                    
                    <div class="form-group" style="flex: 1;">
                        <label class="form-label">Textfarbe</label>
                        <input type="color" name="text_color" id="actionTextColor" class="color-input" value="#ffffff">
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">
                        <i class="fas fa-times"></i>
                        Abbrechen
                    </button>
                    <button type="submit" name="create_action" id="submitBtn" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Speichern
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
    <script>
        // Initialize Sortable
        const sortable = Sortable.create(document.getElementById('sortable-actions'), {
            animation: 150,
            ghostClass: 'sortable-ghost',
            handle: '.drag-handle',
            onEnd: function (evt) {
                updateOrder();
            }
        });
        
        function updateOrder() {
            const items = document.querySelectorAll('.action-item');
            const orders = [];
            
            items.forEach((item, index) => {
                orders.push({
                    id: item.dataset.id,
                    position: index + 1
                });
            });
            
            // Send AJAX request to update order
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            
            const orderInput = document.createElement('input');
            orderInput.type = 'hidden';
            orderInput.name = 'update_order';
            orderInput.value = '1';
            
            const dataInput = document.createElement('input');
            dataInput.type = 'hidden';
            dataInput.name = 'order_data';
            dataInput.value = JSON.stringify(orders);
            
            form.appendChild(orderInput);
            form.appendChild(dataInput);
            document.body.appendChild(form);
            form.submit();
        }
        
        function openModal() {
            document.getElementById('modalTitle').textContent = 'Neue Schnellaktion';
            document.getElementById('actionForm').reset();
            document.getElementById('actionId').value = '';
            document.getElementById('submitBtn').name = 'create_action';
            document.getElementById('submitBtn').innerHTML = '<i class="fas fa-save"></i> Speichern';
            document.getElementById('actionModal').classList.add('active');
        }
        
        function closeModal() {
            document.getElementById('actionModal').classList.remove('active');
        }
        
        function editAction(action) {
            document.getElementById('modalTitle').textContent = 'Schnellaktion bearbeiten';
            document.getElementById('actionId').value = action.id;
            document.getElementById('actionTitle').value = action.title;
            document.getElementById('actionDescription').value = action.description;
            document.getElementById('actionIcon').value = action.icon;
            document.getElementById('actionUrl').value = action.url;
            document.getElementById('actionTarget').value = action.target;
            document.getElementById('actionBgColor').value = action.background_color;
            document.getElementById('actionTextColor').value = action.text_color;
            document.getElementById('submitBtn').name = 'update_action';
            document.getElementById('submitBtn').innerHTML = '<i class="fas fa-save"></i> Aktualisieren';
            
            // Select current icon
            document.querySelectorAll('.icon-option').forEach(opt => opt.classList.remove('selected'));
            const currentIcon = document.querySelector(`[data-icon="${action.icon}"]`);
            if (currentIcon) currentIcon.classList.add('selected');
            
            document.getElementById('actionModal').classList.add('active');
        }
        
        function selectIcon(icon) {
            document.getElementById('actionIcon').value = icon;
            document.querySelectorAll('.icon-option').forEach(opt => opt.classList.remove('selected'));
            document.querySelector(`[data-icon="${icon}"]`).classList.add('selected');
        }
        
        // Close modal on ESC key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });
        
        // Close modal on background click
        document.getElementById('actionModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    </script>
</body>
</html>