<?php
/**
 * Professional Admin Test Suite - Comprehensive System Testing
 * Erweiterte Test-Suite mit neuen Features
 */

session_start();

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

// Try to use database_optimized if available, otherwise fall back to regular database
if (file_exists('../config/database_optimized.php')) {
    require_once '../config/database_optimized.php';
    $db = DatabaseOptimized::getInstance();
    $conn = $db->getConnection();
} else {
    require_once '../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
}

// Try to load performance monitor if available
$performance_monitor_available = false;
if (file_exists('../includes/performance_monitor.php')) {
    require_once '../includes/performance_monitor.php';
    $performance_monitor_available = true;
}

require_once 'includes/helpers.php';

// User-Daten
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';

if ($current_user_id) {
    if (method_exists($db, 'getUserData')) {
        $user_data = $db->getUserData($current_user_id);
        if ($user_data) {
            $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
            $current_user_role = $user_data['role'] ?? 'admin';
        }
    } else {
        $stmt = $conn->prepare("
            SELECT u.full_name, u.username, u.role
            FROM admin_users u
            WHERE u.id = ?
        ");
        $stmt->execute([$current_user_id]);
        $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($user_data) {
            $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
            $current_user_role = $user_data['role'] ?? 'admin';
        }
    }
}

// Theme
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Admin Role Verification
if ($current_user_role !== 'admin') {
    header('Location: index.php');
    exit;
}

// Handle Export
if (isset($_GET['export']) && $_GET['export'] === 'json') {
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="test_results_' . date('Y-m-d_H-i-s') . '.json"');
    // Will be set later
}

// Helper function to format bytes
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    return round($bytes, $precision) . ' ' . $units[$i];
}

// 🧪 Comprehensive Test Suite
$test_results = [];
$overall_score = 0;
$total_tests = 0;
$test_start_time = microtime(true);

// Test 1: Database Connection Performance
$start = microtime(true);
try {
    if (method_exists($db, 'fetchColumn')) {
        $db->fetchColumn("SELECT COUNT(*) FROM admin_users WHERE role = 'admin'");
    } else {
        $stmt = $conn->prepare("SELECT COUNT(*) FROM admin_users WHERE role = 'admin'");
        $stmt->execute();
        $stmt->fetchColumn();
    }
    $db_time = (microtime(true) - $start) * 1000;
    $test_results['database_connection'] = [
        'name' => 'Datenbankverbindung',
        'status' => 'success',
        'time' => $db_time,
        'score' => $db_time < 10 ? 100 : ($db_time < 50 ? 80 : 60),
        'description' => 'Verbindung zur MySQL-Datenbank',
        'details' => 'PDO-Verbindung erfolgreich hergestellt',
        'icon' => 'fas fa-database',
        'color' => $db_time < 10 ? 'success' : ($db_time < 50 ? 'warning' : 'error'),
        'category' => 'database',
        'test_key' => 'database_connection'
    ];
} catch (Exception $e) {
    $test_results['database_connection'] = [
        'name' => 'Datenbankverbindung',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Verbindung zur MySQL-Datenbank',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-database',
        'color' => 'error',
        'category' => 'database',
        'test_key' => 'database_connection'
    ];
}

// Test 2: Admin User Management
$start = microtime(true);
try {
    $stmt = $conn->prepare("
        SELECT u.*, a.name as autohaus_name 
        FROM admin_users u 
        LEFT JOIN autohaus a ON u.autohaus_id = a.id 
        WHERE u.role = 'admin' 
        ORDER BY u.created_at DESC
        LIMIT 10
    ");
    $stmt->execute();
    $admin_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $user_time = (microtime(true) - $start) * 1000;
    $test_results['admin_users'] = [
        'name' => 'Admin-Benutzer-Verwaltung',
        'status' => 'success',
        'time' => $user_time,
        'score' => $user_time < 20 ? 100 : ($user_time < 100 ? 80 : 60),
        'description' => 'Admin-Benutzer abfragen und verwalten',
        'details' => count($admin_users) . ' Admin-Benutzer gefunden',
        'icon' => 'fas fa-users-cog',
        'color' => $user_time < 20 ? 'success' : ($user_time < 100 ? 'warning' : 'error'),
        'count' => count($admin_users),
        'category' => 'database',
        'test_key' => 'admin_users'
    ];
} catch (Exception $e) {
    $test_results['admin_users'] = [
        'name' => 'Admin-Benutzer-Verwaltung',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Admin-Benutzer abfragen und verwalten',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-users-cog',
        'color' => 'error',
        'count' => 0,
        'category' => 'database',
        'test_key' => 'admin_users'
    ];
}

// Test 3: System Statistics
$start = microtime(true);
try {
    $stats = [];
    if (method_exists($db, 'getDashboardStats')) {
        $stats = $db->getDashboardStats();
    } else {
        // Fallback: Basic stats
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM appointments WHERE DATE(slot_date) = CURDATE()");
        $stmt->execute();
        $stats['today_appointments'] = $stmt->fetch()['count'] ?? 0;
    }
    $stats_time = (microtime(true) - $start) * 1000;
    $test_results['system_stats'] = [
        'name' => 'System-Statistiken',
        'status' => 'success',
        'time' => $stats_time,
        'score' => $stats_time < 30 ? 100 : ($stats_time < 150 ? 80 : 60),
        'description' => 'Dashboard-Statistiken berechnen',
        'details' => 'Alle Statistiken erfolgreich geladen',
        'icon' => 'fas fa-chart-bar',
        'color' => $stats_time < 30 ? 'success' : ($stats_time < 150 ? 'warning' : 'error'),
        'data' => $stats,
        'category' => 'performance',
        'test_key' => 'system_stats'
    ];
} catch (Exception $e) {
    $test_results['system_stats'] = [
        'name' => 'System-Statistiken',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Dashboard-Statistiken berechnen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-chart-bar',
        'color' => 'error',
        'category' => 'performance',
        'test_key' => 'system_stats'
    ];
}

// Test 4: Performance Monitoring
$start = microtime(true);
try {
    $performance_stats = [];
    if ($performance_monitor_available && class_exists('PerformanceMonitor')) {
        $performance_stats = PerformanceMonitor::getStats();
    }
    $perf_time = (microtime(true) - $start) * 1000;
    $test_results['performance_monitoring'] = [
        'name' => 'Performance-Monitoring',
        'status' => 'success',
        'time' => $perf_time,
        'score' => $perf_time < 15 ? 100 : ($perf_time < 75 ? 80 : 60),
        'description' => 'Performance-Daten sammeln und analysieren',
        'details' => count($performance_stats) . ' Performance-Einträge',
        'icon' => 'fas fa-tachometer-alt',
        'color' => $perf_time < 15 ? 'success' : ($perf_time < 75 ? 'warning' : 'error'),
        'count' => count($performance_stats),
        'category' => 'performance',
        'test_key' => 'performance_monitoring'
    ];
} catch (Exception $e) {
    $test_results['performance_monitoring'] = [
        'name' => 'Performance-Monitoring',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Performance-Daten sammeln und analysieren',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-tachometer-alt',
        'color' => 'error',
        'count' => 0,
        'category' => 'performance',
        'test_key' => 'performance_monitoring'
    ];
}

// Test 5: Cache Performance
$start = microtime(true);
try {
    $cache_available = false;
    if (method_exists($db, 'getCachedData')) {
        $cached_data = $db->getCachedData('admin_test_cache', function() {
            return ['test' => 'cached_data', 'timestamp' => time()];
        }, 60);
        $cache_available = true;
    }
    $cache_time = (microtime(true) - $start) * 1000;
    $test_results['cache_performance'] = [
        'name' => 'Cache-System',
        'status' => $cache_available ? 'success' : 'warning',
        'time' => $cache_time,
        'score' => $cache_available ? ($cache_time < 5 ? 100 : ($cache_time < 25 ? 80 : 60)) : 50,
        'description' => 'In-Memory-Cache testen',
        'details' => $cache_available ? 'Cache-Test erfolgreich durchgeführt' : 'Cache-System nicht verfügbar',
        'icon' => 'fas fa-memory',
        'color' => $cache_available ? ($cache_time < 5 ? 'success' : ($cache_time < 25 ? 'warning' : 'error')) : 'warning',
        'category' => 'performance',
        'test_key' => 'cache_performance'
    ];
} catch (Exception $e) {
    $test_results['cache_performance'] = [
        'name' => 'Cache-System',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'In-Memory-Cache testen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-memory',
        'color' => 'error',
        'category' => 'performance',
        'test_key' => 'cache_performance'
    ];
}

// Test 6: Security Validation
$start = microtime(true);
try {
    $session_secure = isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'];
    $user_authenticated = $current_user_id && $current_user_role === 'admin';
    $security_time = (microtime(true) - $start) * 1000;
    $security_score = ($session_secure && $user_authenticated) ? 100 : 0;
    
    $test_results['security_validation'] = [
        'name' => 'Sicherheitsvalidierung',
        'status' => $security_score === 100 ? 'success' : 'error',
        'time' => $security_time,
        'score' => $security_score,
        'description' => 'Session und Authentifizierung prüfen',
        'details' => $session_secure ? 'Session sicher' : 'Session unsicher',
        'icon' => 'fas fa-shield-alt',
        'color' => $security_score === 100 ? 'success' : 'error',
        'category' => 'security',
        'test_key' => 'security_validation'
    ];
} catch (Exception $e) {
    $test_results['security_validation'] = [
        'name' => 'Sicherheitsvalidierung',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Session und Authentifizierung prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-shield-alt',
        'color' => 'error',
        'category' => 'security',
        'test_key' => 'security_validation'
    ];
}

// Test 7: File System Access
$start = microtime(true);
try {
    $logs_dir_exists = is_dir('../logs');
    $cache_dir_exists = is_dir('../cache');
    $logs_writable = $logs_dir_exists && is_writable('../logs');
    $cache_writable = $cache_dir_exists && is_writable('../cache');
    $file_time = (microtime(true) - $start) * 1000;
    
    $file_score = 0;
    if ($logs_dir_exists) $file_score += 25;
    if ($logs_writable) $file_score += 25;
    if ($cache_dir_exists) $file_score += 25;
    if ($cache_writable) $file_score += 25;
    
    $test_results['file_system'] = [
        'name' => 'Dateisystem-Zugriff',
        'status' => $file_score >= 50 ? 'success' : 'error',
        'time' => $file_time,
        'score' => $file_score,
        'description' => 'Logs- und Cache-Verzeichnisse prüfen',
        'details' => ($logs_writable && $cache_writable) ? 'Alle Verzeichnisse beschreibbar' : 'Einige Verzeichnisse nicht beschreibbar',
        'icon' => 'fas fa-folder-open',
        'color' => $file_score >= 50 ? 'success' : 'error',
        'category' => 'system',
        'test_key' => 'file_system'
    ];
} catch (Exception $e) {
    $test_results['file_system'] = [
        'name' => 'Dateisystem-Zugriff',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Logs- und Cache-Verzeichnisse prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-folder-open',
        'color' => 'error',
        'category' => 'system',
        'test_key' => 'file_system'
    ];
}

// Test 8: RESY API Connection
$start = microtime(true);
try {
    $resy_api_exists = file_exists(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
    $resy_api_readable = $resy_api_exists && is_readable(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
    
    $api_syntax_valid = false;
    if ($resy_api_exists) {
        $api_content = file_get_contents(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
        $api_syntax_valid = strpos($api_content, 'function validateAndUpdateCookies') !== false ||
                           strpos($api_content, 'function performAutomaticCookieManagement') !== false ||
                           strpos($api_content, 'Hybrid-Ansatz') !== false;
    }
    
    $resy_time = (microtime(true) - $start) * 1000;
    $resy_score = 0;
    if ($resy_api_exists) $resy_score += 40;
    if ($resy_api_readable) $resy_score += 30;
    if ($api_syntax_valid) $resy_score += 30;
    
    $test_results['resy_api'] = [
        'name' => 'RESY API (Hybrid-Ansatz)',
        'status' => $resy_score >= 70 ? 'success' : 'error',
        'time' => $resy_time,
        'score' => $resy_score,
        'description' => 'RESY Data Extraction API prüfen',
        'details' => $api_syntax_valid ? 'API mit Hybrid-Ansatz gefunden' : 'API nicht gefunden oder veraltet',
        'icon' => 'fas fa-cloud-download-alt',
        'color' => $resy_score >= 70 ? 'success' : 'error',
        'version' => $api_syntax_valid ? 'v3 (Hybrid-Ansatz)' : 'unbekannt',
        'category' => 'api',
        'test_key' => 'resy_api'
    ];
} catch (Exception $e) {
    $test_results['resy_api'] = [
        'name' => 'RESY API (Hybrid-Ansatz)',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'RESY Data Extraction API prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-cloud-download-alt',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'resy_api'
    ];
}

// Test 9: Cookie Management System
$start = microtime(true);
try {
    $cookie_file_exists = file_exists(sys_get_temp_dir() . '/resy_cookies_breakthrough.txt');
    $cookie_file_writable = is_writable(sys_get_temp_dir());
    
    $cookie_functions_exist = false;
    if (file_exists(__DIR__ . '/../api/resy_data_extraction_breakthrough.php')) {
        $api_content = file_get_contents(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
        $cookie_functions_exist = strpos($api_content, 'function validateAndUpdateCookies') !== false ||
                                 strpos($api_content, 'function performAutomaticCookieManagement') !== false ||
                                 strpos($api_content, 'function extractCookiesFromFile') !== false;
    }
    
    $cookie_time = (microtime(true) - $start) * 1000;
    $cookie_score = 0;
    if ($cookie_file_writable) $cookie_score += 40;
    if ($cookie_functions_exist) $cookie_score += 40;
    if ($cookie_file_exists) $cookie_score += 20;
    
    $test_results['cookie_management'] = [
        'name' => 'Cookie-Management (Hybrid)',
        'status' => $cookie_score >= 70 ? 'success' : 'error',
        'time' => $cookie_time,
        'score' => $cookie_score,
        'description' => 'Automatische Cookie-Verwaltung prüfen',
        'details' => $cookie_functions_exist ? 'Hybrid-Ansatz Cookie-Management aktiv' : 'Cookie-Management nicht verfügbar',
        'icon' => 'fas fa-cookie-bite',
        'color' => $cookie_score >= 70 ? 'success' : 'error',
        'cookie_file' => $cookie_file_exists ? 'vorhanden' : 'nicht vorhanden',
        'category' => 'api',
        'test_key' => 'cookie_management'
    ];
} catch (Exception $e) {
    $test_results['cookie_management'] = [
        'name' => 'Cookie-Management (Hybrid)',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Automatische Cookie-Verwaltung prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-cookie-bite',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'cookie_management'
    ];
}

// Test 10: RESY System Connectivity
$start = microtime(true);
try {
    $resy_base_url = "https://resy.dtm-reifen.de";
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $resy_base_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $connect_time = curl_getinfo($ch, CURLINFO_CONNECT_TIME);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    $resy_connect_time = (microtime(true) - $start) * 1000;
    $resy_connect_score = 0;
    
    if ($http_code === 200 && empty($curl_error)) {
        $resy_connect_score += 50;
        if ($connect_time < 2) $resy_connect_score += 30;
        if ($connect_time < 5) $resy_connect_score += 20;
    }
    
    $test_results['resy_connectivity'] = [
        'name' => 'RESY-System Verbindung',
        'status' => $resy_connect_score >= 70 ? 'success' : 'error',
        'time' => $resy_connect_time,
        'score' => $resy_connect_score,
        'description' => 'Verbindung zum RESY-System testen',
        'details' => ($http_code === 200 && empty($curl_error)) ? "RESY-System erreichbar (HTTP $http_code)" : "RESY-System nicht erreichbar (HTTP $http_code" . ($curl_error ? ", Fehler: $curl_error" : "") . ")",
        'icon' => 'fas fa-globe',
        'color' => $resy_connect_score >= 70 ? 'success' : 'error',
        'response_time' => round($connect_time, 2) . 's',
        'category' => 'api',
        'test_key' => 'resy_connectivity'
    ];
} catch (Exception $e) {
    $test_results['resy_connectivity'] = [
        'name' => 'RESY-System Verbindung',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Verbindung zum RESY-System testen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-globe',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'resy_connectivity'
    ];
}

// Test 11: Image Proxy System
$start = microtime(true);
try {
    $image_proxy_exists = file_exists(__DIR__ . '/../api/resy_image_proxy.php');
    $image_proxy_readable = $image_proxy_exists && is_readable(__DIR__ . '/../api/resy_image_proxy.php');
    
    $image_proxy_valid = false;
    if ($image_proxy_exists) {
        $image_proxy_content = file_get_contents(__DIR__ . '/../api/resy_image_proxy.php');
        $image_proxy_valid = strpos($image_proxy_content, 'resy_image_proxy') !== false ||
                            strpos($image_proxy_content, 'lgfoto_id') !== false;
    }
    
    $image_proxy_time = (microtime(true) - $start) * 1000;
    $image_proxy_score = 0;
    if ($image_proxy_exists) $image_proxy_score += 40;
    if ($image_proxy_readable) $image_proxy_score += 30;
    if ($image_proxy_valid) $image_proxy_score += 30;
    
    $test_results['image_proxy'] = [
        'name' => 'Image Proxy System',
        'status' => $image_proxy_score >= 70 ? 'success' : 'error',
        'time' => $image_proxy_time,
        'score' => $image_proxy_score,
        'description' => 'RESY Image Proxy prüfen',
        'details' => $image_proxy_valid ? 'Image Proxy verfügbar' : 'Image Proxy nicht verfügbar',
        'icon' => 'fas fa-images',
        'color' => $image_proxy_score >= 70 ? 'success' : 'error',
        'category' => 'api',
        'test_key' => 'image_proxy'
    ];
} catch (Exception $e) {
    $test_results['image_proxy'] = [
        'name' => 'Image Proxy System',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'RESY Image Proxy prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-images',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'image_proxy'
    ];
}

// Test 12: Tire Status Check API
$start = microtime(true);
try {
    $tire_status_exists = file_exists(__DIR__ . '/../api/check_tire_status.php');
    $tire_status_readable = $tire_status_exists && is_readable(__DIR__ . '/../api/check_tire_status.php');
    
    $tire_status_valid = false;
    if ($tire_status_exists) {
        $tire_status_content = file_get_contents(__DIR__ . '/../api/check_tire_status.php');
        $tire_status_valid = strpos($tire_status_content, 'storage_number') !== false ||
                            strpos($tire_status_content, 'searchForStorageNumber') !== false;
    }
    
    $tire_status_time = (microtime(true) - $start) * 1000;
    $tire_status_score = 0;
    if ($tire_status_exists) $tire_status_score += 40;
    if ($tire_status_readable) $tire_status_score += 30;
    if ($tire_status_valid) $tire_status_score += 30;
    
    $test_results['tire_status_api'] = [
        'name' => 'Tire Status Check API',
        'status' => $tire_status_score >= 70 ? 'success' : 'error',
        'time' => $tire_status_time,
        'score' => $tire_status_score,
        'description' => 'Tire Status Check API prüfen',
        'details' => $tire_status_valid ? 'Tire Status API verfügbar' : 'Tire Status API nicht verfügbar',
        'icon' => 'fas fa-tire',
        'color' => $tire_status_score >= 70 ? 'success' : 'error',
        'category' => 'api',
        'test_key' => 'tire_status_api'
    ];
} catch (Exception $e) {
    $test_results['tire_status_api'] = [
        'name' => 'Tire Status Check API',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Tire Status Check API prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-tire',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'tire_status_api'
    ];
}

// Test 13: PHP Configuration
$start = microtime(true);
try {
    $php_version = phpversion();
    $php_extensions = get_loaded_extensions();
    $required_extensions = ['pdo', 'pdo_mysql', 'curl', 'json', 'session'];
    $missing_extensions = array_diff($required_extensions, $php_extensions);
    
    $php_config_time = (microtime(true) - $start) * 1000;
    $php_config_score = 100;
    
    if (version_compare($php_version, '8.0', '<')) {
        $php_config_score -= 30;
    }
    if (!empty($missing_extensions)) {
        $php_config_score -= count($missing_extensions) * 20;
    }
    
    $test_results['php_configuration'] = [
        'name' => 'PHP-Konfiguration',
        'status' => $php_config_score >= 80 ? 'success' : 'error',
        'time' => $php_config_time,
        'score' => $php_config_score,
        'description' => 'PHP-Version und Extensions prüfen',
        'details' => empty($missing_extensions) ? "PHP $php_version mit allen Extensions" : "Fehlende Extensions: " . implode(', ', $missing_extensions),
        'icon' => 'fas fa-code',
        'color' => $php_config_score >= 80 ? 'success' : 'error',
        'version' => $php_version,
        'category' => 'system',
        'test_key' => 'php_configuration'
    ];
} catch (Exception $e) {
    $test_results['php_configuration'] = [
        'name' => 'PHP-Konfiguration',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'PHP-Version und Extensions prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-code',
        'color' => 'error',
        'category' => 'system',
        'test_key' => 'php_configuration'
    ];
}

// Test 14: Database Tables
$start = microtime(true);
try {
    $required_tables = ['admin_users', 'autohaus', 'appointments'];
    $existing_tables = [];
    $missing_tables = [];
    
    foreach ($required_tables as $table) {
        try {
            $stmt = $conn->prepare("SELECT COUNT(*) FROM $table LIMIT 1");
            $stmt->execute();
            $existing_tables[] = $table;
        } catch (Exception $e) {
            $missing_tables[] = $table;
        }
    }
    
    $db_tables_time = (microtime(true) - $start) * 1000;
    $db_tables_score = count($existing_tables) * (100 / count($required_tables));
    
    $test_results['database_tables'] = [
        'name' => 'Datenbank-Tabellen',
        'status' => $db_tables_score >= 75 ? 'success' : 'error',
        'time' => $db_tables_time,
        'score' => round($db_tables_score),
        'description' => 'Erforderliche Tabellen prüfen',
        'details' => empty($missing_tables) ? 'Alle Tabellen vorhanden' : 'Fehlende Tabellen: ' . implode(', ', $missing_tables),
        'icon' => 'fas fa-table',
        'color' => $db_tables_score >= 75 ? 'success' : 'error',
        'tables_found' => count($existing_tables),
        'tables_total' => count($required_tables),
        'category' => 'database',
        'test_key' => 'database_tables'
    ];
} catch (Exception $e) {
    $test_results['database_tables'] = [
        'name' => 'Datenbank-Tabellen',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Erforderliche Tabellen prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-table',
        'color' => 'error',
        'category' => 'database',
        'test_key' => 'database_tables'
    ];
}

// Test 15: Memory Usage
$start = microtime(true);
try {
    $memory_limit = ini_get('memory_limit');
    $memory_usage = memory_get_usage(true);
    $memory_peak = memory_get_peak_usage(true);
    
    $memory_limit_bytes = 128 * 1024 * 1024; // Default 128MB
    if (preg_match('/(\d+)([GMK]?)/i', $memory_limit, $matches)) {
        $value = intval($matches[1]);
        $unit = strtoupper($matches[2]);
        switch ($unit) {
            case 'G': $memory_limit_bytes = $value * 1024 * 1024 * 1024; break;
            case 'M': $memory_limit_bytes = $value * 1024 * 1024; break;
            case 'K': $memory_limit_bytes = $value * 1024; break;
            default: $memory_limit_bytes = $value; break;
        }
    }
    
    $memory_time = (microtime(true) - $start) * 1000;
    $memory_score = 100;
    
    if ($memory_usage > $memory_limit_bytes * 0.8) {
        $memory_score -= 30;
    }
    if ($memory_peak > $memory_limit_bytes * 0.9) {
        $memory_score -= 20;
    }
    
    $test_results['memory_usage'] = [
        'name' => 'Speicherverbrauch',
        'status' => $memory_score >= 80 ? 'success' : 'error',
        'time' => $memory_time,
        'score' => $memory_score,
        'description' => 'Memory-Limit und Verbrauch prüfen',
        'details' => "Verbrauch: " . formatBytes($memory_usage) . " / Limit: $memory_limit",
        'icon' => 'fas fa-memory',
        'color' => $memory_score >= 80 ? 'success' : 'error',
        'usage' => formatBytes($memory_usage),
        'limit' => $memory_limit,
        'category' => 'system',
        'test_key' => 'memory_usage'
    ];
} catch (Exception $e) {
    $test_results['memory_usage'] = [
        'name' => 'Speicherverbrauch',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Memory-Limit und Verbrauch prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-memory',
        'color' => 'error',
        'category' => 'system',
        'test_key' => 'memory_usage'
    ];
}

// NEUE TESTS

// Test 16: Email System
$start = microtime(true);
try {
    // Lade EmailSystem falls verfügbar
    $email_system_available = false;
    if (file_exists(__DIR__ . '/../api/email_system.php')) {
        require_once __DIR__ . '/../api/email_system.php';
        $emailSystem = new EmailSystem($conn);
        $email_system_available = $emailSystem->isAvailable();
    }
    
    $email_templates_table = false;
    $email_logs_table = false;
    $email_templates_count = 0;
    $email_logs_count = 0;
    
    try {
        $stmt = $conn->query("SELECT COUNT(*) as count FROM email_templates LIMIT 1");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $email_templates_table = true;
        $email_templates_count = $result['count'] ?? 0;
    } catch (Exception $e) {
        // Versuche Tabellen zu erstellen
        try {
            require_once __DIR__ . '/../simple_database_updates.php';
        } catch (Exception $e2) {}
    }
    
    try {
        $stmt = $conn->query("SELECT COUNT(*) as count FROM email_logs LIMIT 1");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $email_logs_table = true;
        $email_logs_count = $result['count'] ?? 0;
    } catch (Exception $e) {}
    
    $email_time = (microtime(true) - $start) * 1000;
    $email_score = 0;
    if ($email_templates_table) $email_score += 40;
    if ($email_logs_table) $email_score += 40;
    if ($email_system_available) $email_score += 20;
    
    $test_results['email_system'] = [
        'name' => 'E-Mail-System',
        'status' => $email_score >= 50 ? 'success' : 'warning',
        'time' => $email_time,
        'score' => $email_score,
        'description' => 'E-Mail-Templates und Logs prüfen',
        'details' => ($email_templates_table && $email_logs_table) ? 
            "E-Mail-System vollständig konfiguriert ($email_templates_count Templates, $email_logs_count Logs)" : 
            'E-Mail-Tabellen fehlen teilweise',
        'icon' => 'fas fa-envelope',
        'color' => $email_score >= 50 ? 'success' : 'warning',
        'category' => 'system',
        'test_key' => 'email_system'
    ];
} catch (Exception $e) {
    $test_results['email_system'] = [
        'name' => 'E-Mail-System',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'E-Mail-Templates und Logs prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-envelope',
        'color' => 'error',
        'category' => 'system',
        'test_key' => 'email_system'
    ];
}

// Test 17: Cache Manager API
$start = microtime(true);
try {
    $cache_manager_exists = file_exists(__DIR__ . '/../api/cache_manager.php');
    $cache_manager_readable = $cache_manager_exists && is_readable(__DIR__ . '/../api/cache_manager.php');
    
    $cache_manager_valid = false;
    if ($cache_manager_exists) {
        $cache_manager_content = file_get_contents(__DIR__ . '/../api/cache_manager.php');
        $cache_manager_valid = strpos($cache_manager_content, 'class CacheManager') !== false ||
                              strpos($cache_manager_content, 'getCachedWheelsets') !== false;
    }
    
    $cache_manager_time = (microtime(true) - $start) * 1000;
    $cache_manager_score = 0;
    if ($cache_manager_exists) $cache_manager_score += 40;
    if ($cache_manager_readable) $cache_manager_score += 30;
    if ($cache_manager_valid) $cache_manager_score += 30;
    
    $test_results['cache_manager'] = [
        'name' => 'Cache Manager API',
        'status' => $cache_manager_score >= 70 ? 'success' : 'error',
        'time' => $cache_manager_time,
        'score' => $cache_manager_score,
        'description' => 'Cache Manager API prüfen',
        'details' => $cache_manager_valid ? 'Cache Manager verfügbar' : 'Cache Manager nicht verfügbar',
        'icon' => 'fas fa-database',
        'color' => $cache_manager_score >= 70 ? 'success' : 'error',
        'category' => 'api',
        'test_key' => 'cache_manager'
    ];
} catch (Exception $e) {
    $test_results['cache_manager'] = [
        'name' => 'Cache Manager API',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Cache Manager API prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-database',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'cache_manager'
    ];
}

// Test 18: Backup System
$start = microtime(true);
try {
    // Lade BackupSystem falls verfügbar
    $backup_system_available = false;
    $backup_count = 0;
    if (file_exists(__DIR__ . '/../api/backup_system.php')) {
        require_once __DIR__ . '/../api/backup_system.php';
        $backupSystem = new BackupSystem($conn);
        $backup_system_available = $backupSystem->isAvailable();
        if ($backup_system_available) {
            $backups = $backupSystem->listBackups();
            $backup_count = count($backups);
        }
    }
    
    $backup_dir_exists = is_dir('../backups');
    $backup_dir_writable = $backup_dir_exists && is_writable('../backups');
    
    // Erstelle Backup-Verzeichnis falls nicht vorhanden
    if (!$backup_dir_exists) {
        @mkdir('../backups', 0755, true);
        $backup_dir_exists = is_dir('../backups');
        $backup_dir_writable = $backup_dir_exists && is_writable('../backups');
    }
    
    $backup_time = (microtime(true) - $start) * 1000;
    $backup_score = 0;
    if ($backup_dir_exists) $backup_score += 40;
    if ($backup_dir_writable) $backup_score += 40;
    if ($backup_system_available) $backup_score += 20;
    
    $test_results['backup_system'] = [
        'name' => 'Backup-System',
        'status' => $backup_score >= 50 ? 'success' : 'warning',
        'time' => $backup_time,
        'score' => $backup_score,
        'description' => 'Backup-Verzeichnis und System prüfen',
        'details' => $backup_dir_writable ? 
            "Backup-System verfügbar ($backup_count Backups vorhanden)" : 
            'Backup-Verzeichnis nicht verfügbar',
        'icon' => 'fas fa-hdd',
        'color' => $backup_score >= 50 ? 'success' : 'warning',
        'category' => 'system',
        'test_key' => 'backup_system',
        'backup_count' => $backup_count
    ];
} catch (Exception $e) {
    $test_results['backup_system'] = [
        'name' => 'Backup-System',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Backup-Verzeichnis und System prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-hdd',
        'color' => 'error',
        'category' => 'system',
        'test_key' => 'backup_system'
    ];
}

// Test 19: API Endpoints
$start = microtime(true);
try {
    $api_endpoints = [
        'check_tire_status.php' => file_exists(__DIR__ . '/../api/check_tire_status.php'),
        'cache_manager.php' => file_exists(__DIR__ . '/../api/cache_manager.php'),
        'resy_image_proxy.php' => file_exists(__DIR__ . '/../api/resy_image_proxy.php'),
        'backup_system.php' => file_exists(__DIR__ . '/../api/backup_system.php'),
        'email_system.php' => file_exists(__DIR__ . '/../api/email_system.php'),
    ];
    
    $endpoints_found = count(array_filter($api_endpoints));
    $endpoints_total = count($api_endpoints);
    
    $api_endpoints_time = (microtime(true) - $start) * 1000;
    $api_endpoints_score = ($endpoints_found / $endpoints_total) * 100;
    
    $test_results['api_endpoints'] = [
        'name' => 'API-Endpoints',
        'status' => $api_endpoints_score >= 70 ? 'success' : 'warning',
        'time' => $api_endpoints_time,
        'score' => round($api_endpoints_score),
        'description' => 'Wichtige API-Endpoints prüfen',
        'details' => "$endpoints_found von $endpoints_total Endpoints gefunden",
        'icon' => 'fas fa-plug',
        'color' => $api_endpoints_score >= 70 ? 'success' : 'warning',
        'endpoints_found' => $endpoints_found,
        'endpoints_total' => $endpoints_total,
        'category' => 'api',
        'test_key' => 'api_endpoints'
    ];
} catch (Exception $e) {
    $test_results['api_endpoints'] = [
        'name' => 'API-Endpoints',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Wichtige API-Endpoints prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-plug',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'api_endpoints'
    ];
}

// Test 21: P2 API
$start = microtime(true);
try {
    $p2_api_exists = file_exists(__DIR__ . '/includes/p2_api.php');
    $p2_api_readable = $p2_api_exists && is_readable(__DIR__ . '/includes/p2_api.php');
    
    $p2_api_valid = false;
    $p2_api_class_exists = false;
    if ($p2_api_exists) {
        $p2_api_content = file_get_contents(__DIR__ . '/includes/p2_api.php');
        $p2_api_valid = strpos($p2_api_content, 'class P2API') !== false;
        
        if ($p2_api_valid) {
            require_once __DIR__ . '/includes/p2_api.php';
            $p2_api_class_exists = class_exists('P2API');
        }
    }
    
    // Test P2 API Verbindung
    $p2_api_connection = false;
    if ($p2_api_class_exists) {
        try {
            $p2api = new P2API();
            $test_result = $p2api->apiCall('/health', 'GET', [], 2, 0);
            $p2_api_connection = isset($test_result['success']) || isset($test_result['data']);
        } catch (Exception $e) {
            // Connection test failed, but API class exists
        }
    }
    
    $p2_api_time = (microtime(true) - $start) * 1000;
    $p2_api_score = 0;
    if ($p2_api_exists) $p2_api_score += 30;
    if ($p2_api_readable) $p2_api_score += 20;
    if ($p2_api_valid) $p2_api_score += 30;
    if ($p2_api_connection) $p2_api_score += 20;
    
    $test_results['p2_api'] = [
        'name' => 'P2 API System',
        'status' => $p2_api_score >= 70 ? 'success' : ($p2_api_score >= 50 ? 'warning' : 'error'),
        'time' => $p2_api_time,
        'score' => $p2_api_score,
        'description' => 'P2 API für Reifenangebote prüfen',
        'details' => $p2_api_valid ? 
            ($p2_api_connection ? 'P2 API verfügbar und erreichbar' : 'P2 API verfügbar, Verbindung fehlgeschlagen') : 
            'P2 API nicht verfügbar',
        'icon' => 'fas fa-tire',
        'color' => $p2_api_score >= 70 ? 'success' : ($p2_api_score >= 50 ? 'warning' : 'error'),
        'category' => 'api',
        'test_key' => 'p2_api'
    ];
} catch (Exception $e) {
    $test_results['p2_api'] = [
        'name' => 'P2 API System',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'P2 API für Reifenangebote prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-tire',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'p2_api'
    ];
}

// Test 22: RESY Hybrid API (erweitert)
$start = microtime(true);
try {
    $resy_api_exists = file_exists(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
    $resy_api_readable = $resy_api_exists && is_readable(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
    
    $resy_hybrid_features = [];
    if ($resy_api_exists) {
        $api_content = file_get_contents(__DIR__ . '/../api/resy_data_extraction_breakthrough.php');
        
        $resy_hybrid_features = [
            'validateAndUpdateCookies' => strpos($api_content, 'function validateAndUpdateCookies') !== false,
            'performAutomaticCookieManagement' => strpos($api_content, 'function performAutomaticCookieManagement') !== false,
            'extractCookiesFromFile' => strpos($api_content, 'function extractCookiesFromFile') !== false,
            'hybrid_approach' => strpos($api_content, 'Hybrid-Ansatz') !== false,
            'cookie_validation' => strpos($api_content, 'Cookie-Validierung') !== false,
            'automatic_refresh' => strpos($api_content, 'automatische') !== false || strpos($api_content, 'automatic') !== false
        ];
    }
    
    $features_found = count(array_filter($resy_hybrid_features));
    $features_total = count($resy_hybrid_features);
    
    $resy_hybrid_time = (microtime(true) - $start) * 1000;
    $resy_hybrid_score = 0;
    if ($resy_api_exists) $resy_hybrid_score += 20;
    if ($resy_api_readable) $resy_hybrid_score += 20;
    $resy_hybrid_score += ($features_found / $features_total) * 60;
    
    $test_results['resy_hybrid_api'] = [
        'name' => 'RESY Hybrid API (erweitert)',
        'status' => $resy_hybrid_score >= 70 ? 'success' : ($resy_hybrid_score >= 50 ? 'warning' : 'error'),
        'time' => $resy_hybrid_time,
        'score' => round($resy_hybrid_score),
        'description' => 'RESY Hybrid-Ansatz mit Cookie-Management prüfen',
        'details' => "$features_found von $features_total Hybrid-Features gefunden",
        'icon' => 'fas fa-sync-alt',
        'color' => $resy_hybrid_score >= 70 ? 'success' : ($resy_hybrid_score >= 50 ? 'warning' : 'error'),
        'features_found' => $features_found,
        'features_total' => $features_total,
        'category' => 'api',
        'test_key' => 'resy_hybrid_api'
    ];
} catch (Exception $e) {
    $test_results['resy_hybrid_api'] = [
        'name' => 'RESY Hybrid API (erweitert)',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'RESY Hybrid-Ansatz mit Cookie-Management prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-sync-alt',
        'color' => 'error',
        'category' => 'api',
        'test_key' => 'resy_hybrid_api'
    ];
}

// Test 20: Session Configuration
$start = microtime(true);
try {
    // Lade SecureSessionHandler falls verfügbar
    $session_handler_available = false;
    $session_checks = [];
    if (file_exists(__DIR__ . '/../api/session_handler.php')) {
        require_once __DIR__ . '/../api/session_handler.php';
        $session_handler_available = true;
        $session_checks = SecureSessionHandler::isSecure();
    } else {
        // Fallback: Prüfe manuell
        $session_save_path = session_save_path();
        $session_checks = [
            'cookie_httponly' => ini_get('session.cookie_httponly') == '1',
            'cookie_secure' => ini_get('session.cookie_secure') == '1' || !isset($_SERVER['HTTPS']),
            'use_strict_mode' => ini_get('session.use_strict_mode') == '1',
            'use_only_cookies' => ini_get('session.use_only_cookies') == '1',
            'save_path_writable' => is_writable($session_save_path ?: sys_get_temp_dir())
        ];
    }
    
    $session_time = (microtime(true) - $start) * 1000;
    $session_score = 0;
    $checks_passed = 0;
    foreach ($session_checks as $check => $passed) {
        if ($passed) {
            $checks_passed++;
            $session_score += 20;
        }
    }
    
    $test_results['session_config'] = [
        'name' => 'Session-Konfiguration',
        'status' => $session_score >= 70 ? 'success' : 'warning',
        'time' => $session_time,
        'score' => $session_score,
        'description' => 'Session-Sicherheit prüfen',
        'details' => "$checks_passed von " . count($session_checks) . " Sicherheitsprüfungen bestanden" . 
            ($session_handler_available ? ' (SecureSessionHandler aktiv)' : ' (SecureSessionHandler nicht verfügbar)'),
        'icon' => 'fas fa-lock',
        'color' => $session_score >= 70 ? 'success' : 'warning',
        'category' => 'security',
        'test_key' => 'session_config',
        'checks' => $session_checks
    ];
} catch (Exception $e) {
    $test_results['session_config'] = [
        'name' => 'Session-Konfiguration',
        'status' => 'error',
        'time' => 0,
        'score' => 0,
        'description' => 'Session-Sicherheit prüfen',
        'details' => 'Fehler: ' . $e->getMessage(),
        'icon' => 'fas fa-lock',
        'color' => 'error',
        'category' => 'security',
        'test_key' => 'session_config'
    ];
}

// Calculate Overall Score
foreach ($test_results as $test) {
    $overall_score += $test['score'];
    $total_tests++;
}
$overall_score = $total_tests > 0 ? round($overall_score / $total_tests) : 0;

$total_test_time = (microtime(true) - $test_start_time) * 1000;

// Performance Data
$performance_data = [];
if ($performance_monitor_available && class_exists('PerformanceMonitor')) {
    $performance_data = PerformanceMonitor::end();
} else {
    $performance_data = [
        'total_time' => $total_test_time / 1000,
        'memory_used' => memory_get_usage(true)
    ];
}

// Test Recommendations
$recommendations = [];
if ($overall_score < 80) {
    $recommendations[] = [
        'type' => 'warning',
        'title' => 'System-Optimierung erforderlich',
        'description' => 'Einige Tests zeigen Verbesserungspotential.',
        'action' => 'Performance-Dashboard überprüfen'
    ];
}

// Category-based recommendations
$categories = ['database' => 0, 'api' => 0, 'system' => 0, 'performance' => 0, 'security' => 0];
$category_counts = ['database' => 0, 'api' => 0, 'system' => 0, 'performance' => 0, 'security' => 0];

foreach ($test_results as $test) {
    if (isset($test['category']) && isset($categories[$test['category']])) {
        $categories[$test['category']] += $test['score'];
        $category_counts[$test['category']]++;
    }
}

foreach ($categories as $category => $total_score) {
    if ($category_counts[$category] > 0) {
        $avg_score = $total_score / $category_counts[$category];
        if ($avg_score < 70) {
            $recommendations[] = [
                'type' => 'critical',
                'title' => ucfirst($category) . '-Probleme',
                'description' => "Durchschnittlicher Score: " . round($avg_score) . "/100",
                'action' => ucfirst($category) . '-Bereich überprüfen'
            ];
        }
    }
}

// Export JSON if requested
if (isset($_GET['export']) && $_GET['export'] === 'json') {
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="test_results_' . date('Y-m-d_H-i-s') . '.json"');
    echo json_encode([
        'timestamp' => date('Y-m-d H:i:s'),
        'overall_score' => $overall_score,
        'total_tests' => $total_tests,
        'total_time' => $total_test_time,
        'test_results' => $test_results,
        'recommendations' => $recommendations,
        'performance_data' => $performance_data
    ], JSON_PRETTY_PRINT);
    exit;
}
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Test Suite - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/glassmorphism-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-main {
            margin-left: 300px;
            transition: margin-left 0.3s ease;
            width: calc(100% - 300px);
            padding: 2rem;
            min-height: 100vh;
        }

        .smart-sidebar.collapsed + .admin-main,
        .figma-sidebar.collapsed ~ .admin-main {
            margin-left: 80px;
            width: calc(100% - 80px);
        }

        .test-page-header {
            margin-bottom: 2rem;
        }

        .test-page-header h1 {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .test-page-header p {
            color: #64748b;
            font-size: 1rem;
        }

        .score-display {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 2rem;
            margin: 2rem 0;
        }

        .score-circle {
            position: relative;
            width: 150px;
            height: 150px;
        }

        .score-circle-svg {
            width: 100%;
            height: 100%;
            transform: rotate(-90deg);
        }

        .score-circle-bg {
            fill: none;
            stroke: #e2e8f0;
            stroke-width: 10;
        }

        .score-circle-progress {
            fill: none;
            stroke: #10b981;
            stroke-width: 10;
            stroke-linecap: round;
            stroke-dasharray: 471;
            transition: stroke-dashoffset 0.5s ease;
        }

        .score-value {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
        }

        .score-number {
            font-size: 2.5rem;
            font-weight: 800;
            color: #1e293b;
        }

        .score-label {
            font-size: 0.875rem;
            color: #64748b;
            font-weight: 600;
            text-transform: uppercase;
        }

        .summary-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: #ffffff;
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            text-align: center;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.25rem;
        }

        .stat-label {
            font-size: 0.875rem;
            color: #64748b;
            font-weight: 500;
        }

        .category-filter {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
            margin-bottom: 2rem;
        }

        .filter-btn {
            padding: 0.5rem 1rem;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            background: #ffffff;
            color: #64748b;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
        }

        .filter-btn:hover {
            background: #f8fafc;
            border-color: #3b82f6;
        }

        .filter-btn.active {
            background: #3b82f6;
            color: #ffffff;
            border-color: #3b82f6;
        }

        .tests-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .test-card {
            background: #ffffff;
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            transition: transform 0.2s, box-shadow 0.2s;
            position: relative;
            overflow: hidden;
        }

        .test-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .test-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
        }

        .test-card.success::before {
            background: linear-gradient(90deg, #10b981, #059669);
        }

        .test-card.warning::before {
            background: linear-gradient(90deg, #f59e0b, #d97706);
        }

        .test-card.error::before {
            background: linear-gradient(90deg, #ef4444, #dc2626);
        }

        .test-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 1rem;
        }

        .test-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
        }

        .test-status {
            display: flex;
            align-items: center;
            gap: 0.25rem;
            font-size: 0.75rem;
            font-weight: 600;
            padding: 0.25rem 0.5rem;
            border-radius: 6px;
        }

        .test-status.success {
            background: rgba(16, 185, 129, 0.1);
            color: #10b981;
        }

        .test-status.warning {
            background: rgba(245, 158, 11, 0.1);
            color: #f59e0b;
        }

        .test-status.error {
            background: rgba(239, 68, 68, 0.1);
            color: #ef4444;
        }

        .test-name {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }

        .test-description {
            font-size: 0.875rem;
            color: #64748b;
            margin-bottom: 1rem;
        }

        .test-metrics {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }

        .test-time {
            font-size: 1.25rem;
            font-weight: 700;
            color: #1e293b;
        }

        .test-score-badge {
            font-size: 0.875rem;
            font-weight: 600;
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            background: #f8fafc;
            color: #1e293b;
        }

        .test-details {
            font-size: 0.8125rem;
            color: #9ca3af;
            background: #f8fafc;
            padding: 0.75rem;
            border-radius: 8px;
            border-left: 3px solid #3b82f6;
        }

        .card {
            background: #ffffff;
            border-radius: 12px;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
            overflow: hidden;
        }

        .card-header {
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            background: #f8fafc;
        }

        .card-header h2 {
            font-size: 1.25rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .card-body {
            padding: 1.5rem;
        }

        .recommendations-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1rem;
        }

        .recommendation-card {
            background: #f8fafc;
            border-radius: 8px;
            padding: 1.25rem;
            border: 1px solid #e2e8f0;
        }

        .recommendation-card h4 {
            font-size: 1rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }

        .recommendation-card p {
            font-size: 0.875rem;
            color: #64748b;
            margin-bottom: 0.75rem;
        }

        .recommendation-action {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            background: #3b82f6;
            color: white;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .quick-actions {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
            margin-bottom: 2rem;
        }

        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            transition: all 0.2s;
            text-decoration: none;
            background: #3b82f6;
            color: #ffffff;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            background: #2563eb;
        }

        .btn-secondary {
            background: #6b7280;
        }

        .btn-secondary:hover {
            background: #4b5563;
        }

        .btn-success {
            background: #10b981;
        }

        .btn-success:hover {
            background: #059669;
        }

        .btn-sm {
            font-size: 0.75rem;
            padding: 0.5rem 1rem;
        }

        .test-loading {
            opacity: 0.6;
            pointer-events: none;
        }

        .test-loading::after {
            content: '';
            display: inline-block;
            width: 12px;
            height: 12px;
            border: 2px solid #3b82f6;
            border-top-color: transparent;
            border-radius: 50%;
            animation: spin 0.6s linear infinite;
            margin-left: 0.5rem;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        @media (max-width: 768px) {
            .admin-main {
                margin-left: 0 !important;
                width: 100% !important;
                padding: 1rem;
            }

            .tests-grid {
                grid-template-columns: 1fr;
            }

            .summary-stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <div class="test-page-header">
                <h1>
                    <i class="fas fa-vial"></i>
                    System Test Suite
                </h1>
                <p>Umfassende Systemdiagnose und Validierung</p>
            </div>

            <!-- Score Display -->
            <div class="card">
                <div class="card-body">
                    <div class="score-display">
                        <div class="score-circle">
                            <svg class="score-circle-svg" viewBox="0 0 150 150">
                                <circle class="score-circle-bg" cx="75" cy="75" r="70"></circle>
                                <circle class="score-circle-progress" cx="75" cy="75" r="70" 
                                        style="stroke-dashoffset: calc(471 - (471 * <?php echo $overall_score; ?>) / 100);"></circle>
                            </svg>
                            <div class="score-value">
                                <div class="score-number"><?php echo $overall_score; ?></div>
                                <div class="score-label">Score</div>
                            </div>
                        </div>
                        <div>
                            <div class="stat-value"><?php echo count($test_results); ?></div>
                            <div class="stat-label">Tests durchgeführt</div>
                        </div>
                        <div>
                            <div class="stat-value"><?php echo number_format($total_test_time / 1000, 3); ?>s</div>
                            <div class="stat-label">Gesamtlaufzeit</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="quick-actions">
                <a href="?run_tests=1" class="btn">
                    <i class="fas fa-play"></i>
                    Tests erneut ausführen
                </a>
                <a href="?export=json" class="btn btn-secondary">
                    <i class="fas fa-download"></i>
                    Ergebnisse exportieren (JSON)
                </a>
                <a href="../setup_systems.php" class="btn btn-success" target="_blank">
                    <i class="fas fa-cog"></i>
                    Systeme einrichten
                </a>
                <a href="performance.php" class="btn btn-secondary">
                    <i class="fas fa-chart-line"></i>
                    Performance Dashboard
                </a>
                <a href="cache_management.php" class="btn btn-secondary">
                    <i class="fas fa-database"></i>
                    Cache verwalten
                </a>
            </div>

            <!-- Summary Statistics -->
            <div class="summary-stats">
                <div class="stat-card">
                    <div class="stat-value"><?php echo count($test_results); ?></div>
                    <div class="stat-label">Tests durchgeführt</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?php echo count(array_filter($test_results, function($test) { return $test['status'] === 'success'; })); ?></div>
                    <div class="stat-label">Tests erfolgreich</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?php echo count(array_filter($test_results, function($test) { return $test['status'] === 'error'; })); ?></div>
                    <div class="stat-label">Tests fehlgeschlagen</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?php echo formatBytes($performance_data['memory_used'] ?? memory_get_usage(true)); ?></div>
                    <div class="stat-label">Speicherverbrauch</div>
                </div>
            </div>

            <!-- Category Filter -->
            <div class="category-filter">
                <button class="filter-btn active" onclick="filterTests('all')">Alle</button>
                <button class="filter-btn" onclick="filterTests('database')">Datenbank</button>
                <button class="filter-btn" onclick="filterTests('api')">API</button>
                <button class="filter-btn" onclick="filterTests('system')">System</button>
                <button class="filter-btn" onclick="filterTests('performance')">Performance</button>
                <button class="filter-btn" onclick="filterTests('security')">Sicherheit</button>
            </div>

            <!-- Test Results -->
            <div class="tests-grid" id="testsGrid">
                <?php foreach ($test_results as $key => $test): ?>
                <div class="test-card <?php echo $test['color']; ?>" data-category="<?php echo $test['category'] ?? 'other'; ?>">
                    <div class="test-header">
                        <div class="test-icon" style="background: linear-gradient(135deg, 
                            <?php echo $test['color'] === 'success' ? '#10b981, #059669' : 
                                ($test['color'] === 'warning' ? '#f59e0b, #d97706' : 
                                '#ef4444, #dc2626'); ?>);">
                            <i class="<?php echo $test['icon']; ?>"></i>
                        </div>
                        <div class="test-status <?php echo $test['status']; ?>">
                            <i class="fas fa-<?php echo $test['status'] === 'success' ? 'check' : ($test['status'] === 'warning' ? 'exclamation-triangle' : 'times'); ?>"></i>
                            <?php echo ucfirst($test['status']); ?>
                        </div>
                    </div>
                    
                    <h3 class="test-name"><?php echo htmlspecialchars($test['name']); ?></h3>
                    <p class="test-description"><?php echo htmlspecialchars($test['description']); ?></p>
                    
                    <div class="test-metrics">
                        <div class="test-time"><?php echo number_format($test['time'], 2); ?>ms</div>
                        <div class="test-score-badge">Score: <?php echo $test['score']; ?>/100</div>
                    </div>
                    
                    <div class="test-details"><?php echo htmlspecialchars($test['details']); ?></div>
                    
                    <?php if (isset($test['test_key'])): ?>
                    <div style="margin-top: 0.75rem;">
                        <button class="btn btn-sm" onclick="runSingleTest('<?php echo $test['test_key']; ?>', this)" style="font-size: 0.75rem; padding: 0.5rem 1rem;">
                            <i class="fas fa-redo"></i>
                            Test erneut ausführen
                        </button>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>

            <!-- Recommendations -->
            <?php if (!empty($recommendations)): ?>
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-lightbulb"></i> Empfehlungen</h2>
                </div>
                <div class="card-body">
                    <div class="recommendations-grid">
                        <?php foreach ($recommendations as $rec): ?>
                        <div class="recommendation-card">
                            <h4><?php echo htmlspecialchars($rec['title']); ?></h4>
                            <p><?php echo htmlspecialchars($rec['description']); ?></p>
                            <span class="recommendation-action"><?php echo htmlspecialchars($rec['action']); ?></span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php else: ?>
            <div class="card">
                <div class="card-body" style="text-align: center; padding: 3rem;">
                    <i class="fas fa-check-circle" style="font-size: 3rem; color: #10b981; margin-bottom: 1rem;"></i>
                    <h3 style="color: #1e293b; margin-bottom: 0.5rem;">Perfekte Systemleistung!</h3>
                    <p style="color: #64748b;">Alle Tests wurden erfolgreich durchgeführt. Das System läuft optimal.</p>
                </div>
            </div>
            <?php endif; ?>
        </main>
    </div>

    <script>
        function filterTests(category) {
            const testCards = document.querySelectorAll('.test-card');
            const filterBtns = document.querySelectorAll('.filter-btn');
            
            filterBtns.forEach(btn => btn.classList.remove('active'));
            event.target.classList.add('active');
            
            testCards.forEach(card => {
                if (category === 'all' || card.dataset.category === category) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });
        }

        // Animate score circle
        document.addEventListener('DOMContentLoaded', function() {
            const scoreNumber = document.querySelector('.score-number');
            const targetScore = <?php echo $overall_score; ?>;
            let currentScore = 0;
            const duration = 2000;
            const increment = targetScore / (duration / 16);
            
            const timer = setInterval(() => {
                currentScore += increment;
                if (currentScore >= targetScore) {
                    currentScore = targetScore;
                    clearInterval(timer);
                }
                scoreNumber.textContent = Math.floor(currentScore);
            }, 16);
        });

        // Run single test
        function runSingleTest(testKey, button) {
            const testCard = button.closest('.test-card');
            const originalContent = testCard.innerHTML;
            
            // Show loading state
            testCard.classList.add('test-loading');
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Test läuft...';
            
            fetch('api/run_single_test.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'test_key=' + encodeURIComponent(testKey)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('HTTP ' + response.status);
                }
                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                    return response.text().then(text => {
                        throw new Error('Ungültige Antwort: ' + text.substring(0, 100));
                    });
                }
                return response.json();
            })
            .then(data => {
                if (data.success && data.result) {
                    // Update test card
                    const result = data.result;
                    const statusClass = result.color;
                    
                    // Update status
                    const statusEl = testCard.querySelector('.test-status');
                    if (statusEl) {
                        statusEl.className = 'test-status ' + result.status;
                        statusEl.innerHTML = '<i class="fas fa-' + 
                            (result.status === 'success' ? 'check' : (result.status === 'warning' ? 'exclamation-triangle' : 'times')) + 
                            '"></i> ' + result.status.charAt(0).toUpperCase() + result.status.slice(1);
                    }
                    
                    // Update score
                    const scoreEl = testCard.querySelector('.test-score-badge');
                    if (scoreEl) {
                        scoreEl.textContent = 'Score: ' + result.score + '/100';
                    }
                    
                    // Update time
                    const timeEl = testCard.querySelector('.test-time');
                    if (timeEl) {
                        timeEl.textContent = parseFloat(result.time).toFixed(2) + 'ms';
                    }
                    
                    // Update details
                    const detailsEl = testCard.querySelector('.test-details');
                    if (detailsEl) {
                        detailsEl.textContent = result.details;
                    }
                    
                    // Update card class
                    testCard.className = 'test-card ' + statusClass;
                    
                    // Show success message
                    showNotification('Test erfolgreich ausgeführt', 'success');
                    
                    // Reload page after 1 second to update overall score
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification('Test fehlgeschlagen: ' + (data.error || 'Unbekannter Fehler'), 'error');
                    testCard.innerHTML = originalContent;
                }
            })
            .catch(error => {
                showNotification('Fehler beim Ausführen des Tests: ' + error.message, 'error');
                testCard.innerHTML = originalContent;
            })
            .finally(() => {
                testCard.classList.remove('test-loading');
            });
        }

        function showNotification(message, type) {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
                color: white;
                padding: 1rem 1.5rem;
                border-radius: 8px;
                font-weight: 600;
                box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
                z-index: 10000;
                animation: slideIn 0.3s ease;
            `;
            notification.textContent = message;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }
    </script>
</body>
</html>
