<?php
/**
 * API-Endpoint für Etikett-Vorschau
 * Generiert eine 1:1 Vorschau des zu druckenden Etiketts
 */

session_start();
require_once '../config/database.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Nicht autorisiert']);
    exit;
}

// Output-Buffering starten
ob_start();

// JSON-Header setzen
header('Content-Type: application/json; charset=utf-8');

// Fehler nicht anzeigen (für sauberes JSON)
error_reporting(0);
ini_set('display_errors', 0);

// Bereinige alle String-Werte im Response-Array für JSON
function cleanForJson($data) {
    if (is_array($data)) {
        foreach ($data as $key => $value) {
            $data[$key] = cleanForJson($value);
        }
    } elseif (is_string($data)) {
        $encodings = ['Windows-1252', 'CP850', 'ISO-8859-1', 'UTF-8'];
        foreach ($encodings as $enc) {
            $converted = @mb_convert_encoding($data, 'UTF-8', $enc);
            if ($converted !== false && mb_check_encoding($converted, 'UTF-8')) {
                $data = $converted;
                break;
            }
        }
        $data = mb_convert_encoding($data, 'UTF-8', 'UTF-8');
        $data = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $data);
        if (!mb_check_encoding($data, 'UTF-8')) {
            $data = utf8_encode($data);
        }
    }
    return $data;
}

// Generiere Vorschau
function generatePreview($name, $kennzeichen) {
    try {
        $docx_template = __DIR__ . '/../Reifeneinlagerungsetikett.docx';
        
        if (!file_exists($docx_template)) {
            return ['success' => false, 'message' => 'Word-Dokument nicht gefunden'];
        }
        
        // 1. Word-Dokument verarbeiten
        $zip = new ZipArchive();
        $temp_file = sys_get_temp_dir() . '\\etikett_preview_' . uniqid() . '.docx';
        copy($docx_template, $temp_file);
        
        if ($zip->open($temp_file) !== TRUE) {
            return ['success' => false, 'message' => 'Kann DOCX-Datei nicht öffnen'];
        }
        
        $content = $zip->getFromName('word/document.xml');
        if ($content === false) {
            $zip->close();
            return ['success' => false, 'message' => 'Kann document.xml nicht lesen'];
        }
        
        // Platzhalter ersetzen
        $content = str_replace('[name1]', htmlspecialchars($name, ENT_XML1, 'UTF-8'), $content);
        $content = str_replace('[kennzeichen]', htmlspecialchars($kennzeichen, ENT_XML1, 'UTF-8'), $content);
        
        $zip->addFromString('word/document.xml', $content);
        $zip->close();
        
        // 2. Versuche PDF zu generieren und Maße zu extrahieren
        $pdf_path = null;
        $preview_html = null;
        $preview_base64 = null;
        $page_width_mm = null;
        $page_height_mm = null;
        $margin_mm = null;
        $font_size = 14;
        
        if (class_exists('PhpOffice\PhpWord\IOFactory')) {
            try {
                $phpWord = \PhpOffice\PhpWord\IOFactory::load($temp_file);
                
                // Extrahiere Seitenmaße aus PhpWord
                $sections = $phpWord->getSections();
                if (!empty($sections)) {
                    $section = $sections[0];
                    $style = $section->getStyle();
                    
                    // Seitenbreite und -höhe (in Twips)
                    $pageWidth = $style->getPageSizeW();
                    $pageHeight = $style->getPageSizeH();
                    
                    if ($pageWidth && $pageHeight) {
                        // Konvertiere Twips zu mm
                        // 1 Twip = 1/20 Punkt = 1/1440 Zoll
                        // 1 Zoll = 25.4 mm
                        // 1 Twip = 25.4 / 1440 = 0.0176389 mm
                        $page_width_mm = round($pageWidth * 0.0176389, 2);
                        $page_height_mm = round($pageHeight * 0.0176389, 2);
                    }
                    
                    // Ränder (in Twips)
                    $marginTop = $style->getMarginTop();
                    if ($marginTop) {
                        $margin_mm = round($marginTop * 0.0176389, 2);
                    } else {
                        $margin_mm = round($style->getMarginLeft() * 0.0176389, 2);
                    }
                }
                
                // Extrahiere Schriftgröße aus dem ersten Absatz
                $sections = $phpWord->getSections();
                if (!empty($sections)) {
                    $elements = $sections[0]->getElements();
                    foreach ($elements as $element) {
                        if (method_exists($element, 'getElements')) {
                            $paragraphs = $element->getElements();
                            foreach ($paragraphs as $paragraph) {
                                if (method_exists($paragraph, 'getFontStyle')) {
                                    $font = $paragraph->getFontStyle();
                                    if ($font && $font->getSize()) {
                                        $font_size = $font->getSize();
                                        break 2;
                                    }
                                }
                            }
                        }
                    }
                }
                
                $writer = \PhpOffice\PhpWord\IOFactory::createWriter($phpWord, 'PDF');
                $pdf_path = sys_get_temp_dir() . '\\etikett_preview_' . uniqid() . '.pdf';
                $writer->save($pdf_path);
                
                // Versuche PDF als Bild zu konvertieren (falls ImageMagick verfügbar)
                // Das ist die exakteste 1:1 Vorschau!
                if (extension_loaded('imagick') && file_exists($pdf_path)) {
                    try {
                        $imagick = new Imagick();
                        $imagick->setResolution(200, 200); // Sehr hohe Auflösung für exakte Vorschau
                        $imagick->readImage($pdf_path . '[0]'); // Erste Seite
                        $imagick->setImageFormat('png');
                        $imagick->setImageCompressionQuality(100); // Beste Qualität
                        
                        // Extrahiere exakte Maße aus dem PDF
                        $width_px = $imagick->getImageWidth();
                        $height_px = $imagick->getImageHeight();
                        $dpi = 200; // Die gesetzte Auflösung
                        $page_width_mm = round(($width_px / $dpi) * 25.4, 2);
                        $page_height_mm = round(($height_px / $dpi) * 25.4, 2);
                        
                        // Konvertiere zu Base64
                        $preview_base64 = base64_encode($imagick->getImageBlob());
                        $imagick->clear();
                        $imagick->destroy();
                    } catch (Exception $e) {
                        // ImageMagick fehlgeschlagen, verwende HTML-Vorschau
                    }
                }
            } catch (Exception $e) {
                // PDF-Generierung fehlgeschlagen
            }
        }
        
        // 3. Falls kein Bild verfügbar, erstelle HTML-Vorschau mit exakten Maßen
        if (!$preview_base64) {
            // Versuche Maße aus PDF zu extrahieren (wichtig für exakte Vorschau!)
            if (!$page_width_mm || !$page_height_mm) {
                if ($pdf_path && class_exists('Imagick')) {
                    try {
                        $imagick = new Imagick();
                        $imagick->setResolution(72, 72);
                        $imagick->readImage($pdf_path . '[0]');
                        $width_px = $imagick->getImageWidth();
                        $height_px = $imagick->getImageHeight();
                        $dpi = 72; // Standard DPI
                        $page_width_mm = round(($width_px / $dpi) * 25.4, 2);
                        $page_height_mm = round(($height_px / $dpi) * 25.4, 2);
                        $imagick->clear();
                        $imagick->destroy();
                    } catch (Exception $e) {
                        // Fehler beim Lesen der PDF-Maße
                    }
                }
            }
            
            // Fallback: Versuche Maße aus Word-Dokument XML zu extrahieren
            if (!$page_width_mm || !$page_height_mm) {
                $zip2 = new ZipArchive();
                if ($zip2->open($temp_file) === TRUE) {
                    $doc_content = $zip2->getFromName('word/document.xml');
                    
                    // Suche nach Section Properties
                    if (preg_match('/<w:sectPr>(.*?)<\/w:sectPr>/s', $doc_content, $sect_match)) {
                        $sect = $sect_match[1];
                        
                        // Seitenbreite und -höhe
                        if (preg_match('/<w:pgSz[^>]*w:w="([^"]*)"[^>]*w:h="([^"]*)"[^>]*>/', $sect, $size_match)) {
                            $w_twips = intval($size_match[1]);
                            $h_twips = intval($size_match[2]);
                            $page_width_mm = round(($w_twips / 20) / 2.83465, 2);
                            $page_height_mm = round(($h_twips / 20) / 2.83465, 2);
                        }
                        
                        // Seitenränder
                        if (preg_match('/<w:pgMar[^>]*w:top="([^"]*)"[^>]*>/', $sect, $margin_match)) {
                            $margin_twips = intval($margin_match[1]);
                            $margin_mm = round(($margin_twips / 20) / 2.83465, 2);
                        }
                    }
                    
                    // Extrahiere Schriftgröße
                    if ($font_size == 14 && $doc_content) {
                        preg_match_all('/<w:sz[^>]*w:val="([^"]*)"[^>]*>/', $doc_content, $size_matches);
                        if (!empty($size_matches[1])) {
                            $font_size = intval($size_matches[1][0]) / 2; // Halbe Punkte zu Punkten
                        }
                    }
                    
                    $zip2->close();
                }
            }
            
            // Finale Fallback-Werte (falls nichts gefunden wurde)
            // WICHTIG: Exakte Maße aus dem Word-Dokument: 100.01mm × 55.02mm
            // Exakte Seitenränder aus Word-Dokument:
            // - Oben: 17 mm (964 Twips)
            // - Rechts: 25.01 mm (1418 Twips)
            // - Unten: 20 mm (1134 Twips)
            // - Links: 25.01 mm (1418 Twips)
            if (!$page_width_mm || !$page_height_mm) {
                // Exakte Maße aus Reifeneinlagerungsetikett.docx (5670 Twips × 3119 Twips)
                $page_width_mm = 100.01; // Exakte Breite aus Word-Dokument
                $page_height_mm = 55.02; // Exakte Höhe aus Word-Dokument
            }
            // Exakte Ränder aus Word-Dokument verwenden
            // Standard-Ränder aus Word-Dokument (falls nicht extrahiert)
            $margin_top_mm = 17;    // Oben: 17 mm
            $margin_right_mm = 25.01; // Rechts: 25.01 mm
            $margin_bottom_mm = 20; // Unten: 20 mm
            $margin_left_mm = 25.01; // Links: 25.01 mm
            
            // Wenn bereits extrahiert, verwende diese Werte
            if ($margin_mm) {
                $margin_top_mm = $margin_mm;
                $margin_right_mm = $margin_mm;
                $margin_bottom_mm = $margin_mm;
                $margin_left_mm = $margin_mm;
            }
            
            // Erstelle HTML-Vorschau mit EXAKTEN Maßen und Rändern aus dem Word-Dokument
            // WICHTIG: Im Word-Dokument sind beide Platzhalter im GLEICHEN Absatz,
            // aber mit einem expliziten Zeilenumbruch (w:br) zwischen ihnen!
            // Run 0: [name1]
            // Run 1: ZEILENUMBRUCH
            // Run 2: [kennzeichen]
            // Ausrichtung: Links (Standard, nicht explizit definiert)
            // Exakte Seitenränder:
            // - Oben: 17 mm (964 Twips)
            // - Rechts: 25.01 mm (1418 Twips)
            // - Unten: 20 mm (1134 Twips)
            // - Links: 25.01 mm (1418 Twips)
            // Schriftgröße: 14pt, Fettdruck
            // Die Positionierung muss EXAKT wie im Word-Dokument sein - beide Texte
            // stehen an der gleichen horizontalen Position (links), nur vertikal getrennt durch Zeilenumbruch
            $preview_html = '
                <div style="
                    width: ' . $page_width_mm . 'mm;
                    height: ' . $page_height_mm . 'mm;
                    background: white;
                    padding-top: ' . $margin_top_mm . 'mm;
                    padding-right: ' . $margin_right_mm . 'mm;
                    padding-bottom: ' . $margin_bottom_mm . 'mm;
                    padding-left: ' . $margin_left_mm . 'mm;
                    box-sizing: border-box;
                    font-family: \'Times New Roman\', Times, serif;
                    font-size: ' . $font_size . 'pt;
                    line-height: 1.2;
                    color: #000;
                    border: 1px solid #ddd;
                    margin: 0 auto;
                    position: relative;
                    text-align: left;
                ">
                    <div style="
                        font-weight: bold;
                        font-size: ' . $font_size . 'pt;
                        word-wrap: break-word;
                        color: #000;
                        margin: 0;
                        padding: 0;
                        line-height: 1.2;
                        text-align: left;
                        white-space: pre-line;
                    ">' . htmlspecialchars($name, ENT_QUOTES, 'UTF-8') . "\n" . htmlspecialchars($kennzeichen, ENT_QUOTES, 'UTF-8') . '</div>
                </div>';
        }
        
        // Aufräumen
        @unlink($temp_file);
        if ($pdf_path) @unlink($pdf_path);
        
        return [
            'success' => true,
            'preview_type' => $preview_base64 ? 'image' : 'html',
            'preview_image' => $preview_base64 ? 'data:image/png;base64,' . $preview_base64 : null,
            'preview_html' => $preview_html,
            'page_width_mm' => $page_width_mm,
            'page_height_mm' => $page_height_mm,
            'name' => $name,
            'kennzeichen' => $kennzeichen
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Fehler beim Generieren der Vorschau: ' . $e->getMessage()
        ];
    }
}

// API-Handler
try {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'get_preview') {
        ob_clean();
        
        $name = $_POST['name'] ?? '';
        $kennzeichen = $_POST['kennzeichen'] ?? '';
        
        if (empty($name) || empty($kennzeichen)) {
            $response = [
                'success' => false,
                'message' => 'Name und Kennzeichen sind erforderlich'
            ];
        } else {
            $response = generatePreview($name, $kennzeichen);
        }
        
        $response = cleanForJson($response);
        
        $json_options = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_INVALID_UTF8_IGNORE;
        $json_output = json_encode($response, $json_options);
        
        if ($json_output === false) {
            throw new Exception('JSON-Encoding fehlgeschlagen: ' . json_last_error_msg());
        }
        
        ob_clean();
        echo $json_output;
        ob_end_flush();
        exit;
    } else {
        http_response_code(400);
        ob_clean();
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Ungültige Anfrage']);
        ob_end_flush();
        exit;
    }
} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => 'Server-Fehler: ' . $e->getMessage()
    ]);
    ob_end_flush();
    exit;
}

