<?php
/**
 * Admin-Seite zum Verwalten von Benachrichtigungs-Triggern
 * Erlaubt es, Trigger zu aktivieren/deaktivieren und neue hinzuzufügen
 */

session_start();
require_once '../config/database.php';
require_once '../includes/notification_triggers.php';
require_once 'includes/helpers.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';

if ($current_user_id) {
    $stmt = $conn->prepare("SELECT role, username FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_role = $user_data['role'] ?? 'admin';
    $current_user_name = $user_data['username'] ?? 'Administrator';
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Handle actions
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['toggle_trigger'])) {
        $trigger_name = $_POST['trigger_name'] ?? '';
        $enabled = isset($_POST['enabled']) && $_POST['enabled'] == '1';
        
        if ($trigger_name) {
            setTriggerEnabled($trigger_name, $enabled);
            $_SESSION['notification_message'] = "Trigger '$trigger_name' wurde " . ($enabled ? 'aktiviert' : 'deaktiviert') . '.';
            $_SESSION['notification_message_type'] = 'success';
            header('Location: notification_triggers.php'); // POST-Redirect-GET
            exit;
        }
    } elseif (isset($_POST['create_trigger'])) {
        $trigger_name = $_POST['trigger_name'] ?? '';
        $title_template = $_POST['title_template'] ?? '';
        $message_template = $_POST['message_template'] ?? '';
        $type = $_POST['type'] ?? 'info';
        $url = $_POST['url'] ?? '/profile.php';
        $send_push = isset($_POST['send_push']) && $_POST['send_push'] == '1';
        $enabled = isset($_POST['enabled']) && $_POST['enabled'] == '1';
        
        if ($trigger_name && $title_template && $message_template) {
            // Validiere Trigger-Name (nur alphanumerisch und Unterstriche)
            if (!preg_match('/^[a-z0-9_]+$/', $trigger_name)) {
                $_SESSION['notification_message'] = 'Ungültiger Trigger-Name. Nur Kleinbuchstaben, Zahlen und Unterstriche erlaubt.';
                $_SESSION['notification_message_type'] = 'error';
            } else {
                // Prüfe ob Trigger bereits existiert
                $triggers = getAllTriggers();
                if (isset($triggers[$trigger_name])) {
                    $_SESSION['notification_message'] = "Trigger '$trigger_name' existiert bereits.";
                    $_SESSION['notification_message_type'] = 'error';
                } else {
                    // Füge neuen Trigger zur Datei hinzu
                    $trigger_file = __DIR__ . '/../includes/notification_triggers.php';
                    $trigger_content = file_get_contents($trigger_file);
                    
                    // Finde die Stelle vor dem schließenden Array-Bracket
                    $insert_pos = strrpos($trigger_content, '];');
                    if ($insert_pos !== false) {
                        $new_trigger = "\n    '{$trigger_name}' => [\n";
                        $new_trigger .= "        'enabled' => " . ($enabled ? 'true' : 'false') . ",\n";
                        $new_trigger .= "        'title_template' => " . var_export($title_template, true) . ",\n";
                        $new_trigger .= "        'message_template' => " . var_export($message_template, true) . ",\n";
                        $new_trigger .= "        'type' => " . var_export($type, true) . ",\n";
                        $new_trigger .= "        'url' => " . var_export($url, true) . ",\n";
                        $new_trigger .= "        'send_push' => " . ($send_push ? 'true' : 'false') . ",\n";
                        $new_trigger .= "    ],";
                        
                        $trigger_content = substr_replace($trigger_content, $new_trigger . "\n", $insert_pos, 0);
                        
                        if (file_put_contents($trigger_file, $trigger_content)) {
                            // Speichere enabled-Status in Datenbank
                            if ($enabled) {
                                setTriggerEnabled($trigger_name, true);
                            }
                            $_SESSION['notification_message'] = "Trigger '$trigger_name' erfolgreich erstellt.";
                            $_SESSION['notification_message_type'] = 'success';
                        } else {
                            $_SESSION['notification_message'] = 'Fehler beim Speichern des Triggers.';
                            $_SESSION['notification_message_type'] = 'error';
                        }
                    } else {
                        $_SESSION['notification_message'] = 'Fehler beim Lesen der Trigger-Datei.';
                        $_SESSION['notification_message_type'] = 'error';
                    }
                }
            }
            header('Location: notification_triggers.php'); // POST-Redirect-GET
            exit;
        }
    }
}

// Hole Nachricht aus Session (POST-Redirect-GET Pattern)
if (isset($_SESSION['notification_message'])) {
    $message = $_SESSION['notification_message'];
    $message_type = $_SESSION['notification_message_type'] ?? 'info';
    unset($_SESSION['notification_message']);
    unset($_SESSION['notification_message_type']);
}

// Get all triggers
$triggers = getAllTriggers();

// Get trigger names for validation (as JSON) - sicher escaped
$trigger_names = array_keys($triggers);
$triggers_safe = [];
foreach ($triggers as $name => $trigger) {
    $triggers_safe[$name] = [
        'title_template' => htmlspecialchars($trigger['title_template'] ?? '', ENT_QUOTES, 'UTF-8'),
        'message_template' => htmlspecialchars($trigger['message_template'] ?? '', ENT_QUOTES, 'UTF-8'),
        'type' => htmlspecialchars($trigger['type'] ?? 'info', ENT_QUOTES, 'UTF-8'),
        'enabled' => isset($trigger['enabled']) ? (bool)$trigger['enabled'] : false
    ];
}

$current_page = 'notification_triggers';
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Benachrichtigungs-Trigger - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link rel="stylesheet" href="assets/css/admin-zoom.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #003057;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --border: #dee2e6;
            --shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }

        .admin-container {
            display: flex;
            min-height: 100vh;
            background: #f5f7fa;
        }

        .admin-main {
            flex: 1;
            padding: 2rem;
            margin-left: 300px;
        }

        .admin-header {
            margin-bottom: 2rem;
        }

        .admin-header h1 {
            font-size: 2rem;
            font-weight: 700;
            color: #343a40;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .admin-header h1 i {
            color: var(--primary);
        }

        .info-box {
            background: #e3f2fd;
            border-left: 4px solid var(--info);
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .triggers-grid {
            display: grid;
            gap: 1.5rem;
        }

        .trigger-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
        }

        .trigger-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border);
        }

        .trigger-name {
            font-size: 1.25rem;
            font-weight: 600;
            color: #343a40;
        }

        .trigger-status {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 50px;
            height: 24px;
        }

        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }

        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: .4s;
            border-radius: 24px;
        }

        .slider:before {
            position: absolute;
            content: "";
            height: 18px;
            width: 18px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .4s;
            border-radius: 50%;
        }

        input:checked + .slider {
            background-color: var(--success);
        }

        input:checked + .slider:before {
            transform: translateX(26px);
        }

        .trigger-details {
            display: grid;
            gap: 0.75rem;
        }

        .detail-row {
            display: grid;
            grid-template-columns: 150px 1fr;
            gap: 1rem;
            align-items: start;
        }

        .detail-label {
            font-weight: 600;
            color: #6c757d;
            font-size: 0.9rem;
        }

        .detail-value {
            color: #343a40;
        }

        .badge {
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.85rem;
            font-weight: 500;
            display: inline-block;
        }

        .badge-info { background: #d1ecf1; color: #0c5460; }
        .badge-success { background: #d4edda; color: #155724; }
        .badge-warning { background: #fff3cd; color: #856404; }
        .badge-error { background: #f8d7da; color: #721c24; }
        .badge-disabled { background: #e9ecef; color: #6c757d; }

        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 10000;
            align-items: center;
            justify-content: center;
        }

        .modal.active {
            display: flex;
        }

        .modal-content {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            max-width: 800px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.2);
            position: relative;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border);
        }

        .modal-header h2 {
            margin: 0;
            font-size: 1.5rem;
            color: #343a40;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 50%;
            transition: all 0.2s;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .modal-close:hover {
            background: #f8f9fa;
            color: #343a40;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            font-weight: 600;
            color: #343a40;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 0.9rem;
            transition: border-color 0.2s;
            font-family: inherit;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 48, 87, 0.1);
        }

        .form-group small {
            display: block;
            margin-top: 0.25rem;
            color: #6c757d;
            font-size: 0.85rem;
        }

        .form-group textarea {
            resize: vertical;
            min-height: 100px;
        }

        .form-actions {
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
            margin-top: 2rem;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.2s;
        }

        .btn-secondary:hover {
            background: #5a6268;
        }

        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .checkbox-group input[type="checkbox"] {
            width: auto;
        }
        
        .field-help {
            animation: fadeIn 0.3s ease;
            font-size: 0.9rem;
            position: fixed;
            z-index: 10000;
            background: #e3f2fd;
            padding: 1.5rem;
            border-radius: 8px;
            border-left: 4px solid var(--primary);
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            max-width: 400px;
            max-height: 500px;
            overflow-y: auto;
        }
        
        .field-help code {
            background: rgba(0, 48, 87, 0.1);
            padding: 0.2rem 0.4rem;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 0.85em;
        }
        
        .help-arrow {
            position: absolute;
            width: 0;
            height: 0;
            border: 12px solid transparent;
        }
        
        .help-arrow.left {
            left: -24px;
            top: 50%;
            transform: translateY(-50%);
            border-right-color: var(--primary);
        }
        
        .help-arrow.right {
            right: -24px;
            top: 50%;
            transform: translateY(-50%);
            border-left-color: var(--primary);
        }
        
        .help-arrow.top {
            top: -24px;
            left: 50%;
            transform: translateX(-50%);
            border-bottom-color: var(--primary);
        }
        
        .help-arrow.bottom {
            bottom: -24px;
            left: 50%;
            transform: translateX(-50%);
            border-top-color: var(--primary);
        }
        
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateX(-10px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }
        
        @keyframes arrowPulse {
            0%, 100% {
                opacity: 1;
                transform: translateY(-50%) translateX(0);
            }
            50% {
                opacity: 0.7;
                transform: translateY(-50%) translateX(-5px);
            }
        }
        
        .trigger-validation {
            margin-top: 0.5rem;
            padding: 0.75rem;
            border-radius: 8px;
            font-size: 0.85rem;
        }
        
        .trigger-validation.valid {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .trigger-validation.invalid {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .trigger-validation.info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .available-triggers-list {
            margin-top: 0.5rem;
            max-height: 200px;
            overflow-y: auto;
            background: white;
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 0.5rem;
        }
        
        .available-triggers-list .trigger-item {
            padding: 0.5rem;
            border-bottom: 1px solid #f0f0f0;
            cursor: pointer;
            transition: background 0.2s;
        }
        
        .available-triggers-list .trigger-item:hover {
            background: #f8f9fa;
        }
        
        .available-triggers-list .trigger-item:last-child {
            border-bottom: none;
        }
        
        .available-triggers-list .trigger-item strong {
            color: var(--primary);
        }
        
        @media (max-width: 768px) {
            .form-group > div[style*="display: flex"] {
                flex-direction: column !important;
            }
            
            .field-help {
                margin-top: 0.5rem;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/sidebar.php'; ?>
    
    <div class="admin-container">
        <main class="admin-main">
            <div class="admin-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                <h1><i class="fas fa-cogs"></i> Benachrichtigungs-Trigger verwalten</h1>
                <button class="btn btn-primary" onclick="showCreateModal()" style="padding: 0.75rem 1.5rem; background: var(--primary); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; display: flex; align-items: center; gap: 0.5rem;">
                    <i class="fas fa-plus"></i> Neuen Trigger erstellen
                </button>
            </div>

            <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
            <?php endif; ?>

            <div class="info-box">
                <strong><i class="fas fa-info-circle"></i> Info:</strong>
                <p style="margin: 0.5rem 0 0 0;">
                    Hier können Sie Benachrichtigungs-Trigger aktivieren oder deaktivieren. 
                    Neue Trigger können über den Button "Neuen Trigger erstellen" hinzugefügt werden.
                </p>
                <p style="margin: 0.75rem 0 0 0;">
                    <strong>📘 Ausführliche Anleitung:</strong> Siehe <code>NOTIFICATION_TRIGGERS_ANLEITUNG.md</code> im Projekt-Root für eine detaillierte Dokumentation.
                </p>
            </div>

            <div class="triggers-grid">
                <?php foreach ($triggers as $trigger_name => $trigger): ?>
                <div class="trigger-card">
                    <div class="trigger-header">
                        <div class="trigger-name"><?php echo htmlspecialchars($trigger_name); ?></div>
                        <div class="trigger-status">
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Trigger wirklich <?php echo $trigger['enabled'] ? 'deaktivieren' : 'aktivieren'; ?>?');">
                                <input type="hidden" name="trigger_name" value="<?php echo htmlspecialchars($trigger_name); ?>">
                                <input type="hidden" name="enabled" value="<?php echo $trigger['enabled'] ? '0' : '1'; ?>">
                                <label class="toggle-switch">
                                    <input type="checkbox" <?php echo $trigger['enabled'] ? 'checked' : ''; ?> onchange="this.form.submit()">
                                    <span class="slider"></span>
                                </label>
                                <input type="hidden" name="toggle_trigger" value="1">
                            </form>
                            <span class="badge badge-<?php echo $trigger['enabled'] ? 'success' : 'disabled'; ?>">
                                <?php echo $trigger['enabled'] ? 'Aktiviert' : 'Deaktiviert'; ?>
                            </span>
                        </div>
                    </div>
                    <div class="trigger-details">
                        <div class="detail-row">
                            <div class="detail-label">Titel:</div>
                            <div class="detail-value"><?php echo htmlspecialchars($trigger['title_template']); ?></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">Nachricht:</div>
                            <div class="detail-value"><?php echo htmlspecialchars($trigger['message_template']); ?></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">Typ:</div>
                            <div class="detail-value">
                                <span class="badge badge-<?php echo $trigger['type'] ?? 'info'; ?>">
                                    <?php echo htmlspecialchars($trigger['type'] ?? 'info'); ?>
                                </span>
                            </div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">URL:</div>
                            <div class="detail-value"><code><?php echo htmlspecialchars($trigger['url'] ?? '/profile.php'); ?></code></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">Push:</div>
                            <div class="detail-value">
                                <?php echo ($trigger['send_push'] ?? true) ? '<i class="fas fa-check text-success"></i> Ja' : '<i class="fas fa-times text-danger"></i> Nein'; ?>
                            </div>
                        </div>
                        <?php if (isset($trigger['conditions'])): ?>
                        <div class="detail-row">
                            <div class="detail-label">Bedingungen:</div>
                            <div class="detail-value"><i class="fas fa-check-circle text-success"></i> Aktiv</div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </main>
    </div>

    <!-- Create Trigger Modal -->
    <div id="createModal" class="modal">
        <div class="modal-content">
            <div class="modal-header" style="position: absolute; top: 0; left: 0; right: 0; z-index: 10; background: white; padding: 1.5rem 2rem; border-bottom: 1px solid var(--border);">
                <h2>Neuen Trigger erstellen</h2>
                <button class="modal-close" onclick="closeCreateModal()">&times;</button>
            </div>
            
            <div class="modal-form-container" style="margin-top: 80px;">
            <form method="POST" id="createTriggerForm">
                <input type="hidden" name="create_trigger" value="1">
                
                <div class="form-group">
                    <label>Trigger-Name *</label>
                    <input type="text" name="trigger_name" id="trigger_name" required pattern="[a-z0-9_]+" placeholder="z.B. new_feature_available" onfocus="showFieldHelp('trigger_name', this)" onblur="hideFieldHelp('trigger_name')" oninput="validateTriggerName(this.value)">
                    <small>Nur Kleinbuchstaben, Zahlen und Unterstriche erlaubt</small>
                    <div id="trigger_validation" class="trigger-validation" style="display: none;"></div>
                    <div id="available_triggers_list" class="available-triggers-list" style="display: none;"></div>
                </div>
                
                <div class="form-group">
                    <label>Titel-Vorlage *</label>
                    <input type="text" name="title_template" id="title_template" required placeholder="z.B. Neues Feature verfügbar" onfocus="showFieldHelp('title_template', this)" onblur="hideFieldHelp('title_template')">
                    <small>Der Titel der Benachrichtigung. Platzhalter: {key}</small>
                </div>
                
                <div class="form-group">
                    <label>Nachricht-Vorlage *</label>
                    <textarea name="message_template" id="message_template" required placeholder="z.B. Ein neues Feature ist verfügbar: {feature_name}" onfocus="showFieldHelp('message_template', this)" onblur="hideFieldHelp('message_template')"></textarea>
                    <small>Die Nachricht der Benachrichtigung. Platzhalter: {key}</small>
                </div>
                
                <div class="form-group">
                    <label>Typ</label>
                    <select name="type" id="type" onfocus="showFieldHelp('type', this)" onblur="hideFieldHelp('type')">
                        <option value="info">Info</option>
                        <option value="success">Erfolg</option>
                        <option value="warning">Warnung</option>
                        <option value="error">Fehler</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>URL</label>
                    <input type="text" name="url" id="url" value="/profile.php" placeholder="/profile.php" onfocus="showFieldHelp('url', this)" onblur="hideFieldHelp('url')">
                    <small>URL, die beim Klick auf die Benachrichtigung geöffnet wird</small>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" name="send_push" value="1" id="send_push" checked onfocus="showFieldHelp('send_push', this)" onblur="hideFieldHelp('send_push')">
                        <label for="send_push" style="font-weight: normal; margin: 0;">Push-Benachrichtigung senden</label>
                    </div>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" name="enabled" value="1" id="enabled" checked onfocus="showFieldHelp('enabled', this)" onblur="hideFieldHelp('enabled')">
                        <label for="enabled" style="font-weight: normal; margin: 0;">Trigger aktivieren</label>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="button" onclick="closeCreateModal()" class="btn btn-secondary">Abbrechen</button>
                    <button type="submit" class="btn btn-primary">Trigger erstellen</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Help Tooltips (außerhalb des Modals) -->
    <div id="help_trigger_name" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> Trigger-Name</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>Format:</strong> Nur Kleinbuchstaben, Zahlen und Unterstriche</li>
            <li style="margin-top: 1rem;"><strong>Verfügbare Trigger:</strong></li>
            <div id="available_triggers_help" style="margin-top: 0.5rem; max-height: 200px; overflow-y: auto; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php if (empty($triggers)): ?>
                    <div style="padding: 0.5rem; color: #666; font-size: 0.85rem;">Keine Trigger vorhanden</div>
                <?php else: ?>
                    <?php foreach ($triggers as $name => $trigger): ?>
                    <div style="padding: 0.5rem; border-bottom: 1px solid #f0f0f0; font-size: 0.85rem;">
                        <strong style="color: var(--primary);"><?php echo htmlspecialchars($name, ENT_QUOTES, 'UTF-8'); ?></strong>
                        <div style="color: #666; margin-top: 0.25rem;"><?php echo htmlspecialchars($trigger['title_template'] ?? 'Keine Beschreibung', ENT_QUOTES, 'UTF-8'); ?></div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <li style="margin-top: 1rem;"><strong>💡 Tipp:</strong> Verwende aussagekräftige Namen, die den Zweck des Triggers beschreiben</li>
        </ul>
    </div>
    
    <div id="help_title_template" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> Titel-Vorlage</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>Zweck:</strong> Der Titel, der in der Benachrichtigung angezeigt wird</li>
            <li><strong>Platzhalter:</strong> Verwende <code>{key}</code> oder <code>"{key}"</code> für dynamische Inhalte</li>
            <li style="margin-top: 1rem;"><strong>Verfügbare Titel-Vorlagen:</strong></li>
            <div style="margin-top: 0.5rem; max-height: 200px; overflow-y: auto; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php if (empty($triggers)): ?>
                    <div style="padding: 0.5rem; color: #666; font-size: 0.85rem;">Keine Trigger vorhanden</div>
                <?php else: ?>
                    <?php foreach ($triggers as $name => $trigger): ?>
                    <div style="padding: 0.5rem; border-bottom: 1px solid #f0f0f0; font-size: 0.85rem;">
                        <strong style="color: var(--primary);"><?php echo htmlspecialchars($name, ENT_QUOTES, 'UTF-8'); ?></strong>
                        <div style="color: #666; margin-top: 0.25rem;"><code><?php echo htmlspecialchars($trigger['title_template'] ?? 'Keine Vorlage', ENT_QUOTES, 'UTF-8'); ?></code></div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <li style="margin-top: 1rem;"><strong>💡 Tipp:</strong> Halte den Titel kurz und prägnant (max. 50 Zeichen)</li>
        </ul>
    </div>
    
    <div id="help_message_template" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> Nachricht-Vorlage</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>Zweck:</strong> Die Nachricht, die in der Benachrichtigung angezeigt wird</li>
            <li><strong>Platzhalter:</strong> Verwende <code>{key}</code> oder <code>"{key}"</code> für dynamische Inhalte</li>
            <li style="margin-top: 1rem;"><strong>Verfügbare Nachricht-Vorlagen:</strong></li>
            <div style="margin-top: 0.5rem; max-height: 200px; overflow-y: auto; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php if (empty($triggers)): ?>
                    <div style="padding: 0.5rem; color: #666; font-size: 0.85rem;">Keine Trigger vorhanden</div>
                <?php else: ?>
                    <?php foreach ($triggers as $name => $trigger): ?>
                    <div style="padding: 0.5rem; border-bottom: 1px solid #f0f0f0; font-size: 0.85rem;">
                        <strong style="color: var(--primary);"><?php echo htmlspecialchars($name, ENT_QUOTES, 'UTF-8'); ?></strong>
                        <div style="color: #666; margin-top: 0.25rem;"><code><?php echo htmlspecialchars($trigger['message_template'] ?? 'Keine Vorlage', ENT_QUOTES, 'UTF-8'); ?></code></div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <li style="margin-top: 1rem;"><strong>Hinweis:</strong> Platzhalter werden beim Senden durch die übergebenen Daten ersetzt</li>
        </ul>
    </div>
    
    <div id="help_type" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> Typ</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>info</strong> - Informative Benachrichtigung (Standard, blau)</li>
            <li><strong>success</strong> - Erfolgs-Benachrichtigung (grün) - z.B. für Bestätigungen</li>
            <li><strong>warning</strong> - Warnung (gelb) - z.B. für wichtige Hinweise</li>
            <li><strong>error</strong> - Fehler (rot) - z.B. für Ablehnungen oder Probleme</li>
            <li style="margin-top: 1rem;"><strong>Verfügbare Typen in bestehenden Triggern:</strong></li>
            <div style="margin-top: 0.5rem; max-height: 150px; overflow-y: auto; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php 
                $type_counts = [];
                foreach ($triggers as $trigger) {
                    $type = $trigger['type'] ?? 'info';
                    $type_counts[$type] = ($type_counts[$type] ?? 0) + 1;
                }
                if (empty($type_counts)):
                ?>
                    <div style="padding: 0.5rem; color: #666; font-size: 0.85rem;">Keine Trigger vorhanden</div>
                <?php else: ?>
                    <?php foreach ($type_counts as $type => $count): ?>
                    <div style="padding: 0.5rem; border-bottom: 1px solid #f0f0f0; font-size: 0.85rem;">
                        <strong style="color: var(--primary);"><?php echo htmlspecialchars($type, ENT_QUOTES, 'UTF-8'); ?></strong>
                        <span style="color: #666;">(<?php echo (int)$count; ?> Trigger)</span>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <li style="margin-top: 1rem;"><strong>Verwendung:</strong> Der Typ bestimmt die Farbe und das Icon der Benachrichtigung</li>
        </ul>
    </div>
    
    <div id="help_url" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> URL</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>Zweck:</strong> Die Seite, die geöffnet wird, wenn der Benutzer auf die Benachrichtigung klickt</li>
            <li><strong>Standard:</strong> <code>/profile.php</code></li>
            <li style="margin-top: 1rem;"><strong>Verfügbare URLs in bestehenden Triggern:</strong></li>
            <div style="margin-top: 0.5rem; max-height: 200px; overflow-y: auto; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php 
                $url_counts = [];
                foreach ($triggers as $trigger) {
                    $url = $trigger['url'] ?? '/profile.php';
                    $url_counts[$url] = ($url_counts[$url] ?? 0) + 1;
                }
                if (empty($url_counts)):
                ?>
                    <div style="padding: 0.5rem; color: #666; font-size: 0.85rem;">Keine Trigger vorhanden</div>
                <?php else: ?>
                    <?php foreach ($url_counts as $url => $count): ?>
                    <div style="padding: 0.5rem; border-bottom: 1px solid #f0f0f0; font-size: 0.85rem;">
                        <code style="color: var(--primary);"><?php echo htmlspecialchars($url, ENT_QUOTES, 'UTF-8'); ?></code>
                        <span style="color: #666;">(<?php echo (int)$count; ?> Trigger)</span>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <li style="margin-top: 1rem;"><strong>Hinweis:</strong> Relativer Pfad vom Root-Verzeichnis</li>
        </ul>
    </div>
    
    <div id="help_send_push" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> Push-Benachrichtigung</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>✅ Aktiviert:</strong> Benachrichtigung wird als Push-Nachricht an alle Geräte des Benutzers gesendet</li>
            <li><strong>❌ Deaktiviert:</strong> Nur Datenbank-Benachrichtigung (wird in der App angezeigt, aber keine Push-Nachricht)</li>
            <li style="margin-top: 1rem;"><strong>Status in bestehenden Triggern:</strong></li>
            <div style="margin-top: 0.5rem; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php 
                $push_enabled = 0;
                $push_disabled = 0;
                foreach ($triggers as $trigger) {
                    if (($trigger['send_push'] ?? true)) {
                        $push_enabled++;
                    } else {
                        $push_disabled++;
                    }
                }
                ?>
                <div style="padding: 0.5rem; font-size: 0.85rem;">
                    <span style="color: #28a745;"><strong>✅ Aktiviert:</strong> <?php echo (int)$push_enabled; ?> Trigger</span><br>
                    <span style="color: #dc3545;"><strong>❌ Deaktiviert:</strong> <?php echo (int)$push_disabled; ?> Trigger</span>
                </div>
            </div>
            <li style="margin-top: 1rem;"><strong>Hinweis:</strong> Push-Benachrichtigungen funktionieren nur, wenn der Benutzer diese aktiviert hat</li>
        </ul>
    </div>
    
    <div id="help_enabled" class="field-help" style="display: none;">
        <div class="help-arrow left"></div>
        <strong style="color: var(--primary); display: block; margin-bottom: 0.75rem; font-size: 1.1rem;"><i class="fas fa-info-circle"></i> Trigger aktivieren</strong>
        <ul style="margin: 0; padding-left: 1.5rem; font-size: 0.9rem; color: #333; line-height: 1.8;">
            <li><strong>✅ Aktiviert:</strong> Trigger ist sofort einsatzbereit und sendet Benachrichtigungen</li>
            <li><strong>❌ Deaktiviert:</strong> Trigger wird erstellt, aber sendet keine Benachrichtigungen (kann später aktiviert werden)</li>
            <li style="margin-top: 1rem;"><strong>Status in bestehenden Triggern:</strong></li>
            <div style="margin-top: 0.5rem; background: white; padding: 0.5rem; border-radius: 4px; border: 1px solid #ddd;">
                <?php 
                $enabled_count = 0;
                $disabled_count = 0;
                foreach ($triggers as $trigger) {
                    if (($trigger['enabled'] ?? false)) {
                        $enabled_count++;
                    } else {
                        $disabled_count++;
                    }
                }
                ?>
                <div style="padding: 0.5rem; font-size: 0.85rem;">
                    <span style="color: #28a745;"><strong>✅ Aktiviert:</strong> <?php echo (int)$enabled_count; ?> Trigger</span><br>
                    <span style="color: #dc3545;"><strong>❌ Deaktiviert:</strong> <?php echo (int)$disabled_count; ?> Trigger</span>
                </div>
            </div>
            <li style="margin-top: 1rem;"><strong>Hinweis:</strong> Der Status kann jederzeit im Admin-Panel geändert werden</li>
        </ul>
    </div>

    <script>
    function showCreateModal() {
        document.getElementById('createModal').classList.add('active');
    }

    function closeCreateModal() {
        document.getElementById('createModal').classList.remove('active');
    }

    // Close modal on outside click
    document.getElementById('createModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeCreateModal();
        }
    });

    // Close modal on ESC key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeCreateModal();
        }
    });
    
    // Available triggers for validation
    const availableTriggers = <?php echo json_encode($trigger_names); ?>;
    
    // Field Help System
    function showFieldHelp(fieldName, fieldElement) {
        // Hide all help boxes
        document.querySelectorAll('.field-help').forEach(help => {
            help.style.display = 'none';
        });
        
        // Show help for current field
        const helpBox = document.getElementById('help_' + fieldName);
        if (helpBox && fieldElement) {
            positionHelpTooltip(helpBox, fieldElement);
            helpBox.style.display = 'block';
        }
    }
    
    function positionHelpTooltip(helpBox, fieldElement) {
        if (!fieldElement) return;
        
        const fieldRect = fieldElement.getBoundingClientRect();
        const helpRect = helpBox.getBoundingClientRect();
        const viewportWidth = window.innerWidth;
        const viewportHeight = window.innerHeight;
        const spacing = 20;
        
        // Calculate positions
        let left, top;
        let arrowClass = 'left';
        
        // Try to position on the right side first
        if (fieldRect.right + spacing + helpRect.width < viewportWidth) {
            left = fieldRect.right + spacing;
            top = fieldRect.top + (fieldRect.height / 2) - (helpRect.height / 2);
            arrowClass = 'left';
        }
        // Try left side
        else if (fieldRect.left - spacing - helpRect.width > 0) {
            left = fieldRect.left - spacing - helpRect.width;
            top = fieldRect.top + (fieldRect.height / 2) - (helpRect.height / 2);
            arrowClass = 'right';
        }
        // Try below
        else if (fieldRect.bottom + spacing + helpRect.height < viewportHeight) {
            left = fieldRect.left + (fieldRect.width / 2) - (helpRect.width / 2);
            top = fieldRect.bottom + spacing;
            arrowClass = 'top';
        }
        // Try above
        else if (fieldRect.top - spacing - helpRect.height > 0) {
            left = fieldRect.left + (fieldRect.width / 2) - (helpRect.width / 2);
            top = fieldRect.top - spacing - helpRect.height;
            arrowClass = 'bottom';
        }
        // Fallback: center of viewport
        else {
            left = (viewportWidth - helpRect.width) / 2;
            top = (viewportHeight - helpRect.height) / 2;
            arrowClass = 'left';
        }
        
        // Ensure tooltip stays within viewport
        left = Math.max(10, Math.min(left, viewportWidth - helpRect.width - 10));
        top = Math.max(10, Math.min(top, viewportHeight - helpRect.height - 10));
        
        // Apply position
        helpBox.style.left = left + 'px';
        helpBox.style.top = top + 'px';
        
        // Update arrow
        const arrow = helpBox.querySelector('.help-arrow');
        if (arrow) {
            arrow.className = 'help-arrow ' + arrowClass;
        }
    }
    
    function hideFieldHelp(fieldName) {
        // Small delay to allow clicking on help box
        setTimeout(() => {
            const helpBox = document.getElementById('help_' + fieldName);
            if (helpBox && !helpBox.matches(':hover')) {
                helpBox.style.display = 'none';
            }
        }, 200);
    }
    
    // Keep help visible when hovering over it
    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('.field-help').forEach(help => {
            help.addEventListener('mouseenter', function() {
                this.style.display = 'block';
            });
            help.addEventListener('mouseleave', function() {
                this.style.display = 'none';
            });
        });
    });
    
    // Trigger Name Validation
    function validateTriggerName(value) {
        const validationDiv = document.getElementById('trigger_validation');
        const availableList = document.getElementById('available_triggers_list');
        const triggerNameInput = document.getElementById('trigger_name');
        
        if (!value || value.trim() === '') {
            validationDiv.style.display = 'none';
            availableList.style.display = 'none';
            return;
        }
        
        const trimmedValue = value.trim().toLowerCase();
        
        // Check format
        if (!/^[a-z0-9_]+$/.test(trimmedValue)) {
            validationDiv.className = 'trigger-validation invalid';
            validationDiv.innerHTML = '<i class="fas fa-times-circle"></i> <strong>Ungültiges Format:</strong> Nur Kleinbuchstaben, Zahlen und Unterstriche erlaubt';
            validationDiv.style.display = 'block';
            availableList.style.display = 'none';
            triggerNameInput.setCustomValidity('Ungültiges Format');
            return;
        }
        
        // Check if trigger already exists
        if (availableTriggers.includes(trimmedValue)) {
            validationDiv.className = 'trigger-validation invalid';
            validationDiv.innerHTML = '<i class="fas fa-times-circle"></i> <strong>Trigger existiert bereits:</strong> Dieser Trigger-Name wird bereits verwendet';
            validationDiv.style.display = 'block';
            availableList.style.display = 'none';
            triggerNameInput.setCustomValidity('Trigger existiert bereits');
            return;
        }
        
        // Check for similar triggers
        const similarTriggers = availableTriggers.filter(t => 
            t.includes(trimmedValue) || trimmedValue.includes(t)
        );
        
        if (similarTriggers.length > 0) {
            validationDiv.className = 'trigger-validation info';
            validationDiv.innerHTML = '<i class="fas fa-info-circle"></i> <strong>Ähnliche Trigger gefunden:</strong> Möglicherweise meinten Sie einen dieser Trigger?';
            validationDiv.style.display = 'block';
            
            // Show available triggers (sicher escaped)
            const triggersData = <?php echo json_encode($triggers_safe, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT); ?>;
            availableList.innerHTML = similarTriggers.map(trigger => {
                const triggerData = triggersData[trigger];
                const safeName = escapeHtml(trigger);
                const safeTitle = escapeHtml(triggerData?.title_template || 'Keine Beschreibung');
                return `<div class="trigger-item" onclick="fillTriggerName('${safeName.replace(/'/g, "\\'")}')">
                    <strong>${safeName}</strong><br>
                    <small style="color: #666;">${safeTitle}</small>
                </div>`;
            }).join('');
            availableList.style.display = 'block';
        } else {
            validationDiv.className = 'trigger-validation valid';
            validationDiv.innerHTML = '<i class="fas fa-check-circle"></i> <strong>Trigger-Name verfügbar</strong>';
            validationDiv.style.display = 'block';
            availableList.style.display = 'none';
        }
        
        triggerNameInput.setCustomValidity('');
    }
    
    function fillTriggerName(triggerName) {
        // Sicherheit: Nur erlaubte Zeichen
        const safeName = triggerName.replace(/[^a-z0-9_]/g, '');
        document.getElementById('trigger_name').value = safeName;
        validateTriggerName(safeName);
    }
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Update tooltip position on scroll/resize
    let activeFieldElement = null;
    let activeFieldName = null;
    
    function updateTooltipPosition() {
        if (activeFieldElement && activeFieldName) {
            const helpBox = document.getElementById('help_' + activeFieldName);
            if (helpBox && helpBox.style.display !== 'none') {
                positionHelpTooltip(helpBox, activeFieldElement);
            }
        }
    }
    
    window.addEventListener('scroll', updateTooltipPosition, true);
    window.addEventListener('resize', updateTooltipPosition);
    
    // Store active field for position updates
    const originalShowFieldHelp = showFieldHelp;
    showFieldHelp = function(fieldName, fieldElement) {
        activeFieldElement = fieldElement;
        activeFieldName = fieldName;
        originalShowFieldHelp(fieldName, fieldElement);
    };
    </script>
</body>
</html>

