<?php
/**
 * Listet alle verfügbaren WhatsApp-Templates von HelloMateo API auf
 */

header('Content-Type: application/json');
require_once '../config/database.php';

session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht angemeldet']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$apiKey = $input['api_key'] ?? '';
$autohausId = $input['autohaus_id'] ?? null;

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // WICHTIG: Wenn ein API-Key übergeben wurde, verwende diesen immer
    // Nur wenn wirklich KEIN API-Key übergeben wurde, hole ihn aus der DB
    $forceUseProvided = isset($input['force_use_provided_key']) && $input['force_use_provided_key'];
    
    if ($forceUseProvided && !empty($apiKey)) {
        // Verwende den übergebenen API-Key direkt
    } elseif (empty($apiKey) && !isset($input['api_key'])) {
        if ($autohausId) {
            // Autohaus-spezifischer API-Key
            $stmt = $conn->prepare("SELECT whatsapp_api_key FROM autohaus WHERE id = ?");
            $stmt->execute([$autohausId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            $apiKey = $autohaus['whatsapp_api_key'] ?? '';
        }
        
        // Falls immer noch leer, hole globalen API-Key
        if (empty($apiKey)) {
            $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_api_key'");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $apiKey = $result['setting_value'] ?? '';
        }
    }
    
    if (empty($apiKey)) {
        echo json_encode([
            'success' => false,
            'error' => 'Kein API-Key konfiguriert'
        ]);
        exit;
    }
    
    // API-Key bereinigen (Leerzeichen, Zeilenumbrüche entfernen)
    $apiKey = trim($apiKey);
    
    // Versuche verschiedene Endpunkte zum Auflisten von Templates
    $listEndpoints = [
        'https://integration.getmateo.com/api/v1/templates',
        'https://api.hellomateo.com/v1/templates',
        'https://integration.getmateo.com/api/v1/message-templates',
        'https://api.hellomateo.com/v1/message-templates',
    ];
    
    $templates = null;
    $allErrors = [];
    
    foreach ($listEndpoints as $url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json',
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($httpCode === 200 && !empty($response)) {
            $decoded = json_decode($response, true);
            if ($decoded) {
                // Verschiedene mögliche Response-Formate
                if (isset($decoded['data']) && is_array($decoded['data'])) {
                    $templates = $decoded['data'];
                } elseif (isset($decoded['templates']) && is_array($decoded['templates'])) {
                    $templates = $decoded['templates'];
                } elseif (is_array($decoded) && count($decoded) > 0) {
                    $templates = $decoded;
                }
                
                if ($templates) {
                    break;
                }
            }
        } else {
            $errorMsg = "HTTP $httpCode";
            if ($curlError) {
                $errorMsg .= ": $curlError";
            }
            if ($response) {
                $errorResponse = json_decode($response, true);
            // Laut HelloMateo API-Dokumentation: {code, message, hint, status}
            if ($errorResponse) {
                if (isset($errorResponse['message'])) {
                    $errorMsg .= " - " . $errorResponse['message'];
                }
                if (isset($errorResponse['hint'])) {
                    $errorMsg .= " (" . $errorResponse['hint'] . ")";
                }
                if (isset($errorResponse['code'])) {
                    $errorMsg .= " [Code: " . $errorResponse['code'] . "]";
                }
                // Fallback
                if (isset($errorResponse['error'])) {
                    $errorMsg .= " - " . $errorResponse['error'];
                }
            }
            
            // Bei 401/403: Stoppe weitere Versuche
            if ($httpCode === 401 || $httpCode === 403) {
                break;
            }
            }
            $allErrors[] = $errorMsg;
        }
    }
    
    if ($templates && is_array($templates)) {
        // Formatiere Templates für die Anzeige
        $formattedTemplates = [];
        foreach ($templates as $template) {
            $formattedTemplates[] = [
                'id' => $template['id'] ?? $template['template_id'] ?? $template['name'] ?? 'Unbekannt',
                'name' => $template['name'] ?? $template['template_name'] ?? 'Unbekannt',
                'status' => $template['status'] ?? $template['template_status'] ?? 'Unbekannt',
                'language' => $template['language'] ?? $template['template_language'] ?? 'de',
                'category' => $template['category'] ?? $template['template_category'] ?? 'UTILITY',
            ];
        }
        
        echo json_encode([
            'success' => true,
            'templates' => $formattedTemplates,
            'count' => count($formattedTemplates)
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'error' => 'Keine Templates gefunden oder API-Fehler',
            'all_errors' => $allErrors,
            'info' => 'Bitte überprüfen Sie den API-Key. Möglicherweise sind keine Templates in Ihrem HelloMateo-Konto vorhanden oder der API-Endpunkt ist anders.'
        ]);
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => 'Fehler: ' . $e->getMessage()
    ]);
}
?>

