<?php
/**
 * API-Helper für Reifenangebote
 * Kommunikation mit der Tailscale-API (http://100.82.207.72/api/)
 * Unterstützt Multi-Autohaus-System mit individueller API-Konfiguration pro Autohaus
 */

class TireOffersAPI {
    private $baseUrl;
    private $apiKey;
    private $apiKeyRequired;
    private $autohausId;
    private $conn;
    
    public function __construct($autohausId = null) {
        $this->autohausId = $autohausId;
        
        // Lade Datenbankverbindung
        require_once dirname(__DIR__) . '/../config/database.php';
        $db = new Database();
        $this->conn = $db->getConnection();
        
        // Lade API-Konfiguration aus Datenbank
        $this->loadApiConfig();
    }
    
    /**
     * Lädt API-Konfiguration aus der Datenbank für das Autohaus
     */
    private function loadApiConfig() {
        // Fallback-Werte (für Kiel)
        $this->baseUrl = 'http://100.82.207.72:5001/api';
        $this->apiKey = 'bungert-p2-api-key-2024-default';
        $this->apiKeyRequired = false;
        
        // Wenn keine Autohaus-ID angegeben, versuche aus Session zu holen
        if (!$this->autohausId && isset($_SESSION['current_autohaus_id'])) {
            $this->autohausId = $_SESSION['current_autohaus_id'];
        }
        
        // Lade Konfiguration aus Datenbank
        if ($this->autohausId && $this->conn) {
            try {
                $stmt = $this->conn->prepare("
                    SELECT tire_offers_api_url, tire_offers_api_key, tire_offers_api_key_required 
                    FROM autohaus 
                    WHERE id = ? AND is_active = 1
                ");
                $stmt->execute([$this->autohausId]);
                $config = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($config) {
                    // Verwende konfigurierte Werte wenn vorhanden
                    if (!empty($config['tire_offers_api_url'])) {
                        $this->baseUrl = $config['tire_offers_api_url'];
                    }
                    if (!empty($config['tire_offers_api_key'])) {
                        $this->apiKey = $config['tire_offers_api_key'];
                    }
                    if (isset($config['tire_offers_api_key_required'])) {
                        $this->apiKeyRequired = (bool)$config['tire_offers_api_key_required'];
                    }
                }
            } catch (Exception $e) {
                error_log("TireOffersAPI::loadApiConfig Error: " . $e->getMessage());
                // Verwende Fallback-Werte bei Fehler
            }
        }
    }
    
    /**
     * Führt einen API-Call durch mit Retry-Mechanismus
     * Öffentlich für erweiterte Suche
     * @param string $endpoint API-Endpoint
     * @param string $method HTTP-Methode
     * @param array $params Query-Parameter oder POST-Daten
     * @param int $timeout Optional: Custom Timeout (Standard: 5 Sekunden)
     * @param int $maxRetries Optional: Maximale Anzahl Wiederholungsversuche (Standard: 2)
     */
    public function apiCall($endpoint, $method = 'GET', $params = [], $timeout = 5, $maxRetries = 2) {
        $baseUrl = $this->baseUrl . $endpoint;
        $lastError = null;
        $lastHttpCode = 0;
        
        for ($attempt = 0; $attempt <= $maxRetries; $attempt++) {
            try {
                // Baue URL bei jedem Versuch neu auf
                $url = $baseUrl;
                if ($method === 'GET' && !empty($params)) {
                    $url .= '?' . http_build_query($params);
                }
                
                $headers = [
                    'Content-Type: application/json',
                ];
                
                // API-Key nur mitschicken wenn aktiviert
                if ($this->apiKeyRequired) {
                    $headers[] = 'X-API-Key: ' . $this->apiKey;
                }
                
                $ch = curl_init();
                
                curl_setopt_array($ch, [
                    CURLOPT_URL => $url,
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_HTTPHEADER => $headers,
                    CURLOPT_TIMEOUT => $timeout,
                    CURLOPT_CONNECTTIMEOUT => min($timeout, 5), // Erhöht auf 5 Sekunden für langsamere Verbindungen
                    CURLOPT_FRESH_CONNECT => ($attempt > 0), // Neue Verbindung bei Retry
                    CURLOPT_FORBID_REUSE => false,
                    CURLOPT_SSL_VERIFYPEER => false, // Falls HTTPS verwendet wird
                    CURLOPT_SSL_VERIFYHOST => false,
                ]);
                
                if ($method === 'POST' || $method === 'PUT') {
                    if ($method === 'PUT') {
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                    } else {
                        curl_setopt($ch, CURLOPT_POST, true);
                    }
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
                }
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $curlError = curl_error($ch);
                $curlErrno = curl_errno($ch);
                curl_close($ch);
                
                // Erfolgreiche Antwort
                if ($response !== false && $curlError === '' && $httpCode >= 200 && $httpCode < 400) {
                    // Prüfe ob Response HTML ist statt JSON
                    if (strpos(trim($response), '<') === 0 || strpos(strtolower($response), '<html') !== false || strpos(strtolower($response), '<br') !== false) {
                        error_log("TireOffersAPI: HTML-Response erhalten statt JSON. URL: $url");
                        throw new Exception("API antwortet mit HTML statt JSON (möglicher PHP-Fehler auf Server)");
                    }
                    
                    $data = json_decode($response, true);
                    
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        error_log("TireOffersAPI: JSON-Parse-Fehler. URL: $url, Response: " . substr($response, 0, 200));
                        throw new Exception("Ungültige JSON-Antwort von API: " . json_last_error_msg());
                    }
                    
                    return $data;
                }
                
                // Fehlerbehandlung
                if ($curlError) {
                    $lastError = "cURL Fehler ($curlErrno): $curlError";
                    // Retry bei Timeout oder Verbindungsfehlern
                    if ($curlErrno === CURLE_OPERATION_TIMEDOUT || $curlErrno === CURLE_COULDNT_CONNECT) {
                        if ($attempt < $maxRetries) {
                            usleep(500000 * ($attempt + 1)); // Exponentielles Backoff: 0.5s, 1s, 1.5s
                            continue;
                        }
                    }
                    throw new Exception($lastError);
                }
                
                if ($httpCode >= 400) {
                    $lastHttpCode = $httpCode;
                    // Prüfe ob Response HTML ist
                    if (strpos($response, '<') !== false && strpos($response, 'html') !== false) {
                        if (preg_match('/<b>(.*?)<\/b>/', $response, $matches)) {
                            throw new Exception("API-Fehler (HTTP $httpCode): " . strip_tags($matches[1]));
                        }
                        throw new Exception("API-Fehler (HTTP $httpCode): HTML-Response erhalten");
                    }
                    throw new Exception("API-Fehler (HTTP $httpCode): " . substr($response, 0, 200));
                }
                
            } catch (Exception $e) {
                $lastError = $e->getMessage();
                
                // Retry bei Timeout-Fehlern
                if (strpos($lastError, 'timed out') !== false || strpos($lastError, 'CURLE_OPERATION_TIMEDOUT') !== false) {
                    if ($attempt < $maxRetries) {
                        error_log("TireOffersAPI: Retry $attempt/$maxRetries nach Timeout für $url");
                        usleep(500000 * ($attempt + 1)); // Exponentielles Backoff
                        continue;
                    }
                }
                
                // Bei letztem Versuch oder nicht-retrybaren Fehlern: Fehler werfen
                if ($attempt >= $maxRetries) {
                    $errorDetails = "Fehler nach " . ($maxRetries + 1) . " Versuchen";
                    if ($lastHttpCode > 0) {
                        $errorDetails .= " (HTTP $lastHttpCode)";
                    }
                    $errorDetails .= ": $lastError";
                    throw new Exception($errorDetails);
                }
            }
        }
        
        // Sollte nie hier ankommen, aber als Fallback
        throw new Exception("API-Call fehlgeschlagen: $lastError");
    }
    
    /**
     * Normalisiert einen Suchbegriff (entfernt Whitespace, konvertiert zu Upper)
     */
    private function normalizeSearchTerm($term) {
        if (empty($term)) return '';
        // Entferne alle Whitespace-Zeichen und konvertiere zu Upper
        return strtoupper(preg_replace('/\s+/', '', trim($term)));
    }
    
    /**
     * Normalisiert ein Kennzeichen für Vergleich
     */
    private function normalizeLicensePlate($license) {
        if (empty($license)) return '';
        // Entferne alle Whitespace, Bindestriche, Punkte und konvertiere zu Upper
        return strtoupper(preg_replace('/[\s\-\.]+/', '', trim($license)));
    }
    
    /**
     * Sucht Kunden nach Name - OPTIMIERT mit besserer Suche
     */
    public function searchCustomers($searchTerm, $limit = 100) {
        try {
            $params = ['limit' => $limit];
            if (!empty($searchTerm)) {
                // OPTIMIERT: Trimme und normalisiere Suchbegriff
                $params['search'] = trim($searchTerm);
            }
            
            $result = $this->apiCall('/kunden', 'GET', $params);
            
            if (isset($result['success']) && $result['success']) {
                $customers = $result['data'] ?? [];
                
                // OPTIMIERT: Zusätzliche clientseitige Filterung für bessere Treffer
                if (!empty($searchTerm)) {
                    $searchNormalized = $this->normalizeSearchTerm($searchTerm);
                    $searchLower = strtolower(trim($searchTerm));
                    
                    $filtered = [];
                    foreach ($customers as $customer) {
                        $name1 = strtolower(trim($customer['KundenName1'] ?? ''));
                        $name2 = strtolower(trim($customer['KundenVorname'] ?? ''));
                        $fullName = trim($name1 . ' ' . $name2);
                        $fullNameNormalized = $this->normalizeSearchTerm($name1 . $name2);
                        
                        // Prüfe verschiedene Match-Varianten
                        if (stripos($name1, $searchLower) !== false ||
                            stripos($name2, $searchLower) !== false ||
                            stripos($fullName, $searchLower) !== false ||
                            stripos($fullNameNormalized, $searchNormalized) !== false) {
                            $filtered[] = $customer;
                        }
                    }
                    return $filtered;
                }
                
                return $customers;
            }
            
            return [];
        } catch (Exception $e) {
            error_log("TireOffersAPI::searchCustomers Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt Kunden nach Kennzeichen - OPTIMIERT für Performance
     * Nutzt direkt den optimierten /by-license Endpoint (sehr schnell, filtert in DB)
     */
    public function findCustomerByLicensePlate($licensePlate, $days = 2000) {
        try {
            if (empty($licensePlate)) {
                return null;
            }
            
            // OPTIMIERT: Verwende direkt den optimierten Endpoint (filtert in der Datenbank)
            // Das ist VIEL schneller als alle Daten zu laden und dann zu filtern!
            $encodedLicense = rawurlencode(trim($licensePlate));
            $result = $this->apiCall("/auftraege/by-license/$encodedLicense", 'GET', ['days' => $days]);
            
            if (isset($result['success']) && $result['success'] && !empty($result['data'])) {
                // Nimm ersten Treffer (neuesten, da Backend bereits nach DatumAuftrag DESC sortiert)
                return $result['data'][0] ?? null;
            }
            
            return null;
        } catch (Exception $e) {
            error_log("TireOffersAPI::findCustomerByLicensePlate Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Sucht nach VIN (Fahrgestellnummer) - OPTIMIERT für Performance
     */
    public function searchByVIN($vin, $days = 180) {
        try {
            if (empty($vin)) {
                return [];
            }
            
            // Normalisiere VIN (entferne Whitespace)
            $searchVin = $this->normalizeSearchTerm($vin);
            $searchLast6 = strlen($searchVin) >= 6 ? substr($searchVin, -6) : $searchVin;
            
            // OPTIMIERT: Reduzierter Zeitraum für schnellere Suche
            $actualDays = min($days, 180); // Reduziert von 2000 auf 180
            $params = ['days' => $actualDays];
            $result = $this->apiCall('/auftraege', 'GET', $params);
            
            if (isset($result['success']) && $result['success']) {
                $auftraege = $result['data'] ?? [];
                $matches = [];
                
                foreach ($auftraege as $auftrag) {
                    // Prüfe verschiedene VIN-Felder
                    $auftragVin = '';
                    if (isset($auftrag['VIN']) && !empty($auftrag['VIN'])) {
                        $auftragVin = $this->normalizeSearchTerm($auftrag['VIN']);
                    } elseif (isset($auftrag['vin']) && !empty($auftrag['vin'])) {
                        $auftragVin = $this->normalizeSearchTerm($auftrag['vin']);
                    } elseif (isset($auftrag['Fahrgestellnummer']) && !empty($auftrag['Fahrgestellnummer'])) {
                        $auftragVin = $this->normalizeSearchTerm($auftrag['Fahrgestellnummer']);
                    }
                    
                    if (!empty($auftragVin)) {
                        $auftragLast6 = strlen($auftragVin) >= 6 ? substr($auftragVin, -6) : $auftragVin;
                        
                        // Prüfe auf vollständige VIN oder letzte 6 Ziffern
                        if ($auftragVin === $searchVin || $auftragLast6 === $searchLast6) {
                            $matches[] = $auftrag;
                        }
                    }
                }
                
                return $matches;
            }
            
            return [];
        } catch (Exception $e) {
            error_log("TireOffersAPI::searchByVIN Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt alle Aufträge für ein Kennzeichen - OPTIMIERT
     * Nutzt den optimierten Backend-Endpoint /by-license für direkte Filterung in der Datenbank
     */
    public function getAllOrdersByLicensePlate($licensePlate, $days = 90) {
        try {
            // OPTIMIERT: Nutze den optimierten Backend-Endpoint der direkt in der DB filtert
            // Das ist VIEL schneller als alle Daten zu laden und dann zu filtern!
            $maxDays = 2000; // ~5.5 Jahre (2019-2024)
            $actualDays = min($days, $maxDays);
            
            // URL-encode das Kennzeichen für den Pfad
            $encodedLicense = rawurlencode($licensePlate);
            
            // OPTIMIERT: Nutze /by-license Endpoint (filtert direkt in der Datenbank)
            $params = ['days' => $actualDays];
            $result = $this->apiCall("/auftraege/by-license/$encodedLicense", 'GET', $params);
            
            if (isset($result['success']) && $result['success']) {
                $auftraege = $result['data'] ?? [];
                
                // Backend sortiert bereits, aber sicherstellen dass es korrekt ist
                usort($auftraege, function($a, $b) {
                    $dateA = $a['DatumAuftrag'] ?? $a['Datum'] ?? $a['Erstellt'] ?? '';
                    $dateB = $b['DatumAuftrag'] ?? $b['Datum'] ?? $b['Erstellt'] ?? '';
                    return strcmp($dateB, $dateA);
                });
                
                return $auftraege;
            }
            
            // FALLBACK: Falls Endpoint nicht verfügbar, nutze alte Methode
            error_log("Optimierter /by-license Endpoint nicht verfügbar, nutze Fallback");
            $params = ['days' => $actualDays];
            $result = $this->apiCall('/auftraege', 'GET', $params);
            
            if (isset($result['success']) && $result['success']) {
                $auftraege = $result['data'] ?? [];
                
                // Filtere nach Kennzeichen
                $filtered = [];
                $licensePlateUpper = strtoupper(trim($licensePlate));
                
                foreach ($auftraege as $auftrag) {
                    if (isset($auftrag['ZulassungsKz'])) {
                        $auftragLicense = strtoupper(trim($auftrag['ZulassungsKz']));
                        if ($auftragLicense === $licensePlateUpper) {
                            $filtered[] = $auftrag;
                        }
                    }
                }
                
                // Sortiere nach Datum (neueste zuerst)
                usort($filtered, function($a, $b) {
                    $dateA = $a['DatumAuftrag'] ?? $a['Datum'] ?? $a['Erstellt'] ?? '';
                    $dateB = $b['DatumAuftrag'] ?? $b['Datum'] ?? $b['Erstellt'] ?? '';
                    return strcmp($dateB, $dateA);
                });
                
                return $filtered;
            }
            
            return [];
        } catch (Exception $e) {
            error_log("TireOffersAPI::getAllOrdersByLicensePlate Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Holt detaillierte Termininformationen
     */
    public function getTerminDetails($auftragNr) {
        try {
            // OPTIMIERT: Reduziertes Timeout auf 10 Sekunden (schnelleres Fehlschlagen)
            $result = $this->apiCall("/termine/$auftragNr", 'GET', [], 10, 1);
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? null;
            }
            
            return null;
        } catch (Exception $e) {
            error_log("TireOffersAPI::getTerminDetails Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Holt alle Termine
     */
    public function getTermine($days = 30) {
        try {
            $params = ['days' => $days];
            $result = $this->apiCall('/termine', 'GET', $params);
            
            if (isset($result['success']) && $result['success']) {
                return $result['data'] ?? [];
            }
            
            return [];
        } catch (Exception $e) {
            error_log("TireOffersAPI::getTermine Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Prüft API-Verbindung mit detailliertem Debug-Output
     */
    public function checkHealth() {
        return $this->checkHealthWithDetails()['success'];
    }
    
    /**
     * Prüft API-Verbindung mit detaillierten Informationen
     */
    public function checkHealthWithDetails() {
        $result = [
            'success' => false,
            'url' => '',
            'http_code' => 0,
            'error' => '',
            'response' => '',
            'details' => []
        ];
        
        try {
            // Teste verschiedene URL-Varianten
            $urls = [
                $this->baseUrl . '/system/health',
                'http://100.82.207.72:5001/api/system/health',
                'http://100.82.207.72/api/system/health',
            ];
            
            $headers = [
                'Content-Type: application/json',
            ];
            
            if ($this->apiKeyRequired) {
                $headers[] = 'X-API-Key: ' . $this->apiKey;
            }
            
            $lastError = '';
            $lastHttpCode = 0;
            
            foreach ($urls as $url) {
                $result['url'] = $url;
                $result['details'][] = "Teste URL: $url";
                
                $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 3,
            CURLOPT_FRESH_CONNECT => false, // Connection-Reuse aktivieren
            CURLOPT_FORBID_REUSE => false, // Connection-Reuse aktivieren
        ]);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $error = curl_error($ch);
                $curlErrno = curl_errno($ch);
                curl_close($ch);
                
                $lastError = $error;
                $lastHttpCode = $httpCode;
                
                if ($error) {
                    $result['details'][] = "cURL Fehler: $error (Code: $curlErrno)";
                    continue; // Probiere nächste URL
                }
                
                if ($httpCode === 0) {
                    $result['details'][] = "Keine HTTP-Antwort erhalten. Möglicherweise ist der Server nicht erreichbar.";
                    continue;
                }
                
                $result['http_code'] = $httpCode;
                
                if ($httpCode >= 400) {
                    $result['details'][] = "HTTP Fehler: $httpCode";
                    $result['response'] = substr($response, 0, 500);
                    continue;
                }
                
                // Erfolgreiche Verbindung
                $data = json_decode($response, true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $result['details'][] = "JSON Parse Fehler: " . json_last_error_msg();
                    $result['response'] = substr($response, 0, 500);
                    continue;
                }
                
                if (isset($data['status']) && $data['status'] === 'ok') {
                    $result['success'] = true;
                    $result['response'] = $response;
                    $result['details'][] = "✅ API-Verbindung erfolgreich!";
                    return $result;
                } else {
                    $result['details'][] = "API antwortet, aber Status ist nicht 'ok': " . json_encode($data);
                    $result['response'] = $response;
                }
            }
            
            // Alle URLs fehlgeschlagen
            $result['error'] = $lastError ?: "HTTP $lastHttpCode";
            $result['details'][] = "❌ Alle Verbindungsversuche fehlgeschlagen";
            
        } catch (Exception $e) {
            $result['error'] = $e->getMessage();
            $result['details'][] = "Exception: " . $e->getMessage();
            error_log("TireOffersAPI::checkHealthWithDetails Exception: " . $e->getMessage());
        }
        
        return $result;
    }
}

